PonyCheck is a library for property based testing
with tight integration into PonyTest.

## Property Based Testing

In _traditional_ unit testing, the developer specifies one or more input
examples manually for the class or system under test and asserts on certain
output conditions. The difficulty here is to find enough examples to cover
all branches and cases of the class or system under test.

In property based testing, the developer defines a property, a kind of predicate
for the class or system under test that should hold for all kinds or just a
subset of possible input values. The property based testing engine then
generates a big number of random input values and checks if the property holds
for all of them. The developer only needs to specify the possible set of input
values using a Generator.

This testing technique is great for finding edge cases that would easily go
unnoticed with manually constructed test samples. In general it can lead to much
higher coverage than traditional unit testing, with much less code to write.

## How PonyCheck implements Property Based Testing

A property based test in PonyCheck consists of the following:

* A name (mostly for integration into PonyTest)
* One or more generators, depending on how your property is laid out.
  There are tons of them defined in the primitive
  [Generators](pony_check-Generators.md).
* A `property` method that asserts a certain property for each sample
  generated by the [Generator(s)](pony_check-Generator.md) with the help of
  [PropertyHelper](pony_check-PropertyHelper.md), which tries to expose a
  similar API   as [TestHelper](pony_test-TestHelper.md).
* Optionally, the method `params()` can be used to configure how PonyCheck
  executes the property by specifying a custom
  [PropertyParams](pony_check-PropertyParams.md) object.

The classical list-reverse example:

```pony
use "collections"
use "pony_check"

class ListReverseProperty is Property1[List[USize]]
  fun name(): String => "list/reverse"

  fun gen(): Generator[List[USize]] =>
    Generators.list_of[USize](Generators.usize())

  fun property(arg1: List[USize], ph: PropertyHelper) =>
    ph.array_eq[USize](arg1, arg1.reverse().reverse())
```

## Integration into PonyTest

There are two ways of integrating a [Property](pony_check-Property1.md) into
[PonyTest](pony_test--index.md):

1. In order to pass your Property to the PonyTest engine, you need to wrap it
  inside a [Property1UnitTest](pony_check-Property1UnitTest.md).

```pony
  actor Main is TestList
    new create(env: Env) => PonyTest(env, this)

    fun tag tests(test: PonyTest) =>
      test(Property1UnitTest[String](MyStringProperty))
```

2. Run as many [Properties](pony_check-Property1.md) as you wish inside one
  PonyTest [UnitTest](pony_test-UnitTest.md) using the convenience function
  [PonyCheck.for_all](pony_check-PonyCheck.md#for_all) providing a
  [Generator](pony_check-Generator), the [TestHelper](pony_test-TestHelper.md)
  and the actual property function. (Note that the property function is supplied
  in a second application of the result to `for_all`.)

```pony
  class ListReversePropertyWithinAUnitTest is UnitTest
    fun name(): String => "list/reverse/forall"

    fun apply(h: TestHelper) =>
      let gen = recover val Generators.list_of[USize](Generators.usize()) end
      PonyCheck.for_all[List[USize]](gen, h)(
        {(sample, ph) =>
          ph.array_eq[Usize](arg1, arg1.reverse().reverse())
        })
      // ... possibly more properties, using `PonyCheck.for_all`
```

Independently of how you integrate with [PonyTest](pony_test--index.md),
the PonyCheck machinery will instantiate the provided Generator, and will
execute it for a configurable number of samples.

If the property fails using an assertion method of
[PropertyHelper](pony_check-PropertyHelper.md),
the failed example will be shrunken by the generator
to obtain a smaller and more informative, still failing, sample
for reporting.



## Public Types

* [primitive ASCIIAll](pony_check-ASCIIAll.md)
* [primitive ASCIIAllWithNUL](pony_check-ASCIIAllWithNUL.md)
* [primitive ASCIIDigits](pony_check-ASCIIDigits.md)
* [primitive ASCIILetters](pony_check-ASCIILetters.md)
* [primitive ASCIILettersLower](pony_check-ASCIILettersLower.md)
* [primitive ASCIILettersUpper](pony_check-ASCIILettersUpper.md)
* [primitive ASCIINUL](pony_check-ASCIINUL.md)
* [primitive ASCIINonPrintable](pony_check-ASCIINonPrintable.md)
* [primitive ASCIIPrintable](pony_check-ASCIIPrintable.md)
* [primitive ASCIIPunctuation](pony_check-ASCIIPunctuation.md)
* [type ASCIIRange](pony_check-ASCIIRange.md)
* [primitive ASCIIWhiteSpace](pony_check-ASCIIWhiteSpace.md)
* [class CountdownIter](pony_check-CountdownIter.md)
* [class ForAll](pony_check-ForAll.md)
* [class ForAll2](pony_check-ForAll2.md)
* [class ForAll3](pony_check-ForAll3.md)
* [class ForAll4](pony_check-ForAll4.md)
* [trait GenObj](pony_check-GenObj.md)
* [type GenerateResult](pony_check-GenerateResult.md)
* [class Generator](pony_check-Generator.md)
* [primitive Generators](pony_check-Generators.md)
* [trait IntPairProperty](pony_check-IntPairProperty.md)
* [class IntPairPropertySample](pony_check-IntPairPropertySample.md)
* [type IntPairUnitTest](pony_check-IntPairUnitTest.md)
* [trait IntProperty](pony_check-IntProperty.md)
* [class IntPropertySample](pony_check-IntPropertySample.md)
* [type IntUnitTest](pony_check-IntUnitTest.md)
* [primitive PonyCheck](pony_check-PonyCheck.md)
* [class Poperator](pony_check-Poperator.md)
* [trait Property1](pony_check-Property1.md)
* [class Property1UnitTest](pony_check-Property1UnitTest.md)
* [trait Property2](pony_check-Property2.md)
* [class Property2UnitTest](pony_check-Property2UnitTest.md)
* [trait Property3](pony_check-Property3.md)
* [class Property3UnitTest](pony_check-Property3UnitTest.md)
* [trait Property4](pony_check-Property4.md)
* [class Property4UnitTest](pony_check-Property4UnitTest.md)
* [class PropertyHelper](pony_check-PropertyHelper.md)
* [interface PropertyLogger](pony_check-PropertyLogger.md)
* [class PropertyParams](pony_check-PropertyParams.md)
* [interface PropertyResultNotify](pony_check-PropertyResultNotify.md)
* [actor PropertyRunner](pony_check-PropertyRunner.md)
* [class Randomness](pony_check-Randomness.md)
* [type ValueAndShrink](pony_check-ValueAndShrink.md)
* [type WeightedGenerator](pony_check-WeightedGenerator.md)


## Private Types

* [class _EmptyIterator](pony_check-_EmptyIterator.md)
* [interface _IPropertyRunner](pony_check-_IPropertyRunner.md)
* [interface _PropertyRunNotify](pony_check-_PropertyRunNotify.md)
* [type _Round](pony_check-_Round.md)
* [class _Run](pony_check-_Run.md)
* [class _Shrink](pony_check-_Shrink.md)
* [primitive _Stringify](pony_check-_Stringify.md)
* [primitive _StringifyIntArg](pony_check-_StringifyIntArg.md)
