﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/bcm-pricing-calculator/BCMPricingCalculatorRequest.h>
#include <aws/bcm-pricing-calculator/BCMPricingCalculator_EXPORTS.h>
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace BCMPricingCalculator {
namespace Model {

/**
 */
class CreateBillEstimateRequest : public BCMPricingCalculatorRequest {
 public:
  AWS_BCMPRICINGCALCULATOR_API CreateBillEstimateRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateBillEstimate"; }

  AWS_BCMPRICINGCALCULATOR_API Aws::String SerializePayload() const override;

  AWS_BCMPRICINGCALCULATOR_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p> The ID of the Bill Scenario for which you want to create a Bill estimate.
   * </p>
   */
  inline const Aws::String& GetBillScenarioId() const { return m_billScenarioId; }
  inline bool BillScenarioIdHasBeenSet() const { return m_billScenarioIdHasBeenSet; }
  template <typename BillScenarioIdT = Aws::String>
  void SetBillScenarioId(BillScenarioIdT&& value) {
    m_billScenarioIdHasBeenSet = true;
    m_billScenarioId = std::forward<BillScenarioIdT>(value);
  }
  template <typename BillScenarioIdT = Aws::String>
  CreateBillEstimateRequest& WithBillScenarioId(BillScenarioIdT&& value) {
    SetBillScenarioId(std::forward<BillScenarioIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The name of the Bill estimate that will be created. Names must be unique for
   * an account. </p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateBillEstimateRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> A unique, case-sensitive identifier that you provide to ensure the
   * idempotency of the request. </p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreateBillEstimateRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> An optional list of tags to associate with the specified BillEstimate. You
   * can use resource tags to control access to your BillEstimate using IAM policies.
   * Each tag consists of a key and a value, and each key must be unique for the
   * resource. The following restrictions apply to resource tags: </p> <ul> <li>
   * <p>Although the maximum number of array members is 200, you can assign a maximum
   * of 50 user-tags to one resource. The remaining are reserved for Amazon Web
   * Services. </p> </li> <li> <p>The maximum length of a key is 128 characters.</p>
   * </li> <li> <p>The maximum length of a value is 256 characters.</p> </li> <li>
   * <p>Keys and values can only contain alphanumeric characters, spaces, and any of
   * the following: <code>_.:/=+@-</code>.</p> </li> <li> <p>Keys and values are case
   * sensitive.</p> </li> <li> <p>Keys and values are trimmed for any leading or
   * trailing whitespaces.</p> </li> <li> <p>Don't use <code>aws:</code> as a prefix
   * for your keys. This prefix is reserved for Amazon Web Services.</p> </li> </ul>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateBillEstimateRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateBillEstimateRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_billScenarioId;

  Aws::String m_name;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_billScenarioIdHasBeenSet = false;
  bool m_nameHasBeenSet = false;
  bool m_clientTokenHasBeenSet = true;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace BCMPricingCalculator
}  // namespace Aws
