﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/databrew/GlueDataBrew_EXPORTS.h>
#include <aws/databrew/model/Sample.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace GlueDataBrew {
namespace Model {

/**
 * <p>Represents all of the attributes of a DataBrew project.</p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/Project">AWS
 * API Reference</a></p>
 */
class Project {
 public:
  AWS_GLUEDATABREW_API Project() = default;
  AWS_GLUEDATABREW_API Project(Aws::Utils::Json::JsonView jsonValue);
  AWS_GLUEDATABREW_API Project& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_GLUEDATABREW_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The ID of the Amazon Web Services account that owns the project.</p>
   */
  inline const Aws::String& GetAccountId() const { return m_accountId; }
  inline bool AccountIdHasBeenSet() const { return m_accountIdHasBeenSet; }
  template <typename AccountIdT = Aws::String>
  void SetAccountId(AccountIdT&& value) {
    m_accountIdHasBeenSet = true;
    m_accountId = std::forward<AccountIdT>(value);
  }
  template <typename AccountIdT = Aws::String>
  Project& WithAccountId(AccountIdT&& value) {
    SetAccountId(std::forward<AccountIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time that the project was created.</p>
   */
  inline const Aws::Utils::DateTime& GetCreateDate() const { return m_createDate; }
  inline bool CreateDateHasBeenSet() const { return m_createDateHasBeenSet; }
  template <typename CreateDateT = Aws::Utils::DateTime>
  void SetCreateDate(CreateDateT&& value) {
    m_createDateHasBeenSet = true;
    m_createDate = std::forward<CreateDateT>(value);
  }
  template <typename CreateDateT = Aws::Utils::DateTime>
  Project& WithCreateDate(CreateDateT&& value) {
    SetCreateDate(std::forward<CreateDateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the user who crated the project.</p>
   */
  inline const Aws::String& GetCreatedBy() const { return m_createdBy; }
  inline bool CreatedByHasBeenSet() const { return m_createdByHasBeenSet; }
  template <typename CreatedByT = Aws::String>
  void SetCreatedBy(CreatedByT&& value) {
    m_createdByHasBeenSet = true;
    m_createdBy = std::forward<CreatedByT>(value);
  }
  template <typename CreatedByT = Aws::String>
  Project& WithCreatedBy(CreatedByT&& value) {
    SetCreatedBy(std::forward<CreatedByT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The dataset that the project is to act upon.</p>
   */
  inline const Aws::String& GetDatasetName() const { return m_datasetName; }
  inline bool DatasetNameHasBeenSet() const { return m_datasetNameHasBeenSet; }
  template <typename DatasetNameT = Aws::String>
  void SetDatasetName(DatasetNameT&& value) {
    m_datasetNameHasBeenSet = true;
    m_datasetName = std::forward<DatasetNameT>(value);
  }
  template <typename DatasetNameT = Aws::String>
  Project& WithDatasetName(DatasetNameT&& value) {
    SetDatasetName(std::forward<DatasetNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The last modification date and time for the project.</p>
   */
  inline const Aws::Utils::DateTime& GetLastModifiedDate() const { return m_lastModifiedDate; }
  inline bool LastModifiedDateHasBeenSet() const { return m_lastModifiedDateHasBeenSet; }
  template <typename LastModifiedDateT = Aws::Utils::DateTime>
  void SetLastModifiedDate(LastModifiedDateT&& value) {
    m_lastModifiedDateHasBeenSet = true;
    m_lastModifiedDate = std::forward<LastModifiedDateT>(value);
  }
  template <typename LastModifiedDateT = Aws::Utils::DateTime>
  Project& WithLastModifiedDate(LastModifiedDateT&& value) {
    SetLastModifiedDate(std::forward<LastModifiedDateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the user who last modified the project.</p>
   */
  inline const Aws::String& GetLastModifiedBy() const { return m_lastModifiedBy; }
  inline bool LastModifiedByHasBeenSet() const { return m_lastModifiedByHasBeenSet; }
  template <typename LastModifiedByT = Aws::String>
  void SetLastModifiedBy(LastModifiedByT&& value) {
    m_lastModifiedByHasBeenSet = true;
    m_lastModifiedBy = std::forward<LastModifiedByT>(value);
  }
  template <typename LastModifiedByT = Aws::String>
  Project& WithLastModifiedBy(LastModifiedByT&& value) {
    SetLastModifiedBy(std::forward<LastModifiedByT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unique name of a project.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  Project& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of a recipe that will be developed during a project session.</p>
   */
  inline const Aws::String& GetRecipeName() const { return m_recipeName; }
  inline bool RecipeNameHasBeenSet() const { return m_recipeNameHasBeenSet; }
  template <typename RecipeNameT = Aws::String>
  void SetRecipeName(RecipeNameT&& value) {
    m_recipeNameHasBeenSet = true;
    m_recipeName = std::forward<RecipeNameT>(value);
  }
  template <typename RecipeNameT = Aws::String>
  Project& WithRecipeName(RecipeNameT&& value) {
    SetRecipeName(std::forward<RecipeNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) for the project.</p>
   */
  inline const Aws::String& GetResourceArn() const { return m_resourceArn; }
  inline bool ResourceArnHasBeenSet() const { return m_resourceArnHasBeenSet; }
  template <typename ResourceArnT = Aws::String>
  void SetResourceArn(ResourceArnT&& value) {
    m_resourceArnHasBeenSet = true;
    m_resourceArn = std::forward<ResourceArnT>(value);
  }
  template <typename ResourceArnT = Aws::String>
  Project& WithResourceArn(ResourceArnT&& value) {
    SetResourceArn(std::forward<ResourceArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The sample size and sampling type to apply to the data. If this parameter
   * isn't specified, then the sample consists of the first 500 rows from the
   * dataset.</p>
   */
  inline const Sample& GetSample() const { return m_sample; }
  inline bool SampleHasBeenSet() const { return m_sampleHasBeenSet; }
  template <typename SampleT = Sample>
  void SetSample(SampleT&& value) {
    m_sampleHasBeenSet = true;
    m_sample = std::forward<SampleT>(value);
  }
  template <typename SampleT = Sample>
  Project& WithSample(SampleT&& value) {
    SetSample(std::forward<SampleT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Metadata tags that have been applied to the project.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  Project& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  Project& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the role that will be assumed for this
   * project.</p>
   */
  inline const Aws::String& GetRoleArn() const { return m_roleArn; }
  inline bool RoleArnHasBeenSet() const { return m_roleArnHasBeenSet; }
  template <typename RoleArnT = Aws::String>
  void SetRoleArn(RoleArnT&& value) {
    m_roleArnHasBeenSet = true;
    m_roleArn = std::forward<RoleArnT>(value);
  }
  template <typename RoleArnT = Aws::String>
  Project& WithRoleArn(RoleArnT&& value) {
    SetRoleArn(std::forward<RoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the user that opened the project for
   * use.</p>
   */
  inline const Aws::String& GetOpenedBy() const { return m_openedBy; }
  inline bool OpenedByHasBeenSet() const { return m_openedByHasBeenSet; }
  template <typename OpenedByT = Aws::String>
  void SetOpenedBy(OpenedByT&& value) {
    m_openedByHasBeenSet = true;
    m_openedBy = std::forward<OpenedByT>(value);
  }
  template <typename OpenedByT = Aws::String>
  Project& WithOpenedBy(OpenedByT&& value) {
    SetOpenedBy(std::forward<OpenedByT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time when the project was opened.</p>
   */
  inline const Aws::Utils::DateTime& GetOpenDate() const { return m_openDate; }
  inline bool OpenDateHasBeenSet() const { return m_openDateHasBeenSet; }
  template <typename OpenDateT = Aws::Utils::DateTime>
  void SetOpenDate(OpenDateT&& value) {
    m_openDateHasBeenSet = true;
    m_openDate = std::forward<OpenDateT>(value);
  }
  template <typename OpenDateT = Aws::Utils::DateTime>
  Project& WithOpenDate(OpenDateT&& value) {
    SetOpenDate(std::forward<OpenDateT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_accountId;

  Aws::Utils::DateTime m_createDate{};

  Aws::String m_createdBy;

  Aws::String m_datasetName;

  Aws::Utils::DateTime m_lastModifiedDate{};

  Aws::String m_lastModifiedBy;

  Aws::String m_name;

  Aws::String m_recipeName;

  Aws::String m_resourceArn;

  Sample m_sample;

  Aws::Map<Aws::String, Aws::String> m_tags;

  Aws::String m_roleArn;

  Aws::String m_openedBy;

  Aws::Utils::DateTime m_openDate{};
  bool m_accountIdHasBeenSet = false;
  bool m_createDateHasBeenSet = false;
  bool m_createdByHasBeenSet = false;
  bool m_datasetNameHasBeenSet = false;
  bool m_lastModifiedDateHasBeenSet = false;
  bool m_lastModifiedByHasBeenSet = false;
  bool m_nameHasBeenSet = false;
  bool m_recipeNameHasBeenSet = false;
  bool m_resourceArnHasBeenSet = false;
  bool m_sampleHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
  bool m_roleArnHasBeenSet = false;
  bool m_openedByHasBeenSet = false;
  bool m_openDateHasBeenSet = false;
};

}  // namespace Model
}  // namespace GlueDataBrew
}  // namespace Aws
