﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/iotanalytics/IoTAnalytics_EXPORTS.h>
#include <aws/iotanalytics/model/ChannelStatus.h>
#include <aws/iotanalytics/model/ChannelStorage.h>
#include <aws/iotanalytics/model/RetentionPeriod.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace IoTAnalytics {
namespace Model {

/**
 * <p>A collection of data from an MQTT topic. Channels archive the raw,
 * unprocessed messages before publishing the data to a pipeline.</p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/iotanalytics-2017-11-27/Channel">AWS
 * API Reference</a></p>
 */
class Channel {
 public:
  AWS_IOTANALYTICS_API Channel() = default;
  AWS_IOTANALYTICS_API Channel(Aws::Utils::Json::JsonView jsonValue);
  AWS_IOTANALYTICS_API Channel& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_IOTANALYTICS_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The name of the channel.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  Channel& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Where channel data is stored. You can choose one of
   * <code>serviceManagedS3</code> or <code>customerManagedS3</code> storage. If not
   * specified, the default is <code>serviceManagedS3</code>. You can't change this
   * storage option after the channel is created.</p>
   */
  inline const ChannelStorage& GetStorage() const { return m_storage; }
  inline bool StorageHasBeenSet() const { return m_storageHasBeenSet; }
  template <typename StorageT = ChannelStorage>
  void SetStorage(StorageT&& value) {
    m_storageHasBeenSet = true;
    m_storage = std::forward<StorageT>(value);
  }
  template <typename StorageT = ChannelStorage>
  Channel& WithStorage(StorageT&& value) {
    SetStorage(std::forward<StorageT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the channel.</p>
   */
  inline const Aws::String& GetArn() const { return m_arn; }
  inline bool ArnHasBeenSet() const { return m_arnHasBeenSet; }
  template <typename ArnT = Aws::String>
  void SetArn(ArnT&& value) {
    m_arnHasBeenSet = true;
    m_arn = std::forward<ArnT>(value);
  }
  template <typename ArnT = Aws::String>
  Channel& WithArn(ArnT&& value) {
    SetArn(std::forward<ArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The status of the channel.</p>
   */
  inline ChannelStatus GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  inline void SetStatus(ChannelStatus value) {
    m_statusHasBeenSet = true;
    m_status = value;
  }
  inline Channel& WithStatus(ChannelStatus value) {
    SetStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>How long, in days, message data is kept for the channel.</p>
   */
  inline const RetentionPeriod& GetRetentionPeriod() const { return m_retentionPeriod; }
  inline bool RetentionPeriodHasBeenSet() const { return m_retentionPeriodHasBeenSet; }
  template <typename RetentionPeriodT = RetentionPeriod>
  void SetRetentionPeriod(RetentionPeriodT&& value) {
    m_retentionPeriodHasBeenSet = true;
    m_retentionPeriod = std::forward<RetentionPeriodT>(value);
  }
  template <typename RetentionPeriodT = RetentionPeriod>
  Channel& WithRetentionPeriod(RetentionPeriodT&& value) {
    SetRetentionPeriod(std::forward<RetentionPeriodT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>When the channel was created.</p>
   */
  inline const Aws::Utils::DateTime& GetCreationTime() const { return m_creationTime; }
  inline bool CreationTimeHasBeenSet() const { return m_creationTimeHasBeenSet; }
  template <typename CreationTimeT = Aws::Utils::DateTime>
  void SetCreationTime(CreationTimeT&& value) {
    m_creationTimeHasBeenSet = true;
    m_creationTime = std::forward<CreationTimeT>(value);
  }
  template <typename CreationTimeT = Aws::Utils::DateTime>
  Channel& WithCreationTime(CreationTimeT&& value) {
    SetCreationTime(std::forward<CreationTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>When the channel was last updated.</p>
   */
  inline const Aws::Utils::DateTime& GetLastUpdateTime() const { return m_lastUpdateTime; }
  inline bool LastUpdateTimeHasBeenSet() const { return m_lastUpdateTimeHasBeenSet; }
  template <typename LastUpdateTimeT = Aws::Utils::DateTime>
  void SetLastUpdateTime(LastUpdateTimeT&& value) {
    m_lastUpdateTimeHasBeenSet = true;
    m_lastUpdateTime = std::forward<LastUpdateTimeT>(value);
  }
  template <typename LastUpdateTimeT = Aws::Utils::DateTime>
  Channel& WithLastUpdateTime(LastUpdateTimeT&& value) {
    SetLastUpdateTime(std::forward<LastUpdateTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The last time when a new message arrived in the channel.</p> <p>IoT Analytics
   * updates this value at most once per minute for one channel. Hence, the
   * <code>lastMessageArrivalTime</code> value is an approximation.</p> <p>This
   * feature only applies to messages that arrived in the data store after October
   * 23, 2020. </p>
   */
  inline const Aws::Utils::DateTime& GetLastMessageArrivalTime() const { return m_lastMessageArrivalTime; }
  inline bool LastMessageArrivalTimeHasBeenSet() const { return m_lastMessageArrivalTimeHasBeenSet; }
  template <typename LastMessageArrivalTimeT = Aws::Utils::DateTime>
  void SetLastMessageArrivalTime(LastMessageArrivalTimeT&& value) {
    m_lastMessageArrivalTimeHasBeenSet = true;
    m_lastMessageArrivalTime = std::forward<LastMessageArrivalTimeT>(value);
  }
  template <typename LastMessageArrivalTimeT = Aws::Utils::DateTime>
  Channel& WithLastMessageArrivalTime(LastMessageArrivalTimeT&& value) {
    SetLastMessageArrivalTime(std::forward<LastMessageArrivalTimeT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;

  ChannelStorage m_storage;

  Aws::String m_arn;

  ChannelStatus m_status{ChannelStatus::NOT_SET};

  RetentionPeriod m_retentionPeriod;

  Aws::Utils::DateTime m_creationTime{};

  Aws::Utils::DateTime m_lastUpdateTime{};

  Aws::Utils::DateTime m_lastMessageArrivalTime{};
  bool m_nameHasBeenSet = false;
  bool m_storageHasBeenSet = false;
  bool m_arnHasBeenSet = false;
  bool m_statusHasBeenSet = false;
  bool m_retentionPeriodHasBeenSet = false;
  bool m_creationTimeHasBeenSet = false;
  bool m_lastUpdateTimeHasBeenSet = false;
  bool m_lastMessageArrivalTimeHasBeenSet = false;
};

}  // namespace Model
}  // namespace IoTAnalytics
}  // namespace Aws
