﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/pi/PIRequest.h>
#include <aws/pi/PI_EXPORTS.h>
#include <aws/pi/model/ServiceType.h>

#include <utility>

namespace Aws {
namespace PI {
namespace Model {

/**
 */
class ListTagsForResourceRequest : public PIRequest {
 public:
  AWS_PI_API ListTagsForResourceRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListTagsForResource"; }

  AWS_PI_API Aws::String SerializePayload() const override;

  AWS_PI_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>List the tags for the Amazon Web Services service for which Performance
   * Insights returns metrics. Valid value is <code>RDS</code>.</p>
   */
  inline ServiceType GetServiceType() const { return m_serviceType; }
  inline bool ServiceTypeHasBeenSet() const { return m_serviceTypeHasBeenSet; }
  inline void SetServiceType(ServiceType value) {
    m_serviceTypeHasBeenSet = true;
    m_serviceType = value;
  }
  inline ListTagsForResourceRequest& WithServiceType(ServiceType value) {
    SetServiceType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Lists all the tags for the Amazon RDS Performance Insights resource. This
   * value is an Amazon Resource Name (ARN). For information about creating an ARN,
   * see <a
   * href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Tagging.ARN.html#USER_Tagging.ARN.Constructing">
   * Constructing an RDS Amazon Resource Name (ARN)</a>.</p>
   */
  inline const Aws::String& GetResourceARN() const { return m_resourceARN; }
  inline bool ResourceARNHasBeenSet() const { return m_resourceARNHasBeenSet; }
  template <typename ResourceARNT = Aws::String>
  void SetResourceARN(ResourceARNT&& value) {
    m_resourceARNHasBeenSet = true;
    m_resourceARN = std::forward<ResourceARNT>(value);
  }
  template <typename ResourceARNT = Aws::String>
  ListTagsForResourceRequest& WithResourceARN(ResourceARNT&& value) {
    SetResourceARN(std::forward<ResourceARNT>(value));
    return *this;
  }
  ///@}
 private:
  ServiceType m_serviceType{ServiceType::NOT_SET};

  Aws::String m_resourceARN;
  bool m_serviceTypeHasBeenSet = false;
  bool m_resourceARNHasBeenSet = false;
};

}  // namespace Model
}  // namespace PI
}  // namespace Aws
