﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/redshift-data/RedshiftDataAPIServiceRequest.h>
#include <aws/redshift-data/RedshiftDataAPIService_EXPORTS.h>
#include <aws/redshift-data/model/ResultFormatString.h>

#include <utility>

namespace Aws {
namespace RedshiftDataAPIService {
namespace Model {

/**
 */
class BatchExecuteStatementRequest : public RedshiftDataAPIServiceRequest {
 public:
  AWS_REDSHIFTDATAAPISERVICE_API BatchExecuteStatementRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "BatchExecuteStatement"; }

  AWS_REDSHIFTDATAAPISERVICE_API Aws::String SerializePayload() const override;

  AWS_REDSHIFTDATAAPISERVICE_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>One or more SQL statements to run. The SQL statements are run as a single
   * transaction. They run serially in the order of the array. Subsequent SQL
   * statements don't start until the previous statement in the array completes. If
   * any SQL statement fails, then because they are run as one transaction, all work
   * is rolled back.</p>
   */
  inline const Aws::Vector<Aws::String>& GetSqls() const { return m_sqls; }
  inline bool SqlsHasBeenSet() const { return m_sqlsHasBeenSet; }
  template <typename SqlsT = Aws::Vector<Aws::String>>
  void SetSqls(SqlsT&& value) {
    m_sqlsHasBeenSet = true;
    m_sqls = std::forward<SqlsT>(value);
  }
  template <typename SqlsT = Aws::Vector<Aws::String>>
  BatchExecuteStatementRequest& WithSqls(SqlsT&& value) {
    SetSqls(std::forward<SqlsT>(value));
    return *this;
  }
  template <typename SqlsT = Aws::String>
  BatchExecuteStatementRequest& AddSqls(SqlsT&& value) {
    m_sqlsHasBeenSet = true;
    m_sqls.emplace_back(std::forward<SqlsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The cluster identifier. This parameter is required when connecting to a
   * cluster and authenticating using either Secrets Manager or temporary
   * credentials. </p>
   */
  inline const Aws::String& GetClusterIdentifier() const { return m_clusterIdentifier; }
  inline bool ClusterIdentifierHasBeenSet() const { return m_clusterIdentifierHasBeenSet; }
  template <typename ClusterIdentifierT = Aws::String>
  void SetClusterIdentifier(ClusterIdentifierT&& value) {
    m_clusterIdentifierHasBeenSet = true;
    m_clusterIdentifier = std::forward<ClusterIdentifierT>(value);
  }
  template <typename ClusterIdentifierT = Aws::String>
  BatchExecuteStatementRequest& WithClusterIdentifier(ClusterIdentifierT&& value) {
    SetClusterIdentifier(std::forward<ClusterIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name or ARN of the secret that enables access to the database. This
   * parameter is required when authenticating using Secrets Manager. </p>
   */
  inline const Aws::String& GetSecretArn() const { return m_secretArn; }
  inline bool SecretArnHasBeenSet() const { return m_secretArnHasBeenSet; }
  template <typename SecretArnT = Aws::String>
  void SetSecretArn(SecretArnT&& value) {
    m_secretArnHasBeenSet = true;
    m_secretArn = std::forward<SecretArnT>(value);
  }
  template <typename SecretArnT = Aws::String>
  BatchExecuteStatementRequest& WithSecretArn(SecretArnT&& value) {
    SetSecretArn(std::forward<SecretArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The database user name. This parameter is required when connecting to a
   * cluster as a database user and authenticating using temporary credentials. </p>
   */
  inline const Aws::String& GetDbUser() const { return m_dbUser; }
  inline bool DbUserHasBeenSet() const { return m_dbUserHasBeenSet; }
  template <typename DbUserT = Aws::String>
  void SetDbUser(DbUserT&& value) {
    m_dbUserHasBeenSet = true;
    m_dbUser = std::forward<DbUserT>(value);
  }
  template <typename DbUserT = Aws::String>
  BatchExecuteStatementRequest& WithDbUser(DbUserT&& value) {
    SetDbUser(std::forward<DbUserT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the database. This parameter is required when authenticating
   * using either Secrets Manager or temporary credentials. </p>
   */
  inline const Aws::String& GetDatabase() const { return m_database; }
  inline bool DatabaseHasBeenSet() const { return m_databaseHasBeenSet; }
  template <typename DatabaseT = Aws::String>
  void SetDatabase(DatabaseT&& value) {
    m_databaseHasBeenSet = true;
    m_database = std::forward<DatabaseT>(value);
  }
  template <typename DatabaseT = Aws::String>
  BatchExecuteStatementRequest& WithDatabase(DatabaseT&& value) {
    SetDatabase(std::forward<DatabaseT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A value that indicates whether to send an event to the Amazon EventBridge
   * event bus after the SQL statements run. </p>
   */
  inline bool GetWithEvent() const { return m_withEvent; }
  inline bool WithEventHasBeenSet() const { return m_withEventHasBeenSet; }
  inline void SetWithEvent(bool value) {
    m_withEventHasBeenSet = true;
    m_withEvent = value;
  }
  inline BatchExecuteStatementRequest& WithWithEvent(bool value) {
    SetWithEvent(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the SQL statements. You can name the SQL statements when you
   * create them to identify the query. </p>
   */
  inline const Aws::String& GetStatementName() const { return m_statementName; }
  inline bool StatementNameHasBeenSet() const { return m_statementNameHasBeenSet; }
  template <typename StatementNameT = Aws::String>
  void SetStatementName(StatementNameT&& value) {
    m_statementNameHasBeenSet = true;
    m_statementName = std::forward<StatementNameT>(value);
  }
  template <typename StatementNameT = Aws::String>
  BatchExecuteStatementRequest& WithStatementName(StatementNameT&& value) {
    SetStatementName(std::forward<StatementNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The serverless workgroup name or Amazon Resource Name (ARN). This parameter
   * is required when connecting to a serverless workgroup and authenticating using
   * either Secrets Manager or temporary credentials.</p>
   */
  inline const Aws::String& GetWorkgroupName() const { return m_workgroupName; }
  inline bool WorkgroupNameHasBeenSet() const { return m_workgroupNameHasBeenSet; }
  template <typename WorkgroupNameT = Aws::String>
  void SetWorkgroupName(WorkgroupNameT&& value) {
    m_workgroupNameHasBeenSet = true;
    m_workgroupName = std::forward<WorkgroupNameT>(value);
  }
  template <typename WorkgroupNameT = Aws::String>
  BatchExecuteStatementRequest& WithWorkgroupName(WorkgroupNameT&& value) {
    SetWorkgroupName(std::forward<WorkgroupNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique, case-sensitive identifier that you provide to ensure the
   * idempotency of the request.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  BatchExecuteStatementRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The data format of the result of the SQL statement. If no format is
   * specified, the default is JSON.</p>
   */
  inline ResultFormatString GetResultFormat() const { return m_resultFormat; }
  inline bool ResultFormatHasBeenSet() const { return m_resultFormatHasBeenSet; }
  inline void SetResultFormat(ResultFormatString value) {
    m_resultFormatHasBeenSet = true;
    m_resultFormat = value;
  }
  inline BatchExecuteStatementRequest& WithResultFormat(ResultFormatString value) {
    SetResultFormat(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number of seconds to keep the session alive after the query finishes. The
   * maximum time a session can keep alive is 24 hours. After 24 hours, the session
   * is forced closed and the query is terminated.</p>
   */
  inline int GetSessionKeepAliveSeconds() const { return m_sessionKeepAliveSeconds; }
  inline bool SessionKeepAliveSecondsHasBeenSet() const { return m_sessionKeepAliveSecondsHasBeenSet; }
  inline void SetSessionKeepAliveSeconds(int value) {
    m_sessionKeepAliveSecondsHasBeenSet = true;
    m_sessionKeepAliveSeconds = value;
  }
  inline BatchExecuteStatementRequest& WithSessionKeepAliveSeconds(int value) {
    SetSessionKeepAliveSeconds(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The session identifier of the query.</p>
   */
  inline const Aws::String& GetSessionId() const { return m_sessionId; }
  inline bool SessionIdHasBeenSet() const { return m_sessionIdHasBeenSet; }
  template <typename SessionIdT = Aws::String>
  void SetSessionId(SessionIdT&& value) {
    m_sessionIdHasBeenSet = true;
    m_sessionId = std::forward<SessionIdT>(value);
  }
  template <typename SessionIdT = Aws::String>
  BatchExecuteStatementRequest& WithSessionId(SessionIdT&& value) {
    SetSessionId(std::forward<SessionIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::Vector<Aws::String> m_sqls;

  Aws::String m_clusterIdentifier;

  Aws::String m_secretArn;

  Aws::String m_dbUser;

  Aws::String m_database;

  bool m_withEvent{false};

  Aws::String m_statementName;

  Aws::String m_workgroupName;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};

  ResultFormatString m_resultFormat{ResultFormatString::NOT_SET};

  int m_sessionKeepAliveSeconds{0};

  Aws::String m_sessionId;
  bool m_sqlsHasBeenSet = false;
  bool m_clusterIdentifierHasBeenSet = false;
  bool m_secretArnHasBeenSet = false;
  bool m_dbUserHasBeenSet = false;
  bool m_databaseHasBeenSet = false;
  bool m_withEventHasBeenSet = false;
  bool m_statementNameHasBeenSet = false;
  bool m_workgroupNameHasBeenSet = false;
  bool m_clientTokenHasBeenSet = true;
  bool m_resultFormatHasBeenSet = false;
  bool m_sessionKeepAliveSecondsHasBeenSet = false;
  bool m_sessionIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace RedshiftDataAPIService
}  // namespace Aws
