﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/firehose/Firehose_EXPORTS.h>
#include <aws/firehose/model/BufferingHints.h>
#include <aws/firehose/model/CloudWatchLoggingOptions.h>
#include <aws/firehose/model/CompressionFormat.h>
#include <aws/firehose/model/DataFormatConversionConfiguration.h>
#include <aws/firehose/model/DynamicPartitioningConfiguration.h>
#include <aws/firehose/model/EncryptionConfiguration.h>
#include <aws/firehose/model/ProcessingConfiguration.h>
#include <aws/firehose/model/S3BackupMode.h>
#include <aws/firehose/model/S3DestinationUpdate.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace Firehose {
namespace Model {

/**
 * <p>Describes an update for a destination in Amazon S3.</p><p><h3>See Also:</h3>
 * <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/firehose-2015-08-04/ExtendedS3DestinationUpdate">AWS
 * API Reference</a></p>
 */
class ExtendedS3DestinationUpdate {
 public:
  AWS_FIREHOSE_API ExtendedS3DestinationUpdate() = default;
  AWS_FIREHOSE_API ExtendedS3DestinationUpdate(Aws::Utils::Json::JsonView jsonValue);
  AWS_FIREHOSE_API ExtendedS3DestinationUpdate& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_FIREHOSE_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the Amazon Web Services credentials. For
   * more information, see <a
   * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon
   * Resource Names (ARNs) and Amazon Web Services Service Namespaces</a>.</p>
   */
  inline const Aws::String& GetRoleARN() const { return m_roleARN; }
  inline bool RoleARNHasBeenSet() const { return m_roleARNHasBeenSet; }
  template <typename RoleARNT = Aws::String>
  void SetRoleARN(RoleARNT&& value) {
    m_roleARNHasBeenSet = true;
    m_roleARN = std::forward<RoleARNT>(value);
  }
  template <typename RoleARNT = Aws::String>
  ExtendedS3DestinationUpdate& WithRoleARN(RoleARNT&& value) {
    SetRoleARN(std::forward<RoleARNT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the S3 bucket. For more information, see <a
   * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon
   * Resource Names (ARNs) and Amazon Web Services Service Namespaces</a>.</p>
   */
  inline const Aws::String& GetBucketARN() const { return m_bucketARN; }
  inline bool BucketARNHasBeenSet() const { return m_bucketARNHasBeenSet; }
  template <typename BucketARNT = Aws::String>
  void SetBucketARN(BucketARNT&& value) {
    m_bucketARNHasBeenSet = true;
    m_bucketARN = std::forward<BucketARNT>(value);
  }
  template <typename BucketARNT = Aws::String>
  ExtendedS3DestinationUpdate& WithBucketARN(BucketARNT&& value) {
    SetBucketARN(std::forward<BucketARNT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The "YYYY/MM/DD/HH" time format prefix is automatically used for delivered
   * Amazon S3 files. You can also specify a custom prefix, as described in <a
   * href="https://docs.aws.amazon.com/firehose/latest/dev/s3-prefixes.html">Custom
   * Prefixes for Amazon S3 Objects</a>.</p>
   */
  inline const Aws::String& GetPrefix() const { return m_prefix; }
  inline bool PrefixHasBeenSet() const { return m_prefixHasBeenSet; }
  template <typename PrefixT = Aws::String>
  void SetPrefix(PrefixT&& value) {
    m_prefixHasBeenSet = true;
    m_prefix = std::forward<PrefixT>(value);
  }
  template <typename PrefixT = Aws::String>
  ExtendedS3DestinationUpdate& WithPrefix(PrefixT&& value) {
    SetPrefix(std::forward<PrefixT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A prefix that Firehose evaluates and adds to failed records before writing
   * them to S3. This prefix appears immediately following the bucket name. For
   * information about how to specify this prefix, see <a
   * href="https://docs.aws.amazon.com/firehose/latest/dev/s3-prefixes.html">Custom
   * Prefixes for Amazon S3 Objects</a>.</p>
   */
  inline const Aws::String& GetErrorOutputPrefix() const { return m_errorOutputPrefix; }
  inline bool ErrorOutputPrefixHasBeenSet() const { return m_errorOutputPrefixHasBeenSet; }
  template <typename ErrorOutputPrefixT = Aws::String>
  void SetErrorOutputPrefix(ErrorOutputPrefixT&& value) {
    m_errorOutputPrefixHasBeenSet = true;
    m_errorOutputPrefix = std::forward<ErrorOutputPrefixT>(value);
  }
  template <typename ErrorOutputPrefixT = Aws::String>
  ExtendedS3DestinationUpdate& WithErrorOutputPrefix(ErrorOutputPrefixT&& value) {
    SetErrorOutputPrefix(std::forward<ErrorOutputPrefixT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The buffering option.</p>
   */
  inline const BufferingHints& GetBufferingHints() const { return m_bufferingHints; }
  inline bool BufferingHintsHasBeenSet() const { return m_bufferingHintsHasBeenSet; }
  template <typename BufferingHintsT = BufferingHints>
  void SetBufferingHints(BufferingHintsT&& value) {
    m_bufferingHintsHasBeenSet = true;
    m_bufferingHints = std::forward<BufferingHintsT>(value);
  }
  template <typename BufferingHintsT = BufferingHints>
  ExtendedS3DestinationUpdate& WithBufferingHints(BufferingHintsT&& value) {
    SetBufferingHints(std::forward<BufferingHintsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The compression format. If no value is specified, the default is
   * <code>UNCOMPRESSED</code>. </p>
   */
  inline CompressionFormat GetCompressionFormat() const { return m_compressionFormat; }
  inline bool CompressionFormatHasBeenSet() const { return m_compressionFormatHasBeenSet; }
  inline void SetCompressionFormat(CompressionFormat value) {
    m_compressionFormatHasBeenSet = true;
    m_compressionFormat = value;
  }
  inline ExtendedS3DestinationUpdate& WithCompressionFormat(CompressionFormat value) {
    SetCompressionFormat(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The encryption configuration. If no value is specified, the default is no
   * encryption.</p>
   */
  inline const EncryptionConfiguration& GetEncryptionConfiguration() const { return m_encryptionConfiguration; }
  inline bool EncryptionConfigurationHasBeenSet() const { return m_encryptionConfigurationHasBeenSet; }
  template <typename EncryptionConfigurationT = EncryptionConfiguration>
  void SetEncryptionConfiguration(EncryptionConfigurationT&& value) {
    m_encryptionConfigurationHasBeenSet = true;
    m_encryptionConfiguration = std::forward<EncryptionConfigurationT>(value);
  }
  template <typename EncryptionConfigurationT = EncryptionConfiguration>
  ExtendedS3DestinationUpdate& WithEncryptionConfiguration(EncryptionConfigurationT&& value) {
    SetEncryptionConfiguration(std::forward<EncryptionConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon CloudWatch logging options for your Firehose stream.</p>
   */
  inline const CloudWatchLoggingOptions& GetCloudWatchLoggingOptions() const { return m_cloudWatchLoggingOptions; }
  inline bool CloudWatchLoggingOptionsHasBeenSet() const { return m_cloudWatchLoggingOptionsHasBeenSet; }
  template <typename CloudWatchLoggingOptionsT = CloudWatchLoggingOptions>
  void SetCloudWatchLoggingOptions(CloudWatchLoggingOptionsT&& value) {
    m_cloudWatchLoggingOptionsHasBeenSet = true;
    m_cloudWatchLoggingOptions = std::forward<CloudWatchLoggingOptionsT>(value);
  }
  template <typename CloudWatchLoggingOptionsT = CloudWatchLoggingOptions>
  ExtendedS3DestinationUpdate& WithCloudWatchLoggingOptions(CloudWatchLoggingOptionsT&& value) {
    SetCloudWatchLoggingOptions(std::forward<CloudWatchLoggingOptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The data processing configuration.</p>
   */
  inline const ProcessingConfiguration& GetProcessingConfiguration() const { return m_processingConfiguration; }
  inline bool ProcessingConfigurationHasBeenSet() const { return m_processingConfigurationHasBeenSet; }
  template <typename ProcessingConfigurationT = ProcessingConfiguration>
  void SetProcessingConfiguration(ProcessingConfigurationT&& value) {
    m_processingConfigurationHasBeenSet = true;
    m_processingConfiguration = std::forward<ProcessingConfigurationT>(value);
  }
  template <typename ProcessingConfigurationT = ProcessingConfiguration>
  ExtendedS3DestinationUpdate& WithProcessingConfiguration(ProcessingConfigurationT&& value) {
    SetProcessingConfiguration(std::forward<ProcessingConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>You can update a Firehose stream to enable Amazon S3 backup if it is
   * disabled. If backup is enabled, you can't update the Firehose stream to disable
   * it. </p>
   */
  inline S3BackupMode GetS3BackupMode() const { return m_s3BackupMode; }
  inline bool S3BackupModeHasBeenSet() const { return m_s3BackupModeHasBeenSet; }
  inline void SetS3BackupMode(S3BackupMode value) {
    m_s3BackupModeHasBeenSet = true;
    m_s3BackupMode = value;
  }
  inline ExtendedS3DestinationUpdate& WithS3BackupMode(S3BackupMode value) {
    SetS3BackupMode(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon S3 destination for backup.</p>
   */
  inline const S3DestinationUpdate& GetS3BackupUpdate() const { return m_s3BackupUpdate; }
  inline bool S3BackupUpdateHasBeenSet() const { return m_s3BackupUpdateHasBeenSet; }
  template <typename S3BackupUpdateT = S3DestinationUpdate>
  void SetS3BackupUpdate(S3BackupUpdateT&& value) {
    m_s3BackupUpdateHasBeenSet = true;
    m_s3BackupUpdate = std::forward<S3BackupUpdateT>(value);
  }
  template <typename S3BackupUpdateT = S3DestinationUpdate>
  ExtendedS3DestinationUpdate& WithS3BackupUpdate(S3BackupUpdateT&& value) {
    SetS3BackupUpdate(std::forward<S3BackupUpdateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The serializer, deserializer, and schema for converting data from the JSON
   * format to the Parquet or ORC format before writing it to Amazon S3.</p>
   */
  inline const DataFormatConversionConfiguration& GetDataFormatConversionConfiguration() const {
    return m_dataFormatConversionConfiguration;
  }
  inline bool DataFormatConversionConfigurationHasBeenSet() const { return m_dataFormatConversionConfigurationHasBeenSet; }
  template <typename DataFormatConversionConfigurationT = DataFormatConversionConfiguration>
  void SetDataFormatConversionConfiguration(DataFormatConversionConfigurationT&& value) {
    m_dataFormatConversionConfigurationHasBeenSet = true;
    m_dataFormatConversionConfiguration = std::forward<DataFormatConversionConfigurationT>(value);
  }
  template <typename DataFormatConversionConfigurationT = DataFormatConversionConfiguration>
  ExtendedS3DestinationUpdate& WithDataFormatConversionConfiguration(DataFormatConversionConfigurationT&& value) {
    SetDataFormatConversionConfiguration(std::forward<DataFormatConversionConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The configuration of the dynamic partitioning mechanism that creates smaller
   * data sets from the streaming data by partitioning it based on partition keys.
   * Currently, dynamic partitioning is only supported for Amazon S3 destinations.
   * </p>
   */
  inline const DynamicPartitioningConfiguration& GetDynamicPartitioningConfiguration() const { return m_dynamicPartitioningConfiguration; }
  inline bool DynamicPartitioningConfigurationHasBeenSet() const { return m_dynamicPartitioningConfigurationHasBeenSet; }
  template <typename DynamicPartitioningConfigurationT = DynamicPartitioningConfiguration>
  void SetDynamicPartitioningConfiguration(DynamicPartitioningConfigurationT&& value) {
    m_dynamicPartitioningConfigurationHasBeenSet = true;
    m_dynamicPartitioningConfiguration = std::forward<DynamicPartitioningConfigurationT>(value);
  }
  template <typename DynamicPartitioningConfigurationT = DynamicPartitioningConfiguration>
  ExtendedS3DestinationUpdate& WithDynamicPartitioningConfiguration(DynamicPartitioningConfigurationT&& value) {
    SetDynamicPartitioningConfiguration(std::forward<DynamicPartitioningConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specify a file extension. It will override the default file extension</p>
   */
  inline const Aws::String& GetFileExtension() const { return m_fileExtension; }
  inline bool FileExtensionHasBeenSet() const { return m_fileExtensionHasBeenSet; }
  template <typename FileExtensionT = Aws::String>
  void SetFileExtension(FileExtensionT&& value) {
    m_fileExtensionHasBeenSet = true;
    m_fileExtension = std::forward<FileExtensionT>(value);
  }
  template <typename FileExtensionT = Aws::String>
  ExtendedS3DestinationUpdate& WithFileExtension(FileExtensionT&& value) {
    SetFileExtension(std::forward<FileExtensionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The time zone you prefer. UTC is the default.</p>
   */
  inline const Aws::String& GetCustomTimeZone() const { return m_customTimeZone; }
  inline bool CustomTimeZoneHasBeenSet() const { return m_customTimeZoneHasBeenSet; }
  template <typename CustomTimeZoneT = Aws::String>
  void SetCustomTimeZone(CustomTimeZoneT&& value) {
    m_customTimeZoneHasBeenSet = true;
    m_customTimeZone = std::forward<CustomTimeZoneT>(value);
  }
  template <typename CustomTimeZoneT = Aws::String>
  ExtendedS3DestinationUpdate& WithCustomTimeZone(CustomTimeZoneT&& value) {
    SetCustomTimeZone(std::forward<CustomTimeZoneT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_roleARN;

  Aws::String m_bucketARN;

  Aws::String m_prefix;

  Aws::String m_errorOutputPrefix;

  BufferingHints m_bufferingHints;

  CompressionFormat m_compressionFormat{CompressionFormat::NOT_SET};

  EncryptionConfiguration m_encryptionConfiguration;

  CloudWatchLoggingOptions m_cloudWatchLoggingOptions;

  ProcessingConfiguration m_processingConfiguration;

  S3BackupMode m_s3BackupMode{S3BackupMode::NOT_SET};

  S3DestinationUpdate m_s3BackupUpdate;

  DataFormatConversionConfiguration m_dataFormatConversionConfiguration;

  DynamicPartitioningConfiguration m_dynamicPartitioningConfiguration;

  Aws::String m_fileExtension;

  Aws::String m_customTimeZone;
  bool m_roleARNHasBeenSet = false;
  bool m_bucketARNHasBeenSet = false;
  bool m_prefixHasBeenSet = false;
  bool m_errorOutputPrefixHasBeenSet = false;
  bool m_bufferingHintsHasBeenSet = false;
  bool m_compressionFormatHasBeenSet = false;
  bool m_encryptionConfigurationHasBeenSet = false;
  bool m_cloudWatchLoggingOptionsHasBeenSet = false;
  bool m_processingConfigurationHasBeenSet = false;
  bool m_s3BackupModeHasBeenSet = false;
  bool m_s3BackupUpdateHasBeenSet = false;
  bool m_dataFormatConversionConfigurationHasBeenSet = false;
  bool m_dynamicPartitioningConfigurationHasBeenSet = false;
  bool m_fileExtensionHasBeenSet = false;
  bool m_customTimeZoneHasBeenSet = false;
};

}  // namespace Model
}  // namespace Firehose
}  // namespace Aws
