"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.deepCheck = deepCheck;
exports.pruneConfig = exports.foldResults = void 0;
exports.tryParseNDJson = tryParseNDJson;
var _stream = require("stream");
var _utils = require("../../../../../../src/core/server/saved_objects/routes/utils");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

async function tryParseNDJson(content) {
  try {
    const objects = await (0, _utils.createSavedObjectsStreamFromNdJson)(_stream.Readable.from(content));
    return await objects.toArray();
  } catch (err) {
    return null;
  }
}

/**
 * Check IntegrationReader nested dependencies for validity,
 * as a supplement to shallow config validation.
 *
 * @returns a Result indicating whether the integration is valid, holding the integration's config.
 */
async function deepCheck(reader) {
  const configResult = await reader.getConfig();
  if (!configResult.ok) {
    return configResult;
  }

  // Deep checks not included in default config validation
  const assets = await reader.getAssets();
  // Check if there was an error retrieving assets
  if (!assets.ok) {
    return {
      ok: false,
      error: new Error(`Failed to process assets while loading: ${assets.error.message}`)
    };
  }
  // Validate that at least one asset exists for the integration
  if (assets.value.length === 0) {
    return {
      ok: false,
      error: new Error('An integration must have at least one asset')
    };
  }
  return configResult;
}

/**
 * Helper method: Convert an Array<Result<type>> to Result<Array<type>>.
 *
 * @param results The list of results to fold.
 * @returns A single result object with values in an array, or an error result.
 */
const foldResults = results => results.reduce((result, currentValue) => {
  if (!result.ok) {
    return result;
  }
  if (!currentValue.ok) {
    return currentValue;
  }
  result.value.push(currentValue.value);
  return result;
}, {
  ok: true,
  value: []
});

/**
 * Remove all fields from SerializedIntegration not present in IntegrationConfig.
 *
 * @param rawConfig The raw config to prune
 * @returns A config with all data fields removed
 */
exports.foldResults = foldResults;
const pruneConfig = rawConfig => {
  // Hacky workaround: we currently only need to prune 'data' fields, so just remove every 'data'.
  // Lots of risky conversion in this method, so scope it to here and rewrite if more granular
  // pruning is needed.
  const prunePart = part => {
    const result = {};
    for (const [key, value] of Object.entries(part)) {
      if (key === 'data') {
        continue;
      } else if (Array.isArray(value)) {
        result[key] = value.map(item => {
          if (item instanceof Object && item !== null) {
            return prunePart(item);
          }
          return item;
        });
      } else if (value instanceof Object && value !== null) {
        result[key] = prunePart(value);
      } else {
        result[key] = value;
      }
    }
    return result;
  };
  return prunePart(rawConfig);
};
exports.pruneConfig = pruneConfig;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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