/*
 * CDDA2WAV
 *
 * LAST CHANGE:
 *   18.12.93 - first version,	OK
 *   01.01.94 - generalized & clean up HE
 *   13.06.94 - released HE
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "uti.h"
#include "aspidef.h"


/* possible values for UNDERSAMPLING: 1, 2, 3, 4, 6, 7, 12, 14, 28, 49 */
/*				      44 22 14 11  8  6  3.7  3  KHz */
#define UNDERSAMPLING	2


#define NSECTORS	1
#define MAXTRK	30

#define CB_CDDASECTOR	2368
#define CB_QSUBCHANNEL	16
#define CB_CDROMSECTOR	2048
#define CB_AUDIO	(CB_CDDASECTOR-CB_QSUBCHANNEL)

typedef struct TOC {
  BYTE _reserved1;
  BYTE bFlags;
  BYTE bTrack;
  BYTE _reserved2;
  DWORD dwStartSector;
} TOC;

static BYTE bufferCdRom [CB_CDROMSECTOR];
static BYTE bufferCdda [NSECTORS * CB_CDDASECTOR];
static BYTE bufferAudio [NSECTORS * CB_AUDIO];

static BYTE g_track=0xff, g_index=0xff;
static BYTE g_minute=0xff, g_seconds=0xff;

static int nTocEntries;
static TOC g_toc [MAXTRK]; /* 100 */

void EnableAudioMode (BOOL fAudioMode);

/*
 * ---------------------------------------------------------------------
 *  definitions for RIFF output (from Windows MMSYSTEM)
 * ---------------------------------------------------------------------
 */

typedef DWORD FOURCC;	/* a four character code */

typedef struct CHUNKHDR {
  FOURCC ckid;		/* chunk ID */
  DWORD dwSize; 	/* chunk size */
} CHUNKHDR;

#if	0
/* general waveform format structure (information common to all formats) */
typedef struct waveformat_tag {
  WORD wFormatTag;	/* format type */
  WORD nChannels;	/* number of channels (i.e. mono, stereo, etc.) */
  DWORD nSamplesPerSec; /* sample rate */
  DWORD nAvgBytesPerSec;/* for buffer size estimation */
  WORD nBlockAlign;	/* block size of data */
} WAVEFORMAT;
typedef WAVEFORMAT	 *PWAVEFORMAT;
#endif

/* flags for 'wFormatTag' field of WAVEFORMAT */
#define WAVE_FORMAT_PCM 1

/* specific waveform format structure for PCM data */
typedef struct pcmwaveformat_tag {
  WORD wFormatTag;	/* format type */
  WORD nChannels;	/* number of channels (i.e. mono, stereo, etc.) */
  DWORD nSamplesPerSec; /* sample rate */
  DWORD nAvgBytesPerSec;/* for buffer size estimation */
  WORD nBlockAlign;	/* block size of data */
  WORD wBitsPerSample;
} PCMWAVEFORMAT;
typedef PCMWAVEFORMAT *PPCMWAVEFORMAT;


/* MMIO macros */
#define mmioFOURCC(ch0, ch1, ch2, ch3) \
  ((DWORD)(BYTE)(ch0) | ((DWORD)(BYTE)(ch1) << 8) | \
  ((DWORD)(BYTE)(ch2) << 16) | ((DWORD)(BYTE)(ch3) << 24))

#define FOURCC_RIFF	mmioFOURCC ('R', 'I', 'F', 'F')
#define FOURCC_LIST	mmioFOURCC ('L', 'I', 'S', 'T')
#define FOURCC_WAVE	mmioFOURCC ('W', 'A', 'V', 'E')
#define FOURCC_FMT	mmioFOURCC ('f', 'm', 't', ' ')
#define FOURCC_DATA	mmioFOURCC ('d', 'a', 't', 'a')


/* simplified header for standard WAV files */
typedef struct WAVEHDR {
  CHUNKHDR chkRiff;
  FOURCC fccWave;
  CHUNKHDR chkFmt;
  WORD wFormatTag;	/* format type */
  WORD nChannels;	/* number of channels (i.e. mono, stereo, etc.) */
  DWORD nSamplesPerSec; /* sample rate */
  DWORD nAvgBytesPerSec;/* for buffer size estimation */
  WORD nBlockAlign;	/* block size of data */
  WORD wBitsPerSample;
  CHUNKHDR chkData;
} WAVEHDR;

#define IS_STD_WAV_HEADER(waveHdr) ( \
  waveHdr.chkRiff.ckid == FOURCC_RIFF && \
  waveHdr.fccWave == FOURCC_WAVE && \
  waveHdr.chkFmt.ckid == FOURCC_FMT && \
  waveHdr.chkData.ckid == FOURCC_DATA && \
  waveHdr.wFormatTag == WAVE_FORMAT_PCM)



/*
 * ASPI
 */

static SRB_IO srb;
static FILE *audio;


BOOL AspiPrintError (int err)
{
  if (err == ASPIE_SUCCESS) return FALSE;
  printf ("cdrom: %s\n", AspiGetErrorText (err));
  return TRUE;
}

/* actually reverse */
VOID *Swap (VOID *p, int size)
{
  char *pc = p;
  char tmp;

  if (size == 4) {
    tmp = pc [0];
    pc [0] = pc [3];
    pc [3] = tmp;
    tmp = pc [1];
    pc [1] = pc [2];
    pc [2] = tmp;
  } else {
    tmp = pc [0];
    pc [0] = pc [1];
    pc [1] = tmp;
  }
  return p;
}

static WAVEHDR waveHdr;
static DWORD nBytesDone = 0;

/* define size-related entries in wave header, update and close file */
VOID CloseAudio (void)
{
  DWORD cbData = nBytesDone;/* / 4
                 / (44100 / waveHdr.nSamplesPerSec)
                 * waveHdr.nChannels 
                 * ((waveHdr.wBitsPerSample + 7) / 8);*/

  fseek(audio, 0L, SEEK_SET);

  waveHdr.chkRiff.dwSize = cbData + sizeof(WAVEHDR) - sizeof(CHUNKHDR) ;
  waveHdr.chkData.dwSize = cbData;

  fwrite (&waveHdr, sizeof (waveHdr), 1, audio);

  fclose (audio);
}


void Cleanup (void)
{
  CloseAudio ();
  EnableAudioMode (FALSE);
}


void FatalError (char *szMessage, ...)
{
  vprintf (szMessage, (char *)(&szMessage + 1));
  Cleanup();
  exit (1);
}


void OpenAudio (char *fname, unsigned rate, int nBitsPerSample, int channels)
{
  audio = fopen (fname, "wb");

  waveHdr.chkRiff.ckid = FOURCC_RIFF;
  waveHdr.fccWave = FOURCC_WAVE;
  waveHdr.chkFmt.ckid = FOURCC_FMT;
  waveHdr.chkFmt.dwSize = sizeof (PCMWAVEFORMAT);
  waveHdr.wFormatTag = WAVE_FORMAT_PCM;
  waveHdr.nChannels = channels;
  waveHdr.nSamplesPerSec = rate;
  waveHdr.nBlockAlign = channels * ((nBitsPerSample + 7) / 8);
  waveHdr.nAvgBytesPerSec = waveHdr.nBlockAlign * waveHdr.nSamplesPerSec;
  waveHdr.wBitsPerSample = nBitsPerSample;
  waveHdr.chkData.ckid = FOURCC_DATA;

  fwrite (&waveHdr, sizeof (waveHdr), 1, audio);
}


void OpenCdRom (void)
{
  int targetId = 0;

  while (targetId < 7 && AspiGetDeviceType (0, targetId, 0) != DTC_CDROM)
    targetId++;
  if (targetId == 7)
    FatalError ("No CD-ROM drive found.\n");

  memzero (srb);
  srb.srb.bCmd = SRBC_EXECIO;
  srb.srb.bAdapterId = 0;
  srb.srb.bFlags = 0*SRBF_READ;
  srb.bTargetId = (BYTE)targetId;
  srb.bLun = 0;
  srb.lpData = NULL;
  srb.cbData = 0;
  srb.cbSense = 14;
  srb.cCdb = 6;
}


void ReadCdRom (LONG lSector)
{
  int err;
  static BYTE cmd [10] = {0x28, 0, 0, 0, 0, 0, 0, 0, 0, 0};

  cmd [2] = (BYTE)(lSector >> 24);
  cmd [3] = (BYTE)((lSector >> 16) & 0xFF);
  cmd [4] = (BYTE)((lSector >> 8) & 0xFF);
  cmd [5] = (BYTE)(lSector & 0xFF);
  cmd [8] = (BYTE)NSECTORS;

  srb.lpData = bufferCdda;
  srb.cbData = NSECTORS * CB_CDDASECTOR;
  srb.cCdb = 10;
  memcpy (&srb.cdb, cmd, 10);

  err = AspiRequestWait (&srb.srb);
  if (AspiPrintError (err))
    FatalError ("Read CD-ROM failed");
}

static int quiet = 0;

void UpdateTrackData (short p_num)
{
  if (!quiet) printf ("\ntrack: %.2X, ", p_num);
  g_track = (char)p_num;
}


void UpdateIndexData (short p_num)
{
  if (!quiet) printf ("index: %.2X\n", p_num);
  g_index = (char)p_num;
}


void UpdateTimeData (short p_min, short p_sec)
{
  if (!quiet) printf ("time: %.2X:%.2X\r", p_min, p_sec);
  g_minute = (char)p_min;
  g_seconds = (char)p_sec;
}


static int waitforsignal = 0;

void SaveBuffer (unsigned rate, int nBitsPerSample, int channels, 
                 LONG SecsToDo, DWORD *BytesDone)
{
  short i;
  char *pSrc = (char *)bufferCdda;
  BYTE *pDst = (BYTE *)bufferAudio;
  short nSamplesLeftInSector = ((CB_CDDASECTOR-CB_QSUBCHANNEL)/4);
  unsigned undersampling = 44100 / rate;
  int sh_bits = 16 - nBitsPerSample;
  unsigned BytesToDo = (SecsToDo > NSECTORS ? NSECTORS : SecsToDo) * CB_CDDASECTOR;
  BYTE *pStart = pDst;
  static int any_signal = 0;
  char *pSrcStop = pSrc + BytesToDo;

  while (pSrc < pSrcStop) {
    long lsum = 0;
    long rsum = 0;

    for (i=0; i < undersampling; i++) {
      /* LSB l, MSB l, LSB r, MSB r, ... */
      lsum += *((signed short *)pSrc)++;
      rsum += *((signed short *)pSrc)++;
    }
    lsum /= undersampling;
    rsum /= undersampling;

    if (channels == 1) {
      short sum;       /* mono section */
      sum = (lsum + rsum) >> 1;
      if (nBitsPerSample == 8) {
        if ((BYTE)(sum >> 8) != 0) {
          if (!any_signal) pStart = pDst;
          any_signal = 1;
        }
        *pDst++ = (BYTE)(sum >> 8) + (1 << 7);	/* ok */
	    } else {
        sum >>= sh_bits;
        if (sum != 0) {
          if (!any_signal) pStart = pDst;
          any_signal = 1;
        }
        *((short *)pDst)++ = sum; /* ok */
		  }
    } else {
      /* stereo section */
      if (nBitsPerSample == 8) {
        if ((short)(lsum >> 8) != 0 || 
	    (short)(rsum >> 8) != 0) {
          if (!any_signal) pStart = pDst;
          any_signal = 1;
        }
        *pDst++ = (BYTE)(((short) lsum) >> 8) + (1 << 7);
        *pDst++ = (BYTE)(((short) rsum) >> 8) + (1 << 7);	/* ok */
	    } else {
        if ((short)(lsum >> sh_bits) != 0 || 
	    (short)(rsum >> sh_bits) != 0) {
          if (!any_signal) pStart = pDst;
          any_signal = 1;
        }
        *((short *)pDst)++ = (((short) lsum) >> sh_bits);
        *((short *)pDst)++ = (((short) rsum) >> sh_bits); /* ok */
		  }
    }

    nSamplesLeftInSector -= undersampling;
    if (nSamplesLeftInSector == 0) {

      /* analyze Q-sub channel data */
      if (pSrc [1] == 1) {
        if (((BYTE *)pSrc) [2] != g_track)
	        UpdateTrackData (((BYTE *)pSrc) [2]);
	      if (((BYTE *)pSrc) [3] != g_index)
	        UpdateIndexData (((BYTE *)pSrc) [3]);
	      if (((BYTE *)pSrc) [5] != g_seconds ||
      	    ((BYTE *)pSrc) [4] != g_minute)
	        UpdateTimeData (((BYTE *)pSrc) [4], ((BYTE *)pSrc) [5]);
      }

      /* skip Q-sub channel data */
      pSrc += CB_QSUBCHANNEL;
      nSamplesLeftInSector = ((CB_CDDASECTOR - CB_QSUBCHANNEL)/4);
    }
  }
  if (!waitforsignal) pStart = (BYTE *)bufferAudio;

  if (!waitforsignal || any_signal) {
    fwrite (pStart, pDst - pStart, 1, audio);
    *BytesDone += pDst - pStart;
  }
}


void EnableAudioMode (BOOL fAudioMode)
{
  int err;
  static BYTE mode [12] = {0, 0, 0, 8, 0, 0, 0, 0, 0, 0, 0, 0};
  static BYTE cmd [6] = {0x15, 0x10, 0, 0, sizeof (mode), 0};

  if (fAudioMode) {
    mode [4] = 0x82;
    mode [10] = (CB_CDDASECTOR >> 8);
    mode [11] = (CB_CDDASECTOR & 0xFF);
  } else {
    mode [4] = 0x00;
    mode [10] = (CB_CDROMSECTOR >> 8);
    mode [11] = (CB_CDROMSECTOR & 0xFF);
  }

  srb.lpData = mode;
  srb.cbData = sizeof (mode);
  srb.cCdb = 6;
  memcpy (&srb.cdb, cmd, 6);
  err = AspiRequestWait (&srb.srb);
  if (AspiPrintError (err)) {
    if (err == ASPIE_UNITATTENTION)
      err = AspiRequestWait (&srb.srb);
    if (AspiPrintError (err))
      FatalError ("Enable audio mode failed");
  }
}


void ReadToc (void)
{
  int err;
  static BYTE cmd [10] = {
    0x43, 0, 0, 0, 0, 0, 1, CB_CDROMSECTOR >> 8, CB_CDROMSECTOR & 0xFF, 0
  };

  srb.lpData = bufferCdRom;
  srb.cbData = CB_CDROMSECTOR;
  srb.cCdb = 10;
  memcpy (&srb.cdb, cmd, 10);

  err = AspiRequestWait (&srb.srb);
  if (AspiPrintError (err))
    FatalError ("Read TOC failed.");

  nTocEntries = ((bufferCdRom [0] << 8) + bufferCdRom [1] - 2) / 8;
  memcpy (g_toc, bufferCdRom + 4, 8 * nTocEntries);
}


void DisplayToc (void)
{
  int i;
  DWORD dw;
  unsigned mins;
  float secnds;
  extern double fmod(double,double);

  dw = g_toc [nTocEntries-1].dwStartSector;
  Swap (&dw, 4);
  mins= dw / (75L*60);
  secnds= fmod(dw , 60*75) / 75.0;
  printf ("Table of Contents: total tracks:%d, (total time %u:%05.2f)\n",
          nTocEntries-1, mins, secnds);
  for (i=0; i<nTocEntries; i++) {
    if (g_toc [i].bFlags & 4) continue;	/* skip nonaudio tracks */
    if (g_toc [i].bTrack <= MAXTRK) {
      DWORD dw2 = g_toc [i].dwStartSector;
      dw = g_toc [i+1].dwStartSector;
      Swap (&dw, 4);
      Swap (&dw2, 4);
      mins= (dw - dw2) / (75L*60);
      secnds= fmod(dw-dw2 , 60*75) / 75.0;
      printf (" %d.(%u:%05.2f)", g_toc [i].bTrack,mins,secnds);
    }
    if (i < nTocEntries-1)
       if ((i+1) % 5 == 0) printf("\n");
       else putchar (',');
  }
  printf ("\n");
}


LONG GetStartSector (int p_track)
{
  int i;

  for (i=0; i<nTocEntries; i++) {
    if (g_toc [i].bTrack == p_track) {
      DWORD dw = g_toc [i].dwStartSector;
      if (g_toc [i].bFlags & 4)
	return -1;
      Swap (&dw, 4);
      return dw;
    }
  }

  return -1;
}


LONG GetEndSector (int p_track)
{
  int i;

  for (i=1; i<nTocEntries; i++) {
    if (g_toc [i-1].bTrack == p_track) {
      DWORD dw = g_toc [i].dwStartSector;
      if (g_toc [i].bFlags & 4)
	return -1;
      Swap (&dw, 4);
      return dw-1;
    }
  }

  return -1;
}



int FirstTrack (void)
{
  int i;
  
  for (i=0; i<nTocEntries; i++) {
    if (g_toc [i].bTrack != 0xAA &&
	!(g_toc [i].bFlags & 4))
      return g_toc [i].bTrack;
  }
  return 0;
}


LONG NextTrack (int p_offset)
{
  int track = (g_track >> 4) * 10 + (g_track & 15);
  long res;

  track += p_offset;
  res = GetStartSector (track);
  return res == -1 ? GetStartSector (track - p_offset) : res;
}


void usage(void)
{
	fprintf(stderr, "cdda2wav [-c chans] [-s] [-m] [-b bits] [-r rate] [-t track] [-o offset]\n");
	fprintf(stderr, "         [wavfile.wav]\n");
	fprintf(stderr, "cdda2wav copies parts from audio cd's directly to WAV files.\n");
	fprintf(stderr, "It requires a toshiba XM3401 scsi cdrom drive and an aspi driver to work.\n");
	fprintf(stderr, "options: -c channels : set 1 for mono, or 2 for stereo recording.\n");
	fprintf(stderr, "         -s          : set to stereo recording.\n");
	fprintf(stderr, "         -m          : set to mono recording.\n");
	fprintf(stderr, "         -b bits     : set bits per sample per channel (8, 12 or 16 bits).\n");
	fprintf(stderr, "         -r rate     : set rate in samples per second. Possible values are:\n");
	fprintf(stderr, "            44100,22050,14700,11025,7350,3675,3150,1575,900 Hertz.\n");
	fprintf(stderr, "         -t track    : select start track.\n");
	fprintf(stderr, "         -o offset   : start 'offset' sectors behind start track.\n");
	fprintf(stderr, "                       one sector equivalents 1/75 second.\n");
	fprintf(stderr, "         -d duration : set recording time in seconds or 0 for whole track.\n");
	fprintf(stderr, "         -w          : wait for signal, then start recording.\n");
	fprintf(stderr, "         -q          : quiet operation, no screen output.\n");
	fprintf(stderr, "defaults: mono, 16 bit, 22050 Hz, track 1, no offset, 16 seconds, 'audio.wav'\n"
                  "          don't wait for signal, not quiet\n");
  exit(1);
}

int main (int argc, char *argv [])
{
  int args = 0;
  LONG lSector;
  LONG lSector_p1;
  LONG sector_offset = 0;
  LONG nSectorsToDo = (1200 / NSECTORS) * NSECTORS;
  LONG req_space;
  LONG time = nSectorsToDo / 75;
  int track = 1;
  int	channels = 1;
  unsigned rate = 44100 / UNDERSAMPLING;
  int bits = 16;
  char fname[100] = "audio.wav";
#if	defined	(MSDOS) || defined(__MSDOS__)
#include <conio.h>
  int key_pressed = 0;
#endif

  /* command options parsing */
  while (--argc) {
    args++;
    /* channels */
    if (!strncmp("-c",argv[args],2)) {
      if (strlen (argv[args]) > 2) {
	channels = atoi(argv[args]+2);
	continue;
      } else {
        args++;
        if (--argc) {
	  channels = atoi(argv[args]);
	  continue;
	} else usage();
      }
    }
    /* bits */
    if (!strncmp("-b",argv[args],2)) {
      if (strlen (argv[args]) > 2) {
	bits = atoi(argv[args]+2);
	continue;
      } else {
        args++;
        if (--argc) {
	  bits = atoi(argv[args]);
	  continue;
	} else usage();
      }
    }
    /* rate */
    if (!strncmp("-r",argv[args],2)) {
      if (strlen (argv[args]) > 2) {
	rate = atoi(argv[args]+2);
	continue;
      } else {
        args++;
        if (--argc) {
	  rate = atoi(argv[args]);
	  continue;
	} else usage();
      }
    }
    /* start track */
    if (!strncmp("-t",argv[args],2)) {
      if (strlen (argv[args]) > 2) {
	track = atoi(argv[args]+2);
	continue;
      } else {
        args++;
        if (--argc) {
	  track = atoi(argv[args]);
	  continue;
        } else usage();
      }
    }
    /* recording time */
    if (!strncmp("-d",argv[args],2)) {
      if (strlen (argv[args]) > 2) {
	time = atoi(argv[args]+2);
	continue;
      } else {
        args++;
        if (--argc) {
	  time = atoi(argv[args]);
	  continue;
	} else usage();
      }
    }
    /* sector offset */
    if (!strncmp("-o",argv[args],2)) {
      if (strlen (argv[args]) > 2) {
	sector_offset = atol(argv[args]+2);
	continue;
      } else {
        args++;
        if (--argc) {
	  sector_offset = atol(argv[args]);
	  continue;
	} else usage();
      }
    }
    /* stereo */
    if (!strncmp("-s",argv[args],2)) {
      channels = 2;
      continue;
    }
    /* mono */
    if (!strncmp("-m",argv[args],2)) {
      channels = 1;
      continue;
    }
    /* wait for signal */
    if (!strncmp("-w",argv[args],2)) {
      waitforsignal = 1;
      continue;
    }
    /* quiet */
    if (!strncmp("-q",argv[args],2)) {
      quiet = 1;
      continue;
    }
    /* other options unknown */
    if (*argv[args] == '-') usage();

      /* filename given */
      strcpy(fname,argv[args]);

      /* should be the last parameter */
      if (argc > 1) usage();
  }

  /* check all parameters */
  if (channels != 1 && channels != 2) usage();
  if (bits != 8 && bits != 12 && bits != 16) usage();
  if (44100 % rate != 0) usage();
  if (track < 1 || track > 50) usage();
  if (time < 0 || time > 4440) usage();



  if (not AspiInit ())
    FatalError ("Unable to open ASPI device.\n");

  atexit (Cleanup);

  OpenCdRom ();
  EnableAudioMode (TRUE);
  ReadToc ();
  if (!quiet) DisplayToc ();

  if (!FirstTrack ())
    FatalError ("This is no audio disk");

  lSector = GetStartSector (track);
  lSector_p1 = GetEndSector (track) + 1;
  if (lSector < 0)
    FatalError ("track %d not found\n", track);

  lSector += sector_offset;
  /* check against end sector of track */
  if (lSector >= lSector_p1) {
    printf("sector offset exceeds track size (ignored)\n");
    lSector -= sector_offset;
  }

  if (time == 0) {
    /* set time to track time */
    time = ((lSector_p1 - lSector)) / 75;
  }

  remove(fname);
#if	defined	(MSDOS) || defined(__MSDOS__)
  /* calculate required disk space */
  req_space = sizeof(WAVEHDR) + (bits + 7)/8 * channels * (LONG)rate * time;

  /* check disk space */
  {
#include <dos.h>
#include <ctype.h>
     LONG avail;
     unsigned drive;
#ifdef TURBOC
     struct dfree d;
#else
     struct diskfree_t d;
#endif

     if (fname[1] != ':') {
       /* get default */
       drive = 0;
     } else {
       /* get specified drive */
       drive = toupper(fname[0]) - '@';
     }

#ifdef TURBOC
     getdfree(drive, &d);
     avail = (long) d.df_avail
           * (long) d.df_bsec
           * (long) d.df_sclus;
#else
     _dos_getdiskfree(drive, &d);
     avail = (long) d.avail_clusters
           * (long) d.bytes_per_sector
           * (long) d.sectors_per_cluster;
#endif

     if (req_space > avail) {
       if (!quiet) fprintf(stderr, "unsufficient disk space! reducing recording length.\n");
       time = (avail - sizeof(WAVEHDR)) / 
               ((bits + 7)/8 * channels * (LONG)rate);
       req_space = sizeof(WAVEHDR) + (bits + 7)/8 * channels * (LONG)rate * time;
       if (!quiet) fprintf(stderr, "maximum possible are %ld seconds <=> %ld Bytes.\n",
               time,req_space);
     }
  }
#endif

  /* calculate # of sectors to read */
  nSectorsToDo = time * 75;

  if (!quiet) printf("recording %lu seconds %s with %d bits @ %u Hz ->'%s'...\n"
				              ,time ,channels == 1 ? "mono":"stereo", bits, rate, fname);

  OpenAudio (fname, rate, bits, channels);

  while (nSectorsToDo > 0 
#if	defined	(MSDOS) || defined(__MSDOS__)
         && !(key_pressed = kbhit())
#endif
                         ) {
    ReadCdRom (lSector);
    lSector += NSECTORS;
    SaveBuffer (rate, bits, channels, nSectorsToDo, &nBytesDone);
    nSectorsToDo -= NSECTORS;
  }

#if	defined	(MSDOS) || defined(__MSDOS__)
  if (key_pressed) {
    getch();
    if (!quiet) printf("\naborted.\n");
  }
#endif

  CloseAudio ();
  EnableAudioMode (FALSE);

  return 0;
}
