/* -*- c++ -*- */
/*
 * Copyright 2004 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <usrp0_sink_c.h>
#include <gr_io_signature.h>
#include <usrp0.h>


usrp0_sink_c_sptr
usrp0_make_sink_c (int which_board, unsigned int interp_rate) throw (std::runtime_error)
{
  return usrp0_sink_c_sptr (new usrp0_sink_c (which_board, interp_rate));
}


usrp0_sink_c::usrp0_sink_c (int which_board, unsigned int interp_rate) throw (std::runtime_error)
  : gr_sync_block ("usrp0_sink_c",
		   gr_make_io_signature (1, 1, sizeof (gr_complex)),
		   gr_make_io_signature (0, 0, 0)),
    d_nunderruns (0)
{
  d_usrp = usrp0_tx::make (which_board);
  if (d_usrp == 0)
    throw std::runtime_error ("can't open usrp0");

  if (!d_usrp->set_interp_rate (interp_rate)){
    fprintf (stderr, "invalid interp_rate\n");
    throw std::runtime_error ("invalid interp_rate");
  }

  // All calls to d_usrp->write must be multiples of 512 bytes.
  // We jack this up to 4k to reduce overhead.

  set_output_multiple (4 * 1024 / sizeof (gr_complex));
}

usrp0_sink_c::~usrp0_sink_c ()
{
  delete d_usrp;
}

int
usrp0_sink_c::work (int noutput_items,
		   gr_vector_const_void_star &input_items,
		   gr_vector_void_star &output_items)
{
  assert (input_items.size () == 1);

  gr_complex	*in = (gr_complex *) input_items[0];

  static const int BUFSIZE = 8 * 1024;
  short 	outbuf[BUFSIZE];			// 16KB writes
  int		obi = 0;
  int	 	n;
  bool		underrun;
  
  for (n = 0; n < noutput_items; n++){
    outbuf[obi+0] = (short) real(in[n]); // saturate?
    outbuf[obi+1] = (short) imag(in[n]); // saturate?
    obi += 2;
    if (obi >= BUFSIZE){		// flush
      if (d_usrp->write (outbuf, obi * sizeof (short), &underrun) != obi * (int) sizeof (short)){
	return -1;	// indicate we're done
      }
      if (underrun){
	d_nunderruns++;
	fprintf (stderr, "usrp0_sink: underrun\n");
      }
      obi = 0;
    }
  }

  if (obi != 0){
    if (d_usrp->write (outbuf,
		       obi * sizeof (short), &underrun) != (obi * (int) sizeof (short))){
      return -1;	// indicate we're done
    }
    if (underrun){
      d_nunderruns++;
      fprintf (stderr, "usrp0_sink: underrun");
    }
  }

  return noutput_items;
}

bool
usrp0_sink_c::set_interp_rate (unsigned int rate)
{
  return d_usrp->set_interp_rate (rate);
}

bool
usrp0_sink_c::set_tx_freq (int channel, double freq)
{
  return d_usrp->set_tx_freq (channel, freq);
}

long
usrp0_sink_c::dac_freq () const
{
  return (long) d_usrp->get_oscillator_freq ();	// 125e6
}

unsigned int
usrp0_sink_c::interp_rate () const
{
  return d_usrp->get_interp_rate ();
}

double
usrp0_sink_c::tx_freq (int channel) const
{
  return d_usrp->get_tx_freq (channel);
}
