#include "testutils.h"
#include "knuth-lfib.h"

#define MSG1 "None so blind as those who will not see"
#define MSG2 "Fortune knocks once at every man's door"

static void
test_rsa_sign_tr(struct rsa_public_key *pub,
	     struct rsa_private_key *key,
	     unsigned di_length,
	     const uint8_t *di,
	     mpz_t expected)
{
  mpz_t signature;
  struct knuth_lfib_ctx lfib;

  knuth_lfib_init(&lfib, 1111);

  mpz_init(signature);
  mpz_set_ui (signature, 17);
  /* Try bad private key */
  mpz_add_ui(key->p, key->p, 2);

  ASSERT(!rsa_pkcs1_sign_tr(pub, key,
		    &lfib, (nettle_random_func *) knuth_lfib_random,
		    di_length, di, signature));

  mpz_sub_ui(key->p, key->p, 2);

  ASSERT(!mpz_cmp_ui(signature, 17));

  /* Try the good private key */
  ASSERT(rsa_pkcs1_sign_tr(pub, key,
		    &lfib, (nettle_random_func *) knuth_lfib_random,
		    di_length, di, signature));

  if (verbose)
    {
      fprintf(stderr, "rsa-pkcs1-tr signature: ");
      mpz_out_str(stderr, 16, signature);
      fprintf(stderr, "\nrsa-pkcs1-tr expected: ");
      mpz_out_str(stderr, 16, expected);
      fprintf(stderr, "\n");
    }

  ASSERT (mpz_cmp(signature, expected) == 0);

  /* Try bad data */
  ASSERT (!rsa_pkcs1_verify(pub, 16, (void*)"The magick words", signature));

  /* Try correct data */
  ASSERT (rsa_pkcs1_verify(pub, di_length, di, signature));

  /* Try bad signature */
  mpz_combit(signature, 17);
  ASSERT (!rsa_pkcs1_verify(pub, di_length, di, signature));

  mpz_clear(signature);
}


void
test_main(void)
{
  struct rsa_public_key pub;
  struct rsa_private_key key;

  mpz_t expected;

  mpz_init(expected);

  rsa_private_key_init(&key);
  rsa_public_key_init(&pub);

  test_rsa_set_key_1(&pub, &key);

  /* Test signatures */
  mpz_set_str(expected,
	      "23bd361a622bc35450a30ae332d8dad050c6bad5cae0b61c5bee29"
	      "db876d7fe098472fc7933f2cc1121c0fc8a414c71a98189c66077b"
	      "fe3d58d08dc242bbcfdf2c905b91de0e479a8b4b159a45fe1789db"
	      "abdaf88aa704e7558f24f5a0be382619758061cde89e9c730ac6db"
	      "ef005bfad5a936633e3dfccc37b2214e2",
	      16);

  test_rsa_sign_tr(&pub, &key, LDATA(MSG1), expected);

  mpz_set_str(expected,
	      "15bd817f53501f8eb6693283004546ba14f19dd4da742b1e30a7b2"
	      "1db309cd3f36f821f565d31c1ed2df8b6648dd8bdb218f841f5506"
	      "a437825999aaf3ef77dff9a1fc7631ce0e99f851f4166ae67caed9"
	      "820a764e44274a898ddd72f4c7115ba49d332f4fa929ee6dce4d61"
	      "39fef8d8d25d4c9ff857689846e5cac26d", 16);

  test_rsa_sign_tr(&pub, &key, LDATA(MSG2), expected);

  /* 777-bit key, generated by
   *
   *   lsh-keygen -a rsa -l 777 -f advanced-hex
   *
   * Interesting because the size of n doesn't equal the sum of the
   * sizes of p and q.
   *
   * (private-key (rsa-pkcs1
   *        (n #013b04440e3eef25 d51c738d508a7fa8 b3445180c342af0f
   *            4cb5a789047300e2 cfc5c5450974cfc2 448aeaaa7f43c374
   *            c9a3b038b181f2d1 0f1a2327fd2c087b a49bf1086969fd2c
   *            d1df3fd69f81fa4b 162cc8bbb363fc95 b7b24b9c53d0c67e
   *            f52b#)
   *        (e #3f1a012d#)
   *        (d #f9bae89dacca6cca c21e0412b4df8355 6fe7c5322bbae8ad
   *            3f11494fd12bc076 d4a7da3050fe109d 2074db09cc6a93b4
   *            745479522558379e a0ddfa74f86c9e9e a22c3b0e93d51447
   *            0feb38105dd35395 63b91ee32776f40c 67b2a175690f7abb
   *            25#)
   *        (p #0b73c990eeda0a2a 2c26416052c85560 0c5c0f5ce86a8326
   *            166acea91786237a 7ff884e66dbfdd3a ab9d9801414c1506
   *            8b#)
   *        (q #1b81c19a62802a41 9c99283331b0badb 08eb0c25ffce0fbf
   *            50017850036f32f3 2132a845b91a5236 61f7b451d587383f
   *            e1#)
   *        (a #0a912fc93a6cca6b 3521725a3065b3be 3c9745e29c93303d
   *            7d29316c6cafa4a2 89945f964fcdea59 1f9d248b0b6734be
   *            c9#)
   *        (b #1658eca933251813 1eb19c77aba13d73 e0b8f4ce986d7615
   *            764c6b0b03c18146 46b7f332c43e05c5 351e09006979ca5b
   *            05#)
   *        (c #0114720dace7b27f 2bf2850c1804869f 79a0aad0ec02e6b4
   *            05e1831619db2f10 bb9b6a8fd5c95df2 eb78f303ea0c0cc8
   *            06#)))
   */

  mpz_set_str(pub.n,
	      "013b04440e3eef25" "d51c738d508a7fa8" "b3445180c342af0f"
	      "4cb5a789047300e2" "cfc5c5450974cfc2" "448aeaaa7f43c374"
	      "c9a3b038b181f2d1" "0f1a2327fd2c087b" "a49bf1086969fd2c"
	      "d1df3fd69f81fa4b" "162cc8bbb363fc95" "b7b24b9c53d0c67e"
	      "f52b", 16);

  mpz_set_str(pub.e, "3f1a012d", 16);

  ASSERT (rsa_public_key_prepare(&pub));

  mpz_set_str(key.p,
	      "0b73c990eeda0a2a" "2c26416052c85560" "0c5c0f5ce86a8326"
	      "166acea91786237a" "7ff884e66dbfdd3a" "ab9d9801414c1506"
	      "8b", 16);

  mpz_set_str(key.q,
	      "1b81c19a62802a41" "9c99283331b0badb" "08eb0c25ffce0fbf"
	      "50017850036f32f3" "2132a845b91a5236" "61f7b451d587383f"
	      "e1", 16);

  mpz_set_str(key.a,
	      "0a912fc93a6cca6b" "3521725a3065b3be" "3c9745e29c93303d"
	      "7d29316c6cafa4a2" "89945f964fcdea59" "1f9d248b0b6734be"
	      "c9", 16);

  mpz_set_str(key.b,
	      "1658eca933251813" "1eb19c77aba13d73" "e0b8f4ce986d7615"
	      "764c6b0b03c18146" "46b7f332c43e05c5" "351e09006979ca5b"
	      "05", 16);

  mpz_set_str(key.c,
	      "0114720dace7b27f" "2bf2850c1804869f" "79a0aad0ec02e6b4"
	      "05e1831619db2f10" "bb9b6a8fd5c95df2" "eb78f303ea0c0cc8"
	      "06", 16);

  ASSERT (rsa_private_key_prepare(&key));
  ASSERT (pub.size == key.size);

  /* Test signatures */
  mpz_set_str(expected,
	      "5493d4d774d03f54de2e9bef6818b74a8e22bbe9628b6dec3f178d"
	      "cc0090dab933a0f24bc9ac8aec949558868ea4f8e29d0248e70717"
	      "6c113e8fbd9f1428c45a120378e32febffd4e9aca8102081bb5b90"
	      "0f2914395a220823b700dbd7bd32646e", 16);

  test_rsa_sign_tr(&pub, &key, LDATA(MSG1), expected);

  /* Test sha1 signature */
  mpz_set_str(expected,
	      "126bbd3d9e8649187069d8b6f0f43bd234a53249f765f616647049"
	      "ca49b2c4cc8049440d5a9f1d10fcdf162b40afd9b761bcd30e2c60"
	      "133b2c6f0504fe447038422b835d35448acb0881e894144e4ccaea"
	      "59ce8dd16d3540fcbb84ebd1c5fb37510", 16);

  test_rsa_sign_tr(&pub, &key, LDATA(MSG2), expected);

  rsa_private_key_clear(&key);
  rsa_public_key_clear(&pub);
  mpz_clear(expected);
}
