#!/usr/bin/env node
/*
 This file is part of GNU Taler
 (C) 2022 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import esbuild from "esbuild";
import path from "node:path";
import fs from "node:fs";

const BASE = process.cwd();

let GIT_ROOT = BASE;
while (!fs.existsSync(path.join(GIT_ROOT, ".git")) && GIT_ROOT !== "/") {
  GIT_ROOT = path.join(GIT_ROOT, "../");
}
const GIT_HASH = GIT_ROOT === "/" ? undefined : git_hash();

let PACKAGE_VERSION = get_version()
function get_version() {
  try {
    return JSON.parse(fs.readFileSync(path.join(BASE, "package.json"))).version;
  } catch {
    return 'not defined'
  }
}

function git_hash() {
  const rev = fs
    .readFileSync(path.join(GIT_ROOT, ".git", "HEAD"))
    .toString()
    .trim()
    .split(/.*[: ]/)
    .slice(-1)[0];
  if (rev.indexOf("/") === -1) {
    return rev;
  } else {
    return fs
      .readFileSync(path.join(GIT_ROOT, ".git", rev))
      .toString()
      .trim();
  }
}

export const buildConfig = {
  entryPoints: ["src/index.ts"],
  outfile: "dist/taler-wallet-cli-bundled.cjs",
  bundle: true,
  minify: false,
  target: ["es2020"],
  format: "cjs",
  platform: "node",
  sourcemap: true,
  inject: ["src/import-meta-url.js"],
  define: {
    __VERSION__: `"${PACKAGE_VERSION}"`,
    __GIT_HASH__: `"${GIT_HASH}"`,
    "walletCoreBuildInfo.implementationSemver": `"${PACKAGE_VERSION}"`,
    "walletCoreBuildInfo.implementationGitHash": `"${GIT_HASH}"`,
    ["import.meta.url"]: "import_meta_url",
  },
};

esbuild.build(buildConfig).catch((e) => {
  console.log(e);
  process.exit(1);
});
