/****************************************************************************
 * KPrintcap - encapsulates processing of /etc/printcap. Allows for adding,
 *  removing and editing printer records.
 *
 *  Copyright (C) 1998 Neil Hart (nhart@magma.ca)
 *
 *  This program is free software; you can redistribute it and/or modify 
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  $Id: kprintcap.h,v 1.7 1998/04/27 19:21:15 nhart Exp $
 *                                                                   
 ****************************************************************************/

#ifndef __KPRINTCAP_H__
#define __KPRINTCAP_H__

#include <stdio.h>
#include <qstring.h>

#include "kprinterrs.h"

#define PRINTCAPTABLESZ 32

struct KPrintcapLine
{
	KPrintcapLine* prev;
	KPrintcapLine* next;
	QString* line;
};

struct KPrintcapTableEntry
{
	KPrintcapLine* firstLine;
	KPrintcapLine* lastLine;
	QString*       printerDescription;
	bool           isLocal;
};

class KPrintcap
{
public:
	KPrintcap();
	~KPrintcap();
	bool init();				// return false if initialisation failed
	int numberOfPrinters();		// how many printers are there?

	// add a new printer record -- return values from kprinterrs.h
	int addNewLocalPrinter(const QString& name, const QString& model, 
		const QString& spooldir, const QString& filter, const QString& device,
		const QString& note);
	int addNewRemotePrinter(const QString& name, const QString& spooldir,
		const QString& host, const QString& remqueue, const QString& note);

	// remove a numbered printer record, and (optionally) remove the 
	// spool directory. Return false if we fail for any reason
	bool removePrinter(int printerNumber, bool removeDirs);

	// get details for a numbered printer record. Note that the return
	// value is used to indicate whether the request was valid (ie. the
	// printer number is within range and we're requesting valid 
	// information -- we can't ask the hostname of a local printer, for
	// example). The information is always written into the variable 
	// passed by reference into the second argument. Note also that for 
	// QString items, we assume that the pointer points to a QString whose
	// value we can simply modify.
	bool getPrinterLocation(int printerNumber, bool* isLocal);
	bool getPrinterName(int printerNumber, QString* name);
	bool getPrinterModel(int printerNumber, QString* model);
	bool getPrinterFilter(int printerNumber, QString* filter);
	bool getPrinterSpooldir(int printerNumber, QString* spooldir);
	bool getPrinterDevice(int printerNumber, QString* device);
	bool getPrinterHost(int printerNumber, QString* host);
	bool getPrinterQueue(int printerNumber, QString* remqueue);
	bool getPrinterNote(int printerNumber, QString* note);
	bool getPrinterDescription(int printerNumber, QString* desc);

	// change details for a numbered printer record. Return false
	//  if we failed in some way
	bool changePrinterName(int printerNumber, const char* name);
	bool changePrinterModel(int printerNumber, const char* model);
	bool changePrinterFilter(int printerNumber, const char* filter);
	bool changePrinterSpooldir(int printerNumber, const char* spooldir,
		bool removeDirs);
	bool changePrinterDevice(int printerNumber, const char* device);
	bool changePrinterHost(int printerNumber, const char* host);
	bool changePrinterQueue(int printerNumber, const char* remqueue);
	bool changePrinterNote(int printerNumber, const char* note);

	// create a default printer name
	bool defaultPrinterName(QString* defaultName);

private:
	FILE* printcap;

	struct KPrintcapTableEntry printcapTable[PRINTCAPTABLESZ];	
	int printcapTableIdx;					// points to next available record

	struct KPrintcapLine* firstPrintcapLine;	// first line of /etc/printcap
	struct KPrintcapLine* lastPrintcapLine;		// last line of /etc/printcap
	void addToLinkedList(struct KPrintcapLine* line);

	// private functions for reading and writing printcap 
	bool readPrintcapIntoList();
	bool writeListToPrintcap();

	// find line containing required information. Return false on failure
	bool findPrinterName(int printerNumber, QString** nameLine);
	bool findPrinterModel(int printerNumber, QString** modelLine);
	bool findPrinterFilter(int printerNumber, QString** filterLine);
	bool findPrinterSpooldir(int printerNumber, QString** spooldirLine);
	bool findPrinterDevice(int printerNumber, QString** deviceLine);
	bool findPrinterHost(int printerNumber, QString** hostLine);
	bool findPrinterQueue(int printerNumber, QString** remqueueLine);
	bool findPrinterNote(int printerNumber, QString** noteLine);

	bool createSpoolDirFile(QString& sd, const char* name, int mode);
	bool removeRangeOfLines(KPrintcapLine* start, KPrintcapLine* end);
	bool createPrinterDescription( int printNum, QString* desc);

	void removeSpooldir( const QString& spooldir );
	int createSpooldir( const QString& spooldir );
};


#endif
