/***************************************************************************
                          modemconnector.cpp  -  description                              
                             -------------------                                         
    begin                : Wed Apr 7 1999                                           
    copyright            : (C) 1999 by Torsten Uhlmann                         
    email                : TUhlmann@gmx.de                                     
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/

#include <string.h>
#include <termios.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdio.h>

#include <stdlib.h>
#include <errno.h>
#include <sys/signal.h>
#include <sys/time.h>


#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>

#include "modemconnector.h"

#define _POSIX_SOURCE 1
#define DEFAULT_BAUDRATE B9600

ModemConnector::ModemConnector(const QString &device,
								const QString &baudrate, const QString &lockfile)
{
  _device 	= device.copy();
  _lockfile = lockfile.copy();
  // set baudrate
  if (baudrate == "2400")
  {
  	_baudrate = B2400;
  } else if (baudrate == "4800")
  {
  	_baudrate = B4800;
  } else if (baudrate == "9600")
  {
  	_baudrate = B9600;
  } else if (baudrate == "19200")
  {
  	_baudrate = B19200;
  } else if (baudrate == "38400")
  {
  	_baudrate = B38400;
  } else if (baudrate == "57600")
  {
  	_baudrate = B57600;
  } else if (baudrate == "115200")
  {
  	_baudrate = B115200;
  } else if (baudrate == "230400")
  {
  	_baudrate = B230400;
  } else 	
  	_baudrate = DEFAULT_BAUDRATE;
}

ModemConnector::~ModemConnector()
{
  tcsetattr(_fd,TCSANOW,&_oldtio); /* restore old modem setings */
  close(_fd);
  _device = "";
  unlink(_lockfile.data());
}

bool ModemConnector::setDevice(const QString &device)
{
  _device.setStr(device.data());
  return true;
}

bool ModemConnector::initSerialLine()
{

	if (!lockSerialLine())
		return false;

  _fd = open(_device.data(), O_RDWR | O_NOCTTY );
  if (_fd <0) {perror(_device.data()); return false; }
 	if(!isatty(_fd))
	{
		close(_fd);
		unlink(_lockfile.data());
		return false;
	}
  tcgetattr(_fd,&_oldtio); /* save current modem settings */
	tcgetattr(_fd,&_newtio);

  //_newtio.c_cflag = _baudrate | CRTSCTS | CS8 | CLOCAL | CREAD;
  _newtio.c_cflag = _baudrate | CRTSCTS | CS8 | CREAD | CLOCAL;
	//_newtio.c_cflag = _baudrate | CRTSCTS | CS8 | CREAD | HUPCL;
  _newtio.c_iflag = INPCK | IGNPAR;
  _newtio.c_oflag = OPOST | ONLCR;
  _newtio.c_lflag = 0;
  _newtio.c_cc[VMIN]=0;
  _newtio.c_cc[VTIME]=10;
  cfsetispeed(&_newtio,_baudrate);
  cfsetospeed(&_newtio,_baudrate);

  tcflush(_fd, TCIOFLUSH);
  if (tcsetattr(_fd,TCSANOW,&_newtio) < 0)
	{
		perror("tcsetattr");
		close(_fd);
		unlink(_lockfile);
		return false;
	}
	
  return true;
}

bool ModemConnector::flushSerialLine()
{
  if (_fd<= 0)
    return false;

  tcflush(_fd, TCIOFLUSH);
  return true;
}

bool ModemConnector::lockSerialLine()
{
	int 	fd;
	char 	text[256];
	char 	pidbuf[32];
	int 	lockpid;

	// fprintf(stderr,"Try to create lockfile %s\n",_lockfile);
	fd=open(_lockfile.data(), O_WRONLY|O_CREAT|O_EXCL, 0644);
	if(fd==-1)
	{
		fprintf(stderr,"Couldn't create lockfile %s\n",_lockfile.data());
		if(errno==EEXIST)
		{
			fd=open(_lockfile.data(),O_RDONLY);
			if(fd==-1)
			{
				fprintf(stderr,"Couldn't open %s : %s\n",_lockfile.data(),strerror(errno));
				return false;
			}
			read(fd,pidbuf,20);
			lockpid=atoi(pidbuf);
			if(kill(lockpid,0)==0)
			{
				//errno=EBUSY;
				close(fd);
				return false;
			}
			close(fd);
			unlink(_lockfile.data());
			fprintf(stderr,"Removing old lockfile\n");
			fd=open(_lockfile,O_RDWR|O_CREAT|O_EXCL,0644);
			if(fd==-1)
			{
				fprintf(stderr,"Unable to create lockfile %s : %s\n",_lockfile.data(),strerror(errno));
				return false;
			}
		}
		else
		{
			fprintf(stderr,"Unable to create lockfile %s : %s\n",_lockfile.data(),strerror(errno));
			return false;
		}
	}
	sprintf(text,"%10d\n",getpid());
	write(fd,text,strlen(text));
	close(fd);

	return true;
}

/*
bool ModemConnector::readChar(unsigned char *c)
{
  if (_fd<= 0)
    return false;

  while (true)
  {
    if (read(_fd,c,1) < 1)
    {
	    *c = 0;
      return false;
    } else {
      return true;
    }
  }
}
*/

int ModemConnector::readC()
{
	unsigned char c[2] = "";
  if (_fd<= 0)
    return -1;

  if (read(_fd,c,1) < 1)
		return -2;
	else
		return c[0];
}

/** read char till ERROR, OK or VCON */
int ModemConnector::readCBlocking()
{

	unsigned char c[2] = "";
	static char 	buffer[8] = "";
	static int 		bufpos = 0; // position of last byte filled
	static bool 	wasEnd = false;
	char 					tmpbuf[8];
	
  if (_fd<= 0)
    return -1;

	if (wasEnd)
	{
		wasEnd = false;
		memset(buffer,0,8);
		bufpos = 0;
		return -2;
	}

	read(_fd,c,1);
	
	if (bufpos < 7)
	{
		buffer[bufpos++] 	= c[0];
		buffer[bufpos] 		= 0;
	} else {
		// buffer is full, shift left and append
		*tmpbuf = 0;
		strncpy(tmpbuf,&buffer[1],6);
		tmpbuf[6] = 0;
		*buffer = 0;
		strcpy(buffer, tmpbuf);
		buffer[6] = c[0];
		buffer[7] = 0;
	}

	if ((strstr(buffer,"\nOK")) ||
			(strstr(buffer,"\nERROR")) ||
			(strstr(buffer,"\nVCON")))
	{
		wasEnd = true;
	}
			
	return c[0];
}

QString	ModemConnector::readStr()
{
	fd_set readfds;
	struct timeval timeout;
	int err = 0;

	int 		c 	= 0;
	QString str = 0;
	
	FD_ZERO(&readfds);  FD_SET(_fd, &readfds);
	timeout.tv_sec = 30; timeout.tv_usec = 0;//timeout.tv_usec = 7000000; /* 7 sec */
	err = select(FD_SETSIZE, &readfds, NULL, NULL, &timeout);
	if (((err == -1) || (!err)))
	{
		debug("ERROR: A timeout occured!!!");
		str = 0;
		return str;
	}

  while ( (c = readCBlocking()) >= 0)
	  str += (unsigned char) c;
	
  return str;
}

bool ModemConnector::writeStr(char *c, int len)
{
  if (_fd<= 0)
    return false;

	flushSerialLine();
  if (write(_fd,c,len) == -1)
    return false;
  else
  {
  	usleep(200000);
    return true;
	}
}

bool 	ModemConnector::writeStr(const QString& s)
{
  if (_fd<= 0)
    return false;

	flushSerialLine();
  if (write(_fd,s.data(),s.length()) == -1)
    return false;
  else
  {
  	usleep(200000);
    return true;
	}
}

QString	ModemConnector::executeCommand(const QString& send, const QString& end)
{
	bool success = false;
	struct termios tmptio, newtio;
	QString result = 0;
	
  tcgetattr(_fd,&tmptio); /* save current modem settings */
  newtio.c_cflag = _baudrate | CRTSCTS | CS8 | CLOCAL | CREAD;
  newtio.c_iflag = INPCK | IGNPAR;
  newtio.c_oflag = 0;
  newtio.c_lflag = 0;
  newtio.c_cc[VMIN]=1;
  newtio.c_cc[VTIME]=100;
  tcflush(_fd, TCIFLUSH);
  tcsetattr(_fd,TCSANOW,&newtio);

 	// Send command
	//fprintf(stderr,"executeCommand(%s,%p,%d,%s)\n",send,receive,len,end);
	writeStr(send);

	if (end.isNull())
	{
		result = "\nOK\n";
		success = true;
		debug("device: do not wait for answer");
	} else {
 		// Read result
		result = readStr();
		if(result.length() > 0)
		{
			if(result.find(end.data()))
			{
				success = true;
			} else
				if(result.find("\nERROR"))
				{
					success = false;
				} else
					if(result.find("\nVCON"))
					{
						success = false;
					}
		}
	}
	
	flushSerialLine();
	
  tcsetattr(_fd,TCSANOW,&tmptio); /* restore old modem settings */

  if (success)
		return result;
	else
		return 0;
}

bool ModemConnector::executeSimpleCommand(const QString& send, const QString& end)
{
	QString s = executeCommand(send, end);
	if (s.isNull())
		return false;
	else
		return true;
}












