/***************************************************************************
 *   Copyright (C) 2006 by Thomas Kadauke                                  *
 *   tkadauke@gmx.de                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,      *
 *   Boston, MA 02110-1301, USA.                                           *
 ***************************************************************************/

#ifndef WORKFLOWDOCUMENT_P_H
#define WORKFLOWDOCUMENT_P_H

// Qt includes
#include <qobject.h>
#include <qcstring.h>
#include <qtimer.h>
#include <qvaluevector.h>
#include <qvaluelist.h>
#include <qmap.h>

// WorKflow includes
#include "document.h"
#include "command.h"
#include "commandinfo.h"

namespace WorKflow
{
  // Action classes for undo / redo
  class Action
  {
  public:
    Action(Document* document)
      : m_document(document) {}
    virtual ~Action() {}

    virtual QString name() = 0;
    virtual bool execute() = 0;
    virtual bool unexecute() = 0;

    inline Document* document() { return m_document; }

  private:
    Document* m_document;
  };

  class CompositeAction : public Action
  {
  public:
    CompositeAction(Document* document)
      : Action(document) {}
    virtual ~CompositeAction();

    virtual bool execute();
    virtual bool unexecute();

  protected:
    void addAction(Action* action);

  private:
    typedef QValueList<Action*> ActionList;
    ActionList m_actions;
  };

  class InsertCommandAction : public Action
  {
  public:
    InsertCommandAction(Document* document, int row, Command* cmd);
    ~InsertCommandAction();

    virtual QString name();
    virtual bool execute();
    virtual bool unexecute();

    inline void setOwnsCommand(bool owns) { m_ownsCommand = owns; }

  private:
    int m_row;
    Command* m_command;

    bool m_ownsCommand;
  };

  class RemoveCommandAction : public Action
  {
  public:
    RemoveCommandAction(Document* document, int row);

    virtual QString name();
    virtual bool execute();
    virtual bool unexecute();

  private:
    int m_row;
    Command* m_command;
  };

  class MoveCommandAction : public CompositeAction
  {
  public:
    MoveCommandAction(Document* document, int sourceRow, int destRow);

    virtual QString name();

  private:
    Command* m_command;
  };

  class RemoveRangeAction : public CompositeAction
  {
  public:
    RemoveRangeAction(Document* document, int startRow, int endRow);

    virtual QString name();
  };

  class ModifyCommandAction : public Action
  {
  public:
    ModifyCommandAction(Document* document, Command* command, const CommandState& oldState, const CommandState& newState);

    virtual QString name();
    virtual bool execute();
    virtual bool unexecute();

  private:
    Command* m_command;
    CommandState m_oldState;
    CommandState m_newState;
    bool m_dontReallyExecute;
  };

  class CutRangeAction : public RemoveRangeAction
  {
  public:
    CutRangeAction(Document* document, int startRow, int endRow);

    virtual QString name();
  };

  class PasteCommandAction : public Action
  {
  public:
    PasteCommandAction(Document* document, int row, const CommandInfoList& commands);
    ~PasteCommandAction();

    virtual QString name();
    virtual bool execute();
    virtual bool unexecute();

  private:
    int m_row;
    CommandInfoList m_commands;
  };


  typedef QValueVector<Command*> Commands;
  typedef QValueVector<CommandState> States;
  typedef QValueList<Action*> ActionList;


  // private class
  class Document::Private : public QObject
  {
    Q_OBJECT

  public:
    typedef QMap<QString, Value> Variables;

    Private(Document* doc);

    void execAction(Action* action);
    void updateCommandState(int row);

    bool modified;

    Commands commands;
    States commandStates;
    bool executing;

    ActionList undoList;
    ActionList redoList;
    QTimer modificationTimer;
    Command* modifiedCommand;
    Command* currentCommand;
    bool undoing;

    QString searchText;
    Command* searchCommand;
    Parameter* searchParameter;

    KURL url;

    Variables variables;
    ApplicationManager* appManager;

  public slots:
    void slotCommandModified(Command* command);
    void modificationTimeout();
    void slotProgress(double prog, const QString& task);
    void abort();

  signals:
    void progress(double total, double current, const QString& command, const QString& task);

  private:
    Document* p;
  };
}

#endif
