/***************************************************************************
 *   Copyright (C) 2006 by Thomas Kadauke                                  *
 *   tkadauke@gmx.de                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,      *
 *   Boston, MA 02110-1301, USA.                                           *
 ***************************************************************************/

// KDE include
#include <kdebug.h>

// Qt includes
#include <qdom.h>
#include <qfile.h>

// WorKflow includes
#include "librarydescription.h"
#include "applicationdescription.h"
#include "datatype.h"
#include "category.h"
#include "commanddescription.h"
#include "conversion.h"

using namespace WorKflow;

LibraryDescription::LibraryDescription()
  : XMLDescription()
{
}

LibraryDescription::LibraryDescription(const QString& xmlFile)
  : XMLDescription(), m_xmlFile(xmlFile)
{
  readXML();
}

LibraryDescription::~LibraryDescription()
{
  while (!m_applications.isEmpty())
    delete m_applications.first();

  while (!m_categories.isEmpty())
    delete m_categories.first();

  while (!m_commands.isEmpty())
    delete m_commands.first();

  while (!m_conversions.isEmpty())
    delete m_conversions.first();

  while (!m_datatypes.isEmpty())
    delete m_datatypes.first();
}

void LibraryDescription::addApplication(ApplicationDescription* app)
{
  m_applications.append(app);
}

void LibraryDescription::addCategory(Category* cat)
{
  m_categories.append(cat);
}

void LibraryDescription::addCommand(CommandDescription* cmd)
{
  m_commands.append(cmd);
}

void LibraryDescription::addDatatype(Datatype* type)
{
  m_datatypes.append(type);
}

void LibraryDescription::addConversion(Conversion* conv)
{
  m_conversions.append(conv);
}

void LibraryDescription::removeApplication(ApplicationDescription* app)
{
  m_applications.erase(m_applications.find(app));
}

void LibraryDescription::removeCategory(Category* cat)
{
  m_categories.erase(m_categories.find(cat));
}

void LibraryDescription::removeCommand(CommandDescription* cmd)
{
  m_commands.erase(m_commands.find(cmd));
}

void LibraryDescription::removeDatatype(Datatype* type)
{
  m_datatypes.erase(m_datatypes.find(type));
}

void LibraryDescription::removeConversion(Conversion* conv)
{
  m_conversions.erase(m_conversions.find(conv));
}

LibraryDescription::ApplicationList LibraryDescription::applications() const
{
  return m_applications;
}

LibraryDescription::CategoryList LibraryDescription::categories() const
{
  return m_categories;
}

LibraryDescription::CommandList LibraryDescription::commands() const
{
  return m_commands;
}

LibraryDescription::DatatypeList LibraryDescription::datatypes() const
{
  return m_datatypes;
}

LibraryDescription::ConversionList LibraryDescription::conversions() const
{
  return m_conversions;
}

void LibraryDescription::readXML()
{
  QDomDocument doc("");
  QFile file(m_xmlFile);
  if (!file.open(IO_ReadOnly)) {
    kdWarning() << "Could not read XML file \"" << m_xmlFile << "\"!" << endl;
    return;
  }
  if (!doc.setContent(&file)) {
    kdWarning() << "Could not parse XML file \"" << m_xmlFile << "\"!" << endl;
    file.close();
    return;
  }
  file.close();

  QDomElement docElem = doc.documentElement();

  XMLDescription::readXML(docElem);

  QDomNode n = docElem.firstChild();
  while (!n.isNull()) {
    QDomElement e = n.toElement();
    if (!e.isNull()) {
      XMLDescription* desc = 0;
      if (e.tagName() == "datatype") {
        desc = new Datatype(this);
      } else if (e.tagName() == "application") {
        desc = new ApplicationDescription(this);
      } else if (e.tagName() == "conversion") {
        desc = new Conversion(this);
      } else if (e.tagName() == "category") {
        desc = new Category(this);
      } else if (e.tagName() == "command") {
        desc = new CommandDescription(this);
      }

      if (desc) {
        desc->readXML(e);
      }
    }
    n = n.nextSibling();
  }
}

QString LibraryDescription::writeXML()
{
  QDomDocument doc("wflib");
  QDomElement docElem = doc.createElement("workflowlibrary");

  XMLDescription::writeXML(doc, docElem);

  for (ApplicationList::ConstIterator i = m_applications.begin(); i != m_applications.end(); ++i) {
    (*i)->writeXML(doc, docElem);
  }

  for (CategoryList::ConstIterator i = m_categories.begin(); i != m_categories.end(); ++i) {
    (*i)->writeXML(doc, docElem);
  }

  for (DatatypeList::ConstIterator i = m_datatypes.begin(); i != m_datatypes.end(); ++i) {
    (*i)->writeXML(doc, docElem);
  }

  for (ConversionList::ConstIterator i = m_conversions.begin(); i != m_conversions.end(); ++i) {
    (*i)->writeXML(doc, docElem);
  }

  for (CommandList::ConstIterator i = m_commands.begin(); i != m_commands.end(); ++i) {
    (*i)->writeXML(doc, docElem);
  }

  doc.appendChild(docElem);

  return doc.toString(4);
}
