/* Copyright (C) 2002 Thorsten Kukuk
   Author: Thorsten Kukuk <kukuk@suse.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License version 2 as
   published by the Free Software Foundation.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <getopt.h>
#include <unistd.h>
#include <string.h>
#include <syslog.h>
#include <signal.h>
#include <libgen.h>
#include <locale.h>
#include <libintl.h>
#include <sys/stat.h>
#include <sys/resource.h>

#ifndef _
#define _(String) gettext (String)
#endif

#include "main.h"
#include "public.h"
#include "error_codes.h"

#ifndef GPASSWD_PROGRAM
#define GPASSWD_PROGRAM "gpasswd"
#endif

#ifndef CHFN_PROGRAM
#define CHFN_PROGRAM "chfn"
#endif

#ifndef CHSH_PROGRAM
#define CHSH_PROGRAM "chsh"
#endif

/* Print the version information.  */
void
print_version (const char *program)
{
  fprintf (stdout, "%s (%s) %s\n", program, PACKAGE, VERSION);
  fprintf (stdout, gettext ("\
Copyright (C) %s Thorsten Kukuk.\n\
This is free software; see the source for copying conditions.  There is NO\n\
warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n\
"), "2002");
  /* fprintf (stdout, _("Written by %s.\n"), "Thorsten Kukuk"); */
}

void
print_error (const char *program)
{
  fprintf (stderr,
           _("Try `%s --help' or `%s --usage' for more information.\n"),
           program, program);
}

static void
init_environment (void)
{
  struct rlimit rlim;

  /* Don't create a core file.  */
  rlim.rlim_cur = rlim.rlim_max = 0;
  setrlimit (RLIMIT_CORE, &rlim);

  /* Set all limits to unlimited to avoid to run in any
     problems later.  */
  rlim.rlim_cur = rlim.rlim_max = RLIM_INFINITY;
  setrlimit (RLIMIT_AS, &rlim);
  setrlimit (RLIMIT_CPU, &rlim);
  setrlimit (RLIMIT_DATA, &rlim);
  setrlimit (RLIMIT_FSIZE, &rlim);
  setrlimit (RLIMIT_NOFILE, &rlim);
  setrlimit (RLIMIT_RSS, &rlim);
  setrlimit (RLIMIT_STACK, &rlim);

  /* Ignore all signals which can make trouble later.  */
  signal (SIGALRM, SIG_IGN);
  signal (SIGXFSZ, SIG_IGN);
  signal (SIGHUP, SIG_IGN);
  signal (SIGINT, SIG_IGN);
  signal (SIGPIPE, SIG_IGN);
  signal (SIGQUIT, SIG_IGN);
  signal (SIGTERM, SIG_IGN);
  signal (SIGTSTP, SIG_IGN);
  signal (SIGTTOU, SIG_IGN);

  umask (077);
}

int
main (int argc, char **argv)
{
  int retval;
  char *prog;

  setlocale (LC_ALL, "");
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);

  prog = basename (argv[0]);

  openlog (prog, LOG_PID, LOG_AUTHPRIV);

  /* Before going any further, raise the ulimit and ignore
     signals.  */

  init_environment ();

  if (strcmp ("chsh", prog) == 0 || strcmp ("ypchsh", prog) == 0)
    return chsh_main (prog, argc, argv);

  if (strcmp ("chfn", prog) == 0 || strcmp ("ypchfn", prog) == 0)
    return chfn_main (prog, argc, argv);

  /* easy way to get ride of the first argument.  */
  if (argc > 1 && argv[1][0] == '-' && strchr ("gfs", argv[1][1]))
    {
      char buf[200];

      setuid (getuid ());
      switch (argv[1][1])
	{
	case 'g':
	  argv[1] = GPASSWD_PROGRAM;  /* XXX warning: const */
	  break;
	case 'f':
	  argv[1] = CHFN_PROGRAM;  /* XXX warning: const */
	  break;
	case 's':
	  argv[1] = CHSH_PROGRAM;  /* XXX warning: const */
	  break;
	default:
	  /* If this happens we have a real problem. */
	  abort ();
	}
      snprintf (buf, sizeof buf, _("passwd: Cannot execute %s"), argv[1]);
      execvp(argv[1], &argv[1]);
      perror(buf);
      syslog (LOG_ERR, "Cannot execute %s", argv[1]);
      closelog ();
      return E_FAILURE;
    }

  retval = passwd_main (prog, argc, argv);
  closelog ();
  return retval;
}
