/*	$NetBSD: osiop_sbdio.c,v 1.4 2008/05/14 13:29:28 tsutsui Exp $	*/

/*-
 * Copyright (c) 2001, 2005 Izumi Tsutsui.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: osiop_sbdio.c,v 1.4 2008/05/14 13:29:28 tsutsui Exp $");

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/device.h>

#include <machine/bus.h>
#include <machine/sbdiovar.h>

#include <dev/scsipi/scsi_all.h>
#include <dev/scsipi/scsipi_all.h>
#include <dev/scsipi/scsiconf.h>

#include <dev/ic/osiopreg.h>
#include <dev/ic/osiopvar.h>

int osiop_sbdio_match(device_t, cfdata_t, void *);
void osiop_sbdio_attach(device_t, device_t, void *);
int osiop_sbdio_intr(void *);

CFATTACH_DECL_NEW(osiop_sbdio, sizeof(struct osiop_softc),
    osiop_sbdio_match, osiop_sbdio_attach, NULL, NULL);

int
osiop_sbdio_match(device_t parent, cfdata_t cf, void *aux)
{
	struct sbdio_attach_args *sa = aux;

	return strcmp(sa->sa_name, "osiop") ? 0 : 1;
}

void
osiop_sbdio_attach(device_t parent, device_t self, void *aux)
{
	struct osiop_softc *sc = device_private(self);
	struct sbdio_attach_args *sa = aux;
	int error, scid;

	sc->sc_dev = self;
	sc->sc_dmat = sa->sa_dmat;
	sc->sc_bst  = sa->sa_bust;

	error = bus_space_map(sc->sc_bst, sa->sa_addr1, OSIOP_NREGS, 0,
	    &sc->sc_reg);
	if (error != 0) {
		aprint_error(": can't map registers, error = %d\n", error);
		return;
	}

	sc->sc_clock_freq = 25;	/* MHz */
	sc->sc_ctest7 = 0;
	sc->sc_dcntl = OSIOP_DCNTL_EA;
	if (sa->sa_flags == 0x0001) /* TR2A */
		sc->sc_ctest4 = OSIOP_CTEST4_MUX; /* Host bus multiplex mode */

	scid = ffs(osiop_read_1(sc, OSIOP_SCID));
	if (scid == 0)
		scid = 7;
	else
		scid--;
	sc->sc_id = scid;

	intr_establish(sa->sa_irq, osiop_sbdio_intr, sc);

	osiop_attach(sc);
}

int
osiop_sbdio_intr(void *arg)
{
	struct osiop_softc *sc = arg;
	uint8_t istat;

	if (sc->sc_flags & OSIOP_INTSOFF)
		return 0;

	istat = osiop_read_1(sc, OSIOP_ISTAT);

	if ((istat & (OSIOP_ISTAT_SIP | OSIOP_ISTAT_DIP)) == 0)
		return 0;

	/* Save interrupt details for the back-end interrupt handler */
	sc->sc_sstat0 = osiop_read_1(sc, OSIOP_SSTAT0);
	sc->sc_istat = istat;
	sc->sc_dstat = osiop_read_1(sc, OSIOP_DSTAT);

	osiop_intr(sc);

	return 1;
}
