/* $NetBSD$ */

/*-
 * Copyright (c) 2010 Jared D. McNeill <jmcneill@invisible.ca>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD$");

#include <sys/types.h>
#include <sys/param.h>
#include <sys/conf.h>
#include <sys/buf.h>
#include <sys/kmem.h>
#include <sys/kernel.h>
#include <sys/device.h>
#include <sys/proc.h>
#include <sys/condvar.h>
#include <sys/kthread.h>
#include <sys/select.h>
#include <sys/audioio.h>
#include <sys/file.h>
#include <sys/filedesc.h>
#include <sys/vnode.h>
#include <sys/module.h>

#include <dev/audiovar.h>
#include <dev/auconv.h>
#include <dev/audio_if.h>

#ifdef _MODULE
#include "kmixervar.h"
#else
#include <dev/kmixer/kmixervar.h>
#endif

/* XXX */
extern int	audio_check_params(struct audio_params *);

static int	kmixer_match(device_t, cfdata_t, void *);
static void	kmixer_attach(device_t, device_t, void *);
static int	kmixer_detach(device_t, int);

static struct kmixer_chan * kmixer_alloc_chan(struct kmixer_softc *);
static void	kmixer_free_chan(struct kmixer_chan *);

static int	kmixer_audiodev_probe(struct kmixer_softc *);
static int	kmixer_audiodev_configure(struct kmixer_softc *, int);

static bool	kmixer_mix_isempty(struct kmixer_softc *);
static int16_t	kmixer_mix_buffers(struct kmixer_softc *);
static void	kmixer_write_buffers(struct kmixer_softc *, int16_t);

static void	kmixer_auconv_init(struct kmixer_softc *);
static int	kmixer_auconv_setinfo(struct kmixer_chan *,
				      struct audio_info *);

static void	kmixer_filter_append(stream_filter_list_t *,
				     stream_filter_factory_t,
				     const audio_params_t *);
static void	kmixer_filter_set(stream_filter_list_t *, int,
				  stream_filter_factory_t,
				  const audio_params_t *);
static void	kmixer_filter_prepend(stream_filter_list_t *,
				      stream_filter_factory_t,
				      const audio_params_t *);

static struct kmixer_audiodev_format {
	struct audio_params	params;
} kmixer_audiodev_formats[] = {
	{
	  /* 44.1kHz 2ch slinear16_le */
	  .params = {
	    .sample_rate = 44100,
	    .encoding = AUDIO_ENCODING_SLINEAR_LE,
	    .precision = 16,
	    .validbits = 16,
	    .channels = 2,
	  },
	},
};

dev_type_open(kmixer_open);

const struct cdevsw kmixer_cdevsw = {
	.d_open = kmixer_open,
	.d_close = noclose,
	.d_read = noread,
	.d_write = nowrite,
	.d_ioctl = noioctl,
	.d_stop = nostop,
	.d_tty = notty,
	.d_poll = nopoll,
	.d_mmap = nommap,
	.d_kqfilter = nokqfilter,
	.d_flag = D_OTHER,
};

static int	kmixer_chan_read(struct file *, off_t *, struct uio *,
				 kauth_cred_t, int);
static int	kmixer_chan_write(struct file *, off_t *, struct uio *,
				  kauth_cred_t, int);
static int	kmixer_chan_ioctl(struct file *, u_long, void *);
static int	kmixer_chan_poll(struct file *, int);
static int	kmixer_chan_close(struct file *);

static void	kmixer_thread(void *);

static const struct fileops kmixer_fileops = {
	.fo_read = kmixer_chan_read,
	.fo_write = kmixer_chan_write,
	.fo_ioctl = kmixer_chan_ioctl,
	.fo_fcntl = fnullop_fcntl,
	.fo_poll = kmixer_chan_poll,
	.fo_stat = fbadop_stat,
	.fo_close = kmixer_chan_close,
	.fo_kqfilter = fnullop_kqfilter,
	.fo_drain = fnullop_drain,
};

CFATTACH_DECL_NEW(kmixer, sizeof(struct kmixer_softc),
    kmixer_match, kmixer_attach, kmixer_detach, NULL);

extern const struct cdevsw audio_cdevsw;
extern struct cfdriver kmixer_cd;

static int
kmixer_match(device_t parent, cfdata_t cfdata, void *opaque)
{
	return 1;
}

static void
kmixer_attach(device_t parent, device_t self, void *opaque)
{
	struct kmixer_softc *sc = device_private(self);
	int mj, mn;

	aprint_naive("\n");
	aprint_normal("\n");

	sc->sc_dev = self;
	TAILQ_INIT(&sc->sc_chans);
	cv_init(&sc->sc_condvar, device_xname(self));
	mutex_init(&sc->sc_mutex, MUTEX_DEFAULT, IPL_NONE);

	if (!pmf_device_register(self, NULL, NULL))
		aprint_error_dev(self, "couldn't establish power handler\n");

	mj = cdevsw_lookup_major(&audio_cdevsw);
	if (mj == -1) {
		aprint_error_dev(self, "couldn't lookup audio major");
		return;
	}
	mn = device_unit(parent) + SOUND_DEVICE;
	sc->sc_audiodev = makedev(mj, mn);
	sc->sc_paramsidx = -1;
	aprint_debug_dev(self, "using audio device (%d,%d)\n",
	    major(sc->sc_audiodev), minor(sc->sc_audiodev));

	if (kmixer_audiodev_probe(sc) != 0) {
		aprint_error_dev(self,
		    "kernel doesn't support format required for audio hw\n");
		return;
	}

	kmixer_auconv_init(sc);

	kthread_create(PRI_NONE, KTHREAD_MPSAFE, NULL,
	    kmixer_thread, sc, &sc->sc_lwp, device_xname(sc->sc_dev));
}

static int
kmixer_detach(device_t self, int flags)
{
	struct kmixer_softc *sc = device_private(self);
	int cmaj, mn;

	cmaj = cdevsw_lookup_major(&kmixer_cdevsw);
	mn = device_unit(self);
	vdevgone(cmaj, mn, mn, VCHR);

	sc->sc_running = false;
	mutex_enter(&sc->sc_mutex);
	while (sc->sc_stopped == false)
		cv_wait(&sc->sc_condvar, &sc->sc_mutex);
	mutex_exit(&sc->sc_mutex);
	while (sc->sc_dead == false)
		kpause("kmixerdt", false, mstohz(10), NULL);

	pmf_device_deregister(self);

	mutex_destroy(&sc->sc_mutex);
	cv_destroy(&sc->sc_condvar);

	if (sc->sc_encodings)
		auconv_delete_encodings(sc->sc_encodings);

	return 0;
}

int
kmixer_open(dev_t dev, int flags, int fmt, struct lwp *l)
{
	struct kmixer_softc *sc;
	struct kmixer_chan *ch;
	struct file *fp;
	int err, fd;

	sc = device_lookup_private(&kmixer_cd, minor(dev));
	if (sc == NULL) {
		printf("kmixer: device is gone, dev = 0x%x\n", dev);
		return ENXIO;
	}

	if (sc->sc_audiodev == 0)
		return ENODEV;

	ch = kmixer_alloc_chan(sc);
	if (ch == NULL)
		return ENOMEM;

	err = fd_allocfile(&fp, &fd);
	if (err) {
		kmixer_free_chan(ch);
		return err;
	}

	return fd_clone(fp, fd, flags, &kmixer_fileops, ch);
}

static struct kmixer_chan *
kmixer_alloc_chan(struct kmixer_softc *sc)
{
	struct kmixer_chan *ch;
	int err;

	ch = kmem_alloc(sizeof(*ch), KM_SLEEP);
	if (ch == NULL)
		return NULL;

	ch->ch_softc = sc;
	memset(&ch->ch_buffer, 0, sizeof(ch->ch_buffer));
	cv_init(&ch->ch_condvar, device_xname(sc->sc_dev));
	mutex_init(&ch->ch_mutex, MUTEX_DEFAULT, IPL_NONE);

	mutex_enter(&sc->sc_mutex);
	if (TAILQ_EMPTY(&sc->sc_chans)) {
		/*
		 * If this is the first audio channel, open the backing
		 * audio device.
		 */
		err = cdev_open(sc->sc_audiodev, FREAD|FWRITE, 0, &lwp0);
		if (err == 0)
			kmixer_audiodev_configure(sc, sc->sc_paramsidx);
		if (err != 0) {
			mutex_exit(&sc->sc_mutex);
			kmem_free(ch, sizeof(*ch));
			aprint_error_dev(sc->sc_dev,
			    "couldn't open audio device (%i,%i): %d\n",
			    major(sc->sc_audiodev), minor(sc->sc_audiodev),
			    err);
			return NULL;
		}
	}
	TAILQ_INSERT_TAIL(&sc->sc_chans, ch, ch_entry);
	mutex_exit(&sc->sc_mutex);

	return ch;
}

static void
kmixer_free_chan(struct kmixer_chan *ch)
{
	struct kmixer_softc *sc = ch->ch_softc;
	int err;

	mutex_enter(&sc->sc_mutex);
	TAILQ_REMOVE(&sc->sc_chans, ch, ch_entry);
	if (TAILQ_EMPTY(&sc->sc_chans)) {
		/*
		 * If this was the last audio channel, close the backing
		 * audio device.
		 */
		err = cdev_close(sc->sc_audiodev, FREAD|FWRITE, 0, &lwp0);
		if (err)
			aprint_error_dev(sc->sc_dev,
			    "couldn't close audio device: %d\n", err);
	}
	if (ch->ch_reading == true)
		sc->sc_reading = false;
	mutex_exit(&sc->sc_mutex);

	mutex_destroy(&ch->ch_mutex);
	cv_destroy(&ch->ch_condvar);
	kmem_free(ch, sizeof(*ch));
}

static int
kmixer_audiodev_configure(struct kmixer_softc *sc, int index)
{
	struct audio_params *params;
	struct audio_info ai;

	if (index < 0 || index > __arraycount(kmixer_audiodev_formats))
		return EINVAL;

	params = &kmixer_audiodev_formats[index].params;
	AUDIO_INITINFO(&ai);
	ai.play.sample_rate = params->sample_rate;
	ai.play.channels = params->channels;
	ai.play.precision = params->precision;
	ai.play.encoding = params->encoding;
	ai.record.sample_rate = params->sample_rate;
	ai.record.channels = params->channels;
	ai.record.precision = params->precision;
	ai.record.encoding = params->encoding;
	ai.mode = AUMODE_PLAY|AUMODE_RECORD;

	return cdev_ioctl(sc->sc_audiodev, AUDIO_SETINFO,
		    &ai, FREAD|FWRITE, &lwp0);
}

static int
kmixer_audiodev_probe(struct kmixer_softc *sc)
{
	int i, err;

	err = cdev_open(sc->sc_audiodev, FREAD|FWRITE, 0, &lwp0);
	if (err != 0)
		return err;

	err = EINVAL;
	for (i = 0; i < __arraycount(kmixer_audiodev_formats); i++) {
		if (kmixer_audiodev_configure(sc, i) == 0) {
			aprint_debug_dev(sc->sc_dev,
			    "format %d Hz %d-bit %d-ch slinear_le\n",
			    kmixer_audiodev_formats[i].params.sample_rate,
			    kmixer_audiodev_formats[i].params.validbits,
			    kmixer_audiodev_formats[i].params.channels);
			sc->sc_paramsidx = i;
			err = 0;
			break;
		}
	}

	cdev_close(sc->sc_audiodev, FREAD|FWRITE, 0, &lwp0);

	return err;
}

static int
kmixer_chan_read(struct file *fp, off_t *offp, struct uio *uio,
    kauth_cred_t cred, int flags)
{
	struct kmixer_chan *ch = fp->f_data;
	struct kmixer_softc *sc = ch->ch_softc;

	if (ch->ch_reading == false) {
		mutex_enter(&sc->sc_mutex);
		if (sc->sc_reading == false)
			ch->ch_reading = sc->sc_reading = true;
		mutex_exit(&sc->sc_mutex);

		if (ch->ch_reading == false)
			return EBUSY;
	}

	return cdev_read(sc->sc_audiodev, uio, FREAD|FWRITE);
}

static int
kmixer_chan_write(struct file *fp, off_t *offp, struct uio *uio,
    kauth_cred_t cred, int flags)
{
	struct kmixer_chan *ch = fp->f_data;
	struct kmixer_softc *sc = ch->ch_softc;
	struct kmixer_buffer *kb = &ch->ch_buffer;
	uint8_t *buf = (uint8_t *)kb->kb_buffer;
	int cnt;
	int len, avail;

retry:
	if (uio->uio_resid == 0)
		return 0;
 
	while ((avail = (KMIXER_BUFSIZE - kb->kb_resid)) < 2) {
		if (sc->sc_stopped)
			return EIO;
		if (kpause("kmixer", true, mstohz(10), NULL) == EINTR)
			return EINTR;
	}

	cnt = uio->uio_resid;
	if (cnt > avail)
		cnt = avail;

	//mutex_enter(&ch->ch_mutex);
	len = KMIXER_BUFSIZE - kb->kb_wpos;
	if (len > cnt)
		len = cnt;
	if (len > 0) {
		uiomove(buf + kb->kb_wpos, len, uio);
		kb->kb_wpos += len;
		if (kb->kb_wpos == KMIXER_BUFSIZE)
			kb->kb_wpos = 0;
		kb->kb_resid += len;
		avail -= len;
		//cv_broadcast(&ch->ch_condvar);
		//mutex_exit(&ch->ch_mutex);
		goto retry;
	}

	//cv_broadcast(&ch->ch_condvar);
	//mutex_exit(&ch->ch_mutex);

	return 0;
}

static int
kmixer_chan_ioctl(struct file *fp, u_long cmd, void *data)
{
	struct kmixer_chan *ch = fp->f_data;
	struct kmixer_softc *sc = ch->ch_softc;

	switch (cmd) {
	case AUDIO_SETINFO:
		return kmixer_auconv_setinfo(ch, (struct audio_info *)data);
	case AUDIO_DRAIN:
		while (ch->ch_buffer.kb_resid >= 2)
			if (kpause("kmixdrain", true, mstohz(10), NULL)
			    == EINTR)
				break;
		
		return 0;
	default:
		return cdev_ioctl(sc->sc_audiodev, cmd, data,
		    fp->f_flag, &lwp0);
	}
}

static int
kmixer_chan_poll(struct file *fp, int events)
{
	return 0;
}

static int
kmixer_chan_close(struct file *fp)
{
	struct kmixer_chan *ch = fp->f_data;

	while (ch->ch_buffer.kb_resid >= 2)
		if (kpause("kmixclose", true, mstohz(10), NULL) == EINTR)
			break;

	kmixer_free_chan(ch);

	return 0;
}

static void
kmixer_thread(void *opaque)
{
	struct kmixer_softc *sc = opaque;

	sc->sc_running = true;
	while (sc->sc_running == true) {
		mutex_enter(&sc->sc_mutex);
		if (kmixer_mix_isempty(sc) == true) {
			cv_timedwait_sig(&sc->sc_condvar, &sc->sc_mutex,
			    mstohz(10));
		}

		if (sc->sc_running == true &&
		    kmixer_mix_isempty(sc) == false) {
			int16_t sample = kmixer_mix_buffers(sc);
			mutex_exit(&sc->sc_mutex);
			kmixer_write_buffers(sc, sample);
		} else
			mutex_exit(&sc->sc_mutex);
	}

	mutex_enter(&sc->sc_mutex);
	sc->sc_stopped = true;
	cv_broadcast(&sc->sc_condvar);
	mutex_exit(&sc->sc_mutex);

	sc->sc_dead = true;
	kthread_exit(0);
}

static bool
kmixer_mix_isempty(struct kmixer_softc *sc)
{
	struct kmixer_chan *ch;
	int nchans = 0, achans = 0;

	if (TAILQ_EMPTY(&sc->sc_chans))
		return true;

	TAILQ_FOREACH(ch, &sc->sc_chans, ch_entry) {
		++nchans;
		//if (ch->ch_buffer.kb_resid >= KMIXER_BUFSIZE / 2)
		if (ch->ch_buffer.kb_resid >= 2)
			++achans;
	}

	//return (achans == nchans ? false : true);
	return achans == 0;
}

static int16_t
kmixer_mix_buffers(struct kmixer_softc *sc)
{
	struct kmixer_chan *ch;
	struct kmixer_buffer *kb;
	int64_t sample = 0;
	int nsamples = 0, nchans = 0;

	TAILQ_FOREACH(ch, &sc->sc_chans, ch_entry) {
		kb = &ch->ch_buffer;
		++nchans;
		if (kb->kb_resid < 2)
			continue;
		sample += *(int16_t *)&kb->kb_buffer[kb->kb_rpos];
		kb->kb_resid -= 2;
		kb->kb_rpos += 2;
		if (kb->kb_rpos == KMIXER_BUFSIZE)
			kb->kb_rpos = 0;
		++nsamples;
	}

#if 0
	sample /= nsamples;
#endif

	if (sample > SHRT_MAX)
		sample = SHRT_MAX;
	else if (sample < SHRT_MIN)
		sample = SHRT_MIN;

	return (int16_t)sample;
}

static void
kmixer_write_buffers(struct kmixer_softc *sc, int16_t sample)
{
	struct uio uio;
	struct iovec iovec;

	memset(&uio, 0, sizeof(uio));
	memset(&iovec, 0, sizeof(iovec));

	uio.uio_iovcnt = 1;
	uio.uio_resid = sizeof(sample);
	uio.uio_rw = UIO_WRITE;
	uio.uio_iov = &iovec;
	UIO_SETUP_SYSSPACE(&uio);
	iovec.iov_len = sizeof(sample);
	iovec.iov_base = &sample;

	cdev_write(sc->sc_audiodev, &uio, FREAD|FWRITE);
}

static void
kmixer_auconv_init(struct kmixer_softc *sc)
{
	struct audio_params *params =
	    &kmixer_audiodev_formats[sc->sc_paramsidx].params;
	int err;

	sc->sc_formats[0].driver_data = NULL;
	sc->sc_formats[0].mode = AUMODE_PLAY | AUMODE_RECORD;
	sc->sc_formats[0].encoding = params->encoding;
	sc->sc_formats[0].validbits = params->validbits;
	sc->sc_formats[0].precision = params->precision;
	sc->sc_formats[0].channels = params->channels;
	switch (params->channels) {
	case 1:
		sc->sc_formats[0].channel_mask = AUFMT_MONAURAL;
		break;
	case 2:
		sc->sc_formats[0].channel_mask = AUFMT_STEREO;
		break;
	case 4:
		sc->sc_formats[0].channel_mask = AUFMT_SURROUND4;
		break;
	case 6:
		sc->sc_formats[0].channel_mask = AUFMT_DOLBY_5_1;
		break;
	default:
		sc->sc_formats[0].channel_mask = AUFMT_UNKNOWN_POSITION;
		break;
	}
	sc->sc_formats[0].frequency_type = 1;
	sc->sc_formats[0].frequency[0] = params->sample_rate;

	err = auconv_create_encodings(sc->sc_formats, 1, &sc->sc_encodings);
	if (err) {
		aprint_error_dev(sc->sc_dev, "couldn't create encodings\n");
		return;
	}
}

static int
kmixer_auconv_setinfo(struct kmixer_chan *ch, struct audio_info *ai)
{
#define SPECIFIED(x)	(x != ~0)
	struct kmixer_softc *sc = ch->ch_softc;
	struct audio_params play;
	stream_filter_list_t pfilters;
	int err;

	if (ai->mode & (AUMODE_PLAY|AUMODE_PLAY_ALL)) {
		memset(&play, 0, sizeof(play));
		if (SPECIFIED(ai->play.sample_rate))
			play.sample_rate = ai->play.sample_rate;
		if (SPECIFIED(ai->play.channels))
			play.channels = ai->play.channels;
		if (SPECIFIED(ai->play.precision)) {
			play.precision = ai->play.precision;
			play.validbits = ai->play.precision;	/* XXX */
		}
		if (SPECIFIED(ai->play.encoding)) 
			play.encoding = ai->play.encoding;

		err = audio_check_params(&play);
		if (err)
			return err;

		memset(&pfilters, 0, sizeof(pfilters));
		pfilters.append = kmixer_filter_append;
		pfilters.prepend = kmixer_filter_prepend;
		pfilters.set = kmixer_filter_set;

		err = auconv_set_converter(sc->sc_formats, 1, AUMODE_PLAY,
		    &play, true, &pfilters);
		if (err < 0) {
			aprint_error_dev(sc->sc_dev,
			    "couldn't set stream converter\n");
			return EINVAL;
		}

		ch->ch_pfil = pfilters;
	}

	return 0;
#undef SPECIFIED
}

static void
kmixer_filter_append(stream_filter_list_t *list,
    stream_filter_factory_t factory, const audio_params_t *param)
{
	if (list->req_size >= AUDIO_MAX_FILTERS)
		return;
	list->filters[list->req_size].factory = factory;
	list->filters[list->req_size].param = *param;
	list->req_size++;
}
     
static void
kmixer_filter_set(stream_filter_list_t *list, int i,
    stream_filter_factory_t factory, const audio_params_t *param)
{
	if (i < 0 || i >= AUDIO_MAX_FILTERS)
		return;
	list->filters[i].factory = factory;
	list->filters[i].param = *param;
	if (list->req_size <= i)
		list->req_size = i + 1;
}

static void
kmixer_filter_prepend(stream_filter_list_t *list,
    stream_filter_factory_t factory, const audio_params_t *param)
{
	if (list->req_size >= AUDIO_MAX_FILTERS)
		return;
	memmove(&list->filters[1], &list->filters[0],
	    sizeof(struct stream_filter_req) * list->req_size);
	list->filters[0].factory = factory;
	list->filters[0].param = *param;
	list->req_size++;
}

#ifdef _MODULE
MODULE(MODULE_CLASS_DRIVER, kmixer, NULL);

CFDRIVER_DECL(kmixer, DV_DULL, NULL);
extern struct cfattach kmixer_ca;
static struct cfparent audioparent = { "audiocore", "audio", -1 };
static struct cfdata kmixer_cfdata[] = {
	{
		.cf_name = "kmixer",
		.cf_atname = "kmixer",
		.cf_unit = 0,
		.cf_fstate = FSTATE_STAR,
		.cf_loc = NULL,
		.cf_flags = 0,
		.cf_pspec = &audioparent,
	},
	{ NULL }
};

static int
kmixer_modcmd(modcmd_t cmd, void *arg)
{
	int bmajor = -1, cmajor = -1;
	int err;

	switch (cmd) {
	case MODULE_CMD_INIT:
		err = config_cfdriver_attach(&kmixer_cd);
		if (err)
			return err;
		err = config_cfattach_attach("kmixer", &kmixer_ca);
		if (err) {
			config_cfdriver_detach(&kmixer_cd);
			return err;
		}
		err = config_cfdata_attach(kmixer_cfdata, 1);
		if (err) {
			config_cfattach_detach("kmixer", &kmixer_ca);
			config_cfdriver_detach(&kmixer_cd);
			return err;
		}
		err = devsw_attach("kmixer",
		    NULL, &bmajor, &kmixer_cdevsw, &cmajor);
		if (err) {
			config_cfdata_detach(kmixer_cfdata);
			config_cfattach_detach("kmixer", &kmixer_ca);
			config_cfdriver_detach(&kmixer_cd);
		}
		return 0;
	case MODULE_CMD_FINI:
		devsw_detach(NULL, &kmixer_cdevsw);
		err = config_cfdata_detach(kmixer_cfdata);
		if (err)
			return err;
		config_cfattach_detach("kmixer", &kmixer_ca);
		config_cfdriver_detach(&kmixer_cd);
		return 0;
	default:
		return ENOTTY;
	}
}
#endif
