/* **********************************************************
 * Copyright (C) 1998-2000 VMware, Inc.
 * All Rights Reserved
 * $Id: iocontrols.h,v 1.4 2003/02/16 17:42:22 bad Exp $
 * **********************************************************/


/*
 * iocontrols.h
 *
 *        The driver io controls.
 */

#ifndef _IOCONTROLS_H_
#define _IOCONTROLS_H_

#define INCLUDE_ALLOW_USERLEVEL
#define INCLUDE_ALLOW_MONITOR
#define INCLUDE_ALLOW_MODULE
#define INCLUDE_ALLOW_VMKERNEL
#include "includeCheck.h"


/*
 * Driver version.
 *
 * Increment major version when you make an incompatible change.
 * Compatibility goes both ways (old driver with new executable
 * as well as new driver with old executable).
 */

#define VMMON_VERSION		(10 << 16 | 0)
#define VMMON_VERSION_MAJOR(v)	((uint32) (v) >> 16)
#define VMMON_VERSION_MINOR(v)	((uint16) (v))


/*
 * ENOMEM returned after MAX_VMS virtual machines created
 */

#ifdef VMX86_SERVER
#define MAX_VMS 50
#else
#define MAX_VMS 24
#endif



typedef enum IOCTLCmd {
   IOCTLCMD_NONE = 200,
   IOCTLCMD_VERSION,
   IOCTLCMD_CREATE_VM,
   IOCTLCMD_BIND_VM,
   IOCTLCMD_RELEASE_VM,
   IOCTLCMD_GET_NUM_VMS,
   IOCTLCMD_INIT_VM,
   IOCTLCMD_LATE_INIT_VM,
   IOCTLCMD_RUN_VM,
   IOCTLCMD_LOOK_UP_MPN,
   IOCTLCMD_LOCK_PAGE,
   IOCTLCMD_UNLOCK_PAGE,
   IOCTLCMD_APIC_BASE,
   IOCTLCMD_IOAPIC_BASE,
   IOCTLCMD_GET_STATS,
   IOCTLCMD_SET_STATS,
   IOCTLCMD_GET_HARD_LIMIT,
   IOCTLCMD_SET_HARD_LIMIT,
   IOCTLCMD_GET_MEM_INFO,
   IOCTLCMD_SET_MEM_INFO,
   IOCTLCMD_PAE_ENABLED,
   IOCTLCMD_GET_TOTAL_MEM_USAGE,

   /*
    * Keep host-specific calls at the end so they can be undefined
    * without renumbering the common calls.
    */

#if defined(linux) || defined(__FreeBSD__) || defined(__NetBSD__)
   IOCTLCMD_SET_UID,		// VMX86_DEVEL only
   IOCTLCMD_IS_MP_SAFE,
   IOCTLCMD_GET_MHZ_ESTIMATE,
   IOCTLCMD_ALLOW_CORE_DUMP,

   IOCTLCMD_CHECK_MEMORY,

   IOCTLCMD_BROADCAST_IPI,	// SMP 2.2.8+ only

   IOCTLCMD_REGISTER_PASSTHROUGH_IO,
   IOCTLCMD_REGISTER_PASSTHROUGH_IRQ,
   IOCTLCMD_FREE_PASSTHROUGH_IO,
   IOCTLCMD_FREE_PASSTHROUGH_IRQ,
   IOCTLCMD_START_PASSTHROUGH,
   IOCTLCMD_STOP_PASSTHROUGH,
   IOCTLCMD_QUERY_PASSTHROUGH,

   IOCTLCMD_REGISTER_PERFCTR,
   IOCTLCMD_START_PERFCTR,
   IOCTLCMD_STOP_PERFCTR,
   IOCTLCMD_RELEASE_PERFCTR,

   IOCTLCMD_ALLOC_LOW_PAGES,
   IOCTLCMD_FREE_LOW_PAGES,

#ifdef VMX86_SERVER
   IOCTLCMD_ALLOC_MACHINE_MEMORY,
   IOCTLCMD_POWERON_SLACK_MEMORY,
   IOCTLCMD_ASYNC_ACTIONS,
   IOCTLCMD_FILE_OPEN,
   IOCTLCMD_FILE_CLOSE,
   IOCTLCMD_SET_MMAP_LAST,
#endif
#endif

#ifdef _WIN32
   IOCTLCMD_BEEP,
   IOCTLCMD_HARD_LIMIT_MONITOR_STATUS,	// Windows 2000 only
   IOCTLCMD_BLUE_SCREEN,	// USE_BLUE_SCREEN only
   IOCTLCMD_CHANGE_HARD_LIMIT,
   IOCTLCMD_GET_PCI_INFO,
#endif

#ifdef VMX86_SERVER
   /* bora-root/fileutils/ls needs to be recompiled if the following
      definition ever changes */
   IOCTLCMD_VMFS_GET_LENGTH = 1000,
   IOCTLCMD_VMFS_PHYS_MEM_IO,
#endif
   IOCTLCMD_ADD_WORLD_ARG,
   IOCTLCMD_LAST
} IOCTLCmd;


/*
 * Windows ioctl definitions.
 *
 * We use the IRP Information field for the return value
 * of IOCTLCMD_RUN, to be faster since it is use a lot.
 */

#ifdef _WIN32

#define FILE_DEVICE_VMX86        0x8101
#define VMX86_IOCTL_BASE_INDEX   0x801
#define VMIOCTL_BUFFERED(name) \
      CTL_CODE(FILE_DEVICE_VMX86, \
	       VMX86_IOCTL_BASE_INDEX+IOCTLCMD_##name, \
	       METHOD_BUFFERED, \
	       FILE_ANY_ACCESS)
#define VMIOCTL_NEITHER(name) \
      CTL_CODE(FILE_DEVICE_VMX86, \
	       VMX86_IOCTL_BASE_INDEX+IOCTLCMD_##name, \
	       METHOD_NEITHER, \
	       FILE_ANY_ACCESS)

#define IOCTL_VMX86_VERSION		VMIOCTL_BUFFERED(VERSION)
#define IOCTL_VMX86_CREATE_VM		VMIOCTL_BUFFERED(CREATE_VM)
#define IOCTL_VMX86_BIND_VM		VMIOCTL_BUFFERED(BIND_VM)
#define IOCTL_VMX86_RELEASE_VM		VMIOCTL_BUFFERED(RELEASE_VM)
#define IOCTL_VMX86_GET_NUM_VMS		VMIOCTL_BUFFERED(GET_NUM_VMS)
#define IOCTL_VMX86_INIT_VM		VMIOCTL_BUFFERED(INIT_VM)
#define IOCTL_VMX86_LATE_INIT_VM	VMIOCTL_BUFFERED(LATE_INIT_VM)
#define IOCTL_VMX86_RUN_VM		VMIOCTL_NEITHER(RUN_VM)
#define IOCTL_VMX86_LOOK_UP_MPN		VMIOCTL_BUFFERED(LOOK_UP_MPN)
#define IOCTL_VMX86_LOCK_PAGE		VMIOCTL_BUFFERED(LOCK_PAGE)
#define IOCTL_VMX86_UNLOCK_PAGE		VMIOCTL_BUFFERED(UNLOCK_PAGE)
#define IOCTL_VMX86_APIC_BASE		VMIOCTL_BUFFERED(APIC_BASE)
#define IOCTL_VMX86_IOAPIC_BASE		VMIOCTL_BUFFERED(IOAPIC_BASE)
#define IOCTL_VMX86_GET_STATS		VMIOCTL_BUFFERED(GET_STATS)
#define IOCTL_VMX86_SET_STATS		VMIOCTL_BUFFERED(SET_STATS)
#define IOCTL_VMX86_GET_HARD_LIMIT	VMIOCTL_BUFFERED(GET_HARD_LIMIT)
#define IOCTL_VMX86_SET_HARD_LIMIT	VMIOCTL_BUFFERED(SET_HARD_LIMIT)
#define IOCTL_VMX86_GET_MEM_INFO	VMIOCTL_BUFFERED(GET_MEM_INFO)
#define IOCTL_VMX86_SET_MEM_INFO	VMIOCTL_BUFFERED(SET_MEM_INFO)
#define IOCTL_VMX86_PAE_ENABLED		VMIOCTL_BUFFERED(PAE_ENABLED)

#define IOCTL_VMX86_BEEP		VMIOCTL_BUFFERED(BEEP)
#define IOCTL_VMX86_HARD_LIMIT_MONITOR_STATUS	VMIOCTL_BUFFERED(HARD_LIMIT_MONITOR_STATUS)
#define IOCTL_VMX86_CHANGE_HARD_LIMIT   VMIOCTL_BUFFERED(CHANGE_HARD_LIMIT)
#define IOCTL_VMX86_GET_PCI_INFO        VMIOCTL_BUFFERED(GET_PCI_INFO)
#ifdef USE_BLUE_SCREEN
#define IOCTL_VMX86_BLUE_SCREEN		VMIOCTL_BUFFERED(BLUE_SCREEN)
#endif

#define IOCTL_VMX86_GET_VM_LIST		VMIOCTL_BUFFERED(GET_VM_LIST)
#define IOCTL_VMX86_GET_VM_INFO		VMIOCTL_BUFFERED(GET_VM_INFO)
#define IOCTL_VMX86_SET_VM_INFO		VMIOCTL_BUFFERED(SET_VM_INFO)
#define IOCTL_VMX86_GET_TOTAL_MEM_USAGE	 VMIOCTL_BUFFERED(GET_TOTAL_MEM_USAGE)

#endif


/*
 * Structure for GET_STATS and SET_STATS.
 */

typedef struct {
  uint32 userLockedPages;	/* current count of pages locked by app */
  uint32 maxUserLockedPages;	/* max over time of pages locked by app */
  uint32 monitorLockedPages;	/* current count of pages locked by monitor */
  uint32 maxMonitorLockedPages;	/* max over time of pages locked by monitor */
} VMX86Stats;


/*
 * Return codes from lock page if lock page failed.
 *
 *    PAGE_LOCK_FAILED              The host refused to lock a page.
 *    PAGE_LOCK_LIMIT_EXCEEDED      We have reached the limit of locked
 *                                  pages for all VMs
 */

#define PAGE_LOCK_FAILED                  0
#define PAGE_LOCK_LIMIT_EXCEEDED         -1


/*
 * Flags sent into APICBASE ioctl
 */

#define APIC_FLAG_DISABLE_NMI       0x00000001
#define APIC_FLAG_PROBE             0x00000002
#define APIC_FLAG_FORCE_ENABLE      0x00000004


/*
 * Structure sent into REGISTER_PERFCOUNTERS ioctl
 */

#define PERFCTR_MAX_COUNTERS        2
#define PERFCTR_INVALID_EVENT       0
#define PERFCTR_INVALID_IRQ         -1

typedef struct PerfCtrInfo {
   uint32 eventNum;
   uint32 samplingRate;
} PerfCtrInfo;

typedef struct PerfCtrRegisterArgs {
   PerfCtrInfo counterInfo[PERFCTR_MAX_COUNTERS];
   int irq;
   Bool totalOnly;
} PerfCtrRegisterArgs;


#ifdef VMX86_SERVER
#define FILE_LENGTH_ARGS_NAMELEN 256
/* Structure to get file length */
typedef struct VMFSFileLengthArgs {
   char fileName[FILE_LENGTH_ARGS_NAMELEN];
   long long length;
} VMFSFileLengthArgs;
#endif

typedef struct VMGetPCIInfoArgs {
   int busNumber;
   int deviceNumber;
   int functionNumber;
} VMGetPCIInfoArgs;

typedef struct VMGetPCIInfoResult {
   int bus;
   int slot;
   int vendorID;
   int deviceID;
   int baseClass;
   int subClass;
   int interruptLine;
   int interruptVector;
} VMGetPCIInfoResult;

#endif // ifndef _IOCONTROLS_H_
