################################################################################
#    iGMT: Interactive Mapping of Geoscientific Datasets.                      #
#               Easy access to GMT via a Tcl/Tk GUI                            #
#                                                                              #
#    Copyright (C) 1998 - 2000 Thorsten W. Becker, Alexander Braun             #
#                                                                              #
#    This program is free software; you can redistribute it and/or modify      #
#    it under the terms of the GNU General Public License as published by      #
#    the Free Software Foundation; either version 2 of the License, or         #
#    (at your option) any later version.                                       #
#                                                                              #
#    This program is distributed in the hope that it will be useful,           #
#    but WITHOUT ANY WARRANTY; without even the implied warranty of            #
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             #
#    GNU General Public License for more details.                              #
#                                                                              #
#    In addition, iGMT is not to be used for any military purposes or          #
#    for any other activities that are against the spirit as expressed         #
#    in the Student Pugwash Pledge (www.spusa.org/pugwash/pledge.html).        #
#                                                                              #
#    You should have received a copy of the GNU General Public License         #
#    along with this program; see the file COPYING.  If not, write to          #
#    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,          #
#    Boston, MA 02111-1307, USA.                                               #
#                                                                              #
################################################################################

################################################################################
# igmt_plotting.tcl -- routines used for plotting
#
# part of the iGMT package
#
################################################################################


################################################################################
# main procedures for plotting
# produce the GMT script and thus the postscript file
################################################################################

proc mk_ps {} {
    global west east south north proj ps_filename \
	papersize pscoast_color  batchfile no_convert_complaints  \
	portrait ticks  headermessage   raster_bounds \
	river boundary resolution shoreline batcherr \
	pscoast_add raster_dataset plot_title shell_to_use \
	retval poly_data poly_color poly_symbol \
	poly_linewidth poly_symbol_size poly_parameter \
	raster_data raster_colormap \
	gmtbins temp_grid_filename verbose \
	temp_shade_filename topocolor raster_resolution \
	lon0 lat0 use_img2latlon_batch \
	custom_projection legend title_font_size \
	title_font saved polygon_dataset  \
	gridlines mercatorlimit annotation \
	psc_linewidth batch_pid colormap \
	shading mapscale  ps_offset \
	use_pstext_for_title show_gmt_logo  \
	gmt_version igmt_version  wet_areas  \
	igmt_root gmt_version_boundary our_awk \ 
	

    set headermessage "iGMT: Creating postscript file."

    # this routine works by 
    # a) defining strings such as the projection
    # b) assembling a sequence of strings calling the GMT commands within the script
    # c) writing and executing the script


    # initialize the strings to be formed later
    set scale_unit_string "" 

    
    # use psvelomeca or the separate programs psmeca 
    # and psvelo for plotting
    if { $gmt_version < $gmt_version_boundary } {
	set psmeca    psvelomeca
	set psvelo    psvelomeca
    } else { # please keep in mind that psvelo was fairly buggy for a while
       	set psmeca "psmeca -o"
	set psvelo psvelo
    }

    ################################################################################
    # calculate the size of the page and other geometrical stuff
   
    if { $portrait  } { set maxmap $papersize(1)  } else { set maxmap $papersize(2) }
    set span(1) [ expr ($east  - $west) ]
    set span(2) [ expr ($north - $south) ]
    set mean(1) [ expr ($east  + $west)/2.0 ]
    set mean(2) [ expr ($north + $south)/2.0 ]
    # standard parallels 
    foreach i { 1 2 } {
	set std_par($i) [ expr $south+0.33*$span($i)]
    }
    # distance of name tags from location to be named
    set tagdist [ expr $span(1)/1000. ]
    foreach i { 1 2 } { set interval($i) [ expr $span($i)/($ticks($i)+1) ] }
    foreach i { 1 2 } { 
	if { $interval($i) > 0.5 } { 
	    set i_format($i) 0
	} elseif { $interval($i) > 0.05 } { 
	    set i_format($i) 1 
	} elseif { $interval($i) > 0.005 } { 
	    set i_format($i) 2 
	} elseif { $interval($i) > 0.0005 } { 
	    set i_format($i) 3
	} else {
	    set i_format($i) 4
	}
	set i_format($i) "%0[ format 3.%ilf $i_format($i) ]"
	# spacing of annotation around map
	set aspacing($i) [ format $i_format($i) [ expr $interval($i) ]]
	# want it to be even numbers
	if { ($aspacing($i) > 1)  && ([ expr $aspacing($i) % 2 ] != 0) } {
	    incr aspacing($i)  
	}
	# spacing of grid
	set gspacing($i) [ format %5.3lf  [ expr [ format %lf $aspacing($i)]/2.0 ] ]
	# spacing of black/white intervals
	set fspacing($i) [ format %5.3lf  [ expr [ format %lf $aspacing($i)]/2.0 ] ]
    }

    
    # mapscale hack, originally from Simon Mc Clusky

    set mscale 10.0
    if  { $span(1) > 1.0   } { set mscale 20.0   } 
    if  { $span(1) > 2.0   } { set mscale 50.0   } 
    if  { $span(1) > 5.0   } { set mscale 100.0  } 
    if  { $span(1) > 10.0  } { set mscale 200.0  } 
    if  { $span(1) > 20.0  } { set mscale 500.0  } 
    if  { $span(1) > 50.0  } { set mscale 1000.0 } 
    if  { $span(1) > 100.0 } { set mscale 2000.0 } 
    if  { $span(1) > 200.0 } { set mscale 5000.0 } 

    # create the -B string for boundary annotation and grid lines
    

    if { $use_pstext_for_title } { set tmp_plot_title "" } else { set tmp_plot_title $plot_title }
    if { ($gridlines )&&($annotation) } {
	set frame_box "-Ba$aspacing(1)f$fspacing(1)g$gspacing(1)/a$aspacing(2)f$fspacing(2)g$gspacing(2)"
    } elseif { ($gridlines == 0)&&($annotation) } {
	set frame_box "-Ba$aspacing(1)f$fspacing(1)/a$aspacing(2)f$fspacing(2)"
    } elseif { ($gridlines )&&($annotation == 0) } {
	set frame_box "-Bg$gspacing(1)/g$gspacing(2)"
    } else { set frame_box "-B" }
    # add title, even if empty
    set frame_box "[ format %s:.\"$plot_title\": $frame_box ]"
    # add type of annotation
    if { $annotation == 1 } { 
	set frame_box "[ format %sWESN $frame_box  ]"
    } elseif { $annotation == 2 } {
	set frame_box "[ format %sWeSn $frame_box  ]"
    }


    set proj(2) [ expr ($span(1)>$span(2))?($maxmap/$span(1)):($maxmap/$span(2)) ]

    
    #
    # colors for sea, land and rivers
    #
    set sea_c   "-S[ format %03i/%03i/%03i $pscoast_color(2,1) $pscoast_color(2,2) $pscoast_color(2,3)]"
    set land_c  "-G[ format %03i/%03i/%03i $pscoast_color(1,1) $pscoast_color(1,2) $pscoast_color(1,3)]"
    set river_c "[ format %03i/%03i/%03i   $pscoast_color(3,1) $pscoast_color(3,2) $pscoast_color(3,3)]"
    
    ################################################################################
    # deal with the different projections and try to fit the map to the paper
    # adjust options accordingly
    # set the plotting region (different from the data set region for limited datasets)
    # here because some projection redefine the corners
    #
    
    switch $proj(1) {
	"B"  { # Albers conic equal-area
	    set prp "-JB[ format %03f/%03f/%03f/%03f/%03f $mean(1) $mean(2) $std_par(1) $std_par(2) [ expr 0.8*$maxmap ]]"
	}
	"D"  { # Equidistant Conic projection
	    set prp "-JD[ format %03f/%03f/%03f/%03f/%03f $mean(1) $mean(2) $std_par(1) $std_par(2) [ expr 0.8*$maxmap ]]"
	}
	"Y"  { # General Cylindrical Projections
	    puts "iGMT: General Cylindrical Projection: using $lat0 as reference latitude"
	    set prp "-JY[ format %03f/%03f/%03f $mean(1) $lat0 [ expr 0.8*$maxmap ]]"
	}
	"C"  { # Cassini
	    set prp "-JC[ format %03f/%03f/%03f  $mean(1) $mean(2) [ expr 0.8*$maxmap ]]"
	}
	"M"  { # Mercator
	    set prp "-JM[ format %03f/%03f/%03f $mean(1) $lat0 [ expr 0.8*$maxmap ]]" 
	    puts "iGMT: Mercator projection, using $lat0 as reference latitude"
	    if { $north > $mercatorlimit(1) } { 
		set north $mercatorlimit(1)
		dialog .d {Projection warning} \
		    "Since the Mercator projection was chosen, iGMT limits the northern boundary to $mercatorlimit(1)."  warning 0 {OK} 
	    }
	    if { $south < $mercatorlimit(2) } { 
		set south $mercatorlimit(2)
		dialog .d {Projection warning} \
		    "Since the Mercator projection was chosen, iGMT limits the southern boundary to $mercatorlimit(2)."  warning 0 {OK} 
	    }
	}
	"X"  { 	# linear projection
	    set prp "-JX[ format %03fd [ expr 0.8*$maxmap ]]" 
	}
	"Q"  { 	# equidistant cylindrical
	    set prp "-JQ$lon0/[format %03f [ expr 0.8*$maxmap ]]"
	}
	"A" { 	# azimuthal projections, Lambert
	    set prp "-JA[ format %03f/%03f/%03f $mean(1) $mean(2) [ expr 0.8*$maxmap ]]" 
	    if { ( $west == -180 ) && ( $east == 180 ) && ( $south == -90 ) && ( $north == 90 ) } {
		set west 360.0
		set east 0.0
	    } else { 
		# choose the rectangle technique
		set reg "-R$west/$south/$east/[ format %sr $north]"
	    }
	}
	"S" { 	# azimuthal projections, Stereographic
	    if { $mean(2) > 0.0 } {
		set prp "-JS[ format %0lf/90/%03f $mean(1) [ expr 0.8*$maxmap ]]" 
		if { $span(2) > 85.0 } {
		    set south [ expr $north - 90.0 ]
		    if { $south < 0.0 } { set south 0.0 }
		    dialog .d {Projection warning} \
			"Reducing the southern boundary to $south for the stereographic projection." \
			warning 0 {OK} 
		}
	    } else {
		set prp "-JS[ format %0lf/-90/%03f $mean(1) [ expr 0.8*$maxmap ]]" 
		if { $span(2) > 85.0 } {
		    set north [ expr $north - 90.0 ]
		    if { $north > 0.0 } { set north 0.0 }
		    dialog .d {Projection warning} \
			"Reducing the northern boundary to $north for the stereographic projection." \
			warning 0 {OK} 
		}
	    }
	}
	"G" -
	"E" { # orthographic and azimuthal equidistant
	    set prp "-J$proj(1)$lon0/$lat0/[ format %03f [ expr 0.8*$maxmap ]]" 
	    set east 360.0 ; set west 0.0
	    set south -90.0 ; set north 90.0
	}
	"H" -
	"N" -
	"I" -
	"W" { # Hammer, Mollweide, Robinson and Sinusoidal projections
	    set prp "-J$proj(1)$lon0/[format %03f [ expr 0.8*$maxmap ]]" 
	}
	"C" {
	    set prp $custom_projection
	}
	default { 
	    error "Projection $proj(1) is not implemented!"  
	    set prp "-JX[format %03f [ expr 0.8*$maxmap ]]" 
	   
	}
    }
    ################################################################################@
    # set the plotting region
    set region "[ format -R%g/%g/%g/%g $west $east $south $north]"

    ################################################################################
    # obtain raster data set specific settgins such as
    # the data_region variable: adjust data region for grdimage for different raster data sets
    #                if raster data sets cover whole earth, will be the same as $region
    # the increment (resolution)
    # and the resample on/off switch
    # routine is in igmt_datasets.tcl
    #
    obtain_raster_data_specs data_region inc resample $raster_dataset $raster_resolution \
				    span  raster_bounds $west $east $south $north 

    ################################################################################
    # set some pscoast options
    #
    # this is the switch to activate the pscoast line
    set addsomepscoast 0
    foreach i { 1 2 3 4 5 } { set addsomepscoast [ expr $addsomepscoast + $river($i) ] }
    set addsomepscoast [ expr $addsomepscoast + $shoreline ]
    foreach i { 1 2  } { set addsomepscoast [ expr $addsomepscoast + $boundary($i) ] }

    if { $shoreline  } { 
	set sl "-W$psc_linewidth(3)/[ format %03i $pscoast_color(4,1)]/[ format %03i $pscoast_color(4,2)]/[ format %03i $pscoast_color(4,3)]" 
    } else { set sl "" }
    if { $river(1)  } { 
	set river_polygons "-I1/$psc_linewidth(1)/$river_c" 
    } else { set river_polygons "" }
    if { $river(2)  } { 
	set river_polygons "$river_polygons -I2/$psc_linewidth(1)/$river_c " 
    }
    if { $river(3)  } { 
	set river_polygons "$river_polygons -I3/$psc_linewidth(1)/$river_c " 
    }
    if { $river(4)  } { 
	set river_polygons "$river_polygons -I4/$psc_linewidth(1)/$river_c " 
    }
    if { $river(5) } {
	set river_polygons "$river_polygons -S$river_c" 
    }
    if { $boundary(1)  } { 
	set natbound_polygons "-N1/$psc_linewidth(2)/000/000/000" 
    } else { set natbound_polygons "" }
    if { $boundary(2)  } { set natbound_polygons "$natbound_polygons -N2/$psc_linewidth(2)/000/000/000 " }

    # create the resolution string for pscoast

    if { $resolution == "cred" } {
	set resolution_string "-Dc -A50000"
    } else {
	set resolution_string "-D$resolution"
    }
   
    if { $portrait  } { set port "-P" } else { set port "" }
    # 
    # adjust offset for some datasets
    if { ($portrait) && ($legend) && ($raster_dataset==10) } {
	set offsets "-X$ps_offset(1) -Y[ format %g [ expr $ps_offset(2)+1 ]]"
    } else {
	set offsets "-X$ps_offset(1) -Y$ps_offset(2)"
    }


    ################################################################################
    # produce the gmt script strings
    # by creating an array of strings that correspond to lines in the final script
    #

    # header
    set nrs 1
    set nrs [ add_to_script  $nrs gmtstring "\#!$shell_to_use" ]
    set nrs [ draw_script_line $nrs gmtstring ]
    set nrs [ add_to_script  $nrs gmtstring "\# Script produced with iGMT, version $igmt_version"]
    set nrs [ draw_script_line $nrs gmtstring ]
    set nrs [ add_to_script  $nrs gmtstring "\# All commands are in ksh resp. bash syntax:"]
    set nrs [ add_to_script  $nrs gmtstring "\# \"\#\" leads a comment line, \"\\\" means line continuation"]
    set nr [ add_to_script  $nrs gmtstring "\# and variables are declared as \"a=1\" and referenced as \"echo \$a\""]
    set nrs [ draw_script_line $nrs gmtstring ]
    
    
    ################################################################################
    # begin the  block of shell variables that are set within the 
    # script

    set nrs [ add_to_script  $nrs gmtstring "\# The following variables are used for all the GMT commands."]
    set nrs [ add_to_script  $nrs gmtstring "\# Modify the settings here, further down all references in the script"]
    set nrs [ add_to_script  $nrs gmtstring "\# such as \"\$region\" are replaced by the shell."]

    if { $gmtbins != "" } { 
	set nrs [ add_to_script  $nrs gmtstring "\# Location of the GMT binaries: "]
	set nrs [ add_to_script  $nrs gmtstring "gmtbin='$gmtbins'"]
    }

    set nrs [ add_to_script  $nrs gmtstring "\# Geographical variables: "]
    #  this sets the variable for the projection
    set nrs [ add_to_script  $nrs gmtstring "\# Set the projection "]
    set nrs [ add_to_script  $nrs gmtstring "projection='$prp'"]

    # this sets the variable for the region
    set nrs [ add_to_script  $nrs gmtstring "\# Set the plotting region (west/east/south/north boundaries)"]
    set nrs [ add_to_script  $nrs gmtstring "region=$region"]
    # set the data region
    set nrs [ add_to_script  $nrs gmtstring "\# Set the data region (might be different from  plotting region)"]
    set nrs [ add_to_script  $nrs gmtstring "data_region=$data_region"]
    if { $inc != "" } {
	# set the raster data resolution
	set nrs [ add_to_script  $nrs gmtstring "\# Set the raster data resolution"]
	set nrs [ add_to_script  $nrs gmtstring "raster_inc=$inc"]
    }

    # annotation
    set nrs [ add_to_script  $nrs gmtstring "\# type of annotation "]
    set nrs [ add_to_script  $nrs gmtstring "annotation=$frame_box"]
    


    set nrs [ add_to_script  $nrs gmtstring "\# Postscript output: "]
    # this sets the variable for the output file
    set nrs [ add_to_script  $nrs gmtstring "\# Set the name of the output (postscript) file "]
    set nrs [ add_to_script  $nrs gmtstring "ps_filename='$ps_filename'"]
    # ps offset setting
    set nrs [ add_to_script  $nrs gmtstring "\# Set the X and Y offset on the postscript plot (in inches)"]
    set nrs [ add_to_script  $nrs gmtstring "offsets='$offsets'"]
    # portrait/landscape swith
    set nrs [ add_to_script  $nrs gmtstring "\# set this to -P for portrait mode, else landscape"]
    set nrs [ add_to_script  $nrs gmtstring "portrait='$port'"]
    # verbosity
    set nrs [ add_to_script  $nrs gmtstring "\# set this to -V for verbose GMT output, else leave blank for quite mode"]
    set nrs [ add_to_script  $nrs gmtstring "verbose='$verbose'"]
    

    
    if { $raster_dataset != 1 } {
	# raster data set variable that are not used if only pscoast coverage selected 
	set nrs [ add_to_script  $nrs gmtstring "\# raster data set specific: "]
	# this sets the variable for the colormap
	set nrs [ add_to_script  $nrs gmtstring "\# Set the name of the colormap used for imaging a grid file"]
	set nrs [ add_to_script  $nrs gmtstring "colormap='$colormap'"]
	# this sets the variable for the output grd file
	set nrs [ add_to_script  $nrs gmtstring "\# Set the name of the temporary grd file output"]
	set nrs [ add_to_script  $nrs gmtstring "temp_grid_filename='$temp_grid_filename'"]
	# this sets the variable for the shade output
	set nrs [ add_to_script  $nrs gmtstring "\# Set the name of the temporary shade file output"]
	set nrs [ add_to_script  $nrs gmtstring "temp_shade_filename='$temp_shade_filename'"]
    }
    

    



    ################################################################################
    # now GMT commands
    
    
    set nrs [ draw_script_line $nrs gmtstring ]
    set nrs [ add_to_script  $nrs gmtstring "\# GMT commands follow"]
    set nrs [ draw_script_line $nrs gmtstring ]
    
    ################################################################################
    # first the RASTER DATA SETS
    ################################################################################

    # all raster data sets besides 'pscoast only' (which is of course not really a
    # raster data set) need another call to pscoast to add shoreline etc.
    if { $raster_dataset != 1 } { set wantpscoast 1 } else { set wantpscoast 0 }


    

    # new philosophy: always create a temporary grid file that other procedures
    # such as 'create grid image' and 'create colormap' operate on
    # has advantages in terms of speed, too


    switch $raster_dataset {
	"1" { # only pscoast features are plotted, i.e. land and sea are shown in a different color
	    puts "iGMT: Invoking pscoast."
	    # this is the syntax for adding a line to the script
	    set nrs [ add_to_script  $nrs gmtstring "\# Use pscoast to plot a map with different colors for land and sea"]
	    # do it like this to add a long line 
	    set line "\$gmtbin/pscoast  $sl  \$region \$projection $resolution_string $natbound_polygons $river_polygons \\\n\t"
	    set line "$line $sea_c $land_c \$portrait \\\n\t"
	    set nrs [ add_to_script  $nrs gmtstring "$line -K $pscoast_add  \$offsets > \$ps_filename" ]
	    # we do not need another pscoast call
	    set wantpscoast 0
	}
	"2" { # ETOPO5 topography and bathymetry
	    # create strings for GMT version dependent commands such as psscale with -B option
	    version_dependent_strings   "m"  scale_unit_string 
	    puts "iGMT: Plotting ETOPO5, resolution $inc, max is $raster_bounds(2,6) m"
	    set nrs [ add_to_script  $nrs gmtstring \
			  "\# Create a temporary grid file from the ETOPO5 data set." ]
	    set nrs [ add_to_script  $nrs gmtstring \
			  "\$gmtbin/grdraster 1 -G\$temp_grid_filename \$verbose \$data_region \$projection  \\\n\t\$raster_inc" ]
	    # plot the temporary grid file
	    set nrs [ image_temp_grid $nrs gmtstring $shading $verbose $legend $papersize(1) $maxmap\
			  $scale_unit_string "-C\$colormap" 1 ]

	}
	"3" { # Smith & Sandwell /GTOPO30 data set 
	    puts "iGMT: Smith & Sandwell/GTOPO30, resolution $inc, max is $raster_bounds(3,6) m "
	    # create strings for GMT version dependent commands such as psscale with -B option
	    version_dependent_strings   "m"  scale_unit_string 
	    set nrs [ add_to_script  $nrs gmtstring \
			  "\# Plotting GTOPO30 topography " ]
	    if { $use_img2latlon_batch } { # use our own script for circumventing the absence of img2latlongrd
       		set nrs [ add_to_script  $nrs gmtstring \
			      "\# Create a temporary grid file using the img2latlongrd.batch script" ]
		set nrs [ add_to_script  $nrs gmtstring \
			      "\# pipe the script's output to the trash so that it doesn't confuse iGMT" ]
		set line "$igmt_root/img2latlongrd.batch $raster_data(3) \\\n\t"
		set line "$line \$data_region \$raster_inc \$temp_grid_filename -S1 \$gmtbin > /dev/null" 
		set nrs [ add_to_script  $nrs gmtstring $line ]
			      
	    } else {
		set nrs [ add_to_script  $nrs gmtstring \
			      "\# Create a temporary grid file using img2latlongrd" ]
		set nrs [ add_to_script  $nrs gmtstring \
			      "\$gmtbin/img2latlongrd $raster_data(3) \$raster_inc \$data_region -G\$temp_grid_filename -T1 -S1" ]
	    }
	    # plot the temporary grid file
	    set nrs [ image_temp_grid $nrs gmtstring $shading $verbose \
			  $legend  $papersize(1) $maxmap $scale_unit_string  "-C\$colormap" 1 ]
	}
	"4" { # seafloor age 
	    puts  "iGMT: Seafloor age, resolution fixed to avoid extrapolation"
	    # create strings for GMT version dependent commands such as psscale with -B option
	    version_dependent_strings   "Ma"  scale_unit_string 
	    # create temporary grid file
	    set nrs [ create_temp_grid $nrs gmtstring "sea-floor age" \
			  $raster_data(4)  $data_region $resample $inc \
			  $temp_grid_filename $region 1 ]
	    # plot the temporary grid file
	    set nrs [ image_temp_grid $nrs gmtstring $shading $verbose \
			  $legend $papersize(1) $maxmap $scale_unit_string  "-C\$colormap" 1 ]
	    # pscoast land coverage 
	    set nrs [ add_to_script  $nrs gmtstring "\# Use pscoast for land overlay" ]
	    set line "\$gmtbin/pscoast  $sl  \$region \$projection $resolution_string"
	    set line "$line $natbound_polygons $river_polygons \$portrait  \\\n\t"
	    set nrs [ add_to_script  $nrs gmtstring \
			  "$line -O -K $land_c $pscoast_add \$verbose >> \$ps_filename" ]

	} 
	"5" { 	# free air gravity anomalies on sea
	    puts "iGMT: Freeair gravity, resolution $inc, max is $raster_bounds(5,6) m"
	    # create strings for GMT version dependent commands such as psscale with -B option
	    version_dependent_strings   "mGal"  scale_unit_string 
	    set nrs [ add_to_script  $nrs gmtstring "\# Plotting seafloor gravity" ]

	    # comment this out and the stuff below in for the grdcut version
	    if { $use_img2latlon_batch } { # use our own script for circumventing the absence of img2latlongrd
		set nrs [ add_to_script  $nrs gmtstring "\# Create a grid file using the img2latlongrd.batch script" ]
		set nrs [ add_to_script  $nrs gmtstring \
			      "\# pipe the script's output to the trash so that it doesn't confuse iGMT" ]
		set line "$igmt_root/img2latlongrd.batch $raster_data(5)  \\\n\t"
		set line "$line \$data_region \$raster_inc \$temp_grid_filename -S0.1 \$gmtbin  > /dev/null"
		set nrs [ add_to_script  $nrs gmtstring $line ]
	    } else { # use the old img2latlongrd
		set nrs [ add_to_script  $nrs gmtstring "\# Create a grid file using img2latlongrd" ]
		set nrs [ add_to_script  $nrs gmtstring \
			  "\$gmtbin/img2latlongrd $raster_data(5)  \\\n\t$raster_inc \$data_region -G\$temp_grid_filename -T1" ]
	    }
	    # uncommment this for the grdcut version where you access a fixed resolution grdfile that has
	    # been produced once and for all
	    #	    
	    #	    set nrs [ add_to_script  $nrs gmtstring "\# Create a grid file using grdcut" ]
	    #	    set nrs [ add_to_script  $nrs gmtstring "\$gmtbin/grdcut $freeair_grav_data $data_region \$verbose -G\$temp_grid_filename" ]

	    # plot the temporary grid file
	    set nrs [ image_temp_grid $nrs gmtstring $shading $verbose \
			  $legend $papersize(1) $maxmap $scale_unit_string  "-C\$colormap" 1 ]

	    # pscoast land coverage 
	    set nrs [ add_to_script  $nrs gmtstring "\# Use pscoast for land overlay" ]
	    set line "\$gmtbin/pscoast  $sl  \$region \$projection "
	    set line "$line $resolution_string $natbound_polygons $river_polygons \$portrait  \\\n\t"
	    set nrs [ add_to_script  $nrs gmtstring \
			  "$line -O -K $land_c $pscoast_add \$verbose >> \$ps_filename" ]

	}
	"6" { # geoid data from EGM360 geopotential model
            puts "iGMT: EGM360 geoid, resolution  $inc, max is $raster_bounds(6,6) m"
	    version_dependent_strings   "m"  scale_unit_string 
	    set nrs [ create_temp_grid $nrs gmtstring "geoid" \
			  $raster_data(6)  $data_region $resample $inc \
			  $temp_grid_filename $region 1 ]
	    set nrs [ image_temp_grid $nrs gmtstring $shading $verbose \
			  $legend $papersize(1) $maxmap $scale_unit_string  "-C\$colormap" 1 ]
	} 

	"7" { 	# custom raster data file
	    puts "iGMT: Custom raster data file"
	    puts "iGMT: $raster_data(7)"
	    puts "iGMT: resolution $inc"
	    # create strings for GMT version dependent commands such as psscale with -B option
	    version_dependent_strings   "custom"  scale_unit_string 
	    set nrs [ create_temp_grid $nrs gmtstring "custom raster data" \
			  $raster_data(7)  "" $resample $inc \
			  $temp_grid_filename $region 1 ]

	    set nrs [ image_temp_grid $nrs gmtstring $shading $verbose \
			  $legend $papersize(1) $maxmap $scale_unit_string  "-C\$colormap" 1 ]

	    # comment this out if you want land coverage	    
	    # set nrs [ add_to_script  $nrs gmtstring "\# Use pscoast for land coverage 
	    # and features such as national boundaries." ]
	    # set line "$pscoast  $sl  \$region \$projection $resolution_string $natbound_polygons $river_polygons \$portrait  \\\n\t"
	    # set nrs [ add_to_script  $nrs gmtstring "$line -O -K $land_c $pscoast_add \$verbose >> \$ps_filename" ]

	}
	"8" { #Laske and Masters (1997) sediment thickness 1 by 1 degree grid
            puts  "iGMT: sediment thickness, resolution  $inc, max is $raster_bounds(8,6) m"
	    set resample 1
            version_dependent_strings   "km"  scale_unit_string
	    set nrs [ create_temp_grid $nrs gmtstring "sediment thickness" \
			  $raster_data(8)  $data_region $resample $inc \
			  $temp_grid_filename $region 1 ]
	    set nrs [ image_temp_grid $nrs gmtstring $shading $verbose \
			  $legend $papersize(1) $maxmap $scale_unit_string  "-C\$colormap" 1 ]

        }
	"9" { 	# global free-air gravity anomalies from the geopotential model EGM360
            puts "iGMT: EGM360 freeair gravity, resolution  $inc, max is $raster_bounds(9,6) m"
	    # create strings for GMT version dependent commands such as psscale with -B option
	    version_dependent_strings   "mGal"  scale_unit_string 
	    set nrs [ create_temp_grid $nrs gmtstring "freeair gravity from geoid" \
			  $raster_data(9)  $data_region $resample $inc \
			  $temp_grid_filename $region 1 ]
	    set nrs [ image_temp_grid $nrs gmtstring $shading $verbose \
			  $legend $papersize(1) $maxmap $scale_unit_string  "-C\$colormap" 1 ]
	}
	"10" { # ETOPO5 bathymetry and GSHAP peak ground acceleration
	    # create strings for GMT version dependent commands such as psscale with -B option
	    version_dependent_strings   "m/s@+2@+"  scale_unit_string 
	    # check resolution for resampling
	    puts "iGMT: Plotting ETOPO5 and GSHAP PGA, resolution $inc, max is $raster_bounds(10,6) m"
	    # create bathymetry backdrop
	    set nrs [ add_to_script  $nrs gmtstring \
			  "\# Create a temporary grid file from the ETOPO5 data set." ]
	    # adjust the raster resolution for ETOPO5
	    set  tmpinc [ grdraster_adjust $raster_resolution $raster_bounds(2,6) "ETOPO5"  $west $east $south $north ]
	    set nrs [ add_to_script  $nrs gmtstring \
		  "\$gmtbin/grdraster 1 -G\$temp_grid_filename \$verbose \$data_region \$projection  \\\n\t$tmpinc" ]
	    set nrs [ image_temp_grid $nrs gmtstring 1 $verbose 0 $papersize(1) $maxmap\
			  $scale_unit_string  "-C$raster_colormap(11)" 1 ]
	    # begin clip path
	    set nrs [ add_to_script  $nrs gmtstring \
			  "\# begin coastline clipping" ]
	    set nrs [ add_to_script  $nrs gmtstring \
			  "\$gmtbin/pscoast \$region \$projection -O -K $resolution_string -Gc >> \$ps_filename" ]
	    # plot actual data
	    set nrs [ create_temp_grid $nrs gmtstring "GSHAP peak ground acceleration" \
			  $raster_data(10)  $data_region 1 $inc $temp_grid_filename $region 0 ]
	    set nrs [ image_temp_grid $nrs gmtstring $shading $verbose 0 $papersize(1) $maxmap\
			  $scale_unit_string  "-C\$colormap" 0 ]
	    # end clip path
	    set nrs [ add_to_script  $nrs gmtstring \
			  "\# end coastline clipping" ]
	    set nrs [ add_to_script  $nrs gmtstring \
			  "\$gmtbin/pscoast \$region \$projection -O -K -Q >> \$ps_filename" ]
	    if { $legend } {
		# add a colorbar here, else it would be outside clipping path
		set nrs [ add_to_script  $nrs gmtstring "\# Add a scale beneath the plot." ]
		# change size of font
		set nrs [ add_to_script  $nrs gmtstring "\$gmtbin/gmtset LABEL_FONT_SIZE 9p"]
		set line "\$gmtbin/psscale -C\$colormap -D[ expr $papersize(1)*0.38]/-0.5/[expr $maxmap*0.75]/0.2h"
		set line "$line \\\n\t-B:\"GSHAP Peak Ground Acceleration - 10% probability of exceedance\\\n"
		set line "$line in 50 years (475-years return-period)\":/:\"m/s@+ 2@\":\\\n\t -L"
		set nrs [ add_to_script $nrs gmtstring "$line  \$verbose -O -K  >> \$ps_filename"]
		# reset
		if { ($gmt_version == 3.0) || ($gmt_version == 3.1) } {
		    set tmpstring "24" 
		} elseif { $gmt_version == 3.2 } {
		    set tmpstring "16p"
		} else {
		    set tmpstring "24p"
		}
		set nrs [ add_to_script  $nrs gmtstring "\$gmtbin/gmtset LABEL_FONT_SIZE $tmpstring"]
	    }
	}



	default { error "Raster dataset $raster_dataset is not implemented!" ; set raster_dataset "1" }
    }
    
    
    if { ($addsomepscoast) && ($wantpscoast) } {
	set nrs [ add_to_script  $nrs gmtstring \
		      "\# Use pscoast for possible features such as national boundaries." ]
	set line "\$gmtbin/pscoast  $sl  \$region \$projection $resolution_string $natbound_polygons $river_polygons \$portrait   \\\n\t"
	set nrs [ add_to_script  $nrs gmtstring "$line -O -K $pscoast_add \$verbose >> \$ps_filename" ]
    }

    ################################################################################
    # now the polygon datasets, all optional and "superimposable"
    ################################################################################



    # NUVEL plate boundaries, dataset P1

    if { $polygon_dataset(1)  } { 
	set nrs [ add_to_script  $nrs gmtstring "\# Add NUVEL1 plate boundaries to the plot." ]
	set line "\$gmtbin/psxy $poly_data(1) -: \$verbose \\\n\t"
	set line "$line -M \$region \$projection -O -K  "
	add_linetype_string  line poly_linewidth poly_color 1 
	set line "$line   >> \$ps_filename"
	set nrs [ add_to_script  $nrs gmtstring "$line" ] 
    }
    
    # slab contour data, dataset P9
    if { $polygon_dataset(9)  } { 
	set nrs [ add_to_script  $nrs gmtstring "\# Add slab contours to the plot." ]
	set line "\$gmtbin/psxy $poly_data(9)  \$verbose \\\n\t"
	set line "$line -M \$region \$projection -O -K  "
	add_linetype_string  line poly_linewidth poly_color 9
	set line "$line   >> \$ps_filename"
	set nrs [ add_to_script  $nrs gmtstring "$line" ] 
    }

    # Steinberger hotspots, P7

    if { $polygon_dataset(7)  } { 
	set nrs [ add_to_script  $nrs gmtstring "\# Add hotspot locations from the Steinberger compilation." ]
	set line "$our_awk '{if((\$1!=\"\#\")&&(\$1!=\"\"))print(\$5,\$4,$poly_symbol_size(7)*$maxmap)}'"
	set line "$line \\\n\t $poly_data(7) | \\\n\t"
	set line "$line  \$gmtbin/psxy \$verbose \$region \$projection "
	add_scolor_string line poly_color 7 
	set line "$line -O -K $poly_symbol(7)   >> \$ps_filename"
	set nrs [ add_to_script  $nrs gmtstring "$line" ]
	if { $poly_parameter(7,1) } { # add the name of the hotspot
	    set nrs [ add_to_script  $nrs gmtstring "\# Add name tags for the hotspot locations." ]
	    set line "$our_awk '{if((\$1!=\"\#\")&&(\$1!=\"\"))printf(\"%g %g %g %g %g %g %s\\n\",\$5-$tagdist*$maxmap,\$4,10,0,15,3,\$1)}'"
	    set line "$line \\\n\t $poly_data(7) | \\\n\t"
	    set line "$line \$gmtbin/pstext \$region \$projection"
	    add_scolor_string line poly_color 7 
	    set line "$line -O -K >> \$ps_filename"
	    set nrs [ add_to_script  $nrs gmtstring "$line" ]
	}
    }


    # volcano list from the Smithsonian Institution, P8

    if { $polygon_dataset(8)  } { 
	set nrs [ add_to_script  $nrs gmtstring "\# Add volcano locations from the Smithsonian." ]
	set line "$our_awk '{if((\$1!=\"#\")&&(\$1!=\"\"))print substr(\$0,61,20)}' $poly_data(8) |\\\n\t"
	set line "$line $our_awk '{la=\$1;lo=\$3;if(\$2==\"S\")la*=-1;if(\$4==\"W\")lo*=-1;print(lo,la)}'|\\\n\t"
	set line "$line $our_awk '{print(\$1,\$2,$poly_symbol_size(8)*$maxmap)}' | \\\n\t"
	set line "$line  \$gmtbin/psxy \$verbose \$region \$projection "
	add_scolor_string line poly_color 8
	set line "$line -O -K $poly_symbol(8)   >> \$ps_filename"
	set nrs [ add_to_script  $nrs gmtstring "$line" ]
	if { $poly_parameter(8,1) } { # add the name of the volcano
	    set nrs [ add_to_script  $nrs gmtstring "\# Add name tags for the volcano locations." ]
	    set line "$our_awk '{if((\$1!=\"#\")&&(\$1!=\"\"))print(substr(\$0,61,20),substr(\$0,10,30))}' $poly_data(8) |\\\n\t"
	    set line "$line $our_awk '{la=\$1;lo=\$3;if(\$2==\"S\")la*=-1;if(\$4==\"W\")lo*=-1;print(lo,la,\$5,\$6,\$7,\$8)}'|\\\n\t"
	    set line "$line $our_awk '{print(\$1-0.08*$maxmap,\$2,10,0,15,3,\$3,\$4,\$5,\$6)}'| \\\n\t"
	    set line "$line \$gmtbin/pstext \$region \$projection"
	    add_scolor_string line poly_color 8 
	    set line "$line  -O -K >> \$ps_filename"
	    set nrs [ add_to_script  $nrs gmtstring "$line" ]
	}
    }

    
    # CMT solutions, P6

    if { $polygon_dataset(6)  } { 
 	set nrs [ add_to_script  $nrs gmtstring "\# Add CMT solutions in psmeca format to the plot." ]
 	set line "$our_awk '{if((\$1!=\"\#\")&&(\$1!=\"\"))print(\$0)}' $poly_data(6) | \\\n\t"
	set line "$line $our_awk '{print(\$1,\$2,\$3,\$4,\$5,\$6,\$7,\$8,\$9,\$10,\$1,\$2,\$11)}' | \\\n\t"
	set line "$line \$gmtbin/$psmeca \$region \$verbose "
	add_scolor_string line poly_color 6
 	set line "$line \\\n\t \$projection -O -K -Sc[format %03f [ expr $poly_symbol_size(6)*$maxmap ]] >> \$ps_filename"
 	set nrs [ add_to_script  $nrs gmtstring "$line" ] 
    }
    
    # significant quakes, take only the ones with magnitude assignment, P2

    if { $polygon_dataset(2)  } { 
	set nrs [ add_to_script  $nrs gmtstring "\# Add significant NGDC earthquakes with magnitudes to the plot" ]
	set line "$our_awk '{if((\$1!=\"\#\")&&(\$1!=\"*\")&&(\$1!=\"\")&&(\$9 != \"0\"))print(\$7,\$6,\$9*$poly_symbol_size(2)*$maxmap)}'"
	set line "$line \\\n\t $poly_data(2) | \\\n\t"
	set line "$line  \$gmtbin/psxy \$verbose \$region \$projection -O "
	add_scolor_string line poly_color 2
	set line "$line -K $poly_symbol(2)   >> \$ps_filename"
	set nrs [ add_to_script  $nrs gmtstring "$line" ]
    }

    # USGS/NEIS PDE catalog quakes above 5, 1973-1998, P3

    if { $polygon_dataset(3)  } { 
	set nrs [ add_to_script  $nrs gmtstring "\# Add significant USGS/NEIC 73-98 >5 " ]
	set line "$our_awk '{if((\$1!=\"\#\")&&(\$1!=\"\"))print(\$7,\$6,\$9*$poly_symbol_size(3)*$maxmap)}'"
	set line "$line \\\n\t $poly_data(3) | \\\n\t"
	set line "$line  \$gmtbin/psxy \$verbose \$region \$projection -O "
	add_scolor_string line poly_color 3
	set line "$line  -K $poly_symbol(3)   >> \$ps_filename"
	set nrs [ add_to_script  $nrs gmtstring "$line" ]
    }
    
    # first and second custom data sets P4 and P5
    
    foreach i { 4 5 } {
	if { $polygon_dataset($i)  } then { 
	    if { $poly_parameter($i,1) == "" } then { 
		dialog .d {File warning}  "Custom xy filename 1 not defined, skipping this polygon set!"  warning 0 {OK} 
	    } else {
		
		if { $poly_parameter($i,3) == "" } then { # no size column, use fixed xy data plotting
		    set nrs [ add_to_script  $nrs gmtstring "\# Add custom xy-data from $poly_parameter($i,5) " ]
		    set line "$our_awk '{if((\$1!=\"\#\")&&(\$1!=\"\"))print(\$$poly_parameter($i,1),\$$poly_parameter($i,2),5.*$poly_parameter($i,4)*$maxmap*$poly_symbol_size($i))}'" 
		} else { # $custom1(3) column contains sizes
		    set nrs [ add_to_script  $nrs gmtstring "\# Add custom xys-data from $poly_parameter($i,5) " ]
		    set line "$our_awk '{if((\$1!=\"\#\")&&(\$1!=\"\"))print(\$$poly_parameter($i,1),\$$poly_parameter($i,2),\$$poly_parameter($i,3)*$poly_parameter($i,4)*$maxmap*$poly_symbol_size($i))}'" 
		} 
		set line "$line \\\n\t $poly_parameter($i,5) | \\\n\t"
		set line "$line  \$gmtbin/psxy \$verbose \$region \$projection -O "
		
		set line "$line  -K $poly_symbol($i)"
		if { $poly_symbol($i) == "-L" } {
		    set line "$line -M"
		    add_linetype_string line poly_linewidth poly_color $i
		} else {
		    add_scolor_string line poly_color $i 
		}
		set line "$line >> \$ps_filename"
		set nrs [ add_to_script  $nrs gmtstring "$line" ]
	    }
	}
    }
    
      

    ################################################################################
    #
    # Velocity solutions, P10/P11 using psxy and psvelo. originally by Simone McClusky
    #

    if { $polygon_dataset(10)  } { 
	#
	# first site locations
 	set nrs [ add_to_script  $nrs gmtstring "\# Add station positions in psxy format to the plot." ] 
 	set line "$our_awk '{if((\$1!=\"\#\")&&(\$1!=\"\")&&(\$7<=ms)&&(\$8<=ms))print(\$1,\$2,$poly_symbol_size(10)*$maxmap)}' ms=$poly_parameter(10,5) $poly_data(10) |\\\n\t"
	set line "$line \$gmtbin/psxy \$verbose \$region \$projection -O "
	add_linetype_string line poly_linewidth poly_color 10
	set line "$line  -K $poly_symbol(10)   >> \$ps_filename" 
 	set nrs [ add_to_script  $nrs gmtstring "$line" ] 
	#
	# now vectors, first set arrow scale
	set astring "-A[format %5.4f/%5.4f/%5.4f [expr $poly_linewidth(10)/100.0] [expr $poly_linewidth(10)/20.0] [expr $poly_linewidth(10)/60.0]]"
	# velocity part of the scaling string
	set sstring "[format %03f [ expr $poly_parameter(10,1)/25.4]]"
	
 	set nrs [ add_to_script  $nrs gmtstring "\# Add Velocity solutions in psvelo format to the plot." ] 
 	set line "$our_awk '{if((\$1!=\"\#\")&&(\$1!=\"\")&&(\$7<=ms)&&(\$8<=ms))print(\$1,\$2,\$3,\$4,\$7,\$8,\$9,substr(\$13,1,4))}' ms=$poly_parameter(10,5) $poly_data(10) |\\\n\t"
	set line "$line \$gmtbin/$psvelo \$region \$verbose "
	add_linetype_string line poly_linewidth poly_color 11
	add_scolor_string line poly_color 11
	set line "$line $astring\\\n\t"
 	set line "$line \$projection -O -K -Se[ format %s/%03f/%03i $sstring $poly_parameter(10,3) $poly_parameter(10,4)] -D[format %03f $poly_parameter(10,2)] >> \$ps_filename"
 	set nrs [ add_to_script  $nrs gmtstring "$line" ] 

	# legend

 	set nrs [ add_to_script  $nrs gmtstring "\# Add Velocity scale vector to the plot." ]
	if { $poly_parameter(10,6) != 0. } {
	    set line "echo [ expr $east-$span(1)/5.0 ] [ expr $south+$span(2)/12.0 ] 0.0 $poly_parameter(10,6) 0.0 0.0 0.0 $poly_parameter(10,6) mm/yr |\\\n\t"
	    set line "$line \$gmtbin/$psvelo \$region \$verbose \$projection -O -K -Se[format %s/.95/15 $sstring ]" 
	    add_scolor_string line poly_color 11
	    set line "$line $astring >> \$ps_filename"
	    set nrs [ add_to_script  $nrs gmtstring "$line" ]
	}
    }
  
    # City names, P12/P13

    if { $polygon_dataset(12)  } { 
	set nrs [ add_to_script  $nrs gmtstring "\# Add city locations and names" ]
	set line "$our_awk '{if((\$1!=\"\#\")&&(\$1!=\"\"))print(\$1,\$2,$poly_symbol_size(12)*$maxmap)}'"
	set city_code [ expr 12 + $poly_parameter(12,1) ]
	set line "$line \\\n\t $poly_data($city_code) | \\\n\t"
	set line "$line  \$gmtbin/psxy \$verbose \$region \$projection -O "
	add_scolor_string line poly_color 12
	set line "$line -K $poly_symbol(12)   >> \$ps_filename"
	set nrs [ add_to_script  $nrs gmtstring "$line" ]
	if { $poly_parameter(12,2) } { # add the name of the city
	    set nrs [ add_to_script  $nrs gmtstring "\# Add name tags for the city locations." ]
	    set line "$our_awk '{if((\$1!=\"\#\")&&(\$1!=\"\"))printf(\"%g %g %g %g %g %g %s\\n\",\$1-$tagdist*$maxmap,\$2,10,0,15,3,\$3)}'"
	    set line "$line \\\n\t  $poly_data($city_code) | \\\n\t"
	    set line "$line \$gmtbin/pstext \$region \$projection"
	    add_scolor_string line poly_color 12
	    set line "$line -O -K >> \$ps_filename"
	    set nrs [ add_to_script  $nrs gmtstring "$line" ]
	}
    }
    
    # World Stress Map data, P15/16/17/18

    if { $polygon_dataset(15)  } { 
	# set colors and type selection string
	foreach i { 1 2 3 4 } {
	    set j [ expr 14 + $i ]
	    set wsm_color_string($i) \
		"[format %03i/%03i/%03i $poly_color($j,1) $poly_color($j,2) $poly_color($j,3)]"
	}
	set scr_tp_str "t1=$poly_parameter(15,1) t2=$poly_parameter(15,2) t3=$poly_parameter(15,3) t4=$poly_parameter(15,4) t5=$poly_parameter(15,5)"
	
	if { $poly_parameter(15,7) == 1 } { # plot different vectors for extension and compressional mechanism derived data
	    set wsm_veclen [  expr $maxmap*$poly_symbol_size(15)*0.7 ]
	    
	    set nrs [ add_to_script  $nrs gmtstring "\# Add WSM data points" ]
	    set nrs [ add_to_script  $nrs gmtstring "\# extensional (maybe with some strike slip component)" ]
	    set line "$our_awk -f $igmt_root/sortwsm.awk pregime=1 pquality=$poly_parameter(15,6)"
	    set line "$line \\\n\t  $scr_tp_str \\\n\t $poly_data(15) | \\\n\t"
	    set line "$line \$gmtbin/$psvelo \$region \$projection \$verbose "
	    set line "$line -W$poly_linewidth(15)/$wsm_color_string(1) [ format -Sx%f $wsm_veclen] "
	    set line "$line -O -K >> \$ps_filename"
	    set nrs [ add_to_script  $nrs gmtstring "$line" ]
	    
	    set nrs [ add_to_script  $nrs gmtstring "\# pure strike slip" ]
	    set line "$our_awk -f $igmt_root/sortwsm.awk pregime=2 pquality=$poly_parameter(15,6)"
	    set line "$line \\\n\t  $scr_tp_str \\\n\t $poly_data(15)  | \\\n\t"
	    set line "$line \$gmtbin/$psvelo \$region \$projection \$verbose "
	    set line "$line -W$poly_linewidth(15)/$wsm_color_string(2) [ format -Sx%f $wsm_veclen] "
	    set line "$line -O -K >> \$ps_filename"
	    set nrs [ add_to_script  $nrs gmtstring "$line" ]
	    
	    set nrs [ add_to_script  $nrs gmtstring "\# compressional (maybe with strike slip component)" ]
	    set line "$our_awk -f $igmt_root/sortwsm.awk pregime=3 pquality=$poly_parameter(15,6) "
	    set line "$line \\\n\t  $scr_tp_str \\\n\t $poly_data(15)  | \\\n\t"
	    set line "$line \$gmtbin/$psvelo \$region \$projection \$verbose "
	    set line "$line -W$poly_linewidth(15)/$wsm_color_string(3) [ format -Sx%f $wsm_veclen] "
	    set line "$line -O -K >> \$ps_filename"
	    set nrs [ add_to_script  $nrs gmtstring "$line" ]
	    
	    set nrs [ add_to_script  $nrs gmtstring "\# undetermined " ]
	    set line "$our_awk -f $igmt_root/sortwsm.awk pregime=4 pquality=$poly_parameter(15,6) "
	    set line "$line \\\n\t  $scr_tp_str \\\n\t $poly_data(15) | \\\n\t"
	    set line "$line \$gmtbin/$psvelo \$region \$projection \$verbose "
	    set line "$line -W$poly_linewidth(15)/$wsm_color_string(4) [ format -Sx%f $wsm_veclen] "
	    set line "$line -O -K >> \$ps_filename"
	    set nrs [ add_to_script  $nrs gmtstring "$line" ]

	} else  { # plot single line along compressional axis
	    set wsm_veclen [  expr $maxmap*$poly_symbol_size(15)*0.5 ]
	    set wsm_vec_with [ expr 0.02*$poly_linewidth(15)/0.5 ]
	    set nrs [ add_to_script  $nrs gmtstring "\# Add WSM data points" ]

	    set nrs [ add_to_script  $nrs gmtstring "\# extensional (maybe with some strike slip component)" ]
	    set line "$our_awk -f $igmt_root/sortwsm.awk pregime=1 pquality=$poly_parameter(15,6) "
	    set line "$line \\\n\t $scr_tp_str  \\\n\t $poly_data(15) | \\\n\t"
	    set line "$line $our_awk '{printf(\"%g %g %g %g\\n%g %g %g %g\\n\",\$1,\$2,\$5,f,\$1,\$2,\$5+180,f);}' f=$wsm_veclen | \\\n\t"
	    set line "$line \$gmtbin/psxy \$region \$projection \$verbose -SV$wsm_vec_with/0./0. "
	    set line "$line -G$wsm_color_string(1)"
	    set line "$line -O -K >> \$ps_filename"
	    set nrs [ add_to_script  $nrs gmtstring "$line" ]

	   
	    set nrs [ add_to_script  $nrs gmtstring "\# strike slip" ]
	    set line "$our_awk -f $igmt_root/sortwsm.awk pregime=2 pquality=$poly_parameter(15,6) "
	    set line "$line \\\n\t $scr_tp_str  \\\n\t $poly_data(15) | \\\n\t"
	    set line "$line $our_awk '{printf(\"%g %g %g %g\\n%g %g %g %g\\n\",\$1,\$2,\$5,f,\$1,\$2,\$5+180,f);}' f=$wsm_veclen | \\\n\t"
	    set line "$line \$gmtbin/psxy \$region \$projection \$verbose -SV$wsm_vec_with/0./0. "
	    set line "$line -G$wsm_color_string(2)"
	    set line "$line -O -K >> \$ps_filename"
	    set nrs [ add_to_script  $nrs gmtstring "$line" ]
	    

	    set nrs [ add_to_script  $nrs gmtstring "\# compressional (with strike-slip)" ]
	    set line "$our_awk -f $igmt_root/sortwsm.awk pregime=3 pquality=$poly_parameter(15,6) "
	    set line "$line \\\n\t $scr_tp_str  \\\n\t $poly_data(15) | \\\n\t"
	    set line "$line $our_awk '{printf(\"%g %g %g %g\\n%g %g %g %g\\n\",\$1,\$2,\$5,f,\$1,\$2,\$5+180,f);}' f=$wsm_veclen | \\\n\t"
	    set line "$line \$gmtbin/psxy \$region \$projection \$verbose -SV$wsm_vec_with/0./0. "
	    set line "$line -G$wsm_color_string(3)"
	    set line "$line -O -K >> \$ps_filename"
	    set nrs [ add_to_script  $nrs gmtstring "$line" ]
	    
	    set nrs [ add_to_script  $nrs gmtstring "\# undetermined" ]
	    set line "$our_awk -f $igmt_root/sortwsm.awk pregime=4 pquality=$poly_parameter(15,6) "
	    set line "$line \\\n\t $scr_tp_str  \\\n\t $poly_data(15) | \\\n\t"
	    set line "$line $our_awk '{printf(\"%g %g %g %g\\n%g %g %g %g\\n\",\$1,\$2,\$5,f,\$1,\$2,\$5+180,f);}' f=$wsm_veclen | \\\n\t"
	    set line "$line \$gmtbin/psxy \$region \$projection \$verbose -SV$wsm_vec_with/0./0. "
	    set line "$line -G$wsm_color_string(4)"
	    set line "$line -O -K >> \$ps_filename"
	    set nrs [ add_to_script  $nrs gmtstring "$line" ]


	}
    }
    # 
    # P19/20: plotting of vector fields given as grid files
    # 
    if { $polygon_dataset(19)  } { 
	# make sure field and reference vectors have same color
	foreach i { 1 2 3 } { 
	    set poly_color(20,$i) $poly_color(19,$i)
	}
	puts "iGMT: plotting vector field based on $poly_data(19) and"
	puts "iGMT: $poly_data(20)"
	set tmpvecsize "-Q[format %g/%g/%g [ expr 0.015*$poly_linewidth(19) ] [ expr 0.06*$poly_linewidth(19) ] [ expr 0.045*$poly_linewidth(19) ]]"
	set nrs [ add_to_script  $nrs gmtstring "\# add vector field" ]
	set line "\$gmtbin/grdvector $poly_data(19)\\\n\t$poly_data(20)\\\n\t"
	set line "$line -T \$region \$projection \$verbose\\\n\t"
	set line "$line -I$poly_parameter(19,4) -S$poly_parameter(19,1) $tmpvecsize\\\n\t"
	add_scolor_string line poly_color 19
	set line "$line  -O -K >> \$ps_filename"
	set nrs [ add_to_script  $nrs gmtstring $line ]
	if { $poly_parameter(19,5) } {
	    set nrs [ add_to_script  $nrs gmtstring "\# add reference vector" ]
	    set tmpstring "\"$poly_parameter(19,2) $poly_parameter(19,3)\""
	    set line "echo [ expr $west + 0.15*$span(1) ] [ expr $south + 0.08*$span(2) ] "
	    set line "$line $poly_parameter(19,2) 0 0 0 0 $tmpstring | \\\n\t"
	    set line "$line\$gmtbin/$psvelo -Se[ format %g/0.95/12 [ expr 1./$poly_parameter(19,1)]]"
	    add_linetype_string  line poly_linewidth poly_color 20
	    add_scolor_string line poly_color 20
	    set line "$line  \$region \\\n\t\$projection \$verbose -N -O -K >> \$ps_filename"
	    set nrs [ add_to_script  $nrs gmtstring $line ]
	}

    }
    
    ################################################################################
    # End of polygon data


    # possible extra title item
    # uncomment this for an attempt to plot the plot_title using pstext instead of the -B version below
    # title location is some percent of vertical plot size above northern boundary

    if { ($use_pstext_for_title) && ($plot_title != "") } {
	set line "echo  [ expr $mean(1)*1.02 ]  [ expr $north+0.07*$span(2)] $title_font_size 0 $title_font  6 \"$plot_title\" | "
	set line "$line $pstext \$region \$projection "
	set nrs [ add_to_script  $nrs gmtstring "$line \$portrait -N -O -K >> \$ps_filename" ]
    } 
    
  
    
    ################################################################################
    # add a frame for the map using psbasemap and complete the plot
    #

    set nrs [ add_to_script  $nrs gmtstring "\# Use psbasemap for basic map layout, possible title"]
    set nrs [ add_to_script  $nrs gmtstring "\#     and complete the plot"]
    set line "\$gmtbin/psbasemap  \$region \$verbose \$projection \\\n\t \$portrait \$annotation \\\n\t"
    #    if { $show_gmt_logo } { set line "$line -U\"iGMT $igmt_version\"" }
    if { $show_gmt_logo } { set line "$line -U/0.0/-0.47/\"iGMT $igmt_version\"" }
    if { $mapscale } { 
	set nrs [ add_to_script  $nrs gmtstring "\#     map-scale is correct at mid latitude"]
	if { $mapscale == 2 } { # fancy
	    set line "$line -Lfx1.1/0.4/$mean(2)/[ format %05.0lf $mscale]"
	} else { # plain
	    set line "$line -Lx1.1/0.4/$mean(2)/[ format %05.0lf $mscale]"

	}
    }
    set nrs [ add_to_script  $nrs gmtstring "$line -O $pscoast_add  >> \$ps_filename" ]

    
    ################################################################################
    # add a commented out line that would delete the temporary files
    #
    
    set nrs [ add_to_script  $nrs gmtstring "\# comment out (remove the \#) the line after next to delete temporary files automatically"]
    set nrs [ add_to_script  $nrs gmtstring "\# remove temporary files" ]
    set nrs [ add_to_script  $nrs gmtstring "\#rm -f \$temp_grid_filename \$temp_shade_filename 2> /dev/null" ]


    
    ################################################################################
    # print the script strings into a file


    set file [open $batchfile w]
    foreach i [lsort -integer [ array names gmtstring ]] {
	puts $file $gmtstring($i)
    }
    close $file

    ################################################################################
    # execute the GMT script and save error messages
    # allow for process being killed 
    # return 1 for success, 0 is killed 
    #
    # idea of command batch handling from Alexandre Ferrieux
    #
   
    
    # show a kill panel
   
    set headermessage "iGMT: Running the script..."
    update idletasks
    set retval 1
    # obtain top level geometry
    set w_x_pos [expr [ winfo x . ] + 100 ]
    set w_y_pos [expr [ winfo y . ] + 50 ]
    toplevel .panel 
    wm title .panel "Script control..." 
    wm geometry .panel +$w_x_pos+$w_y_pos
    # deal with batch aborts
    proc sad_return {  } { 
	global  headermessage 
	set headermessage "iGMT: Script was interrupted."
	update idletasks	
	catch { exec rm ./.gmtcommands 2> /dev/null &}
	destroy .panel
    }
    proc happy_return {   } {
	global headermessage batchfile  
	set headermessage "iGMT: Done, commands in $batchfile."
	update idletasks
	catch { exec rm ./.gmtcommands 2> /dev/null &}
	destroy .panel
    }

    # set the file_pointer to the piped batchfile command

    if { [ catch  [  list set file_pipe [open "| $batchfile 2> $batcherr" r] ] ] } {
	error "While trying to execute $batchfile. Make sure that the file exists and is executable (usually checked by the iGMT start-up sequence). If you want to recreate the file, type \"touch $batchfile\" and \"chmod +x $batchfile\" in a shell window. Quitting and restarting might be a better idea."
	sad_return
	set retval 0
    } 
    set batch_pid [pid $file_pipe]
    fileevent $file_pipe readable "happy_return"
    label .panel.label -text "Now running the GMT script\n$batchfile.\nThis might take a while. If errors occur or raster data sets look strange,\n check the stderr output (under menu item \"Scripting option\") first.\nInterrupt the background job with the \"CANCEL\" button below.\n\n"
    button .panel.kill -text "CANCEL" -relief groove -command { 
	# try to kill child processes first
	catch { exec ps -f | grep $batch_pid | $our_awk "{printf(\"%i \",\$2)}" } tmp_var 
	if { $tmp_var != "" } {
	    foreach child $tmp_var {
		catch { exec kill -9 $child }
	    }
	}
	# now kill the batch itself, if not already killed
	catch { exec kill -9 $batch_pid  }
	sad_return
	set retval 0
    }
    pack .panel.label .panel.kill -side top
    grab set .panel
    focus .panel
    tkwait window .panel
    catch { close $file_pipe }

    return $retval
}


################################################################################
# convert the postscript file to be able to display it, 
# only GIF is supported so far
################################################################################

proc conv_ps {} {
    global gif_filename ps_filename headermessage portrait ps_to_gif_command_landscape\
	ps_to_gif_command_portrait no_convert_complaints

    set headermessage "iGMT: Converting $ps_filename to $gif_filename..."

    update idletasks 
    if { $portrait == 0 } { 
	set command_string "exec $ps_to_gif_command_landscape"
    } else {
	set command_string "exec $ps_to_gif_command_portrait" 
    }	

    catch {  eval $command_string  } output 

    if { $no_convert_complaints  } {
       set headermessage "iGMT: Converting probably done."
    } else {
	if { $output != "" } {
	    dialog .d {PS to GIF conversion error...} \
		"PS to GIF conversion might have failed, the error message was\n$output\nLikely causes are:\na) the conversion program can not be found by iGMT (wrong path),\nb) the script file did not produce proper postscript output, or,\nc) the temporary GIF file that iGMT tries to write ($gif_filename)\ncan not be created.\nYou might still be able to use iGMT by only creating Postscript output."  warning 0 {OK}
	    puts $output
	    set headermessage "iGMT: PS to GIF conversion failed."
        } else {
	       set headermessage "iGMT: Converting done."
        }
    }
    set headermessage "iGMT: Converting done."
    update idletasks
    return
}


################################################################################
# delete the old and display the new GIF image in the lower frame
################################################################################


proc refresh_image {} {
    global  map_image gif_filename
    image delete map_image
    image create photo map_image -file $gif_filename
}


################################################################################
# procedure to plot the temporary grid file
################################################################################
proc image_temp_grid { nrstr array shading verbose legend papersize1 maxmap\
			   scale_unit_string colormap start_plot } {
    upvar $array ar 
    
    if { $shading } { # gradient shading for relief is on
	set nrstr [ add_to_script  $nrstr ar "\# Create a shade file." ]
	set nrstr [ add_to_script  $nrstr ar "\$gmtbin/grdgradient \$temp_grid_filename -A65 $verbose -G\$temp_shade_filename -Nt" ]
	
	set nrstr [ add_to_script  $nrstr ar "\# Use grdimage to create a raster map." ]
	set line "\$gmtbin/grdimage \$temp_grid_filename \$region \$projection \\\n\t$colormap \\\n\t-I\$temp_shade_filename \\\n\t"
	if { $start_plot } {
	    set nrstr [ add_to_script  $nrstr ar  "$line \$verbose \$portrait  -K \$offsets > \$ps_filename" ]
	} else {
	    set nrstr [ add_to_script  $nrstr ar  "$line \$verbose \$portrait  -K -O >> \$ps_filename" ]
	}
	if { $legend  } { # add a colorbar
	    set nrstr [ add_to_script  $nrstr ar "\# Add a scale beneath the plot." ]
	    set line "\$gmtbin/psscale $colormap -D[ expr $papersize1*0.38]/-0.5/[expr $maxmap*0.75]/0.2h"
	    set line "$line $scale_unit_string \\\n\t -L -I\$temp_shade_filename"
	    set nrstr [ add_to_script $nrstr ar "$line  \$verbose -O -K  >> \$ps_filename"]
	}
    } else { # flat representation as in defaults
	set nrstr [ add_to_script  $nrstr ar "\# Use grdimage to create a raster map." ]
	set line "\$gmtbin/grdimage \$temp_grid_filename \$region \$projection \\\n\t$colormap  \\\n\t\$verbose "
	if { $start_plot } {
	    set nrstr [ add_to_script  $nrstr ar  "$line \$portrait  \$offsets -K > \$ps_filename" ]
	} else {
	    set nrstr [ add_to_script  $nrstr ar  "$line \$portrait   -K -O >> \$ps_filename" ]
	}
	if { $legend  } { # add a colorbar
	    set nrstr [ add_to_script  $nrstr ar "\# Add a scale beneath the plot." ]
	    set line "\$gmtbin/psscale $colormap \\\n\t-D[ expr $papersize1*0.38]/-0.5/[expr $maxmap*0.75]/0.2h"
	    set line "$line $scale_unit_string \\\n\t -L"
	    set nrstr [ add_to_script $nrstr ar "$line  \$verbose -O -K  >> \$ps_filename"]
	}
    }
   
    return $nrstr
}

################################################################################
# procedure to add a line to the script string

proc add_to_script { nrstr array string } {
    upvar $array ar 
    set ar($nrstr) $string
    incr nrstr
    return $nrstr
}
################################################################################
# procedure for adding a beautiful line with 80 dashes to the script
proc draw_script_line { nrstr array } {
    upvar $array ar 
    set a "\#\#\#\#\#\#\#\#\#\#\#\#\#\#\#\#\#\#\#\#"
    set ar($nrstr) [ format "%s%s%s%s" $a $a $a $a ]
    incr nrstr
    return $nrstr
}

################################################################################    
# produce strings called by the plotting routines 
# that GMT versions 3.0 and 3.x handle differently

proc version_dependent_strings { unit b_string } {
    upvar $b_string s
    global gmt_version
    
    # string for the units next to the psscale scale
    if { $gmt_version == 3.0 } {
	set s "-B:\"$unit\":" 
    } else {
	set s "-B/:\"$unit\":" 
    }
}


################################################################################
#
# create temporary grid files for plotting
#

proc create_temp_grid {  nrstr array data_decription_string data \
			     limit_region resample spacing temp_grid_filename region \
			     use_data_region } {
    upvar $array ar 
    global gmtbins
    
    set nrstr [ add_to_script  $nrstr ar "\# Plotting $data_decription_string" ]
    if { $resample } { 
	set nrstr [ add_to_script  $nrstr ar "\# resample data at grid spacing \$raster_inc" ]
	if { $use_data_region } {
	    catch { exec $gmtbins/grdsample $data $limit_region -G$temp_grid_filename $spacing } output
	} else {
	    catch { exec $gmtbins/grdsample $data $region -G$temp_grid_filename $spacing } output
	}
	if { $output != "" } {
	    puts $output
	    dialog .d {grdrcut/sample warning} \
		"grdsample would not execute correctly.\nA likely reason is that the plotting region is not entirely within your data.\nCheck your grd-file using grdinfo, script will now probably crash"  warning 0 {OK} 
	}
	if { $use_data_region } {
	    set nrstr [ add_to_script  $nrstr ar "\$gmtbin/grdsample $data \\\n\t\$data_region \$raster_inc \$verbose -G\$temp_grid_filename" ]
	} else {
	    set nrstr [ add_to_script  $nrstr ar "\$gmtbin/grdsample $data \\\n\t\$region \$raster_inc \$verbose -G\$temp_grid_filename" ]
	}
    } else {
	set nrstr [ add_to_script  $nrstr ar "\# Create a temporary (smaller) grid file using grdcut" ]
	if { $use_data_region } {
	    set nrstr [ add_to_script  $nrstr ar "\$gmtbin/grdcut $data \\\n\t\$data_region \$verbose -G\$temp_grid_filename" ]
	} else {
	    set nrstr [ add_to_script  $nrstr ar "\$gmtbin/grdcut $data \\\n\t\$region \$verbose -G\$temp_grid_filename" ]
	}
    }
    return $nrstr
}

################################################################################
# procedures to produce string with polygone linewidth and color specifications
# or symbol fill colors
proc add_linetype_string { ref_line pl pc i } {
    upvar $ref_line line
    upvar $pl poly_linewidth
    upvar $pc poly_color
    set line "$line -W$poly_linewidth($i)/[format %03i/%03i/%03i $poly_color($i,1) $poly_color($i,2) $poly_color($i,3)]"
}
# add a string with a fill color specification
proc add_scolor_string { ref_line pc i } {
    upvar $ref_line line
    upvar $pc poly_color
    set line "$line -G[format %03i/%03i/%03i $poly_color($i,1) $poly_color($i,2) $poly_color($i,3)]"
}
