#---------------------------------------------------------------------
# BETCL_safe_script.tcl		- Intitialization of safe backend
#				  Tcl interpreter.
#---------------------------------------------------------------------

if {$HAVE_LOADTCL} {

#---------------------------------------------------------------------
# loadtcl			- The Tcl command to force another
#				  source from the table betcl_source
#				  to be loaded into the interpreter.
#---------------------------------------------------------------------
proc loadtcl {args} {
    foreach srcname $args {
        betcl_load $srcname
    }
}

} ;# HAVE_LOADTCL


if {$HAVE_EXECSQL} {

#---------------------------------------------------------------------
# SQLquote              - Quote literals properly
#---------------------------------------------------------------------
proc SQLquote {val} {
    regsub -all {\\} $val {\\\\} val
    regsub -all {'}  $val {\\'} val
    regsub -all "\n" $val {\\n} val
    regsub -all "\t" $val {\\t} val
    return $val
}



#---------------------------------------------------------------------
# EXECSQL		- Internal wrapper for SQL execution
#---------------------------------------------------------------------
proc EXECSQL {cmd args} {
    #-----------------------------------------------------------
    # Check if there's a special handling for this
    # SQL statement
    #-----------------------------------------------------------
    set xcmd _EXECSQL_cmd_$cmd
    if {[info procs $xcmd] == ""} {
	#-----------------------------------------------------------
	# No special handling - execute directly
	#-----------------------------------------------------------
	set query [join $args]
	regsub -all "\n" $query " " query
	set res [betcl_execsql "$cmd $query"]
	return [betcl_sqlresult $res -clear]
    }

    #-----------------------------------------------------------
    # Call the special handling procedure
    #-----------------------------------------------------------
    return [$xcmd $args]
}


#---------------------------------------------------------------------
# _EXECSQL_cmd_select	- Special handling of select statement
#---------------------------------------------------------------------
proc _EXECSQL_cmd_select {query} {
    #-----------------------------------------------------------
    # Check for 'single' keyword and build the real query
    #-----------------------------------------------------------
    if {[lindex $query 0] == "single"} {
        set single 1
	set query [join [lrange $query 1 end]]
    } else {
        set single 0
	set cmd [lindex $query end]
	set query [join [lreplace $query end end]]
    }
    regsub -all "\n" $query " " query

    #-----------------------------------------------------------
    # Execute the query
    #-----------------------------------------------------------
    set res [betcl_execsql "select $query"]

    #-----------------------------------------------------------
    # Create the attibute list from the result buffer.
    # Upvar the attributes up the the interpreter level
    # where the original 'select' came from and initialize
    # the variables to an empty string.
    #-----------------------------------------------------------
    set attrlist {}
    foreach attrd [betcl_sqlresult $res -attributes] {
        set attr [lindex $attrd 0]
	lappend attrlist $attr
	upvar 2 $attr val_$attr
	set val_$attr {}
    }

    if {$single} {
	#-----------------------------------------------------------
	# On 'select single ...' set the attribute variables
	# to the values of the first tuple (if we got anything)
	#-----------------------------------------------------------
        if {[betcl_sqlresult $res -numtuples] > 0} {
	    set data [betcl_sqlresult $res -gettuple 0]
	} else {
	    set data {}
	}
	set i 0
	foreach attr $attrlist {
	    set val_$attr [lindex $data $i]
	    incr i
	}
    } else {
	#-----------------------------------------------------------
	# On normal 'select ...' set the attribute variables
	# inside a loop over all tuples and evaluate the command.
	#-----------------------------------------------------------
	set n [betcl_sqlresult $res -numtuples]
	for {set t 0} {$t < $n} {incr t} {
	    set data [betcl_sqlresult $res -gettuple $t]
	    set i 0
	    foreach attr $attrlist {
		set val_$attr [lindex $data $i]
		incr i
	    }
	    uplevel 2 $cmd
	} 
    }

    #-----------------------------------------------------------
    # Clear the result buffer and return the number of
    # tuples we got for this query.
    #-----------------------------------------------------------
    return [betcl_sqlresult $res -numtuples -clear]
}


} ;# HAVE_EXECSQL



