/**********************************************************************
 * betcl.c		- Postgres95 backend module for functions
 *			  written in Tcl
 *
 * IDENTIFICATION
 *    $Header: /home/wieck/src/tcltk/extensions/pqatcl/src/libbetcl/RCS/betcl.c,v 0.2 1996/10/09 08:40:37 wieck Exp $
 *
 *    This software is copyrighted by Jan Wieck - Hamburg.
 *
 *    The author hereby grants permission  to  use,  copy,  modify,
 *    distribute,  and  license this software and its documentation
 *    for any purpose, provided that existing copyright notices are
 *    retained  in  all  copies  and  that  this notice is included
 *    verbatim in any distributions. No written agreement, license,
 *    or  royalty  fee  is required for any of the authorized uses.
 *    Modifications to this software may be  copyrighted  by  their
 *    author  and  need  not  follow  the licensing terms described
 *    here, provided that the new terms are  clearly  indicated  on
 *    the first page of each file where they apply.
 *
 *    IN NO EVENT SHALL THE AUTHOR OR DISTRIBUTORS BE LIABLE TO ANY
 *    PARTY  FOR  DIRECT,   INDIRECT,   SPECIAL,   INCIDENTAL,   OR
 *    CONSEQUENTIAL   DAMAGES  ARISING  OUT  OF  THE  USE  OF  THIS
 *    SOFTWARE, ITS DOCUMENTATION, OR ANY DERIVATIVES THEREOF, EVEN
 *    IF  THE  AUTHOR  HAVE BEEN ADVISED OF THE POSSIBILITY OF SUCH
 *    DAMAGE.
 *
 *    THE  AUTHOR  AND  DISTRIBUTORS  SPECIFICALLY   DISCLAIM   ANY
 *    WARRANTIES,  INCLUDING,  BUT  NOT  LIMITED  TO,  THE  IMPLIED
 *    WARRANTIES  OF  MERCHANTABILITY,  FITNESS  FOR  A  PARTICULAR
 *    PURPOSE,  AND NON-INFRINGEMENT.  THIS SOFTWARE IS PROVIDED ON
 *    AN "AS IS" BASIS, AND THE AUTHOR  AND  DISTRIBUTORS  HAVE  NO
 *    OBLIGATION   TO   PROVIDE   MAINTENANCE,   SUPPORT,  UPDATES,
 *    ENHANCEMENTS, OR MODIFICATIONS.
 *
 * HISTORY
 * $Log: betcl.c,v $
 * Revision 0.2  1996/10/09 08:40:37  wieck
 * Fixed bug in getting attribute descriptions where no
 * tuples are returned.
 *
 * Revision 0.1  1996/10/01 06:56:55  postgres95
 * Initial revision
 *
 *
 **********************************************************************/

#include <tcl.h>

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <setjmp.h>

#include "libpq/libpq.h"
#include "libpq/libpq-be.h"
#include "c.h"
#include "postgres.h"
#include "utils/elog.h"

#include "betcl.h"

/**********************************************************************
 * Global variables
 *
 * betcl_firstcall	- True until the first time we are called.
 * betcl_interp_level	- How deep is the nesting of Tcl-SQL-Tcl-...
 * betcl_restart_in_progress - If we are in transaction abort.
 * betcl_reset_required - Sourcing a script failed or the database
 *			   told us about a change of a script. When
 *			   the last interp_level returns we destroy
 *			   the interpreter and start from scratch.
 * betcl_at_load	- During scrip load SQL queries are forbidden.
 *
 **********************************************************************/
static int		betcl_firstcall = 1;
static int		betcl_interp_level = 0;
static int		betcl_restart_in_progress = 0;
static int		betcl_reset_required = 0;
static int		betcl_at_load = 0;
static Tcl_Interp	*betcl_hold_interp = NULL;
static Tcl_Interp	*betcl_main_interp = NULL;
static Tcl_Interp	*betcl_safe_interp = NULL;

static Tcl_HashTable	*betcl_proc_hash = NULL;
static Tcl_HashTable	*betcl_source_hash = NULL;
static Tcl_HashTable	*betcl_result_hash = NULL;
static int		betcl_result_count = 0;


/**********************************************************************
 * betcl_initall()		- Ensure that all initialization is
 *				  is done.
 **********************************************************************/
static
void betcl_initall(void)
{
    Tcl_HashEntry	*hashent;
    Tcl_HashSearch	hashsearch;
    betcl_proc_desc	*proc_desc;
    betcl_source_desc	*source_desc;
    betcl_result	*result;
    int			i;

    /************************************************************
     * Only once or after told by betcl_reset()
     ************************************************************/
    if(!betcl_firstcall) {
        return;
    }

    if(betcl_hold_interp == NULL) {
        betcl_hold_interp = Tcl_CreateInterp();
	if(betcl_hold_interp == NULL) {
	    elog(WARN, "betcl: cannot create Tcl 'hold' interpreter");
	}
    }

    /************************************************************
     * Tidy up an earlier abort situation
     ************************************************************/
    Debug((stderr, "betcl: remove old interpreters and hash tables\n"));
    if(betcl_main_interp != NULL) {
	Tcl_DeleteInterp(betcl_main_interp);
	betcl_main_interp = NULL;
    }
    if(betcl_safe_interp != NULL) {
	Tcl_DeleteInterp(betcl_safe_interp);
	betcl_safe_interp = NULL;
    }
    if(betcl_proc_hash != NULL) {
	hashent = Tcl_FirstHashEntry(betcl_proc_hash, &hashsearch);
	while(hashent != NULL) {
	    proc_desc = (betcl_proc_desc *)Tcl_GetHashValue(hashent);
	    free(proc_desc);
	    hashent = Tcl_NextHashEntry(&hashsearch);
	}
	Tcl_DeleteHashTable(betcl_proc_hash);
	free(betcl_proc_hash);
	betcl_proc_hash = NULL;
    }
    if(betcl_source_hash != NULL) {
	hashent = Tcl_FirstHashEntry(betcl_source_hash, &hashsearch);
	while(hashent != NULL) {
	    source_desc = (betcl_source_desc *)Tcl_GetHashValue(hashent);
	    free(source_desc);
	    hashent = Tcl_NextHashEntry(&hashsearch);
	}
	Tcl_DeleteHashTable(betcl_source_hash);
	free(betcl_source_hash);
	betcl_source_hash = NULL;
    }
    if(betcl_result_hash != NULL) {
	hashent = Tcl_FirstHashEntry(betcl_result_hash, &hashsearch);
	while(hashent != NULL) {
	    result = (betcl_result *)Tcl_GetHashValue(hashent);
	    if(result->num_tuples > 0) {
	        Tcl_DStringFree(&(result->attributes));
		for(i = 0; i < result->num_tuples; i++) {
		    free(result->tuple_strings[i]);
		}
		free(result->tuple_strings);
	    }
	    free(result);
	    hashent = Tcl_NextHashEntry(&hashsearch);
	}
	Tcl_DeleteHashTable(betcl_result_hash);
	free(betcl_result_hash);
	betcl_result_hash = NULL;
    }

    /************************************************************
     * Now create the interpreters and anything else
     ************************************************************/
    betcl_initmain();
    betcl_initsafe();

    betcl_firstcall = 0;
    return;
}


/**********************************************************************
 * betcl_initmain()		- Initialize the main interpreter
 *				  and the hash tables
 **********************************************************************/
static
void betcl_initmain(void)
{
    int		fd;
    int		sz;
    char	buf[4096];
    Tcl_DString	main_script;

    Debug((stderr, "betcl: initialize main interpreter\n"));

    /************************************************************
     * Create the main interp and initialize the hash tables
     ************************************************************/
    if((betcl_main_interp = Tcl_CreateInterp()) == NULL) {
        elog(WARN, "betcl: cannot create main Tcl interpreter");
    }

    betcl_proc_hash   = (Tcl_HashTable *)malloc(sizeof(Tcl_HashTable));
    betcl_source_hash = (Tcl_HashTable *)malloc(sizeof(Tcl_HashTable));
    betcl_result_hash = (Tcl_HashTable *)malloc(sizeof(Tcl_HashTable));
    Tcl_InitHashTable(betcl_proc_hash, TCL_STRING_KEYS);
    Tcl_InitHashTable(betcl_source_hash, TCL_STRING_KEYS);
    Tcl_InitHashTable(betcl_result_hash, TCL_STRING_KEYS);

    /************************************************************
     * Read or setup the main interpreter init script
     ************************************************************/
    Tcl_DStringInit(&main_script);
    if((fd = open(BETCL_MAIN_SCRIPT, O_RDONLY, 0)) >= 0) {
	while((sz = read(fd, buf, sizeof(buf))) > 0) {
	    Tcl_DStringAppend(&main_script, buf, sz);
	}
	close(fd);
    } else {
        Tcl_DStringAppend(&main_script, 
		"proc init_slave_interp {path user} {\n", -1);
        Tcl_DStringAppend(&main_script, 
		"    interp share {} stderr $path\n", -1);
        Tcl_DStringAppend(&main_script, 
		"    interp eval $path set PGUSER $user\n", -1);
        Tcl_DStringAppend(&main_script, 
		"    interp eval $path set HAVE_STDOUT 0\n", -1);
        Tcl_DStringAppend(&main_script, 
		"    interp eval $path set HAVE_STDERR 1\n", -1);
        Tcl_DStringAppend(&main_script, 
		"    interp eval $path set HAVE_EXECSQL 0\n", -1);
        Tcl_DStringAppend(&main_script, 
		"    interp eval $path set HAVE_LOADTCL 0\n", -1);
        Tcl_DStringAppend(&main_script, 
		"}\n", -1);
    }

    /************************************************************
     * Evaluate that in the main interpreter
     ************************************************************/
    if(Tcl_Eval(betcl_main_interp, 
	    Tcl_DStringValue(&main_script)) != TCL_OK) {
	elog(WARN, "betcl: error in main interpreter init script - %s",
		betcl_main_interp->result);
    }
    Tcl_DStringFree(&main_script);

    /************************************************************
     * Install the commands to enable load and sql in the slave
     ************************************************************/
    Tcl_CreateCommand(betcl_main_interp, "betcl_install_load", 
	    betcl_install_load, NULL, NULL);
    Tcl_CreateCommand(betcl_main_interp, "betcl_install_sql", 
	    betcl_install_sql, NULL, NULL);

    return;
}


/**********************************************************************
 * betcl_initsafe()		- Initialize the safe interpreter and
 *				  install the SQL commands
 **********************************************************************/
static
void betcl_initsafe(void)
{
    char	init_cmd[512];

    Debug((stderr, "betcl: initialize safe interpreter\n"));

    /************************************************************
     * Create the safe slave interpreter
     ************************************************************/
    if((betcl_safe_interp = 
	    Tcl_CreateSlave(betcl_main_interp, "safe", 1)) == NULL) {
	elog(WARN, "betcl: cannot create safe interpreter");
    }

    /************************************************************
     * Call the init proc in the master interpreter
     ************************************************************/
    sprintf(init_cmd, "init_slave_interp safe {%s}", GetPgUserName());
    if(Tcl_Eval(betcl_main_interp, init_cmd) != TCL_OK) {
	elog(WARN, "betcl: init_slave_interp failed - %s",
		betcl_main_interp->result);
    }

    return;
}


/**********************************************************************
 * betcl_reset()		- Set the reset flag and if no
 *				  Tcl proc is active do so.
 **********************************************************************/
int4 betcl_reset(void)
{
    betcl_reset_required = 1;

    if(betcl_interp_level > 0) {
	Debug((stderr, "betcl: reset delayed\n"));
        return 0;
    }

    betcl_firstcall = 1;
    betcl_initall();
    betcl_reset_required = 0;

    return 0;
}


/**********************************************************************
 * betcl_load_source()		- Get a source text from the betcl_source
 *				  relation and put it into the interpreter
 **********************************************************************/
static
void betcl_load_source(char *srcname)
{
    Tcl_HashEntry	*hashent;
    int			hashnew;
    betcl_source_desc	*source_desc;
    char		*srctext;
    char		query[512];
    char		*pqres;
    PortalBuffer	*portal;
    int			tclrc;
    char		*errmsg;
    int			srcsize;
    int			i;

    /************************************************************
     * Lookup if we already have this source in the interpreter
     ************************************************************/
    if((hashent = Tcl_FindHashEntry(betcl_source_hash, srcname)) != NULL) {
	/************************************************************
	 * Anything is fine
	 ************************************************************/
	return;
    }

    Debug((stderr, "betcl: load source '%s'\n", srcname));

    /************************************************************
     * Get the source text from betcl_source
     ************************************************************/
    sprintf(query, 
    	"select S.srctext, S.srcseq from betcl_source S \
		where S.srcname = '%s' \
		order by srcseq", srcname);
    pqres = (char *)PQexec(query);
    if(*pqres != 'P') {
        elog(WARN, "betcl: No portal on lookup for '%s' in betcl_source",
		srcname);
    }
    portal = PQparray(++pqres);
    if(PQntuples(portal) < 1) {
        PQclear(pqres);
	elog(WARN, "betcl: Tcl source '%s' not found in betcl_source", 
		srcname);
    }
    srcsize = 1;
    for(i = 0; i < PQntuples(portal); i++) {
        srcsize += strlen(PQgetvalue(portal, i, 0));
    }
    srctext = palloc(srcsize);
    srctext[0] = '\0';
    for(i = 0; i < PQntuples(portal); i++) {
        strcat(srctext, PQgetvalue(portal, i, 0));
    }

    /************************************************************
     * Create the source description block and hash it
     ************************************************************/
    source_desc = (betcl_source_desc *)malloc(sizeof(betcl_source_desc));
    strcpy(source_desc->srcname, srcname);

    hashent = Tcl_CreateHashEntry(betcl_source_hash, source_desc->srcname,
    	&hashnew);
    Tcl_SetHashValue(hashent, (ClientData)source_desc);

    /************************************************************
     * Put the source into the safe interpreter
     ************************************************************/
    betcl_at_load = 1;
    betcl_interp_level++;
    tclrc = Tcl_Eval(betcl_safe_interp, srctext);
    betcl_interp_level--;
    betcl_at_load = 0;
    pfree(srctext);
    PQclear(pqres);

    /************************************************************
     * On errors during source reset the interpreter
     ************************************************************/
    if(tclrc != TCL_OK) {
        fprintf(stderr, "betcl: error while sourcing '%s'\n", srcname);
	errmsg = palloc(strlen(betcl_safe_interp->result) + 1);
	strcpy(errmsg, betcl_safe_interp->result);
	if(betcl_restart_in_progress) {
	    if(betcl_interp_level == 0) {
	        Debug((stderr, "betcl: back on level 0 - restart complete\n"));
		betcl_restart_in_progress = 0;
		betcl_reset();
		betcl_longjmp(Warn_restart, 1);
	    }
	    Debug((stderr, "betcl: restart still in progress\n"));
	    betcl_longjmp(Warn_restart, 1);
	}
	betcl_reset();
	elog(WARN, "betcl: error during load of '%s' - %s",
		srcname, errmsg);
    }

    return;
}


/**********************************************************************
 * betcl_load_proc()		- Load a Tcl procedure
 **********************************************************************/
static
void betcl_load_proc(NameData *proname, int4 nargs)
{
    Tcl_HashEntry	*hashent;
    char		hashkey[NAMEDATALEN + 1];
    int			hashnew;
    betcl_proc_desc	*proc_desc;
    char		query[512];
    char		*pqres;
    PortalBuffer	*portal;
    int			pronargs;
    char		srcname[NAMEDATALEN + 1];

    /************************************************************
     * Lookup if we already have this procedure in the interpreter
     ************************************************************/
    strncpy(hashkey, (char *)proname, NAMEDATALEN);
    hashkey[NAMEDATALEN] = '\0';
    if((hashent = Tcl_FindHashEntry(betcl_proc_hash, hashkey)) != NULL) {
	/************************************************************
	 * If so check the # of call arguments
	 ************************************************************/
        proc_desc = (betcl_proc_desc *)Tcl_GetHashValue(hashent);
	if(proc_desc->pronargs != nargs) {
	    elog(WARN, 
	    "betcl: Wrong # of arguments on call to '%s' (%d) - should be %d",
	    hashkey, nargs, proc_desc->pronargs);
	}
	/************************************************************
	 * Anything is fine
	 ************************************************************/
	return;
    }

    Debug((stderr, "betcl: load proc '%s'\n", hashkey));

    /************************************************************
     * Get the procedure description from betcl_proc and 
     * check the # of call arguments
     ************************************************************/
    sprintf(query, 
    	"select pronargs, prosrc from betcl_proc where proname = '%s'",
	hashkey);
    pqres = (char *)PQexec(query);
    if(*pqres != 'P') {
        elog(WARN, "betcl: No portal on lookup for '%s' in betcl_proc",
		hashkey);
    }
    portal = PQparray(++pqres);
    if(PQntuples(portal) < 1) {
        PQclear(pqres);
	elog(WARN, "betcl: Tcl procedure '%s' not found in betcl_proc", 
		hashkey);
    }
    pronargs = atoi(PQgetvalue(portal, 0, 0));
    if(pronargs != nargs) {
	PQclear(pqres);
	elog(WARN, 
	"betcl: Wrong # of arguments on call to '%s' (%d) - should be %d",
	hashkey, nargs, pronargs);
    }

    /************************************************************
     * Load the source
     ************************************************************/
    strcpy(srcname, PQgetvalue(portal, 0, 1));
    PQclear(pqres);

    betcl_load_source(srcname);

    /************************************************************
     * Create the hash entry for this procedure
     ************************************************************/
    proc_desc = (betcl_proc_desc *)malloc(sizeof(betcl_proc_desc));
    strcpy(proc_desc->proname, hashkey);
    proc_desc->pronargs = pronargs;
    hashent = Tcl_CreateHashEntry(betcl_proc_hash, proc_desc->proname,
    	&hashnew);
    Tcl_SetHashValue(hashent, (ClientData)proc_desc);

    return;
}


/**********************************************************************
 * betcl_call()		- Handle a call of a Tcl procedure
 *			  from the backend function manager
 **********************************************************************/
text *betcl_call(NameData *proname, int4 nargs,
	text *a1, text *a2, text *a3, text *a4, text *a5, text *a6)
{
    text		*retval;
    int			retlen;
    char		tclproname[NAMEDATALEN + 1];
    Tcl_DString		tclcmd;
    int			tclrc;
    Tcl_HashEntry	*hashent;
    Tcl_HashSearch	hashsearch;
    betcl_result	*result;
    int			i;

    Debug((stderr, "betcl: betcl_call()\n"));

    /************************************************************
     * Make sure all interpreters are there
     ************************************************************/
    betcl_initall();

    /************************************************************
     * Load the proc into the safe interpreter
     ************************************************************/
    betcl_load_proc(proname, nargs);

    /************************************************************
     * Build the command
     ************************************************************/
    Tcl_DStringInit(&tclcmd);
    strncpy(tclproname, (char *)proname, NAMEDATALEN);
    tclproname[NAMEDATALEN] = '\0';
    Tcl_DStringAppend(&tclcmd, tclproname, -1);

    if(nargs > 0) {
        Tcl_DStringAppend(&tclcmd, " {", 2);
	Tcl_DStringAppend(&tclcmd, VARDATA(a1), VARSIZE(a1) - VARHDRSZ);
	Tcl_DStringAppend(&tclcmd, "}", 1);
    if(nargs > 1) {
        Tcl_DStringAppend(&tclcmd, " {", 2);
	Tcl_DStringAppend(&tclcmd, VARDATA(a2), VARSIZE(a2) - VARHDRSZ);
	Tcl_DStringAppend(&tclcmd, "}", 1);
    if(nargs > 2) {
        Tcl_DStringAppend(&tclcmd, " {", 2);
	Tcl_DStringAppend(&tclcmd, VARDATA(a3), VARSIZE(a3) - VARHDRSZ);
	Tcl_DStringAppend(&tclcmd, "}", 1);
    if(nargs > 3) {
        Tcl_DStringAppend(&tclcmd, " {", 2);
	Tcl_DStringAppend(&tclcmd, VARDATA(a4), VARSIZE(a4) - VARHDRSZ);
	Tcl_DStringAppend(&tclcmd, "}", 1);
    if(nargs > 4) {
        Tcl_DStringAppend(&tclcmd, " {", 2);
	Tcl_DStringAppend(&tclcmd, VARDATA(a5), VARSIZE(a5) - VARHDRSZ);
	Tcl_DStringAppend(&tclcmd, "}", 1);
    if(nargs > 5) {
        Tcl_DStringAppend(&tclcmd, " {", 2);
	Tcl_DStringAppend(&tclcmd, VARDATA(a6), VARSIZE(a6) - VARHDRSZ);
	Tcl_DStringAppend(&tclcmd, "}", 1);
    }}}}}}

    /************************************************************
     * Evaluate it in the safe interpreter
     ************************************************************/
    betcl_interp_level++;
    tclrc = Tcl_Eval(betcl_safe_interp, Tcl_DStringValue(&tclcmd));
    Tcl_DStringFree(&tclcmd);

    /************************************************************
     * Free result buffers not cleared by this proc
     ************************************************************/
    hashent = Tcl_FirstHashEntry(betcl_result_hash, &hashsearch);
    while(hashent != NULL) {
	result = (betcl_result *)Tcl_GetHashValue(hashent);
	if(result->interp_level == betcl_interp_level) {
	    Debug((stderr, "free fogotten result buffer %s\n",
	    	result->handle));
	    if(result->num_tuples > 0) {
	        Tcl_DStringFree(&(result->attributes));
		for(i = 0; i < result->num_tuples; i++) {
		    free(result->tuple_strings[i]);
		}
		free(result->tuple_strings);
	    }
	    Tcl_DeleteHashEntry(hashent);
	    free(result);
	}
        hashent = Tcl_NextHashEntry(&hashsearch);
    }
    betcl_interp_level--;

    /************************************************************
     * Handle Tcl errors
     ************************************************************/
    if(tclrc != TCL_OK) {
	/************************************************************
	 * Only generate a transaction abort if this is the
	 * first error in this query
	 ************************************************************/
        if(!betcl_restart_in_progress) {
	    Debug((stderr, "betcl: got initial Tcl error - begin restart\n"));
	    if(betcl_interp_level > 0) {
		betcl_restart_in_progress = 1;
	    }
	    elog(WARN, "betcl: TclError %s", betcl_safe_interp->result);
	}
	/************************************************************
	 * This might be a Tcl error caused by a failed query inside
	 * the script (or maybe nested very deeply). Get up to the
	 * next level of error handling without a warning.
	 ************************************************************/
	Debug((stderr, "betcl: Tcl error during restart - %s\n",
		betcl_safe_interp->result));
	if(betcl_interp_level == 0) {
	    Debug((stderr, "betcl: Back on level 0 - restart complete\n"));
	    betcl_restart_in_progress = 0;
	    if(betcl_reset_required) {
	        betcl_reset();
	    }
	}
	betcl_longjmp(Warn_restart, 1);
    }

    /************************************************************
     * Get the result from the Tcl script
     ************************************************************/
    retlen = strlen(betcl_safe_interp->result);
    retval = (text *)palloc(retlen + VARHDRSZ + 1);
    memcpy(VARDATA(retval), betcl_safe_interp->result, retlen + 1);
    VARSIZE(retval) = retlen + VARHDRSZ;
    Tcl_ResetResult(betcl_safe_interp);

    /************************************************************
     * If required reset the safe interpreter
     ************************************************************/
    if(betcl_reset_required) {
        betcl_reset();
    }

    /************************************************************
     * Return the result
     ************************************************************/
    return retval;
}


/**********************************************************************
 * betcl_execsql()		- Callback for queries from the
 *				  Tcl interpreter
 **********************************************************************/
static
int betcl_execsql(ClientData cdata, Tcl_Interp *interp,
	int argc, char *argv[])
{
    betcl_jmp_buf	restart_save;
    char		*pqres;
    PortalBuffer	*portal;
    betcl_result	*result;
    Tcl_HashEntry	*hashent;
    int			hashnew;
    int			i;
    int			j;
    int			n;
    Tcl_DString		tuple_data;
    char		buf[512];
    char		*attr_data;

    /************************************************************
     * Check the syntax
     ************************************************************/
    if(argc != 2) {
        Tcl_SetResult(interp, "syntax error - betcl_execsql <query>",
		TCL_VOLATILE);
        return TCL_ERROR;
    }

    /************************************************************
     * No SQL calls during procedure load
     ************************************************************/
    if(betcl_at_load) {
        Tcl_SetResult(interp, "SQL queries not allowed during procedure load",
		TCL_VOLATILE);
        return TCL_ERROR;
    }

    /************************************************************
     * Prepare for Transaction abort and restart
     ************************************************************/
    memcpy(&restart_save, &Warn_restart, sizeof(Warn_restart));
    if(betcl_setjmp(Warn_restart) != 0) {
        Debug((stderr, "betcl: error on query '%s' - begin restart\n", 
		argv[1]));
	memcpy(&Warn_restart, &restart_save, sizeof(Warn_restart));
	Tcl_SetResult(interp, "Transaction abort", TCL_VOLATILE);
	betcl_restart_in_progress = 1;
	return TCL_ERROR;
    }

    /************************************************************
     * Execute the query, restore the original restart handler
     * and check for error
     ************************************************************/
    pqres = PQexec(argv[1]);
    memcpy(&Warn_restart, &restart_save, sizeof(Warn_restart));
    if(*pqres == 'E') {
        Tcl_AppendResult(interp, "ERROR: ", ++pqres, NULL);
	return TCL_ERROR;
    }

    /************************************************************
     * Setup a new result buffer and hash it. Set the result
     * of the interpreter to the result handle.
     ************************************************************/
    result = (betcl_result *)malloc(sizeof(betcl_result));
    memset(result, 0, sizeof(betcl_result));
    sprintf(result->handle, "betcl_%d", betcl_result_count++);
    result->interp_level = betcl_interp_level;
    hashent = Tcl_CreateHashEntry(betcl_result_hash, 
    		result->handle, &hashnew);
    Tcl_SetHashValue(hashent, (ClientData)result);
    Tcl_SetResult(interp, result->handle, TCL_VOLATILE);

    /************************************************************
     * Now examine if the query returned data
     ************************************************************/
    if(*pqres != 'P') {
	return TCL_OK;
    }

    /************************************************************
     * Get the portal and check if we got data
     ************************************************************/
    portal = PQparray(++pqres);
    result->num_tuples = PQntuples(portal);
    if(result->num_tuples == 0) {
	PQclear(pqres);
	return TCL_OK;
    }

    /************************************************************
     * There is data - get the attribute description. We only
     * examine the first of all tuples - so tuple groups may
     * wire things up.
     ************************************************************/
    result->num_attrs = PQnfields(portal, 0);
    Tcl_DStringInit(&(result->attributes));

    for(i = 0; i < result->num_attrs; i++) {
        sprintf(buf, "%s %d %d", PQfname(portal, 0, i),
				 PQftype(portal, 0, i),
				 PQfsize(portal, 0, i));
        Tcl_DStringAppendElement(&(result->attributes), buf);
    }

    /************************************************************
     * Get the data from the portal into the result->tuple_strings
     ************************************************************/
    result->tuple_strings = 
    	(char **)malloc(sizeof(char *) * result->num_tuples);

    for(i = 0; i < result->num_tuples; i++) {
        Tcl_DStringInit(&tuple_data);
	n = PQnfields(portal, i);
	if(n > result->num_attrs) {
	    n = result->num_attrs;
	}
	for(j = 0; j < n; j++) {
	    attr_data = PQgetvalue(portal, i, j);
	    if(attr_data == NULL) {
	        attr_data = "";
	    }
	    Tcl_DStringAppendElement(&tuple_data, attr_data);
	}
	while(j < result->num_attrs) {
	    Tcl_DStringAppendElement(&tuple_data, "");
	}

	result->tuple_strings[i] = 
		malloc(strlen(Tcl_DStringValue(&tuple_data)) + 1);
        strcpy(result->tuple_strings[i], Tcl_DStringValue(&tuple_data));
	Tcl_DStringFree(&tuple_data);
    }


    /************************************************************
     * Thats it.
     ************************************************************/
    PQclear(pqres);
    return TCL_OK;
}


/**********************************************************************
 * betcl_sqlresult()		- Examine a result buffer
 **********************************************************************/
static
int betcl_sqlresult(ClientData cdata, Tcl_Interp *interp,
	int argc, char *argv[])
{
    int			f_numtuples = 0;
    int			f_attributes = 0;
    int			f_tuplenum = -1;
    int			f_clear = 0;
    int			n_switches = 0;
    Tcl_HashEntry	*hashent;
    Tcl_HashSearch	hashsearch;
    betcl_result	*result;
    int			i;
    char		buf[64];

    /************************************************************
     * Check if we have at least two arguments
     ************************************************************/
    if(argc < 3) {
        Tcl_SetResult(interp, "syntax error - betcl_result <result> options",
		TCL_VOLATILE);
        return TCL_ERROR;
    }

    /************************************************************
     * Get the result buffer
     ************************************************************/
    hashent = Tcl_FindHashEntry(betcl_result_hash, argv[1]);
    if(hashent == NULL) {
        Tcl_AppendResult(interp, "invalid result handle '",
		argv[1], "'", NULL);
        return TCL_ERROR;
    }
    result = (betcl_result *)Tcl_GetHashValue(hashent);

    /************************************************************
     * Get the options
     ************************************************************/
    for(i = 2; i < argc; i++) {
	if(!strcmp(argv[i], "-gettuple")) {
	    if(++i == argc) {
	        Tcl_SetResult(interp, "-numtuples requires a value",
			TCL_VOLATILE);
	        return TCL_ERROR;
	    }
	    if(Tcl_GetInt(interp, argv[i], &f_tuplenum) != TCL_OK) {
	        return TCL_ERROR;
	    }
	    if(f_tuplenum < 0 || f_tuplenum >= result->num_tuples) {
	        Tcl_AppendResult(interp, "tuple number '", argv[i],
			"' out of range", NULL);
	        return TCL_ERROR;
	    }
	    n_switches++;
	    continue;
	}
	if(!strcmp(argv[i], "-numtuples")) {
	    f_numtuples = 1;
	    n_switches++;
	    continue;
	}
	if(!strcmp(argv[i], "-attributes")) {
	    f_attributes = 1;
	    n_switches++;
	    continue;
	}
        if(!strcmp(argv[i], "-clear")) {
	    f_clear = 1;
	    n_switches++;
	    continue;
	}

	Tcl_AppendResult(interp, "unknown option '", argv[i], 
		"'", NULL);
	return TCL_ERROR;
    }

    /************************************************************
     * Final syntax check
     ************************************************************/
    if(n_switches > 1) {
        if(n_switches != 2 || f_clear == 0) {
	    Tcl_SetResult(interp, 
	        "except for -clear all options are exclusive",
		TCL_VOLATILE);
	    return TCL_ERROR;
	}
    }

    /************************************************************
     * Set the result for the option
     ************************************************************/
    if(f_tuplenum >= 0) {
        Tcl_SetResult(interp, result->tuple_strings[f_tuplenum],
		TCL_VOLATILE);
    } else {
        if(f_numtuples) {
	    sprintf(buf, "%d", result->num_tuples);
	    Tcl_SetResult(interp, buf, TCL_VOLATILE);
	} else {
	    if(f_attributes) {
		if(result->num_tuples == 0) {
		    Tcl_ResetResult(interp);
		} else {
	            Tcl_DStringResult(interp, &(result->attributes));
		}
	    }
	}
    }

    /************************************************************
     * Clear the result buffer if requested
     ************************************************************/
    if(f_clear) {
        if(result->num_tuples != 0) {
	    Tcl_DStringFree(&(result->attributes));
	    for(i = 0; i < result->num_tuples; i++) {
	        free(result->tuple_strings[i]);
	    }
	    free(result->tuple_strings);
	}
	Tcl_DeleteHashEntry(hashent);
	free(result);
    }

    return TCL_OK;
}


/**********************************************************************
 * betcl_load()		- The postgres visible load function
 **********************************************************************/
int4 betcl_load(NameData *srcname)
{
    char	buf[NAMEDATALEN + 1];

    /************************************************************
     * Make sure all interpreters are there
     * initialize the module here.
     ************************************************************/
    betcl_initall();

    /************************************************************
     * Load the source
     ************************************************************/
    strncpy(buf, (char *)srcname, NAMEDATALEN);
    buf[NAMEDATALEN] = '\0';

    betcl_load_source(buf);

    return 0;
}


/**********************************************************************
 * betcl_tclload()		- Load another source explicitly
 *				  from inside a Tcl function
 **********************************************************************/
static
int betcl_tclload(ClientData cdata, Tcl_Interp *interp,
	int argc, char *argv[])
{
    betcl_jmp_buf	restart_save;

    if(argc != 2) {
        Tcl_SetResult(interp, "syntax error - betcl_load <srcname>",
		TCL_VOLATILE);
	return TCL_ERROR;
    }

    /************************************************************
     * Prepare for Transaction abort and restart
     ************************************************************/
    memcpy(&restart_save, &Warn_restart, sizeof(Warn_restart));
    if(betcl_setjmp(Warn_restart) != 0) {
        Debug((stderr, "betcl: error on load '%s' - begin restart\n", 
		argv[1]));
	memcpy(&Warn_restart, &restart_save, sizeof(Warn_restart));
	Tcl_SetResult(interp, "Transaction abort", TCL_VOLATILE);
	betcl_restart_in_progress = 1;
	return TCL_ERROR;
    }

    /************************************************************
     * Load the source and restore the restart address
     ************************************************************/
    betcl_load((NameData *)argv[1]);
    memcpy(&Warn_restart, &restart_save, sizeof(Warn_restart));

    return TCL_OK;
}


/**********************************************************************
 * betcl_install_load()		- Install the command for source
 *				  loading in the safe interpreter
 **********************************************************************/
static
int betcl_install_load(ClientData cdata, Tcl_Interp *interp,
	int argc, char *argv[])
{
    if(betcl_safe_interp == NULL) {
        Tcl_SetResult(interp, "slave interpreter not available yet",
		TCL_VOLATILE);
        return TCL_ERROR;
    }

    Tcl_CreateCommand(betcl_safe_interp, "betcl_load",
	    betcl_tclload, NULL, NULL);

    return TCL_OK;
}


/**********************************************************************
 * betcl_install_sql()		- Install the commands for SQL query
 *				  execution in the safe interpreter.
 **********************************************************************/
static
int betcl_install_sql(ClientData cdata, Tcl_Interp *interp,
	int argc, char *argv[])
{
    if(betcl_safe_interp == NULL) {
        Tcl_SetResult(interp, "slave interpreter not available yet",
		TCL_VOLATILE);
        return TCL_ERROR;
    }

    Tcl_CreateCommand(betcl_safe_interp, "betcl_execsql", 
	    betcl_execsql, NULL, NULL);
    Tcl_CreateCommand(betcl_safe_interp, "betcl_sqlresult", 
	    betcl_sqlresult, NULL, NULL);

    return TCL_OK;
}


