#!/bin/sh
# Start wish for this application \
exec wish "$0" "$@"

package require Sqlpqa


#---------------------------------------------------------------------
# Set the default values for database connection
#---------------------------------------------------------------------
if {[catch {set conn_dbuser $env(USER)}]} {
    set conn_dbuser "?"
}
set conn_dbname	$conn_dbuser
set conn_dbhost "localhost"
set conn_dbport ""

set conn_dbname wieck95

wm title . "Postgres95 Tcl Function manager"


#---------------------------------------------------------------------
# Create the frames
#---------------------------------------------------------------------
frame	.all	-borderwidth 0
frame	.all.mf	-borderwidth 2 -relief flat
frame	.all.lf -borderwidth 2 -relief sunken
frame	.all.sf -borderwidth 2 -relief flat
pack	.all.mf -side top -fill x
pack	.all.sf -side bottom -fill x -expand true
pack	.all.lf -side top -fill both -expand true
pack	.all	-side top -fill both -expand true

#---------------------------------------------------------------------
# Create the Database menu
#---------------------------------------------------------------------
menubutton	.all.mf.db	-text "Database" -width 14	\
	-anchor w -borderwidth 0 -menu .all.mf.db.menu
pack	.all.mf.db -side left
menu	.all.mf.db.menu -tearoff false
	.all.mf.db.menu add command -label "Connect"		\
		-command db_connect
	.all.mf.db.menu add command -label "Disconnet"		\
		-command db_disconnect -state disabled
	.all.mf.db.menu add separator
	.all.mf.db.menu add command -label "Quit"		\
		-command {destroy .}

#---------------------------------------------------------------------
# Create the Procedure menu
#---------------------------------------------------------------------
menubutton	.all.mf.proc	-text "Procedure" -width 14	\
	-anchor w -borderwidth 0 -menu .all.mf.proc.menu
pack	.all.mf.proc -side left
menu	.all.mf.proc.menu -tearoff false
	.all.mf.proc.menu add command -label "New"		\
		-command dbproc_new -state disabled
	.all.mf.proc.menu add command -label "Show"		\
		-command dbproc_show -state disabled
	.all.mf.proc.menu add command -label "Edit"		\
		-command dbproc_edit -state disabled
	.all.mf.proc.menu add command -label "Delete"		\
		-command dbproc_delete -state disabled
	.all.mf.proc.menu add separator
	.all.mf.proc.menu add command -label "Activate"		\
		-command dbproc_activate -state disabled
	.all.mf.proc.menu add command -label "Deactivate"	\
		-command dbproc_deactivate -state disabled

#---------------------------------------------------------------------
# Create the listboxes, the frames and the labels for the procedure list
#---------------------------------------------------------------------
frame	.all.lf.proname -borderwidth 0
label	.all.lf.proname.lbl -text "Procedure name:" -anchor w	\
	-borderwidth 0
listbox	.all.lf.proname.lst -width 32 -font fixed -height 30	\
	-borderwidth 1 -relief sunken				\
	-selectmode single

bind .all.lf.proname.lst <ButtonRelease-1> "+sel_proname"
bind .all.lf.proname.lst <Double-Button-1> "+dblsel_proname"

pack	.all.lf.proname.lbl -side top -fill x
pack	.all.lf.proname.lst -side top -fill y -expand true
pack	.all.lf.proname -side left -fill y -expand true

frame	.all.lf.proact -borderwidth 0
label	.all.lf.proact.lbl -text "A:" -anchor w -borderwidth 0
listbox	.all.lf.proact.lst -width 2 -font fixed			\
	-borderwidth 1 -relief sunken
bind .all.lf.proact.lst <ButtonRelease-1> {
    .all.lf.proact.lst selection clear 0 end
    menu_proc_disable
}
pack	.all.lf.proact.lbl -side top -fill x
pack	.all.lf.proact.lst -side top -fill y -expand true
pack	.all.lf.proact -side left -fill y -expand true

frame	.all.lf.prosrc -borderwidth 0
label	.all.lf.prosrc.lbl -text "Procedure source:" -anchor w -borderwidth 0
listbox	.all.lf.prosrc.lst -width 32 -font fixed		\
	-yscrollcommand {.all.lf.scrolly set}			\
	-borderwidth 1 -relief sunken
pack	.all.lf.prosrc.lbl -side top -fill x
pack	.all.lf.prosrc.lst -side top -fill y -expand true
pack	.all.lf.prosrc -side left -fill y -expand true

frame	.all.lf.prodesc -borderwidth 0
label	.all.lf.prodesc.lbl -text "Description:" -anchor w -borderwidth 0
listbox	.all.lf.prodesc.lst -width 40 -font fixed			\
	-borderwidth 1 -relief sunken
pack	.all.lf.prodesc.lbl -side top -fill x
pack	.all.lf.prodesc.lst -side top -fill y -expand true
pack	.all.lf.prodesc -side left -fill y -expand true

#---------------------------------------------------------------------
# Create the scrollbar for the procedure list
#---------------------------------------------------------------------
scrollbar .all.lf.scrolly -orient vertical -width 11			\
	-borderwidth 2 -relief sunken -command proclist_scrolly
pack	.all.lf.scrolly -side right -fill y -expand true
proc proclist_scrolly {args} {
    eval .all.lf.proname.lst yview $args
    eval .all.lf.proact.lst yview $args
    eval .all.lf.prosrc.lst yview $args
    eval .all.lf.prodesc.lst yview $args
}

label	.all.sf.msg -textvariable Msg -borderwidth 1 -relief sunken	\
	-anchor w
pack 	.all.sf.msg -side top -fill x -expand true


#---------------------------------------------------------------------
# message			- Display a new text in the status
#---------------------------------------------------------------------
proc message {args} {
    global Msg
    set Msg [join $args]
    update idletasks
}


#---------------------------------------------------------------------
# db_connect			- Connect to a database
#---------------------------------------------------------------------
proc db_connect {} {
    global	conn_db
    global	conn_dbname
    global	conn_dbhost
    global	conn_dbport
    global	conn_dbuser
    global	conn_action

    toplevel	.dbconn
    frame	.dbconn.all -borderwidth 2 -relief sunken
    pack	.dbconn.all

    label	.dbconn.all.toplbl -text "Connect to database"

    label	.dbconn.all.lbldbname -text "Database name: "
    label	.dbconn.all.lbldbuser -text "Database user: "
    label	.dbconn.all.lbldbhost -text "Database host: "
    label	.dbconn.all.lbldbport -text "Database port: "

    entry	.dbconn.all.entdbname -textvariable conn_dbname		\
    		-width 32 -borderwidth 1 -relief sunken
    entry	.dbconn.all.entdbuser -textvariable conn_dbuser		\
    		-width 32 -borderwidth 1 -relief sunken
    entry	.dbconn.all.entdbhost -textvariable conn_dbhost		\
    		-width 32 -borderwidth 1 -relief sunken
    entry	.dbconn.all.entdbport -textvariable conn_dbport		\
    		-width 32 -borderwidth 1 -relief sunken

    frame	.dbconn.all.bf -borderwidth 4 -relief flat
    button	.dbconn.all.bf.connect -borderwidth 4 -relief groove	\
    		-text "CONNECT" -width 10 -command {
		    global conn_action
		    set conn_action "OK"
		    destroy .dbconn
		}
    button	.dbconn.all.bf.cancel -borderwidth 2 -relief raised	\
    		-text "CANCEL" -width 10 -command {
		    global conn_action
		    set conn_action "CANCEL"
		    destroy .dbconn
		}
    pack	.dbconn.all.bf.connect -side left -padx 4
    pack	.dbconn.all.bf.cancel  -side left -padx 4

    grid configure .dbconn.all.toplbl -in .dbconn.all			\
    		-column 0 -row 0 -sticky nsew -padx 4 -pady 4		\
		-columnspan 2

    grid configure .dbconn.all.lbldbname -in .dbconn.all		\
    		-column 0 -row 1 -sticky w -padx 2 -pady 2
    grid configure .dbconn.all.lbldbuser -in .dbconn.all		\
    		-column 0 -row 2 -sticky w -padx 2 -pady 2
    grid configure .dbconn.all.lbldbhost -in .dbconn.all		\
    		-column 0 -row 3 -sticky w -padx 2 -pady 2
    grid configure .dbconn.all.lbldbport -in .dbconn.all		\
    		-column 0 -row 4 -sticky w -padx 2 -pady 2

    grid configure .dbconn.all.entdbname -in .dbconn.all		\
    		-column 1 -row 1 -sticky ew -padx 2 -pady 2
    grid configure .dbconn.all.entdbuser -in .dbconn.all		\
    		-column 1 -row 2 -sticky ew -padx 2 -pady 2
    grid configure .dbconn.all.entdbhost -in .dbconn.all		\
    		-column 1 -row 3 -sticky ew -padx 2 -pady 2
    grid configure .dbconn.all.entdbport -in .dbconn.all		\
    		-column 1 -row 4 -sticky ew -padx 2 -pady 2

    grid configure .dbconn.all.bf -in .dbconn.all			\
    		-column 0 -row 5 -columnspan 2

    #-----------------------------------------------------------
    # Bindings and focus
    #-----------------------------------------------------------
    bind .dbconn.all.entdbname <Key-Return> {.dbconn.all.bf.connect invoke}
    bind .dbconn.all.entdbuser <Key-Return> {.dbconn.all.bf.connect invoke}
    bind .dbconn.all.entdbhost <Key-Return> {.dbconn.all.bf.connect invoke}
    bind .dbconn.all.entdbport <Key-Return> {.dbconn.all.bf.connect invoke}
    bind .dbconn.all.entdbname <Key-Escape> {.dbconn.all.bf.cancel invoke}
    bind .dbconn.all.entdbuser <Key-Escape> {.dbconn.all.bf.cancel invoke}
    bind .dbconn.all.entdbhost <Key-Escape> {.dbconn.all.bf.cancel invoke}
    bind .dbconn.all.entdbport <Key-Escape> {.dbconn.all.bf.cancel invoke}

    #-----------------------------------------------------------
    # Place the popup
    #-----------------------------------------------------------
    wm transient .dbconn .
    wm withdraw .dbconn
    update idletasks
    set rx [winfo rootx .]
    set ry [winfo rooty .]
    set rw [winfo width .]
    set tw [winfo reqwidth .dbconn]
    set x [expr $rx + $rw / 2 - $tw / 2]
    set y [expr $ry + 50]
    wm geometry .dbconn +${x}+${y}
    wm deiconify .dbconn
    focus .dbconn.all.entdbname
    grab .dbconn.all

    #-----------------------------------------------------------
    # Wait for the user input
    #-----------------------------------------------------------
    set conn_action ""
    vwait conn_action
    if {$conn_action == "CANCEL"} {
        return
    }

    #-----------------------------------------------------------
    # Try to connect to the database
    #-----------------------------------------------------------
    set cmd "SQLconnect -dbname \"$conn_dbname\" -user \"$conn_dbuser\""
    append cmd " -host \"$conn_dbhost\""
    if {$conn_dbport != ""} {
        append cmd " -port \"$conn_dbport\""
    }

    if {[catch {set conn_db [eval $cmd]} errmsg]} {
	error_window "Connection failed!" $errmsg
	return
    }

    #
    # Now we have a valid connection - change the menu states
    #
    .all.mf.db.menu entryconfigure 0 -state disabled	;# Database-Connect
    .all.mf.db.menu entryconfigure 1 -state normal	;# Database-Disconnect

    .all.mf.proc.menu entryconfigure 0 -state normal	;# Proc-New

    message "Connected to database '$conn_dbname'"
    wm title . "${conn_dbuser} on database ${conn_dbname} @${conn_dbhost}"

    update idletasks

    dbproc_rescanlist
}


#---------------------------------------------------------------------
# db_disconnect			- Disconnect from the database
#---------------------------------------------------------------------
proc db_disconnect {} {
    global	conn_db
    global	conn_dbname

    SQLdisconnect $conn_db
    dbproc_clearlist
    .all.mf.db.menu entryconfigure 0 -state normal	;# Database-Connect
    .all.mf.db.menu entryconfigure 1 -state disabled	;# Database-Disconnect

    menu_proc_disable
    .all.mf.proc.menu entryconfigure 0 -state disabled	;# Proc-New

    message "Disconnected from database '$conn_dbname'"
    wm title . "Postgres95 Tcl Function manager"
}


proc menu_proc_disable {} {
    .all.mf.proc.menu entryconfigure 1 -state disabled	;# Proc-Show
    .all.mf.proc.menu entryconfigure 2 -state disabled	;# Proc-Edit
    .all.mf.proc.menu entryconfigure 3 -state disabled	;# Proc-Remove
    .all.mf.proc.menu entryconfigure 5 -state disabled	;# Proc-Activate
    .all.mf.proc.menu entryconfigure 6 -state disabled	;# Proc-Deactivate
}


#---------------------------------------------------------------------
# error_window				- Display a window with a msg
#---------------------------------------------------------------------
proc error_window {title errmsg} {
    toplevel .err
    frame .err.all -borderwidth 2 -relief sunken
    label .err.all.top -text "$title"				\
	    -borderwidth 4 -relief flat

    frame .err.all.msg -borderwidth 0
    text .err.all.msg.txt -borderwidth 1 -relief sunken		\
	    -width 60 -height 8					\
	    -wrap word 						\
	    -yscrollcommand {.err.all.msg.scrolly set}
    .err.all.msg.txt insert end $errmsg
    .err.all.msg.txt configure -state disabled
    scrollbar .err.all.msg.scrolly -orient vertical		\
	    -width 11 -borderwidth 2 -relief sunken		\
	    -command {.err.all.msg.txt yview}
    pack .err.all.msg.scrolly -side right -fill y -expand true
    pack .err.all.msg.txt -side right -fill both -expand true

    button .err.all.ok -text "OK" -borderwidth 4 		\
	    -width 4						\
	    -relief groove -command {destroy .err}
    bind .err.all.ok <Key-Return> {destroy .err}
    pack .err.all.top -side top
    pack .err.all.msg -side top -fill both -expand true
    pack .err.all.ok -side top
    pack .err.all

    #-----------------------------------------------------------
    # Place the errormsg
    #-----------------------------------------------------------
    wm transient .err .
    wm withdraw .err
    update idletasks
    set rx [winfo rootx .]
    set ry [winfo rooty .]
    set rw [winfo width .]
    set tw [winfo reqwidth .err]
    set x [expr $rx + $rw / 2 - $tw / 2]
    set y [expr $ry + 50]
    wm geometry .err +${x}+${y}
    wm deiconify .err
    focus .err.all.ok
    grab .err.all

    tkwait window .err
}


#---------------------------------------------------------------------
# dbproc_clearlist		- Clear the proc list
#---------------------------------------------------------------------
proc dbproc_clearlist {} {
    global proc_list
    global proc_source
    global proc_active
    global proc_desc

    .all.lf.proname.lst delete 0 end
    .all.lf.proact.lst  delete 0 end
    .all.lf.prosrc.lst  delete 0 end
    .all.lf.prodesc.lst delete 0 end

    set proc_list ""
    catch {unset proc_source}
    catch {unset proc_active}
    catch {unset proc_desc}
}


#---------------------------------------------------------------------
# dbproc_rescanlist 		- Update the proc list
#---------------------------------------------------------------------
proc dbproc_rescanlist {} {
    global	proc_list
    global	proc_source
    global	proc_active
    global	proc_desc

    .all configure -cursor watch
    update idletasks

    dbproc_clearlist
    set proc_source(dummy) ""
    set proc_active(dummy) ""
    set proc_desc(dummy) ""
    unset proc_source(dummy)
    unset proc_active(dummy)
    unset proc_desc(dummy)
    update idletasks

    if {[catch {dbproc_getlist} errmsg]} {
        catch {db_disconnect}
	error_window "Cannot get current procedure list!" $errmsg
	return
    }

    foreach proname $proc_list {
        .all.lf.proname.lst insert end $proname
        .all.lf.proact.lst insert end $proc_active($proname)
        .all.lf.prosrc.lst insert end $proc_source($proname)
        .all.lf.prodesc.lst insert end $proc_desc($proname)
    }

    .all configure -cursor {}
}


#---------------------------------------------------------------------
# dbproc_getlist		- Query the database about Tcl procs
#---------------------------------------------------------------------
proc dbproc_getlist {} {
    global	conn_db
    global	proc_list
    global	proc_source
    global	proc_active
    global	proc_desc

    #-----------------------------------------------------------
    # Get all procedures
    #-----------------------------------------------------------
    $conn_db select * from betcl_proc order by prosrc, proname {
	lappend proc_list $proname
        set proc_active($proname) " "
	set proc_source($proname) $prosrc
	set proc_desc($proname) $prodesc
    }

    #-----------------------------------------------------------
    # Lookup if functions with the procedure names exist
    #-----------------------------------------------------------
    $conn_db select proname from pg_proc {
        if {[info exists proc_active($proname)]} {
	    set proc_active($proname) "X"
	}
    }
}


#---------------------------------------------------------------------
# dbproc_new			- Prompt for new proc name and source
#---------------------------------------------------------------------
proc dbproc_new {} {
    global	conn_db
    global	new_action
    global	new_proname
    global	new_prosrc

    set new_proname ""
    set new_prosrc ""

    toplevel	.new
    frame	.new.all -borderwidth 2 -relief sunken

    label	.new.all.toplbl -text "New Tcl Procedure"	\
    		-borderwidth 4 -relief flat

    label	.new.all.lblproname -text "Proc name: "		\
    		-anchor w -borderwidth 0
    label	.new.all.lblprosrc -text "Source name: "	\
    		-anchor w -borderwidth 0

    entry	.new.all.entproname -textvariable new_proname	\
    		-width 32 -borderwidth 1 -relief sunken
    entry	.new.all.entprosrc -textvariable new_prosrc	\
    		-width 32 -borderwidth 1 -relief sunken

    frame	.new.all.bf -borderwidth 4 -relief flat
    button	.new.all.bf.ok -text "OK" -width 10		\
    		-borderwidth 4 -relief groove -command {
		    global new_action

		    set new_action "OK"
		    destroy .new
		}
    button	.new.all.bf.cancel -text "CANCEL" -width 10	\
    		-borderwidth 2 -relief raised -command {
		    global new_action
		    set new_action "CANCEL"
		    destroy .new
		}

    bind .new.all.entproname <Key-Return> {.new.all.bf.ok invoke}
    bind .new.all.entprosrc <Key-Return> {.new.all.bf.ok invoke}
    bind .new.all.entproname <Key-Escape> {.new.all.bf.cancel invoke}
    bind .new.all.entprosrc <Key-Escape> {.new.all.bf.cancel invoke}

    pack	.new.all.bf.ok .new.all.bf.cancel -side left -padx 4

    grid configure .new.all.toplbl -in .new.all			\
    		-column 0 -row 0 -columnspan 2

    grid configure .new.all.lblproname -in .new.all		\
    		-column 0 -row 1 -sticky w
    grid configure .new.all.lblprosrc -in .new.all		\
    		-column 0 -row 2 -sticky w

    grid configure .new.all.entproname -in .new.all		\
    		-column 1 -row 1 -sticky ew
    grid configure .new.all.entprosrc -in .new.all		\
    		-column 1 -row 2 -sticky ew

    grid configure .new.all.bf -in .new.all			\
    		-column 0 -row 3 -columnspan 2

    pack .new.all

    #-----------------------------------------------------------
    # Place the popup
    #-----------------------------------------------------------
    wm transient .new .
    wm withdraw .new
    update idletasks
    set rx [winfo rootx .]
    set ry [winfo rooty .]
    set rw [winfo width .]
    set tw [winfo reqwidth .new]
    set x [expr $rx + $rw / 2 - $tw / 2]
    set y [expr $ry + 50]
    wm geometry .new +${x}+${y}
    wm deiconify .new
    focus .new.all.entproname
    grab .new.all

    #-----------------------------------------------------------
    # Wait for the user input
    #-----------------------------------------------------------
    message "Specify new procedure name and source"
    set new_action ""
    vwait new_action
    if {$new_action == "CANCEL"} {
	message ""
        return
    }
    message ""

    #-----------------------------------------------------------
    # Check that the user specified anything and that the
    # procedure doesn't exist already
    #-----------------------------------------------------------
    set proname [SQLquote [string trim $new_proname]]
    set prosrc  [SQLquote [string trim $new_prosrc]]

    if {$proname == ""} {
        error_window "Missing procedure name" \
		"You must specify a name for the new procedure"
        return
    }
    if {$prosrc == ""} {
        error_window "Missing procedure source" \
		"You must specify the source name for the new procedure"
        return
    }

    if {[$conn_db select single proname as dummy from betcl_proc \
    		where proname = '$proname'] != 0} {
        error_window "Already exists!"	\
		"A procedure named '$proname' already exists"
        return
    }

    #-----------------------------------------------------------
    # Check that there isn't a function declaration in
    # postgres with that name
    #-----------------------------------------------------------
    if {[$conn_db select single proname as dummy from pg_proc	\
    		where proname = '$proname'] != 0} {
        error_window "Already exists!"	\
		[join \
		"A function named '$proname' already exists in pg_proc!
		This might be a built in, sql or C language user function"]
        return
    }

    edit_procedure $proname $prosrc 1
}


#---------------------------------------------------------------------
# edit_procedure		- Edit a procedure definition
#---------------------------------------------------------------------
proc edit_procedure {proname prosrc editmode} {
    global	conn_db
    global	edit_proname
    global	edit_prorett
    global	edit_proargt1
    global	edit_proargt2
    global	edit_proargt3
    global	edit_proargt4
    global	edit_proargt5
    global	edit_proargt6
    global	edit_prodesc
    global	edit_prosrc
    global	edit_action

    .all configure -cursor watch
    message ""

    #-----------------------------------------------------------
    # Get the info about the procedure into global variables
    #-----------------------------------------------------------
    set n [$conn_db select single		\
	prorett  as edit_prorett,		\
    	proargt1 as edit_proargt1,		\
    	proargt2 as edit_proargt2,		\
    	proargt3 as edit_proargt3,		\
    	proargt4 as edit_proargt4,		\
    	proargt5 as edit_proargt5,		\
    	proargt6 as edit_proargt6,		\
	prodesc  as edit_prodesc		\
	from betcl_proc where proname = '$proname']

    if {$n == 0} {
        set edit_prorett "text"
    }

    set edit_proname $proname
    set edit_prosrc $prosrc

    #-----------------------------------------------------------
    # Get the source text
    #-----------------------------------------------------------
    set edit_srctext ""
    $conn_db select srcseq, srctext from betcl_source		\
    		where srcname = '$prosrc' order by srcseq {
        append edit_srctext $srctext
    }


    #-----------------------------------------------------------
    # Create the edit toplevel and the frames
    #-----------------------------------------------------------
    toplevel	.edit
    wm title	.edit "Edit Tcl procedure '$proname'"
    frame	.edit.all -borderwidth 0
    frame	.edit.all.bf -borderwidth 2 -relief flat
    frame	.edit.all.hf -borderwidth 2 -relief sunken
    frame	.edit.all.ef -borderwidth 0
    frame	.edit.all.sf -borderwidth 2 -relief flat

    pack	.edit.all.bf -side top -fill x
    pack	.edit.all.hf -side top -fill x
    pack	.edit.all.sf -side bottom -fill x
    pack	.edit.all.ef -side top -fill both -expand true
    pack	.edit.all -side top -fill both -expand true

    #-----------------------------------------------------------
    # Create the buttons depending on the edit mode
    #-----------------------------------------------------------
    if {$editmode} {
	set editbg "-background white"
        set state normal
	button .edit.all.bf.save -text "SAVE" -width 10		\
		-borderwidth 2 -relief raised -command {
		    global edit_action
		    set edit_action "SAVE"
		}
        pack .edit.all.bf.save -side left -padx 2
	button .edit.all.bf.cancel -text "CANCEL" -width 10		\
		-borderwidth 2 -relief raised -command {
		    global edit_action
		    set edit_action "CANCEL"
		    destroy .edit
		}
        pack .edit.all.bf.cancel -side left -padx 2
    } else {
	set editbg ""
        set state disabled
	button .edit.all.bf.back -text "BACK" -width 10		\
		-borderwidth 2 -relief raised -command {
		    global edit_action
		    set edit_action "CANCEL"
		    destroy .edit
		}
        pack .edit.all.bf.back -side left -padx 2
    }

    #-----------------------------------------------------------
    # Create the header information windows
    #-----------------------------------------------------------
    label	.edit.all.hf.lblproname -text "Procedure: "		\
    		-borderwidth 0 -anchor w
    label	.edit.all.hf.lblprodesc -text "Description: "		\
    		-borderwidth 0 -anchor w
    label	.edit.all.hf.lblprorett -text "Return type: "		\
    		-borderwidth 0 -anchor w
    label	.edit.all.hf.lblproargt -text "Arg types: "		\
    		-borderwidth 0 -anchor w
    label	.edit.all.hf.lblprosrc -text "Source: "		\
    		-borderwidth 0 -anchor w

    label	.edit.all.hf.entproname -textvariable edit_proname	\
    		-borderwidth 1 -relief sunken -width 32			\
		-anchor w -background yellow
    eval entry	.edit.all.hf.entprodesc -textvariable edit_prodesc	\
    		-borderwidth 1 -relief sunken -state $state $editbg
    eval entry	.edit.all.hf.entprorett -textvariable edit_prorett	\
    		-borderwidth 1 -relief sunken -width 32			\
		-state $state $editbg
    eval entry	.edit.all.hf.entproargt1 -textvariable edit_proargt1	\
    		-borderwidth 1 -relief sunken -width 32			\
		-state $state $editbg
    eval entry	.edit.all.hf.entproargt2 -textvariable edit_proargt2	\
    		-borderwidth 1 -relief sunken -width 32			\
		-state $state $editbg
    eval entry	.edit.all.hf.entproargt3 -textvariable edit_proargt3	\
    		-borderwidth 1 -relief sunken -width 32			\
		-state $state $editbg
    eval entry	.edit.all.hf.entproargt4 -textvariable edit_proargt4	\
    		-borderwidth 1 -relief sunken -width 32			\
		-state $state $editbg
    eval entry	.edit.all.hf.entproargt5 -textvariable edit_proargt5	\
    		-borderwidth 1 -relief sunken -width 32			\
		-state $state $editbg
    eval entry	.edit.all.hf.entproargt6 -textvariable edit_proargt6	\
    		-borderwidth 1 -relief sunken -width 32			\
		-state $state $editbg
    label	.edit.all.hf.entprosrc -textvariable edit_prosrc	\
    		-borderwidth 1 -relief sunken -width 32 -anchor w

    #-----------------------------------------------------------
    # Layout the header information
    #-----------------------------------------------------------
    grid columnconfigure .edit.all.hf 0 -weight 0
    grid columnconfigure .edit.all.hf 1 -weight 0
    grid columnconfigure .edit.all.hf 2 -weight 0
    grid columnconfigure .edit.all.hf 3 -weight 0
    grid columnconfigure .edit.all.hf 4 -weight 1

    grid configure .edit.all.hf.lblproname -in .edit.all.hf		\
    		-column 0 -row 0 -sticky ew
    grid configure .edit.all.hf.lblprodesc -in .edit.all.hf		\
    		-column 0 -row 1 -sticky ew
    grid configure .edit.all.hf.lblprorett -in .edit.all.hf		\
    		-column 0 -row 2 -sticky ew
    grid configure .edit.all.hf.lblproargt -in .edit.all.hf		\
    		-column 0 -row 3 -sticky ew
    grid configure .edit.all.hf.lblprosrc -in .edit.all.hf		\
    		-column 0 -row 5 -sticky ew

    grid configure .edit.all.hf.entproname -in .edit.all.hf		\
    		-column 1 -row 0
    grid configure .edit.all.hf.entprodesc -in .edit.all.hf		\
    		-column 1 -row 1 -sticky ew -columnspan 4
    grid configure .edit.all.hf.entprorett -in .edit.all.hf		\
    		-column 1 -row 2

    grid configure .edit.all.hf.entproargt1 -in .edit.all.hf		\
    		-column 1 -row 3
    grid configure .edit.all.hf.entproargt2 -in .edit.all.hf		\
    		-column 2 -row 3
    grid configure .edit.all.hf.entproargt3 -in .edit.all.hf		\
    		-column 3 -row 3
    grid configure .edit.all.hf.entproargt4 -in .edit.all.hf		\
    		-column 1 -row 4
    grid configure .edit.all.hf.entproargt5 -in .edit.all.hf		\
    		-column 2 -row 4
    grid configure .edit.all.hf.entproargt6 -in .edit.all.hf		\
    		-column 3 -row 4

    grid configure .edit.all.hf.entprosrc -in .edit.all.hf		\
    		-column 1 -row 5


    #-----------------------------------------------------------
    # Create the source editing window
    #-----------------------------------------------------------
    text	.edit.all.ef.txt -width 80 -height 20			\
    		-borderwidth 2 -relief sunken -wrap none		\
		-yscrollcommand {.edit.all.ef.scrolly set}		\
		-xscrollcommand {.edit.all.ef.scrollx set}
    .edit.all.ef.txt insert end $edit_srctext
    .edit.all.ef.txt mark set insert 1.0
    .edit.all.ef.txt configure -state $state
    bind .edit.all.ef.txt <KeyRelease-Return> {+edit_autoindent}
    scrollbar	.edit.all.ef.scrolly -orient vertical -width 11		\
    		-borderwidth 2 -relief sunken				\
		-command {.edit.all.ef.txt yview}
    scrollbar	.edit.all.ef.scrollx -orient horizontal -width 11	\
    		-borderwidth 2 -relief sunken				\
		-command {.edit.all.ef.txt xview}
    pack .edit.all.ef.scrollx -side bottom -fill x
    pack .edit.all.ef.scrolly -side right -fill y
    pack .edit.all.ef.txt -side left -fill both -expand true

    #-----------------------------------------------------------
    # Create the editor message line
    #-----------------------------------------------------------
    label	.edit.all.sf.msg -text "" -borderwidth 1 -relief sunken	\
    		-anchor w
    pack	.edit.all.sf.msg -side top -fill x -expand true

    #-----------------------------------------------------------
    # Place the editor
    #-----------------------------------------------------------
    wm transient .edit .
    wm withdraw .edit
    update idletasks
    set rx [winfo rootx .]
    set ry [winfo rooty .]
    set rw [winfo width .]
    set tw [winfo reqwidth .edit]
    set x [expr $rx + $rw / 2 - $tw / 2]
    set y $ry
    wm geometry .edit +${x}+${y}
    wm deiconify .edit
    focus .edit.all.hf.entprodesc
    grab .edit.all

    #-----------------------------------------------------------
    # Wait until the user finished editing/viewing
    #-----------------------------------------------------------
    while {1} {
	vwait edit_action
	if {$edit_action == "SAVE"} {
	    edit_checksave
	}
	if {$edit_action == "CANCEL"} {
	    .all configure -cursor {}
	    return
	}
    }
}


proc edit_autoindent {} {
    set curidx [.edit.all.ef.txt index insert]
    if {[lindex [split $curidx .] 0] < 2} {
        return
    }
    set lidx [.edit.all.ef.txt index "insert -1 line linestart"]
    set line [.edit.all.ef.txt get $lidx "$lidx lineend"]
    regexp {^[ 	]*} $line ins
    .edit.all.ef.txt insert insert $ins
}


#---------------------------------------------------------------------
# edit_checksave		- Check all input and save the proc
#---------------------------------------------------------------------
proc edit_checksave {} {
    global	conn_db
    global	edit_proname
    global	edit_prorett
    global	edit_proargt1
    global	edit_proargt2
    global	edit_proargt3
    global	edit_proargt4
    global	edit_proargt5
    global	edit_proargt6
    global	edit_prodesc
    global	edit_prosrc
    global	edit_action

    #-----------------------------------------------------------
    # Trim all input fields
    #-----------------------------------------------------------
    set edit_prodesc [string trim $edit_prodesc]
    set edit_prorett [string trim $edit_prorett]
    set edit_proargt1 [string trim $edit_proargt1]
    set edit_proargt2 [string trim $edit_proargt2]
    set edit_proargt3 [string trim $edit_proargt3]
    set edit_proargt4 [string trim $edit_proargt4]
    set edit_proargt5 [string trim $edit_proargt5]
    set edit_proargt6 [string trim $edit_proargt6]
    set edit_srctext [.edit.all.ef.txt get 1.0 end]
    regsub "\n\$" $edit_srctext "" edit_srctext

    #-----------------------------------------------------------
    # Force the user to enter a description
    #-----------------------------------------------------------
    if {$edit_prodesc == ""} {
        .edit.all.sf.msg configure -text \
		"Please enter a description line"
        focus .edit.all.hf.entprodesc
	return
    }

    #-----------------------------------------------------------
    # Validate the return type
    #-----------------------------------------------------------
    if {$edit_prorett == ""} {
        .edit.all.sf.msg configure -text \
		"All Postgres function must have a return type"
        focus .edit.all.hf.entprorett
	return
    }
    if {[catch {set rett_conv [edit_checktype $edit_prorett]} errmsg]} {
        .edit.all.sf.msg configure -text $errmsg
        focus .edit.all.hf.entprorett
	return
    }

    set pronargs 0
    #-----------------------------------------------------------
    # Validate the argument type 1
    #-----------------------------------------------------------
    if {$edit_proargt1 != ""} {
	if {[catch {set argt1_conv [edit_checktype $edit_proargt1]} errmsg]} {
	    .edit.all.sf.msg configure -text $errmsg
	    focus .edit.all.hf.entproargt1
	    return
	}
	set pronargs 1
    }

    #-----------------------------------------------------------
    # Validate the argument type 2
    #-----------------------------------------------------------
    if {$edit_proargt2 != ""} {
	if {$pronargs != 1} {
	    .edit.all.sf.msg configure -text \
	    	"Previous argument type empty - arguments must be contigous"
	    focus .edit.all.hf.entproargt2
	    return
	}
	if {[catch {set argt1_conv [edit_checktype $edit_proargt2]} errmsg]} {
	    .edit.all.sf.msg configure -text $errmsg
	    focus .edit.all.hf.entproargt2
	    return
	}
	set pronargs 2
    }

    #-----------------------------------------------------------
    # Validate the argument type 3
    #-----------------------------------------------------------
    if {$edit_proargt3 != ""} {
	if {$pronargs != 2} {
	    .edit.all.sf.msg configure -text \
	    	"Previous argument type empty - arguments must be contigous"
	    focus .edit.all.hf.entproargt3
	    return
	}
	if {[catch {set argt1_conv [edit_checktype $edit_proargt3]} errmsg]} {
	    .edit.all.sf.msg configure -text $errmsg
	    focus .edit.all.hf.entproargt3
	    return
	}
	set pronargs 3
    }

    #-----------------------------------------------------------
    # Validate the argument type 4
    #-----------------------------------------------------------
    if {$edit_proargt4 != ""} {
	if {$pronargs != 3} {
	    .edit.all.sf.msg configure -text \
	    	"Previous argument type empty - arguments must be contigous"
	    focus .edit.all.hf.entproargt4
	    return
	}
	if {[catch {set argt1_conv [edit_checktype $edit_proargt4]} errmsg]} {
	    .edit.all.sf.msg configure -text $errmsg
	    focus .edit.all.hf.entproargt4
	    return
	}
	set pronargs 4
    }

    #-----------------------------------------------------------
    # Validate the argument type 5
    #-----------------------------------------------------------
    if {$edit_proargt5 != ""} {
	if {$pronargs != 4} {
	    .edit.all.sf.msg configure -text \
	    	"Previous argument type empty - arguments must be contigous"
	    focus .edit.all.hf.entproargt5
	    return
	}
	if {[catch {set argt1_conv [edit_checktype $edit_proargt5]} errmsg]} {
	    .edit.all.sf.msg configure -text $errmsg
	    focus .edit.all.hf.entproargt5
	    return
	}
	set pronargs 5
    }

    #-----------------------------------------------------------
    # Validate the argument type 6
    #-----------------------------------------------------------
    if {$edit_proargt6 != ""} {
	if {$pronargs != 5} {
	    .edit.all.sf.msg configure -text \
	    	"Previous argument type empty - arguments must be contigous"
	    focus .edit.all.hf.entproargt6
	    return
	}
	if {[catch {set argt1_conv [edit_checktype $edit_proargt6]} errmsg]} {
	    .edit.all.sf.msg configure -text $errmsg
	    focus .edit.all.hf.entproargt6
	    return
	}
	set pronargs 6
    }

    #-----------------------------------------------------------
    # If the procedure already existed try to drop an existing
    # function for it
    #-----------------------------------------------------------
    set n [$conn_db select single 		\
		pronargs as old_nargs,		\
    		proargt1 as old_argt1,		\
    		proargt2 as old_argt2,		\
    		proargt3 as old_argt3,		\
    		proargt4 as old_argt4,		\
    		proargt5 as old_argt5,		\
    		proargt6 as old_argt6		\
		from betcl_proc where proname = '$edit_proname']
    if {$n > 0} {
        set old_arglist ""
	if {$old_nargs > 0} {append old_arglist $old_argt1}
	if {$old_nargs > 1} {append old_arglist ", $old_argt2"}
	if {$old_nargs > 2} {append old_arglist ", $old_argt3"}
	if {$old_nargs > 3} {append old_arglist ", $old_argt4"}
	if {$old_nargs > 4} {append old_arglist ", $old_argt5"}
	if {$old_nargs > 5} {append old_arglist ", $old_argt6"}
	catch {$conn_db drop function 		\
		[SQLquote $edit_proname] ( [SQLquote $old_arglist] )}
    }

    #-----------------------------------------------------------
    # Anything's fine with the input - save the procedure
    #-----------------------------------------------------------
    $conn_db begin

    $conn_db delete from betcl_proc where proname = '$edit_proname'
    $conn_db delete from betcl_source where srcname = '$edit_prosrc'
    $conn_db insert into betcl_proc values (		\
		'[SQLquote $edit_proname]',		\
		$pronargs,				\
		'[SQLquote $edit_prosrc]',		\
		'[SQLquote $edit_prorett]',		\
		'[SQLquote $edit_proargt1]',		\
		'[SQLquote $edit_proargt2]',		\
		'[SQLquote $edit_proargt3]',		\
		'[SQLquote $edit_proargt4]',		\
		'[SQLquote $edit_proargt5]',		\
		'[SQLquote $edit_proargt6]',		\
		'[SQLquote $edit_prodesc]')

    set seq 0
    while {$edit_srctext != ""} {
        set chunk [string range $edit_srctext 0 7999]
	set edit_srctext [string range $edit_srctext 8000 end]
	$conn_db insert into betcl_source values (	\
		'[SQLquote $edit_prosrc]',		\
		$seq,					\
		'[SQLquote $chunk]')
        incr seq
    }
    $conn_db commit

    message "Procedure '$edit_proname' saved - don't forget to activate"
    set edit_action "CANCEL"
    destroy .edit
    dbproc_rescanlist
}


#---------------------------------------------------------------------
# edit_checktype		- Check a data type
#---------------------------------------------------------------------
proc edit_checktype {type} {
    global conn_db

    set type [SQLquote $type]

    if {[$conn_db select single typinput, typoutput from pg_type	\
    		where typname = '$type'] != 1} {
        return -code error "Postgres type '$type' does not exist"
    }

    if {[$conn_db select single proname from pg_proc 			\
       		where proname = '$typinput'] != 1} {
        return -code error \
"Postgres input conversion function '$typinput' for type '$type' not found"
    }
    if {[$conn_db select single proname from pg_proc 			\
       		where proname = '$typoutput'] != 1} {
        return -code error \
"Postgres output conversion function '$typinput' for type '$type' not found"
    }

    return [list $typinput $typoutput]
}


proc sel_proname {} {
    global proc_active

    set sel [.all.lf.proname.lst curselection]
    if {$sel == ""} {
	.all.mf.proc.menu entryconfigure 1 -state disabled
	.all.mf.proc.menu entryconfigure 2 -state disabled
	.all.mf.proc.menu entryconfigure 3 -state disabled
	.all.mf.proc.menu entryconfigure 5 -state disabled
	.all.mf.proc.menu entryconfigure 6 -state disabled
        return
    }

    set proname [.all.lf.proname.lst get [lindex $sel 0]]
    if {![info exists proc_active($proname)]} {
	.all.mf.proc.menu entryconfigure 1 -state disabled
	.all.mf.proc.menu entryconfigure 2 -state disabled
	.all.mf.proc.menu entryconfigure 3 -state disabled
	.all.mf.proc.menu entryconfigure 5 -state disabled
	.all.mf.proc.menu entryconfigure 6 -state disabled
        return
    }
    .all.mf.proc.menu entryconfigure 1 -state normal
    .all.mf.proc.menu entryconfigure 2 -state normal
    .all.mf.proc.menu entryconfigure 3 -state normal
    if {$proc_active($proname) == " "} {
	.all.mf.proc.menu entryconfigure 5 -state normal
	.all.mf.proc.menu entryconfigure 6 -state disabled
    } else {
	.all.mf.proc.menu entryconfigure 5 -state disabled
	.all.mf.proc.menu entryconfigure 6 -state normal
    }
}


proc dblsel_proname {} {
    dbproc_edit
}


proc dbproc_edit {} {
    global proc_source

    set sel [.all.lf.proname.lst curselection]
    if {$sel == ""} {
        return
    }
    set proname [.all.lf.proname.lst get [lindex $sel 0]]
    if {![info exists proc_source($proname)]} {
        return
    }

    edit_procedure $proname $proc_source($proname) 1
}


proc dbproc_show {} {
    global proc_source

    set sel [.all.lf.proname.lst curselection]
    if {$sel == ""} {
        return
    }
    set proname [.all.lf.proname.lst get [lindex $sel 0]]
    if {![info exists proc_source($proname)]} {
        return
    }

    edit_procedure $proname $proc_source($proname) 0
}


#---------------------------------------------------------------------
# dbproc_activate		- Activate a Tcl function
#---------------------------------------------------------------------
proc dbproc_activate {} {
    global conn_db
    global proc_source

    set sel [.all.lf.proname.lst curselection]
    if {$sel == ""} {
        return
    }
    set proname [.all.lf.proname.lst get [lindex $sel 0]]
    if {![info exists proc_source($proname)]} {
        return
    }

    .all configure -cursor watch

    #-----------------------------------------------------------
    # Get the function description
    #-----------------------------------------------------------
    set n [$conn_db select single 		\
		pronargs as nargs,		\
		prorett  as rett,		\
    		proargt1 as argt1,		\
    		proargt2 as argt2,		\
    		proargt3 as argt3,		\
    		proargt4 as argt4,		\
    		proargt5 as argt5,		\
    		proargt6 as argt6		\
		from betcl_proc where proname = '$proname']
    if {$n == 0} {
        message "Procedure not found!"
	.all configure -cursor {}
	return
    }

    #-----------------------------------------------------------
    # Setup our argument list descriptions
    #-----------------------------------------------------------
    set create_arglist ""
    set conv_arglist ""
    if {$nargs > 0} {
	append create_arglist $argt1
	lappend conv_arglist  $argt1
    }
    if {$nargs > 1} {
	append create_arglist ", $argt2"
	lappend conv_arglist  $argt2
    }
    if {$nargs > 2} {
	append create_arglist ", $argt3"
	lappend conv_arglist  $argt3
    }
    if {$nargs > 3} {
	append create_arglist ", $argt4"
	lappend conv_arglist  $argt4
    }
    if {$nargs > 4} {
	append create_arglist ", $argt5"
	lappend conv_arglist  $argt5
    }
    if {$nargs > 5} {
	append create_arglist ", $argt6"
	lappend conv_arglist  $argt6
    }

    #-----------------------------------------------------------
    # Get the input converter for the returntype
    #-----------------------------------------------------------
    if {[catch {set rett_conv [edit_checktype $rett]} errmsg]} {
        message $errmsg
	.all configure -cursor {}
	return
    }

    #-----------------------------------------------------------
    # Create the SQL command to define the function
    #-----------------------------------------------------------
    set cmd "create function $proname ($create_arglist) "
    append cmd "returns $rett as '"
    if {$rett == "text"} {
        append cmd "select betcl_call("
    } else {
	append cmd "select [lindex $rett_conv 0](textout(betcl_call("
    }
    append cmd "\\'[SQLquote $proname]\\', $nargs, "
    set argidx 1
    foreach argt $conv_arglist {
        if {[catch {set t_conv [edit_checktype $argt]} errmsg]} {
	    message $errmsg
	    .all configure -cursor {}
	    return
	}
	if {$argt == "text"} {
	    append cmd "\$$argidx"
	} else {
	    append cmd "textin([lindex $t_conv 1](\$${argidx}))"
	}
	incr argidx
	if {$argidx < 7} {
	    append cmd ", "
	}
    }
    while {$argidx < 7} {
        append cmd "\\'\\'"
	incr argidx
	if {$argidx < 7} {
	    append cmd ", "
	}
    }
    if {$rett != "text"} {
        append cmd "))"
    }
    append cmd ")' language 'sql'"

    #-----------------------------------------------------------
    # Send this command to the database and update the display
    #-----------------------------------------------------------
    if {[catch {$conn_db $cmd} errmsg]} {
puts stderr "cmd:\n$cmd"
        message "Activation of '$proname' failed"
	error_window "Activation failed!" $errmsg
	.all configure -cursor {}
	return
    }

    message "Function '$proname' succesfully activated"
    dbproc_rescanlist
}


#---------------------------------------------------------------------
# dbproc_deactivate		- Deactivate a Tcl function
#---------------------------------------------------------------------
proc dbproc_deactivate {} {
    global conn_db
    global proc_source

    set sel [.all.lf.proname.lst curselection]
    if {$sel == ""} {
        return
    }
    set proname [.all.lf.proname.lst get [lindex $sel 0]]
    if {![info exists proc_source($proname)]} {
        return
    }

    .all configure -cursor watch

    #-----------------------------------------------------------
    # Get the function description
    #-----------------------------------------------------------
    set n [$conn_db select single 		\
		pronargs as nargs,		\
		prorett  as rett,		\
    		proargt1 as argt1,		\
    		proargt2 as argt2,		\
    		proargt3 as argt3,		\
    		proargt4 as argt4,		\
    		proargt5 as argt5,		\
    		proargt6 as argt6		\
		from betcl_proc where proname = '$proname']
    if {$n == 0} {
        message "Procedure not found!"
	.all configure -cursor {}
	return
    }

    #-----------------------------------------------------------
    # Setup our argument list descriptions
    #-----------------------------------------------------------
    set create_arglist ""
    if {$nargs > 0} {
	append create_arglist $argt1
    }
    if {$nargs > 1} {
	append create_arglist ", $argt2"
    }
    if {$nargs > 2} {
	append create_arglist ", $argt3"
    }
    if {$nargs > 3} {
	append create_arglist ", $argt4"
    }
    if {$nargs > 4} {
	append create_arglist ", $argt5"
    }
    if {$nargs > 5} {
	append create_arglist ", $argt6"
    }

    #-----------------------------------------------------------
    # Create the SQL command to drop the function
    #-----------------------------------------------------------
    set cmd "drop function $proname ($create_arglist)"

    #-----------------------------------------------------------
    # Send this command to the database and update the display
    #-----------------------------------------------------------
    if {[catch {$conn_db $cmd} errmsg]} {
        message "Activation of '$proname' failed"
	error_window "Activation failed!" $errmsg
	.all configure -cursor {}
	return
    }

    message "Function '$proname' succesfully dopped"
    dbproc_rescanlist
}


