#!/bin/sh
#Start tclsh for this script \
exec tclsh "$0" "$@"

#---------------------------------------------------------------------
# betclinstall			- Install Tcl backend support into an
#				  existing database.
#
# IDENTIFICATION
# $Header: /home/wieck/src/tcltk/extensions/pqatcl/src/libbetcl/RCS/betclinstall.tcl,v 1.1 1996/10/09 08:41:59 wieck Exp $
#
# HISTORY
# $Log: betclinstall.tcl,v $
# Revision 1.1  1996/10/09 08:41:59  wieck
# Initial revision
#
#
#---------------------------------------------------------------------

#---------------------------------------------------------------------
# Make sure we have all resources
#
# We need the Sqlpqa package and the HOME and PGDATA environment vars.
#---------------------------------------------------------------------
package require Sqlpqa

if {![info exists env(PGDATA)]} {
    puts stderr "Environment variable PGDATA not set"
    exit 1
}
if {![info exists env(HOME)]} {
    puts stderr "Environment variable HOME not set"
    exit 1
}

#---------------------------------------------------------------------
# Check that some files used later exist
#---------------------------------------------------------------------
set script_main	BETCL_main_script.tcl
set script_safe BETCL_safe_script.tcl
set script_main_path $env(HOME)/lib/$script_main
set script_safe_path $env(HOME)/lib/$script_safe
set betcl_sl_path    $env(HOME)/lib/libbetcl.so.1

if {![file exists $script_main_path]} {
    puts stderr "file '$script_main_path' does not exist"
    exit 1
}
if {![file exists $script_safe_path]} {
    puts stderr "file '$script_safe_path' does not exist"
    exit 1
}
if {![file exists $betcl_sl_path]} {
    puts stderr "file '$betcl_sl_path' does not exist"
    exit 1
}

#---------------------------------------------------------------------
# Get the current user name from the OS for later privilege check
#---------------------------------------------------------------------
set current_user [exec /usr/bin/whoami]

#---------------------------------------------------------------------
# Get the database name and check call syntax
#---------------------------------------------------------------------
if {$argc > 1} {
    puts stderr "usage: $argv0 [dbname]"
    exit 1
}
if {$argc == 0} {
    puts -nonewline "Database to install backend Tcl module in : "
    flush stdout
    if {[catch {set dbname [gets stdin]}]} {
        exit 2
    }
    if {[eof stdin]} {
        puts ""
    }
} else {
    set dbname [lindex $argv 0]
}
if {$dbname == ""} {
    puts stderr "Database name cannot be empty"
    exit 2
}


#---------------------------------------------------------------------
# Connect to the database
#---------------------------------------------------------------------
if {[catch {set db [SQLconnect -dbname $dbname]} errmsg]} {
    puts stderr "Cannot connect to database '$dbname'"
    puts stderr ""
    puts stderr "Diagnostic:"
    puts stderr "$errmsg"
    exit 3
}

#---------------------------------------------------------------------
# Check privileges
#
# The user must be permitted to create databases. If he's also
# permitted to create users, he can install Tcl support into any
# database. Else only into databases owned by himself.
#---------------------------------------------------------------------
set n [$db select single usecreatedb, usesuper from pg_user \
	where usename = '$current_user']
if {$n == 0} {
    puts stderr "Postgres user '$current_user' does not exist"
    SQLdisconnect $db
    exit 4
}
if {$usecreatedb != "t"} {
    puts stderr "User '$current_user' is not allowed to create databases."
    puts stderr "This privilege is required to install betcl functionality."
    puts stderr "Sorry."
    SQLdisconnect $db
    exit 4
}
if {$usesuper != "t"} {
    set n [$db select single U.usename from	\
    		pg_database D, pg_user U	\
		where D.datname = '$dbname'	\
		and D.datdba = U.usesysid ]
    if {$n != 1} {
        puts stderr "Cannot determine owner of database '$dbname'"
	SQLdisconnect $db
	exit 4
    }
    if {$usename != $current_user} {
        puts stderr "User '$current_user' is not a postgres superuser and"
	puts stderr "database '$dbname' is owned by '$usename'."
	puts stderr "Sorry."
	SQLdisconnect $db
	exit 4
    }
}

#---------------------------------------------------------------------
# Make sure there are currently no tables beginning with 'betcl_'
# as a magic flag that the Tcl support is already installed.
#---------------------------------------------------------------------
$db select single count(pg_class.relname) as n where relname ~ '^betcl_'
if {$n != 0} {
    puts stderr "There are already tables beginning with 'betcl_' in"
    puts stderr "database '$dbname'."
    puts stderr "Installation aborted."
    SQLdisconnect $db
    exit 4
}

#---------------------------------------------------------------------
# Copy the Tcl interpreter startup files into the database dir
#---------------------------------------------------------------------
if {[catch {
exec cp $script_main_path $env(PGDATA)/base/$dbname/$script_main
exec cp $script_safe_path $env(PGDATA)/base/$dbname/$script_safe
} errmsg]} {
    puts stderr "Installation aborted due to error:"
    puts stderr "$errmsg"
    SQLdisconnect $db
    exit 5
}

#---------------------------------------------------------------------
# Create the tables and functions for the Tcl support module
#---------------------------------------------------------------------
if {[catch {
$db begin
$db create table betcl_proc (	\
	proname		name,	\
	pronargs	int4,	\
	prosrc		name,	\
	prorett		name,	\
	proargt1	name,	\
	proargt2	name,	\
	proargt3	name,	\
	proargt4	name,	\
	proargt5	name,	\
	proargt6	name,	\
	prodesc		text)
$db create table betcl_source (	\
	srcname		name,	\
	srcseq		int4,	\
	srctext		text)
$db create function betcl_call 						\
	(name, int4, text, text, text, text, text, text)		\
	returns text as '$betcl_sl_path' language 'c'
$db create function betcl_load (name)					\
	returns int4 as '$betcl_sl_path' language 'c'
$db create function betcl_reset ()					\
	returns int4 as '$betcl_sl_path' language 'c'

$db commit
} errmsg]} {
    puts stderr "Installation aborted due to error:"
    puts stderr "$errmsg"
    SQLdisconnect $db
    exit 5
}

#---------------------------------------------------------------------
# That's it.
#---------------------------------------------------------------------
puts "Installation in database '$dbname' successful"

SQLdisconnect $db
exit 0
