#!/bin/sh
#Start tclsh for this script \
exec tclsh "$0" "$@"

#---------------------------------------------------------------------
# betcluninstall		- Uninstall Tcl backend support from a
#				  database.
#
# IDENTIFICATION
# $Header: /home/wieck/src/tcltk/extensions/pqatcl/src/libbetcl/RCS/betcluninstall.tcl,v 1.1 1996/10/09 08:42:19 wieck Exp $
#
# HISTORY
# $Log: betcluninstall.tcl,v $
# Revision 1.1  1996/10/09 08:42:19  wieck
# Initial revision
#
#
#---------------------------------------------------------------------

#---------------------------------------------------------------------
# Make sure we have all resources
#
# We need the Sqlpqa package and the PGDATA environment variable.
#---------------------------------------------------------------------
package require Sqlpqa

if {![info exists env(PGDATA)]} {
    puts stderr "Environment variable PGDATA not set"
    exit 1
}

#---------------------------------------------------------------------
# Get the current user name from the OS for later privilege check
#---------------------------------------------------------------------
set current_user [exec /usr/bin/whoami]

#---------------------------------------------------------------------
# Get the database name and check call syntax
#---------------------------------------------------------------------
if {$argc > 1} {
    puts stderr "usage: $argv0 [dbname]"
    exit 1
}
if {$argc == 0} {
    puts -nonewline "Database to uninstall backend Tcl module from : "
    flush stdout
    if {[catch {set dbname [gets stdin]}]} {
        exit 2
    }
    if {[eof stdin]} {
        puts ""
    }
} else {
    set dbname [lindex $argv 0]
}
if {$dbname == ""} {
    puts stderr "Database name cannot be empty"
    exit 2
}


#---------------------------------------------------------------------
# Check that the Tcl startup scripts exist in the database dir
#---------------------------------------------------------------------
set script_main	BETCL_main_script.tcl
set script_safe BETCL_safe_script.tcl
set script_main_path $env(PGDATA)/base/$dbname/$script_main
set script_safe_path $env(PGDATA)/base/$dbname/$script_safe
if {![file exists $script_main_path]} {
    puts stderr "file '$script_main_path' does not exist"
    exit 1
}
if {![file exists $script_safe_path]} {
    puts stderr "file '$script_safe_path' does not exist"
    exit 1
}

#---------------------------------------------------------------------
# Connect to the database
#---------------------------------------------------------------------
if {[catch {set db [SQLconnect -dbname $dbname]} errmsg]} {
    puts stderr "Cannot connect to database '$dbname'"
    puts stderr ""
    puts stderr "Diagnostic:"
    puts stderr "$errmsg"
    exit 3
}

#---------------------------------------------------------------------
# Check privileges
#
# The user must be permitted to create databases. If he's also
# permitted to create users, he can uninstall Tcl support from any
# database. Else only from databases owned by himself.
#---------------------------------------------------------------------
set n [$db select single usecreatedb, usesuper from pg_user \
	where usename = '$current_user']
if {$n == 0} {
    puts stderr "Postgres user '$current_user' does not exist"
    SQLdisconnect $db
    exit 4
}
if {$usecreatedb != "t"} {
    puts stderr "User '$current_user' is not allowed to create databases."
    puts stderr "This privilege is required to uninstall betcl functionality."
    puts stderr "Sorry."
    SQLdisconnect $db
    exit 4
}
if {$usesuper != "t"} {
    set n [$db select single U.usename from	\
    		pg_database D, pg_user U	\
		where D.datname = '$dbname'	\
		and D.datdba = U.usesysid ]
    if {$n != 1} {
        puts stderr "Cannot determine owner of database '$dbname'"
	SQLdisconnect $db
	exit 4
    }
    if {$usename != $current_user} {
        puts stderr "User '$current_user' is not a postgres superuser and"
	puts stderr "database '$dbname' is owned by '$usename'."
	puts stderr "Sorry."
	SQLdisconnect $db
	exit 4
    }
}

#---------------------------------------------------------------------
# Make sure the table 'betcl_proc' and 'betcl_source' exist
# as a magic flag that the Tcl support is installed.
#---------------------------------------------------------------------
$db select single count(pg_class.relname) as n 	\
	where relname = 'betcl_proc'		\
	   or relname = 'betcl_source'
if {$n != 2} {
    puts stderr "Tables 'betcl_proc' or 'betcl_source' not found in"
    puts stderr "database '$dbname'."
    puts stderr "Uninstallation aborted."
    SQLdisconnect $db
    exit 4
}

#---------------------------------------------------------------------
# Make sure the user really wants this
#---------------------------------------------------------------------
puts ""
puts "WARNING:"
puts "This will not only disable the Tcl support in database '$dbname'."
puts "Since the tables 'betcl_proc' and 'betcl_source' will be dropped,"
puts "all Tcl function sources will get lost!"
puts ""
puts -nonewline "Continue and destroy data \[y/N\] ? "
flush stdout
if {[catch {set ans [gets stdin]}]} {
    puts stderr ""
    puts stderr "Uninstall aborted by user"
    SQLdisconnect $db
    exit 1
}
if {[string toupper [string index [string trim $ans] 0]] != "Y"} {
    puts stderr ""
    puts stderr "Uninstall aborted by user"
    SQLdisconnect $db
    exit 1
}

#---------------------------------------------------------------------
# Drop the tables and functions for the Tcl support module
#---------------------------------------------------------------------
if {[catch {
$db begin
$db drop function betcl_call						\
	(name, int4, text, text, text, text, text, text)
$db drop function betcl_load (name)
$db drop function betcl_reset ()
$db drop table betcl_proc
$db drop table betcl_source

$db commit
} errmsg]} {
    puts stderr "Uninstallation aborted due to error:"
    puts stderr "$errmsg"
    SQLdisconnect $db
    exit 5
}

#---------------------------------------------------------------------
# Remove the Tcl interpreter startup files
#---------------------------------------------------------------------
exec rm -f $env(PGDATA)/base/$dbname/$script_main
exec rm -f $env(PGDATA)/base/$dbname/$script_safe

#---------------------------------------------------------------------
# That's it.
#---------------------------------------------------------------------
puts "Uninstallation in database '$dbname' successful"

SQLdisconnect $db
exit 0



set script_main	BETCL_main_script.tcl
set script_safe BETCL_safe_script.tcl
set betcl_sl_path    $env(HOME)/lib/libbetcl.so.1

if {![file exists $betcl_sl_path]} {
    puts stderr "file '$betcl_sl_path' does not exist"
    exit 1
}

#---------------------------------------------------------------------
# Get the current user name from the OS for later privilege check
#---------------------------------------------------------------------
set current_user [exec /usr/bin/whoami]

#---------------------------------------------------------------------
# Get the database name and check call syntax
#---------------------------------------------------------------------
if {$argc > 1} {
    puts stderr "usage: $argv0 [dbname]"
    exit 1
}
if {$argc == 0} {
    puts -nonewline "Database to install backend Tcl module in : "
    flush stdout
    if {[catch {set dbname [gets stdin]}]} {
        exit 2
    }
    if {[eof stdin]} {
        puts ""
    }
} else {
    set dbname [lindex $argv 0]
}
if {$dbname == ""} {
    puts stderr "Database name cannot be empty"
    exit 2
}


#---------------------------------------------------------------------
# Connect to the database
#---------------------------------------------------------------------
if {[catch {set db [SQLconnect -dbname $dbname]} errmsg]} {
    puts stderr "Cannot connect to database '$dbname'"
    puts stderr ""
    puts stderr "Diagnostic:"
    puts stderr "$errmsg"
    exit 3
}

#---------------------------------------------------------------------
# Check privileges
#
# The user must be permitted to create databases. If he's also
# permitted to create users, he can install Tcl support into any
# database. Else only into databases owned by himself.
#---------------------------------------------------------------------
set n [$db select single usecreatedb, usesuper from pg_user \
	where usename = '$current_user']
if {$n == 0} {
    puts stderr "Postgres user '$current_user' does not exist"
    SQLdisconnect $db
    exit 4
}
if {$usecreatedb != "t"} {
    puts stderr "User '$current_user' is not allowed to create databases."
    puts stderr "This privilege is required to install betcl functionality."
    puts stderr "Sorry."
    SQLdisconnect $db
    exit 4
}
if {$usesuper != "t"} {
    set n [$db select single U.usename from	\
    		pg_database D, pg_user U	\
		where D.datname = '$dbname'	\
		and D.datdba = U.usesysid ]
    if {$n != 1} {
        puts stderr "Cannot determine owner of database '$dbname'"
	SQLdisconnect $db
	exit 4
    }
    if {$usename != $current_user} {
        puts stderr "User '$current_user' is not a postgres superuser and"
	puts stderr "database '$dbname' is owned by '$usename'."
	puts stderr "Sorry."
	SQLdisconnect $db
	exit 4
    }
}

#---------------------------------------------------------------------
# Make sure there are currently no tables beginning with 'betcl_'
# as a magic flag that the Tcl support is already installed.
#---------------------------------------------------------------------
$db select single count(pg_class.relname) as n where relname ~ '^betcl_'
if {$n != 0} {
    puts stderr "There are already tables beginning with 'betcl_' in"
    puts stderr "database '$dbname'."
    puts stderr "Installation aborted."
    SQLdisconnect $db
    exit 4
}

#---------------------------------------------------------------------
# Copy the Tcl interpreter startup files into the database dir
#---------------------------------------------------------------------
if {[catch {
exec cp $script_main_path $env(PGDATA)/base/$dbname/$script_main
exec cp $script_safe_path $env(PGDATA)/base/$dbname/$script_safe
} errmsg]} {
    puts stderr "Installation aborted due to error:"
    puts stderr "$errmsg"
    SQLdisconnect $db
    exit 5
}

#---------------------------------------------------------------------
# Create the tables and functions for the Tcl support module
#---------------------------------------------------------------------
if {[catch {
$db begin
$db create table betcl_proc (	\
	proname		name,	\
	pronargs	int4,	\
	prosrc		name,	\
	prodesc		text)
$db create table betcl_source (	\
	srcname		name,	\
	srcseq		int4,	\
	srctext		text)
$db create function betcl_call 						\
	(name, int4, text, text, text, text, text, text)		\
	returns text as '$betcl_sl_path' language 'c'
$db create function betcl_load (name)					\
	returns int4 as '$betcl_sl_path' language 'c'
$db create function betcl_reset ()					\
	returns int4 as '$betcl_sl_path' language 'c'

$db commit
} errmsg]} {
    puts stderr "Installation aborted due to error:"
    puts stderr "$errmsg"
    SQLdisconnect $db
    exit 5
}

#---------------------------------------------------------------------
# That's it.
#---------------------------------------------------------------------
puts "Installation in database '$dbname' successful"

SQLdisconnect $db
exit 0
