/**********************************************************************
 * $Source: /home/wieck/src/tcltk/extensions/pqatcl/libpqa/RCS/connect.c,v $
 * $Revision: 0.1 $ $State: Stab $
 * $Date: 1996/06/20 17:57:11 $
 * $Author: wieck $
 *
 * DESCRIPTION
 * 	connect.c		- Connect and disconnect database
 *
 * HISTORY
 * $Log: connect.c,v $
 * Revision 0.1  1996/06/20 17:57:11  wieck
 *   Initial revision
 *
 **********************************************************************/
char _RCSID_Pqa_connect[] = "$Id: connect.c,v 0.1 1996/06/20 17:57:11 wieck Stab $";

#include <tcl.h>
#include "pqatcl.h"
#include <pwd.h>


/**********************************************************************
 * This is something from earlier times as we supported fe-auth stuff.
 **********************************************************************/
char	PQerrormsg[PQA_ERRORMSG_LENGTH];

static int pqatcl_create_socket(Tcl_Interp *interp,
		PQA_client_data *cd, 
		PQA_connection **connp,
		char *host, char *port, char *dflport)
{
    PQA_connection	*conn;

    /************************************************************
     * Allocate and initialize the connection descriptor
     ************************************************************/
    conn = (PQA_connection *)ckalloc(sizeof(PQA_connection));
    memset(conn, 0, sizeof(PQA_connection));
    conn->struct_type = PQA_STRUCT_CONNECTION;

    conn->cdata = cd;
    conn->status = PQA_QSTATUS_IDLE;
    Tcl_DStringInit(&(conn->notify_list));

    /************************************************************
     * Establish the connection to the postmaster and switch
     * to binary mode.
     ************************************************************/
    if(port == NULL) port = dflport;
    conn->backend = Tcl_OpenTcpClient(interp,
    			atoi(port), host, NULL, 0, 0);
    if(conn->backend == NULL) {
        Tcl_AppendResult(interp, "\n", NULL);
        Tcl_DStringFree(&(conn->notify_list));
        ckfree(conn);
        return TCL_ERROR;
    }
    if(Tcl_SetChannelOption(interp, conn->backend,
    		"-translation", "binary") == TCL_ERROR) {
        Tcl_AppendResult(interp, "\n", NULL);
        Tcl_DStringFree(&(conn->notify_list));
        ckfree(conn);
        return TCL_ERROR;
    }

    *connp = conn;
    return TCL_OK;
}


/**********************************************************************
 * pqatcl_try_postgres_4_2()	- Try to handshake with V4.2
 **********************************************************************/
static int pqatcl_try_postgres_4_2(Tcl_Interp *interp,
		PQA_connection *conn,
		StartupPacket_4_2 *Startup,
    		int		Msgtype,
    		char		*host)
{
    PacketHdr_4_2 *hdr;
    int		status;
    char	buf[8];
    Tcl_File	sock_file;
    int		sock;
    int		n;

    /************************************************************
     * Send the database request to the postmaster
     ************************************************************/
    hdr = (PacketHdr_4_2 *)Startup;
    hdr->seqno	= htonl(PQA_INITIAL_SEQNO_4_2);
    hdr->connId	= htonl(PQA_INVALID_ID);
    hdr->type	= htonl(Msgtype);
    hdr->len	= htonl(sizeof(StartupPacket_4_2));
    sock_file = Tcl_GetChannelFile(conn->backend, TCL_READABLE);
    sock      = (int)Tcl_GetFileInfo(sock_file, NULL);
    status = send(sock, Startup, sizeof(StartupPacket_4_2), 0);
    if(status < 0) {
        Tcl_AppendResult(interp, "pqa_connect V4.2:",
        	" send(): ", sys_errlist[errno], "\n",
        	NULL);
        Tcl_DStringFree(&(conn->notify_list));
        Tcl_Close(interp, conn->backend);
        ckfree(conn);
        return TCL_ERROR;
    }
    if(status != sizeof(StartupPacket_4_2)) {
        Tcl_AppendResult(interp, "pqa_connect V4.2:", 
        	" send(): unable to send complete startup message\n",
        	NULL);
        Tcl_DStringFree(&(conn->notify_list));
        Tcl_Close(interp, conn->backend);
        ckfree(conn);
        return TCL_ERROR;
    }

    /************************************************************
     * Now send the initial empty query and wait for the result
     ************************************************************/
    conn->backend_version = PQA_DBVERSION_4_2;
    if(pqatcl_send_simple(conn, 'Q', 0, " ") < 0) {
        Tcl_AppendResult(interp, "pqa_connect V4.2:",
        	" connection to database failed -",
        	" cannot send initial empty query\n",
        	NULL);
        Tcl_DStringFree(&(conn->notify_list));
        Tcl_Close(interp, conn->backend);
        ckfree(conn);
        return TCL_ERROR;
    }
    n = Tcl_Read(conn->backend, buf, 1);
    if(n != 1 || buf[0] != 'I') {
        Tcl_AppendResult(interp, "pqa_connect V4.2:",
        	" connection to database failed -",
        	" cannot get reply for initial empty query\n",
        	NULL);
        Tcl_DStringFree(&(conn->notify_list));
        Tcl_Close(interp, conn->backend);
        ckfree(conn);
        return TCL_ERROR;
    }
    n = Tcl_Read(conn->backend, buf, 4);
    if(n != 4) {
        Tcl_AppendResult(interp, "pqa_connect V4.2:",
        	" connection to database failed -",
        	" cannot get reply for initial empty query\n",
        	NULL);
        Tcl_DStringFree(&(conn->notify_list));
        Tcl_Close(interp, conn->backend);
        ckfree(conn);
        return TCL_ERROR;
    }

    return TCL_OK;
}


/**********************************************************************
 * pqatcl_try_postgres_95()	- Try to handshake with postgres95
 **********************************************************************/
static int pqatcl_try_postgres_95(Tcl_Interp *interp,
		PQA_connection *conn,
		StartupPacket_95 *Startup,
    		int		Msgtype,
    		char		*host)
{
    int		status;
    char	buf[8];
    PacketBuf_95 msgbuf;
    char	*msgdata;
    Tcl_File	sock_file;
    int		sock;
    int		n;

    /************************************************************
     * Send the database request to the postmaster
     ************************************************************/
    memset(&msgbuf, 0, sizeof(msgbuf));

    msgbuf.len     = htonl(sizeof(msgbuf));
    msgbuf.msgtype = htonl(Msgtype);
    msgdata = msgbuf.data;
    strncpy(msgdata, Startup->database, sizeof(Startup->database));
    msgdata += sizeof(Startup->database);
    strncpy(msgdata, Startup->user, sizeof(Startup->user));
    msgdata += sizeof(Startup->user);
    strncpy(msgdata, Startup->options, sizeof(Startup->options));
    msgdata += sizeof(Startup->options);
    strncpy(msgdata, Startup->execFile, sizeof(Startup->execFile));
    msgdata += sizeof(Startup->execFile);
    strncpy(msgdata, Startup->tty, sizeof(Startup->tty));
    sock_file = Tcl_GetChannelFile(conn->backend, TCL_READABLE);
    sock      = (int)Tcl_GetFileInfo(sock_file, NULL);

    status = send(sock, &msgbuf, sizeof(msgbuf), 0);
    if(status < 0) {
        Tcl_AppendResult(interp, "pqa_connect V95:",
        	" send(): ", sys_errlist[errno], "\n",
        	NULL);
        Tcl_DStringFree(&(conn->notify_list));
        Tcl_Close(interp, conn->backend);
        ckfree(conn);
        return TCL_ERROR;
    }
    if(status != sizeof(msgbuf)) {
        Tcl_AppendResult(interp, "pqa_connect V95:", 
        	" send(): unable to send complete startup message\n",
        	NULL);
        Tcl_DStringFree(&(conn->notify_list));
        Tcl_Close(interp, conn->backend);
        ckfree(conn);
        return TCL_ERROR;
    }

    /************************************************************
     * Now send the initial empty query and wait for the result
     ************************************************************/
    conn->backend_version = PQA_DBVERSION_95;
    if(pqatcl_send_simple(conn, 'Q', 0, " ") < 0) {
        Tcl_AppendResult(interp, "pqa_connect V95:",
        	" connection to database failed -",
        	" cannot send initial empty query\n",
        	NULL);
        Tcl_DStringFree(&(conn->notify_list));
        Tcl_Close(interp, conn->backend);
        ckfree(conn);
        return TCL_ERROR;
    }
    n = Tcl_Read(conn->backend, buf, 1);
    if(n != 1 || buf[0] != 'I') {
        Tcl_AppendResult(interp, "pqa_connect V95:",
        	" connection to database failed -",
        	" cannot get reply for initial empty query\n",
        	NULL);
        Tcl_DStringFree(&(conn->notify_list));
        Tcl_Close(interp, conn->backend);
        ckfree(conn);
        return TCL_ERROR;
    }
    n = Tcl_Read(conn->backend, buf, 1);
    if(n != 1 || buf[0] != '\0') {
        Tcl_AppendResult(interp, "pqa_connect V95:",
        	" connection to database failed -",
        	" cannot get reply for initial empty query\n",
        	NULL);
        Tcl_DStringFree(&(conn->notify_list));
        Tcl_Close(interp, conn->backend);
        ckfree(conn);
        return TCL_ERROR;
    }

    return TCL_OK;
}


/**********************************************************************
 * Pqa_connect()		- The Tcl visible pqa_connect command
 **********************************************************************/
int Pqa_connect(ClientData cdata, Tcl_Interp *interp, int argc, char *argv[])
{
    static char	*dbname;
    static char	*host;
    static char	*port;
    static char	*user;
    static char	*tty;
    static char	*options;
    static char	*execfile;
    static PQA_argument args[] = {
    	{ "-dbname",   PQA_ARGTYPE_STR, &dbname },
    	{ "-databasename", PQA_ARGTYPE_STR, &dbname },
    	{ "-host",     PQA_ARGTYPE_STR, &host },
    	{ "-port",     PQA_ARGTYPE_STR, &port },
    	{ "-user",     PQA_ARGTYPE_STR, &user },
    	{ "-tty",      PQA_ARGTYPE_STR, &tty },
    	{ "-options",  PQA_ARGTYPE_STR, &options },
    	{ "-execfile", PQA_ARGTYPE_STR, &execfile },
    	{ NULL, 0, NULL }
    };

    PQA_client_data *cd = (PQA_client_data *)cdata;
    Tcl_HashEntry	*hash_ent;
    StartupPacket_4_2	Startup_4_2;
    StartupPacket_95	Startup_95;
    int			Msgtype;
    PQA_connection	*conn;
    int			status;

    /************************************************************
     * Parse options
     ************************************************************/
    dbname = NULL;
    host = NULL;
    port = NULL;
    user = NULL;
    tty = NULL;
    options = NULL;
    execfile = NULL;

    if(pqatcl_get_arguments(interp, args, 1, argc, argv) != TCL_OK) {
        return TCL_ERROR;
    }

    /************************************************************
     * Determine the user name
     ************************************************************/
    if(user == NULL) {
	static struct passwd *pwd;

	pwd = getpwuid(getuid());
	if(pwd) {
	    user = pwd->pw_name;
	}
        if(user == NULL) {
            Tcl_AppendResult(interp, argv[0], 
            	"Unable to determine user name", NULL);
            return TCL_ERROR;
        }
    }
    /************************************************************
     * Determine the database name
     ************************************************************/
    if(dbname == NULL) {
        dbname = getenv("PGDATABASE");
        if(dbname == NULL) {
            dbname = user;
        }
    }
    /************************************************************
     * Fix null options to backend
     ************************************************************/
    if(options == NULL) {
        options = "";
    }
    /************************************************************
     * Fix the hostname and do the address lookup
     ************************************************************/
    if(host == NULL) {
        host = getenv("PGHOST");
        if(host == NULL) {
            host = "localhost";
        }
    }
    /************************************************************
     * Fix null debug tty for backend
     ************************************************************/
    if(tty == NULL) {
        tty = "";
    }
    /************************************************************
     * Fix null exec file for backend process
     ************************************************************/
    if(execfile == NULL) {
        execfile = "";
    }
    /************************************************************
     * Determine the service port for the postmaster
     ************************************************************/
    if(port == NULL) {
        port = getenv("PGPORT");
    }

    /************************************************************
     * File the structure used for the database request
     * to the postmaster
     ************************************************************/
    memset((char *)&Startup_4_2, 0, sizeof(Startup_4_2));
    memset((char *)&Startup_95, 0, sizeof(Startup_95));

    Msgtype	= PQA_STARTUP_MSG;
    strncpy(Startup_4_2.database, dbname, sizeof(Startup_4_2.database));
    strncpy(Startup_4_2.user, user, sizeof(Startup_4_2.user));
    strncpy(Startup_4_2.options, options, sizeof(Startup_4_2.options));
    strncpy(Startup_4_2.execFile, execfile, sizeof(Startup_4_2.execFile));
    strncpy(Startup_4_2.tty, tty, sizeof(Startup_4_2.tty));

    strncpy(Startup_95.database, dbname, sizeof(Startup_95.database));
    strncpy(Startup_95.user, user, sizeof(Startup_95.user));
    strncpy(Startup_95.options, options, sizeof(Startup_95.options));
    strncpy(Startup_95.execFile, execfile, sizeof(Startup_95.execFile));
    strncpy(Startup_95.tty, tty, sizeof(Startup_95.tty));

    /************************************************************
     * Create the connection to the postmaster of V 4.2
     ************************************************************/
    if(pqatcl_create_socket(interp, cd, &conn, host, port, 
    		PQA_POSTPORT_4_2) == TCL_ERROR) {
        Tcl_AppendResult(interp, 
                "connection to V4.2 postmaster failed\n",
        	NULL);
        conn = NULL;
    } else {
	/************************************************************
	 * Try to handshake with postgres 4.2
	 ************************************************************/
	if(pqatcl_try_postgres_4_2(interp, conn, &Startup_4_2, 
		    Msgtype, host) == TCL_ERROR) {
	    Tcl_AppendResult(interp, 
		    "handshake with V4.2 postmaster failed\n",
		    NULL);
	    conn = NULL;
	}
    }

    /************************************************************
     * If that failed try to connect to postgres95
     ************************************************************/
    if(conn == NULL) {
	if(pqatcl_create_socket(interp, cd, &conn, host, port, 
		    PQA_POSTPORT_95) == TCL_ERROR) {
	    Tcl_AppendResult(interp, 
		    "connection to V95 postmaster failed\n",
		    NULL);
	    conn = NULL;
	} else {
	    /************************************************************
	     * Try to handshake with postgres 4.2
	     ************************************************************/
	    if(pqatcl_try_postgres_95(interp, conn, &Startup_95,
			Msgtype, host) == TCL_ERROR) {
		Tcl_AppendResult(interp, 
			"handshake with V95 postmaster failed\n",
			NULL);
		conn = NULL;
	    }
	}
    }

    if(conn == NULL) {
        return TCL_ERROR;
    }

    /************************************************************
     * Switch the Tcl channel to async I/O
     ************************************************************/
    Tcl_SetChannelOption(interp, conn->backend, "-translation", "binary");
    Tcl_SetChannelOption(interp, conn->backend, "-buffering", "none");
    Tcl_SetChannelOption(interp, conn->backend, "-blocking", "no");

    /************************************************************
     * Install the channel handler
     ************************************************************/
    Tcl_CreateChannelHandler(conn->backend,
    	TCL_READABLE, pqatcl_exec_asyncIO, (ClientData)conn);

    /************************************************************
     * Finally create the connection ID and the hash entry
     ************************************************************/
    sprintf(conn->handle, "pqac%d", cd->conn_id++);
    hash_ent = Tcl_CreateHashEntry(&(cd->idtable), conn->handle, &status);
    Tcl_SetHashValue(hash_ent, (ClientData)conn);

    /************************************************************
     * Thats it.
     ************************************************************/
    Tcl_SetResult(interp, conn->handle, TCL_VOLATILE);
    return TCL_OK;
}


/**********************************************************************
 * Pqa_close()			- Close a connection to the backend
 **********************************************************************/
int Pqa_close(ClientData cdata, Tcl_Interp *interp, int argc, char *argv[])
{
    PQA_client_data *cd = (PQA_client_data *)cdata;
    PQA_connection *conn;
    PQA_result *result;
    PQA_large_obj *lobj;

    Tcl_HashEntry	*hash_ent;
    int			i;
    char		resultid[32];

    if(argc != 2) {
        Tcl_AppendResult(interp, argv[0], ": syntax error '",
        	argv[0], " conn'", NULL);
        return TCL_ERROR;
    }

    /************************************************************
     * Get the connection control block by the ID
     ************************************************************/
    if((conn = pqatcl_getconnbyid(interp, cd, argv[1])) == NULL) {
        return TCL_ERROR;
    }

    /************************************************************
     * Remove the hash entry close the connection
     ************************************************************/
    if(conn->status != PQA_QSTATUS_LOST) {
	Tcl_Close(interp, conn->backend);
    }
    hash_ent = Tcl_FindHashEntry(&(cd->idtable), argv[1]);
    Tcl_DeleteHashEntry(hash_ent);
    Tcl_DStringFree(&(conn->notify_list));
    if(conn->notify_command) ckfree(conn->notify_command);

    /************************************************************
     * Remove the backlink from all resultbuffers.
     * For queries currently waiting for the database,
     * create their result ID in the hash table, because the
     * asyncIO function will not be called any more for them.
     * Set the status to ABORT and evaluate their callback.
     ************************************************************/
    result = conn->result_list;
    while(result) {
        if(result->status == PQA_QSTATUS_BUSY) {
            strcpy(resultid, result->handle);
            result->status = PQA_QSTATUS_ABORT;
            if(conn->status != PQA_QSTATUS_LOST) {
		Tcl_DStringAppend(&(result->errormsg),
		    "database connection shut down by pqa_close\n", -1);
            }
            hash_ent = Tcl_CreateHashEntry(&(cd->idtable),
            	result->handle, &i);
            Tcl_SetHashValue(hash_ent, (ClientData)result);
            if(result->async_command) {
                if(Tcl_VarEval(interp, result->async_command, " ",
                	result->handle, NULL) == TCL_ERROR) {
                    Tcl_BackgroundError(interp);
                    Tcl_SetResult(interp, "", TCL_VOLATILE);
                }
                /************************************************************
                 * Special case if the command cleared the
                 * result buffer
                 ************************************************************/
                if(pqatcl_getresultbyid(NULL, cd, resultid) == NULL) {
                    result = conn->result_list;
                    continue;
                }
            }
        }
        result->conn = NULL;
        result = result->next;
    }

    /************************************************************
     * Remove the backlink from all large object channels
     ************************************************************/
    for(lobj = conn->large_obj_list; lobj; lobj = lobj->next) {
        lobj->conn = NULL;
    }

    /************************************************************
     * Finally remove the connection control block
     ************************************************************/
    ckfree(conn);
    return TCL_OK;
}


/**********************************************************************
 * Pqa_status()		- The Tcl command pqa_status
 **********************************************************************/
int Pqa_status(ClientData cdata, Tcl_Interp *interp, int argc, char *argv[])
{
    Tcl_DString	retval;
    static int	f_status;
    static int	f_version;
    static int	f_results;
    static PQA_argument args[] = {
    	{ "-status",       PQA_ARGTYPE_INTVAL1, &f_status },
    	{ "-version",      PQA_ARGTYPE_INTVAL1, &f_version },
    	{ "-results",      PQA_ARGTYPE_INTVAL1, &f_results },
    	{ NULL, 0, NULL }
    };

    PQA_client_data *cd = (PQA_client_data *)cdata;
    PQA_connection	*conn;
    PQA_result		*result;
    Tcl_HashEntry	*hash_ent;
    Tcl_HashSearch	hash_search;

    /************************************************************
     * Check for syntax
     ************************************************************/
    if(argc < 3) {
        Tcl_AppendResult(interp, "syntax error - usage: ",
        	argv[0], " conn options", NULL);
        return TCL_ERROR;
    }

    /************************************************************
     * Parse options
     ************************************************************/
    f_status  = 0;
    f_version = 0;
    f_results = 0;

    if(pqatcl_get_arguments(interp, args, 2, argc, argv) != TCL_OK) {
        return TCL_ERROR;
    }

    /************************************************************
     * Get the connection
     ************************************************************/
    if((conn = pqatcl_getconnbyid(interp, cd, argv[1])) == NULL) {
        return TCL_ERROR;
    }

    /************************************************************
     * Create the return value
     ************************************************************/
    Tcl_DStringInit(&retval);
    if(f_status) {
        switch(conn->status) {
            case PQA_QSTATUS_IDLE:
                Tcl_DStringAppendElement(&retval, "idle");
                break;
            case PQA_QSTATUS_LOST:
                Tcl_DStringAppendElement(&retval, "lost");
                break;
            default:
                Tcl_DStringAppendElement(&retval, "busy");
                break;
        }
    }

    if(f_version) {
        switch(conn->backend_version) {
            case PQA_DBVERSION_4_2:
                Tcl_DStringAppendElement(&retval, "V4.2");
                break;
            case PQA_DBVERSION_95:
                Tcl_DStringAppendElement(&retval, "V95");
                break;
            default:
                Tcl_DStringAppendElement(&retval, "unknown");
                break;
        }
    }

    if(f_results) {
        hash_ent = Tcl_FirstHashEntry(&(cd->idtable), &hash_search);
        while(hash_ent != NULL) {
            result = (PQA_result *)Tcl_GetHashValue(hash_ent);
            if(result->struct_type == PQA_STRUCT_RESULT && 
            		result->conn == conn) {
                Tcl_DStringAppendElement(&retval, result->handle);
            }
            hash_ent = Tcl_NextHashEntry(&hash_search);
        }
    }

    /************************************************************
     * Thats it.
     ************************************************************/
    Tcl_SetResult(interp, Tcl_DStringValue(&retval), TCL_VOLATILE);
    Tcl_DStringFree(&retval);
    return TCL_OK;
}


