#---------------------------------------------------------------------
# Epqpqa.tcl			- Embedded postquel over Pqatcl
#				  package.
#
# IDENTIFICATION
# $Header: /home/wieck/src/tcltk/extensions/pqatcl/libpqa/RCS/epqpqa.tcl,v 1.2 1996/10/09 08:44:29 wieck Exp $
#
# HISTORY
# $Log: epqpqa.tcl,v $
# Revision 1.2  1996/10/09 08:44:29  wieck
# Added large object support as in sqlpqa.tcl.
#
# Revision 1.1  1996/10/09 06:10:09  wieck
# Initial revision
#
#
#---------------------------------------------------------------------


#---------------------------------------------------------------------
# EPQconnect			- Create a database connection and
#				  install the EPQ wrapper command.
#
# As a side effect the required Pqatcl package for the async
# postgres access is loaded here.
#---------------------------------------------------------------------
proc EPQconnect {args} {
    package require Pqatcl

    set conn [eval pqa_connect $args]
    set xcmd [_EPQ_dbh2cmd $conn]
    proc $xcmd {args} "_EPQ_command_ $conn \$args"

    return $xcmd
}


#---------------------------------------------------------------------
# EPQdisconnect			- Close a database connection and
#				  remove the EPQ wrapper command
#				  installed by EPQconnect
#---------------------------------------------------------------------
proc EPQdisconnect {cmd} {
    set dbh [_EPQ_cmd2dbh $cmd]
    rename $cmd {}
    pqa_close $dbh
}


#---------------------------------------------------------------------
# _EPQ_dbh2cmd			- Create the wrapper command name
#				  from the Pqatcl database connection.
#---------------------------------------------------------------------
proc _EPQ_dbh2cmd {dbh} {
    return "_EPQ_cmdwrap_$dbh"
}


#---------------------------------------------------------------------
# _EPQ_cmd2dbh			- Convert from wrapper command name
#				  to Pqatcl database connection.
#---------------------------------------------------------------------
proc _EPQ_cmd2dbh {cmd} {
    if {[info procs $cmd] == ""} {
        return -code error "invalid EPQ connection '$cmd'"
    }
    if {[regexp {^_EPQ_cmdwrap_(.*)$} $cmd {} dbh] == 0} {
        return -code error "ERROR: cannot examine connection in $cmd"
    }

    return $dbh
}


#---------------------------------------------------------------------
# EPQquote		- Quote literals properly
#---------------------------------------------------------------------
proc EPQquote {val} {
    regsub -all {\\} $val {\\\\} val
    regsub -all {"}  $val {\\"} val
    regsub -all "\n" $val {\\n} val
    regsub -all "\t" $val {\\t} val
    return $val
}


#---------------------------------------------------------------------
# _EPQ_single2dbl		- Convert single quotes into double
#				  quotes in the query string.
#
# Double quotes are special characters in Tcl. So it is really hairy
# to write embedded postquel queries directly in Tcl. I just found it
# useful to change the postquel syntax here to accept single quotes
# where double ones are normally required.
#---------------------------------------------------------------------
proc _EPQ_single2dbl {in} {
    set out {}

    while {[regexp {([^']*')(.*)} $in {} l r]} {
        regsub -all {\\.} $l {} cl
        if {![regexp {\'$} $cl]} {
	    regsub -all {"} $l {\"} l
            append out $l
            regsub -all {\\'$} $out {'} out
            set in $r
            continue
        }
	regsub -all {"} $l {\"} l
        regsub {'$} $l {} l
        append out "$l\""
        set in $r
    }

    append out $in
    return $out
}


#---------------------------------------------------------------------
# _EPQ_command_			- The wrapper command for a database
#				  connection simply calls this proc
#				  to carry out the work.
#---------------------------------------------------------------------
proc _EPQ_command_ {dbh argv} {
    #-----------------------------------------------------------
    # Check if we must handle this command in a special way.
    #-----------------------------------------------------------
    set xcmd "_EPQ_cmd_[lindex $argv 0]"
    if {[info procs $xcmd] == ""} {
        #-----------------------------------------------------------
        # There is no special handling procedure. Execute the
        # query as is.
        #-----------------------------------------------------------
        set query [_EPQ_single2dbl [join $argv]]
        if {[catch {set res [pqa_exec $dbh $query]} errmsg]} {
            return -code error $errmsg
        }

        #-----------------------------------------------------------
        # Check for errors from the backend
        #-----------------------------------------------------------
        if {[pqa_result $res -status] != "ok"} {
            return -code error [pqa_result $res -errormsg -clear]
        }

        #-----------------------------------------------------------
        # Return the command identifier value from the backend
        #-----------------------------------------------------------
        return [pqa_result $res -cmdreturn -clear]
    }

    return [$xcmd $dbh $argv]
}


#---------------------------------------------------------------------
# _EPQ_process_result		- Process a result buffer generated
#				  by a query returning tuple data
#				  (like retrieve or fetch).
#---------------------------------------------------------------------
proc _EPQ_process_result {res single cmd} {
    #-----------------------------------------------------------
    # Check for errors
    #-----------------------------------------------------------
    if {[pqa_result $res -status] != "ok"} {
        return -code error [pqa_result $res -errormsg -clear]
    }

    #-----------------------------------------------------------
    # Examine the result buffer about attribute names and
    # link them into the interpreter level of the original
    # query.
    #-----------------------------------------------------------
    set attrlist {}
    foreach att [pqa_result $res -attributes] {
        set a [lindex $att 0]
        lappend attrlist $a
        upvar 4 $a var_$a
        set $a {}
    }

    #-----------------------------------------------------------
    # Set EPQnumtuples now, so the loop can already access it.
    #-----------------------------------------------------------
    upvar 4 EPQnumtuples ntuples
    set ntuples [pqa_result $res -numtuples]

    #-----------------------------------------------------------
    # Depending on the single flag process only the first
    # tuple of the result or all in a loop.
    #-----------------------------------------------------------
    if {$single} {
        if {[pqa_result $res -numtuples] > 0} {
            set data [pqa_result $res -gettuple 0]
        } else {
            set data 0
        }
        set i 0
        foreach a $attrlist {
            set var_$a [lindex $data $i]
            incr i
        }
    } else {
        if {[catch {pqa_loop $res data {
            set i 0
            foreach a $attrlist {
                set var_$a [lindex $data $i]
                incr i
            }
            uplevel 4 $cmd
        }} errmsg]} {
            catch {pqa_result $res -clear}
            return -code error $errmsg
        }
    }

    #-----------------------------------------------------------
    # Return the number of tuples in the result again and
    # return that value.
    #-----------------------------------------------------------
    return [set ntuples [pqa_result $res -numtuples -clear]]
}


#---------------------------------------------------------------------
# _EPQ_cmd_retrieve		- Special handling of retrieve query
#---------------------------------------------------------------------
proc _EPQ_cmd_retrieve {dbh argv} {
    #-----------------------------------------------------------
    # Extra special handling of 'retrieve portal'. This
    # query doesn't return tuple data.
    #-----------------------------------------------------------
    if {[lindex $argv 1] == "portal"} {
        set query [_EPQ_single2dbl [join $argv]]
        if {[catch {set res [pqa_exec $dbh $query]} errmsg]} {
            return -code error $errmsg
        }

        #-----------------------------------------------------------
        # Check for errors from the backend
        #-----------------------------------------------------------
        if {[pqa_result $res -status] != "ok"} {
            return -code error [pqa_result $res -errormsg -clear]
        }

        #-----------------------------------------------------------
        # Return the command identifier value from the backend
        #-----------------------------------------------------------
        return [pqa_result $res -cmdreturn -clear]
    }

    #-----------------------------------------------------------
    # Now check for 'single' keyword
    #-----------------------------------------------------------
    if {[lindex $argv 1] == "single"} {
        set single 1
        set cmd {}
        set query [_EPQ_single2dbl [join [lreplace $argv 1 1]]]
    } else {
        set single 0
        set cmd [lindex $argv end]
        set query [_EPQ_single2dbl [join [lreplace $argv end end]]]
    }

    #-----------------------------------------------------------
    # Execute the query and process the result
    #-----------------------------------------------------------
    if {[catch {set res [pqa_exec $dbh $query]} errmsg]} {
        return -code error $errmsg
    }
    return [_EPQ_process_result $res $single $cmd]
}


#---------------------------------------------------------------------
# _EPQ_cmd_fetch		- Special handling of fetch query
#---------------------------------------------------------------------
proc _EPQ_cmd_fetch {dbh argv} {
    #-----------------------------------------------------------
    # Check for 'single' keyword
    #-----------------------------------------------------------
    if {[lindex $argv 1] == "single"} {
        set single 1
        set cmd {}
        set query [_EPQ_single2dbl [join [lreplace $argv 1 1]]]
    } else {
        set single 0
        set cmd [lindex $argv end]
        set query [_EPQ_single2dbl [join [lreplace $argv end end]]]
    }

    #-----------------------------------------------------------
    # Execute the query and process the result
    #-----------------------------------------------------------
    if {[catch {set res [pqa_exec $dbh $query]} errmsg]} {
        return -code error $errmsg
    }
    return [_EPQ_process_result $res $single $cmd]
}


#---------------------------------------------------------------------
# _EPQ_cmd_describe		- Describe special objects in the
#				  database. This is a special
#				  extension of the postquel language
#				  useful for interactive access throug
#				  a tclsh.
#---------------------------------------------------------------------
proc _EPQ_cmd_describe {dbh argv} {
    set llen [llength $argv]
    if {$llen < 2} {
        return -code error "don't know what to describe"
    }
    set db  [_EPQ_dbh2cmd $dbh]
    set obj [lindex $argv 1]

    switch -- $obj {
    table	-
    class	{
    	    if {$llen != 3} {
    	        return -code error \
    	        	"syntax error - describe $obj <${obj}_name>"
    	    }
    	    set tname [lindex $argv 2]
    	    set ret {}
    	    $db retrieve (A.attnum, A.attname, T.typname, T.typlen)	\
    	        	from C in pg_class, A in pg_attribute,		\
    	        	     T in pg_type				\
    	    		where C.relname = '$tname'			\
    	    		  and A.attrelid = C.oid			\
    	    		  and A.attnum   > 0				\
    	    		  and T.oid      = A.atttypid			\
    	    		sort by attnum {
    	        if {$typlen < 0} {
    	            set typlen "var"
    	        }
    	        lappend ret [list $attname $typname $typlen]
    	    }
    	    if {$EPQnumtuples <= 0} {
    	        return -code error "'$tname': no such $obj"
    	    }
    	    return $ret
    	}

    default	{
    	    return -code error	\
    	    	"don't know how to describe object type '%obj'"
    	}
    }
}


#---------------------------------------------------------------------
# _EPQ_cmd_create	- Special handling of create statement
#			  This implements the 'create large object'
#			  pseudo EPQ syntax.
#---------------------------------------------------------------------
proc _EPQ_cmd_create {dbh query} {
    #-----------------------------------------------------------
    # Preserve the original query quoted and strip the command
    #-----------------------------------------------------------
    set xquery [_EPQ_single2dbl [join $query]]
    set query  [lrange $query 1 end]
    #-----------------------------------------------------------
    # Check for large object operation
    #-----------------------------------------------------------
    if {[regexp "^\[ \t\n\]*large\[ \t\n+\]object" $query]} {
        if {[llength $query] < 4 || [lindex $query 2] != "mode"} {
	    return -code error \
	        {syntax error - 'create large object mode <mode>'}
	}
        set mode ""
	foreach m [lrange $query 3 end] {
	    switch -- $m {
		read	-
	        r	{ lappend mode "INV_READ"
			}
	        write	-
		w	{ lappend mode "INV_WRITE"
			}
		readwrite -
		rw	{ lappend mode "INV_READ" "INV_WRITE"
	    		}
	        archive	{ lappend mode "INV_ARCHIVE"
			}
	        default	{ return -code error \
      {invalid mode - must be a combination of 'read', 'write' and 'archive'}
			}
	    }
	}
	return [pqa_lo_creat $dbh $mode]
    }

    #-----------------------------------------------------------
    # No special handling - execute directly
    #-----------------------------------------------------------
    set res [pqa_exec $dbh "$xquery"]
    if {[pqa_result $res -status] != "ok"} {
	return -code error [pqa_result $res -errormsg -clear]
    }
    return [pqa_result $res -cmdreturn -clear]
}


#---------------------------------------------------------------------
# _EPQ_cmd_open		- Special handling of open statement
#			  This implements the 'open large object'
#			  pseudo EPQ syntax.
#---------------------------------------------------------------------
proc _EPQ_cmd_open {dbh query} {
    #-----------------------------------------------------------
    # Preserve the original query quoted and strip the command
    #-----------------------------------------------------------
    set xquery [_EPQ_single2dbl [join $query]]
    set query  [lrange $query 1 end]
    #-----------------------------------------------------------
    # Check for large object operation
    #-----------------------------------------------------------
    if {[regexp "^\[ \t\n\]*large\[ \t\n+\]object" $query]} {
        if {[llength $query] < 5 || [lindex $query 3] != "mode"} {
	    return -code error \
	        {syntax error - 'open large object <OID> mode <mode>'}
	}
	set oid [lindex $query 2]
        set mode ""
	foreach m [lrange $query 4 end] {
	    switch -- $m {
		read	-
	        r	{ lappend mode "INV_READ"
			}
	        write	-
		w	{ lappend mode "INV_WRITE"
			}
		readwrite -
		rw	{ lappend mode "INV_READ" "INV_WRITE"
	    		}
	        archive	{ lappend mode "INV_ARCHIVE"
			}
	        default	{ return -code error \
      {invalid mode - must be a combination of 'read', 'write' and 'archive'}
			}
	    }
	}
	return [pqa_lo_open $dbh $oid $mode]
    }

    #-----------------------------------------------------------
    # No special handling - execute directly
    #-----------------------------------------------------------
    set res [pqa_exec $dbh "$xquery"]
    if {[pqa_result $res -status] != "ok"} {
	return -code error [pqa_result $res -errormsg -clear]
    }
    return [pqa_result $res -cmdreturn -clear]
}


#---------------------------------------------------------------------
# _EPQ_cmd_remove	- Special handling of remove statement
#			  This implements the 'remove large object'
#			  pseudo EPQ syntax.
#---------------------------------------------------------------------
proc _EPQ_cmd_remove {dbh query} {
    #-----------------------------------------------------------
    # Preserve the original query quoted and strip the command
    #-----------------------------------------------------------
    set xquery [_EPQ_single2dbl [join $query]]
    set query  [lrange $query 1 end]
    #-----------------------------------------------------------
    # Check for large object operation
    #-----------------------------------------------------------
    if {[regexp "^\[ \t\n\]*large\[ \t\n+\]object" $query]} {
        if {[llength $query] != 3} {
	    return -code error \
	        {syntax error - 'remove large object <OID>'}
	}
	set oid [lindex $query 2]
	return [pqa_lo_unlink $dbh $oid]
    }

    #-----------------------------------------------------------
    # No special handling - execute directly
    #-----------------------------------------------------------
    set res [pqa_exec $dbh "$xquery"]
    if {[pqa_result $res -status] != "ok"} {
	return -code error [pqa_result $res -errormsg -clear]
    }
    return [pqa_result $res -cmdreturn -clear]
}


#---------------------------------------------------------------------
# _EPQ_cmd_info		- Implementation of info command
#---------------------------------------------------------------------
proc _EPQ_cmd_info {dbh query} {
    set what [lindex $query 1]
    switch -- $what {
	dbhandle {
		  return $dbh
		}
        default { return -code error \
	  "unknown info '$what' - must be 'dbhandle'"
		}
    }
}


#---------------------------------------------------------------------
# Propagate the package
#---------------------------------------------------------------------
package provide Epqpqa 1.0


