/**********************************************************************
 * $Source: /home/wieck/src/tcltk/extensions/pqatcl/libpqa/RCS/exec.c,v $
 * $Revision: 0.1 $ $State: Stab $
 * $Date: 1996/06/20 17:57:11 $
 * $Author: wieck $
 *
 * DESCRIPTION
 * 	exec.c			- Execution of queries, communication
 *				  with backend and handling of
 *				  asynchronous portal notification
 *
 * HISTORY
 * $Log: exec.c,v $
 * Revision 0.1  1996/06/20 17:57:11  wieck
 *   Initial revision
 *
 **********************************************************************/
char _RCSID_Pqa_exec[] = "$Id: exec.c,v 0.1 1996/06/20 17:57:11 wieck Stab $";

#include <tcl.h>
#include "pqatcl.h"


/**********************************************************************
 * Pqa_exec()			- Execute a query
 **********************************************************************/
int Pqa_exec(ClientData cdata, Tcl_Interp *interp, int argc, char *argv[])
{
    PQA_client_data *cd = (PQA_client_data *)cdata;
    PQA_connection *conn;
    PQA_result	  *result;

    static char	*command;
    static PQA_argument args[] = {
        { "-command", PQA_ARGTYPE_STR, &command },
        { NULL, 0, NULL }
    };

    command = NULL;

    /************************************************************
     * Check if we have at least the connection and the query
     ************************************************************/
    if(argc < 3) {
        Tcl_AppendResult(interp, argv[0], ": syntax error '",
        	argv[0], " conn query ?-command cmd?'", NULL);
        return TCL_ERROR;
    }

    /************************************************************
     * Get the connection
     ************************************************************/
    if((conn = pqatcl_getconnbyid(interp, cd, argv[1])) == NULL) {
        return TCL_ERROR;
    }

    /************************************************************
     * Parse options
     ************************************************************/
    if(pqatcl_get_arguments(interp, args, 3, argc, argv) != TCL_OK) {
        return TCL_ERROR;
    }

    /************************************************************
     * The database connection must be alive
     ************************************************************/
    if(conn->status == PQA_QSTATUS_LOST) {
        Tcl_AppendResult(interp, argv[0], 
        	": connection to database lost", NULL);
        return TCL_ERROR;
    }

    /************************************************************
     * Allocate and initialize the new result buffer
     ************************************************************/
    result = (PQA_result *)ckalloc(sizeof(PQA_result));
    memset(result, 0, sizeof(PQA_result));
    result->struct_type = PQA_STRUCT_RESULT;
    result->conn = conn;
    result->cdata = cd;

    Tcl_DStringInit(&(result->errormsg));
    Tcl_DStringInit(&(result->attributes));
    sprintf(result->handle, "pqar%d", cd->result_id++);
    result->status = PQA_QSTATUS_BUSY;

    result->prev   = NULL;
    result->next   = conn->result_list;
    if(conn->result_list) {
        conn->result_list->prev = result;
    }
    conn->result_list  = result;

    /************************************************************
     * If a command is given remeber that for async execution
     ************************************************************/
    if(command != NULL) {
        result->async_command = ckalloc(strlen(command) + 1);
        strcpy(result->async_command, command);
    }

    /************************************************************
     * Everything is set up right now. But there might be
     * another query in progress - wait until that finished.
     ************************************************************/
    while(conn->status != PQA_QSTATUS_IDLE) {
        Tcl_DoOneEvent(TCL_ALL_EVENTS);

        if(pqatcl_getconnbyid(NULL, cd, argv[1]) == NULL) {
            if(result->async_command == NULL) {
                Tcl_SetResult(interp, result->handle, TCL_VOLATILE);
            }
	    return TCL_OK;
        }

        if(conn->status == PQA_QSTATUS_LOST) {
            if(result->async_command == NULL) {
                Tcl_SetResult(interp, result->handle, TCL_VOLATILE);
            }
	    return TCL_OK;
        }
    }
    conn->status = PQA_QSTATUS_GETID;

    /************************************************************
     * Send the query to the backend
     ************************************************************/
    if(pqatcl_send_simple(conn, 'Q', conn->xactid, argv[2])) {
        Tcl_AppendResult(interp, argv[0], ": ", PQerrormsg, NULL);
        return TCL_ERROR;
    }

    /************************************************************
     * For async queries thats all
     ************************************************************/
    if(command) {
        return TCL_OK;
    }

    /************************************************************
     * Synchronous queries only seem to be synchronous. In fact,
     * they just loop processing events until their query is
     * complete.
     ************************************************************/
    while(result->status == PQA_QSTATUS_BUSY) {
        Tcl_DoOneEvent(TCL_ALL_EVENTS);
    }
    Tcl_SetResult(interp, result->handle, TCL_VOLATILE);
    return TCL_OK;
}


/**********************************************************************
 * Pqa_notify()			- The Tcl pqa_notify command
 **********************************************************************/
int Pqa_notify(ClientData cdata, Tcl_Interp *interp, int argc, char *argv[])
{
    PQA_client_data *cd = (PQA_client_data *)cdata;
    PQA_connection *conn;
    static char	*command;
    static PQA_argument args[] = {
        { "-command", PQA_ARGTYPE_STR, &command },
        { NULL, 0, NULL }
    };

    command = NULL;

    /************************************************************
     * Check if we have at least the connection
     ************************************************************/
    if(argc < 2) {
        Tcl_AppendResult(interp, argv[0], ": syntax error '",
        	argv[0], " conn ?-command cmd?'", NULL);
        return TCL_ERROR;
    }

    /************************************************************
     * Get the connection
     ************************************************************/
    if((conn = pqatcl_getconnbyid(interp, cd, argv[1])) == NULL) {
        return TCL_ERROR;
    }

    /************************************************************
     * Parse option
     ************************************************************/
    if(pqatcl_get_arguments(interp, args, 2, argc, argv) != TCL_OK) {
        return TCL_ERROR;
    }

    /************************************************************
     * If the command option is given, replace the existing
     * notification callback.
     * If then there are notifications pending, immediately
     * evaluate the command. But for consistency (e.g. someone
     * installed a bgerror procedure) report errors as
     * background errors.
     ************************************************************/
    if(command != NULL) {
        if(conn->notify_command) {
            ckfree(conn->notify_command);
            conn->notify_command = NULL;
        }
        if(*command == '\0') {
            return TCL_OK;
        }
        conn->notify_command = ckalloc(strlen(command) + 1);
        strcpy(conn->notify_command, command);

        if(conn->notify_pending > 0 && conn->notify_active == 0) {
            conn->notify_active++;
            if(Tcl_VarEval(cd->interp, conn->notify_command, " ",
		    conn->handle, NULL) == TCL_ERROR) {
                Tcl_BackgroundError(cd->interp);
            }
            if(pqatcl_getconnbyid(NULL, cd, argv[1]) != NULL) {
                conn->notify_active--;
            }
        }

        Tcl_SetResult(cd->interp, "", TCL_VOLATILE);
        return TCL_OK;
    }

    /************************************************************
     * Return the current notify list for the connection
     * and free it.
     ************************************************************/
    Tcl_SetResult(interp, Tcl_DStringValue(&(conn->notify_list)),
    		TCL_VOLATILE);
    Tcl_DStringFree(&(conn->notify_list));
    Tcl_DStringInit(&(conn->notify_list));
    conn->notify_pending = 0;
    return TCL_OK;
}


/**********************************************************************
 * pqatcl_exec_asyncIO()	- Callback when backend sends data
 **********************************************************************/
void pqatcl_exec_asyncIO(ClientData cdata, int mask)
{
    PQA_client_data	*cd;
    PQA_connection	*conn = (PQA_connection *)cdata;
    PQA_result		*result;
    int i;
    int n;
    char buf[4096];
    char connid[32];
    char resultid[32];
    Tcl_HashEntry 	*hash_ent;

    /************************************************************
     * Setup all our pointers and remember the connection ID
     ************************************************************/
    cd = conn->cdata;
    strcpy(connid, conn->handle);
    if(conn->status == PQA_QSTATUS_IDLE ||
       conn->status == PQA_QSTATUS_LOGETID) {
        result = NULL;
    } else {
        result = conn->result_list;
    }

    /************************************************************
     * Get the data from the backend and handle connection loss
     ************************************************************/
    n = Tcl_Read(conn->backend, buf, sizeof(buf));
    if(n <= 0) {
        Tcl_Close(cd->interp, conn->backend);
        conn->status = PQA_QSTATUS_LOST;
        result = conn->result_list;
        while(result) {
            if(result->status == PQA_QSTATUS_BUSY) {
                result->status = PQA_QSTATUS_FATAL;
                Tcl_DStringAppend(&(result->errormsg),
                	"Fatal error: connection to database lost\n", -1);
                hash_ent = Tcl_CreateHashEntry(&(cd->idtable),
                	result->handle, &i);
                Tcl_SetHashValue(hash_ent, (ClientData)result);
                if(result->async_command) {
                    strcpy(resultid, result->handle);
                    if(Tcl_VarEval(cd->interp, result->async_command, " ",
                    		result->handle, NULL) == TCL_ERROR) {
                        Tcl_BackgroundError(cd->interp);
                    }
                    /**************************************************
                     * If the command closed the connection,
                     * everything else is already done by
                     * Pqa_close.
                     **************************************************/
                    if(pqatcl_getconnbyid(NULL, cd, connid) == NULL) {
                        return;
                    }

                    /**************************************************
                     * Scan again if the command cleared this
                     * result buffer
                     **************************************************/
                    if(pqatcl_getresultbyid(NULL, cd, resultid) == NULL) {
                        result = conn->result_list;
                        continue;
                    }
                }
            }
            result = result->next;
        }
        return;
    }

    /************************************************************
     * Process the incoming data and if the query is complete,
     ************************************************************/
    while(pqatcl_exec_getresult(cd->interp, conn, result, buf, n) == 0) {
        if(result) {
            /************************************************************
             * This was data for a query and the query is complete
             * now. So now create the hash entry for the result ID.
             * Reenable execution of queries and call the assotiated
             * asynchronous command (if any).
             ************************************************************/
            hash_ent = Tcl_CreateHashEntry(&(cd->idtable),
            	result->handle, &i);
            Tcl_SetHashValue(hash_ent, (ClientData)result);
            conn->status = PQA_QSTATUS_IDLE;
            if(result->async_command) {
                if(Tcl_VarEval(cd->interp, result->async_command, " ",
                	result->handle, NULL) == TCL_ERROR) {
                    Tcl_BackgroundError(cd->interp);
                }
            }
            result = NULL;
        }
        
        n = 0;

        /************************************************************
         * Maybe the async command closed the connection
         ************************************************************/
        if(pqatcl_getconnbyid(NULL, cd, connid) == NULL) {
            return;
        }
    }

    /************************************************************
     * If the connection is still alive, check for notifications
     ************************************************************/
    if(pqatcl_getconnbyid(NULL, cd, connid) == NULL) {
	return;
    }
    if(conn->notify_pending > 0 && conn->notify_command != NULL &&
    		conn->notify_active == 0) {
        conn->notify_active++;
        if(Tcl_VarEval(cd->interp, conn->notify_command, " ",
        	conn->handle, NULL) == TCL_ERROR) {
            Tcl_BackgroundError(cd->interp);
        }
        if(pqatcl_getconnbyid(NULL, cd, connid) != NULL) {
            conn->notify_active--;
        }
    }
}


/**********************************************************************
 * Inline function to check if we have a complete string
 **********************************************************************/
static inline int have_string(char *cp, int left)
{
    int n;

    for(n = 0; n < left; n++, cp++) {
        if(*cp == '\0') return n;
    }
    return -1;
}

#define need_nbytes_and_string(n)	if(left <= (n)) {	\
					    done++;	\
					    break;	\
					}		\
					if((i = have_string(cp + (n), left - (n))) < 0) { \
					    done++;	\
					    break;	\
					}		\
					cp += (n);	\
					left -= (n);


/**********************************************************************
 * pqatcl_exec_getresult()	- Interpret the data from backend
 **********************************************************************/
int pqatcl_exec_getresult(Tcl_Interp *interp, PQA_connection *conn, 
		PQA_result *result, char *buf, int len)
{
    char	*cp;
    int		left;
    int		id;
    int		i;
    int		shift;
    int		done;
    int		attrtype;
    char	attrbuf[128];
    int		flen;
    static int		xactid = 0;

    if(conn->status == PQA_QSTATUS_LOGETID) {
        return pqatcl_lobj_getresult(interp, conn, buf, len);
    }

    /************************************************************
     * Gcc claimed these two might be used uninitialized. In
     * fact, they're not.
     ************************************************************/
    id = i = 0;

    /************************************************************
     * Collect the data in the temp buffer
     ************************************************************/
    if(len > 0) {
	memcpy(&(conn->recv_buf[conn->recv_have]), buf, len);
	conn->recv_have += len;
    }

    cp = conn->recv_buf;
    left = conn->recv_have;
    done = 0;
    /************************************************************
     * Process the messages as long as there is enough
     * data and we are happy with it
     ************************************************************/
    while(left > 0 && !done) {

        switch(conn->status) {

        case PQA_QSTATUS_IDLE:
        	/************************************************************
        	 * Get the next ID and for V4.2 XactID
        	 ************************************************************/
        	if(conn->backend_version == PQA_DBVERSION_4_2) {
		    if(left < 5) {
			done++;
			break;
		    }
		    id = *cp;
		    xactid = 0;
		    shift = 0;
		    for(i = 1; i < 4; i++) {
			xactid |= (cp[i] & 0xFF) << shift;
			shift += 8;
		    }
        	} else
        	if(conn->backend_version == PQA_DBVERSION_95) {
		    if(left < 1) {
			done++;
			break;
		    }
		    id = *cp;
        	}

        	/************************************************************
        	 * We are IDLE - so only A messages are acceptable
        	 ************************************************************/
        	switch(id) {
        	case 'A':	/* Asynchronous portal notification */
        	    if(conn->backend_version == PQA_DBVERSION_4_2) {
			need_nbytes_and_string(5);
        	    } else
        	    if(conn->backend_version == PQA_DBVERSION_95) {
			need_nbytes_and_string(1);
        	    }
        	    conn->xactid = xactid;
        	    conn->notify_pending++;
        	    Tcl_DStringAppendElement(&(conn->notify_list), cp);
        	    cp += i + 1;
        	    left -= i + 1;
        	    break;

        	default:	/* Unsupported		*/
        	    sprintf(attrbuf, "%X", id);
        	    Tcl_AppendResult(interp, "Pqa: Got Id '0x", attrbuf,
        	    	"' in IDLE mode ???", NULL);
        	    Tcl_BackgroundError(interp);
        	    conn->status = PQA_QSTATUS_LOST;
        	    Tcl_Close(interp, conn->backend);
        	    return 1;
        	}
        	break; /* Done with PQA_QSTATUS_IDLE	*/

        case PQA_QSTATUS_GETID:
        	/************************************************************
        	 * Get the next ID and for V4.2 XactID
        	 ************************************************************/
        	if(conn->backend_version == PQA_DBVERSION_4_2) {
		    if(left < 5) {
			done++;
			break;
		    }
		    id = *cp;
		    xactid = 0;
		    shift = 0;
		    for(i = 1; i < 4; i++) {
			xactid |= (cp[i] & 0xFF) << shift;
			shift += 8;
		    }
        	} else
        	if(conn->backend_version == PQA_DBVERSION_95) {
		    if(left < 1) {
			done++;
			break;
		    }
		    id = *cp;
        	}

        	/************************************************************
        	 * Now process the message according to it's ID
        	 ************************************************************/
        	switch(id) {
        	case 'R':	/* Collect remarks and notices	*/
        	case 'N':
        	    if(conn->backend_version == PQA_DBVERSION_4_2) {
			need_nbytes_and_string(5);
        	    } else
        	    if(conn->backend_version == PQA_DBVERSION_95) {
			need_nbytes_and_string(1);
        	    }
        	    conn->xactid = xactid;
		    Tcl_DStringAppend(&(result->errormsg), cp, -1);
        	    cp += i + 1;
        	    left -= i + 1;
        	    break;

        	case 'C':	/* Query without data complete	*/
        	    if(conn->backend_version == PQA_DBVERSION_4_2) {
			need_nbytes_and_string(5);
        	    } else
        	    if(conn->backend_version == PQA_DBVERSION_95) {
			need_nbytes_and_string(1);
        	    }
        	    conn->xactid = xactid;
		    strcpy(result->c_ret, cp);

		    if(!strcmp(cp, "BEGIN")) {
		        conn->transaction_state = 1;
		    } else
		    if(!strcmp(cp, "END")) {
		        conn->transaction_state = 0;
		    } else
		    if(!strcmp(cp, "ABORT")) {
		        conn->transaction_state = 0;
		    }

		    if(pqatcl_send_simple(conn, 'Q', conn->xactid, " ")) {
			Tcl_DStringAppend(&(result->errormsg), 
			    "Fatal error: pqatcl_send_simple() for dummy query failed",
			    -1);
			result->status = PQA_QSTATUS_FATAL;
			return 0;
		    }
		    conn->dummy_queries++;
        	    cp += i + 1;
        	    left -= i + 1;
        	    break;

        	case 'I':	/* Our dummy queries return	*/
        	    if(conn->backend_version == PQA_DBVERSION_4_2) {
			cp += 5;
			left -= 5;
        	    } else
        	    if(conn->backend_version == PQA_DBVERSION_95) {
			cp += 2;
			left -= 2;
        	    }
        	    conn->xactid = xactid;
		    if(conn->dummy_queries == 0) {
			/**************************************************
			 * OOPS - User did a dummy query
			 **************************************************/
			result->status = PQA_QSTATUS_DONE;
			done++;
			break;
		    }
		    if(--(conn->dummy_queries) == 0) {
			done++;
			result->status = PQA_QSTATUS_DONE;
		    }
        	    break;

        	case 'E':	/* Severe error from backend	*/
        	    if(conn->backend_version == PQA_DBVERSION_4_2) {
			need_nbytes_and_string(5);
        	    } else
        	    if(conn->backend_version == PQA_DBVERSION_95) {
			need_nbytes_and_string(1);
        	    }
        	    conn->xactid = xactid;
		    result->status = PQA_QSTATUS_ERROR;
		    Tcl_DStringAppend(&(result->errormsg), cp, -1);
        	    cp += i + 1;
        	    left -= i + 1;
        	    break;

        	case 'P':	/* Portal for data	*/
        	    if(conn->backend_version == PQA_DBVERSION_4_2) {
			need_nbytes_and_string(5);
        	    } else
        	    if(conn->backend_version == PQA_DBVERSION_95) {
			need_nbytes_and_string(1);
        	    }
        	    conn->xactid = xactid;
        	    conn->status = PQA_QSTATUS_GETPID;
		    strcpy(result->p_ret, cp);
        	    cp += i + 1;
        	    left -= i + 1;
        	    break;

        	case 'A':	/* Asynchronous portal notification */
        	    if(conn->backend_version == PQA_DBVERSION_4_2) {
			need_nbytes_and_string(5);
        	    } else
        	    if(conn->backend_version == PQA_DBVERSION_95) {
			need_nbytes_and_string(1);
        	    }
        	    conn->xactid = xactid;
        	    conn->notify_pending++;
        	    Tcl_DStringAppendElement(&(conn->notify_list), cp);
        	    cp += i + 1;
        	    left -= i + 1;
        	    break;

        	default:	/* Unsupported		*/
        	    result->status = PQA_QSTATUS_FATAL;
        	    Tcl_DStringAppend(&(result->errormsg),
			"Fatal error: Backend protocol error - got ID ", -1);
		    sprintf(attrbuf, "'0x%X'\n", id);
		    Tcl_DStringAppend(&(result->errormsg), attrbuf, -1);
        	    conn->status = PQA_QSTATUS_LOST;
        	    Tcl_Close(interp, conn->backend);
        	    return 1;
        	}
        	break; /* Done with PQA_QSTATUS_GETID	*/

        /************************************************************
         * If a portal is received in normal GETID mode, we
         * switch to another set of messages supported.
         ************************************************************/
        case PQA_QSTATUS_GETPID:
        	/************************************************************
        	 * To get the next ID we need 1 byte
        	 ************************************************************/
        	if(left < 1) {
        	    return 1;
        	}
        	id = *cp;

        	/************************************************************
        	 * Now process the message according to it's ID
        	 ************************************************************/
        	switch(id) {
        	case 'R':	/* Collect remarks and notices	*/
        	case 'N':
        	    if(conn->backend_version == PQA_DBVERSION_4_2) {
			need_nbytes_and_string(5);
        	    } else
        	    if(conn->backend_version == PQA_DBVERSION_95) {
			need_nbytes_and_string(1);
        	    }
        	    Tcl_DStringAppend(&(result->errormsg), cp, -1);
        	    cp += i + 1;
        	    left -= i + 1;
        	    break;

        	case 'C':	/* End the command		*/
        	    if(conn->backend_version == PQA_DBVERSION_4_2) {
			need_nbytes_and_string(5);
        	    } else
        	    if(conn->backend_version == PQA_DBVERSION_95) {
			need_nbytes_and_string(1);
        	    }
        	    strcpy(result->c_ret, cp);
		    if(pqatcl_send_simple(conn, 'Q', conn->xactid, " ")) {
			Tcl_DStringAppend(&(result->errormsg), 
			    "Fatal error: pqatcl_send_simple() for dummy query failed",
			    -1);
			result->status = PQA_QSTATUS_FATAL;
			return 0;
		    }
		    conn->dummy_queries++;
		    conn->status = PQA_QSTATUS_GETID;
        	    cp += i + 1;
        	    left -= i + 1;
        	    break;

        	case 'E':	/* Severe error from Backend	*/
        	    if(conn->backend_version == PQA_DBVERSION_4_2) {
			need_nbytes_and_string(5);
        	    } else
        	    if(conn->backend_version == PQA_DBVERSION_95) {
			need_nbytes_and_string(1);
        	    }
        	    result->status = PQA_QSTATUS_ERROR;
        	    Tcl_DStringAppend(&(result->errormsg), cp, -1);
        	    cp += i + 1;
        	    left -= i + 1;
        	    break;

		case 'T':	/* Tuple description		*/
		    if(result->have_attrs) {
			/****************************************
			 * Currently we don't support multiple
			 * groups of tuples in a single portal.
			 ****************************************/
			result->status = PQA_QSTATUS_FATAL;
			Tcl_DStringAppend(&(result->errormsg),
			    "Fatal: Multiple tuple-groups unsupported", -1);
			return 0;
		    }
		    /****************************************
		     * At least we need the number of fields
		     ****************************************/
		    if(left < 3) {
		        done++;
		        break;
		    }
		    result->num_attrs = (cp[2] << 8) | cp[1];
		    if(result->num_attrs > 0) {
		        /****************************************
		         * There are fields, switch to TATTR mode
		         ****************************************/
		        Tcl_DStringInit(&(result->attributes));
		        conn->status = PQA_QSTATUS_TATTR;
		        conn->field_bmsize = result->num_attrs / 8;
		        if((result->num_attrs % 8) > 0) {
		            conn->field_bmsize++;
		        }
		    }
		    cp += 3;
		    left -= 3;
		    break;

        	case 'D':	/* Data arrives		*/
        	    /****************************************
        	     * At least we need the field bitmap
        	     ****************************************/
        	    if(left < (conn->field_bmsize + 1)) {
        	        done++;
        	        break;
        	    }
        	    memcpy(conn->field_bitmap, ++cp, conn->field_bmsize);
        	    cp += conn->field_bmsize;
        	    left -= conn->field_bmsize + 1;

        	    /****************************************
        	     * If this is the first tuple, preallocate
        	     * 32 Tcl_DString descriptors for the
        	     * incoming tuples
        	     ****************************************/
        	    if(result->alloc_tuples == 0) {
        	        result->tuple_strings = (char **)
        	        	ckalloc(sizeof(char *) * 32);
        	        result->alloc_tuples = 32;
        	    }
        	    /****************************************
        	     * That wasn't enough, duplicate the
        	     * number of tuples every time.
        	     ****************************************/
        	    if(result->num_tuples == result->alloc_tuples) {
        	        char **tmp_data;
        	        tmp_data = (char **)
        	        	ckalloc(sizeof(char *) * 
        	        	result->alloc_tuples * 2);
        	        memcpy(tmp_data, result->tuple_strings,
        	        	sizeof(char *) * result->alloc_tuples);
        	        ckfree(result->tuple_strings);
        	        result->tuple_strings = tmp_data;
        	        result->alloc_tuples *= 2;
        	    }

        	    /****************************************
        	     * Now initialize the next tuple list
        	     * and switch to TDATA mode.
        	     ****************************************/
        	    Tcl_DStringInit(&(result->tuple_current));
        	    conn->status = PQA_QSTATUS_TDATA;
        	    break;

        	case 'A':	/* Asynchronous portal notification */
        	    if(conn->backend_version == PQA_DBVERSION_4_2) {
			need_nbytes_and_string(5);
        	    } else
        	    if(conn->backend_version == PQA_DBVERSION_95) {
			need_nbytes_and_string(1);
        	    }
        	    conn->notify_pending++;
        	    Tcl_DStringAppendElement(&(conn->notify_list), cp);
        	    cp += i + 1;
        	    left -= i + 1;
        	    break;

        	default:	/* Unsupported		*/
        	    result->status = PQA_QSTATUS_FATAL;
        	    Tcl_DStringAppend(&(result->errormsg),
			"Fatal error: Backend protocol error - got ID ", -1);
		    sprintf(attrbuf, "'0x%X'\n", id);
		    Tcl_DStringAppend(&(result->errormsg), attrbuf, -1);
        	    conn->status = PQA_QSTATUS_LOST;
        	    Tcl_Close(interp, conn->backend);
        	    return 1;
        	}
        	break; /* Done with PQA_QSTATUS_GETPID	*/

        /************************************************************
         * We are inside the T message. Append entries
         * to the attribute description list
         ************************************************************/
        case PQA_QSTATUS_TATTR:
        	/**************************************************
        	 * We need a nul terminated string, a four byte
        	 * integer for the type and a two bye integer
        	 * for the length :-) silly format.
        	 **************************************************/
        	if((i = have_string(cp, left)) < 0) {
        	    done++;
        	    break;
        	}
        	if(left < i + 7) {
        	    done++;
        	    break;
        	}

        	/**************************************************
        	 * Append a list of the three values to
        	 * the attribute description
        	 **************************************************/
        	strcpy(attrbuf, cp);
		if(conn->backend_version == PQA_DBVERSION_4_2) {
		    attrbuf[16] = '\0';
		}
        	Tcl_DStringStartSublist(&(result->attributes));
        	Tcl_DStringAppendElement(&(result->attributes), attrbuf);
        	cp += i + 1;
        	left -= i + 1;

        	attrtype = 0;
        	shift = 0;
        	for(i = 0; i < 4; i++) {
        	    attrtype |= (*cp++ & 0xFF) << shift;
        	    shift += 8;
        	}
        	sprintf(attrbuf, "%d", attrtype);
        	left -= 4;
        	Tcl_DStringAppendElement(&(result->attributes), attrbuf);

        	attrtype = (cp[1] << 8) | cp[0];
        	sprintf(attrbuf, "%d", attrtype);
        	cp += 2;
        	left -= 2;
        	Tcl_DStringAppendElement(&(result->attributes), attrbuf);

        	Tcl_DStringEndSublist(&(result->attributes));

        	/**************************************************
        	 * If we have all attributes, switch back
        	 * to GETPID mode to expect the next ID in
        	 * portal mode.
        	 **************************************************/
        	result->have_attrs++;
        	if(result->have_attrs == result->num_attrs) {
        	    conn->status = PQA_QSTATUS_GETPID;
        	}

        	break; /* Done with PQA_QSTATUS_TAPPR	*/

        /************************************************************
         * We are inside the D message. As long as the
         * bitmap say's there are fields, get them.
         ************************************************************/
        case PQA_QSTATUS_TDATA:
        	while(conn->field_no < result->num_attrs) {
        	    if((conn->field_bitmap[conn->field_no / 8])
        	    		&0200) {
        	        char tmpc;
        	        if(left < 4) {
        	            done++;
        	            break;
        	        }
        	        shift = 0;
        	        flen  = 0;
        	        for(i = 0; i < 4; i++) {
        	            flen |= (cp[i] & 0xFF) << shift;
        	            shift += 8;
        	        }
        	        if(left < flen) {
        	            done++;
        	            break;
        	        }
        	        tmpc = cp[flen];
        	        cp[flen] = '\0';
        	        Tcl_DStringAppendElement(
        	            &(result->tuple_current), &cp[4]);
        	        cp[flen] = tmpc;
        	        cp += flen;
        	        left -= flen;
        	    } else {
        	        Tcl_DStringAppendElement(
        	            &(result->tuple_current), "");
        	    }
        	    conn->field_bitmap[conn->field_no++ / 8] <<= 1;
        	}
        	if(conn->field_no == result->num_attrs) {
        	    result->tuple_strings[result->num_tuples] = 
			    ckalloc(strlen(
			    Tcl_DStringValue(&(result->tuple_current))) + 1);
		    strcpy(result->tuple_strings[result->num_tuples],
		    	Tcl_DStringValue(&(result->tuple_current)));
		    Tcl_DStringFree(&(result->tuple_current));
        	    result->num_tuples++;
        	    conn->field_no = 0;
        	    conn->status = PQA_QSTATUS_GETPID;
        	} else {
        	    done++;
        	}

        	break; /* Done with PQA_QSTATUS_TDATA	*/
        }
    }

    /************************************************************
     * Remove the complete interpreted messages from
     * our internal buffer
     ************************************************************/
    if(left > 0) {
        memcpy(conn->recv_buf, cp, left);
        conn->recv_have = left;
    } else {
        conn->recv_have = 0;
    }

    /************************************************************
     * Check if we are through with the query and tell it
     * to the data reader.
     ************************************************************/
    if(result == NULL) {
        return 1;
    }
    if(result->status != PQA_QSTATUS_BUSY) {
        return 0;
    }
    return 1;	/* Need more data */
}


/**********************************************************************
 * pqatcl_send_simple()		- Send simple message format
 *				  to backend
 **********************************************************************/
int pqatcl_send_simple(PQA_connection *conn, char type, int id, char *msg)
{
    char *buf;
    char statbuf[4096];
    int  msglen;
    int  shift;
    int  i;
    int  retcode;

    /************************************************************
     * Allocate the send buffer
     ************************************************************/
    msglen = strlen(msg) + 6;
    if(msglen < 4096) {
        buf = statbuf;
    } else {
        buf = ckalloc(msglen);
    }

    /************************************************************
     * Create the message
     ************************************************************/
    buf[0] = type;
    switch(conn->backend_version) {
    case PQA_DBVERSION_4_2:
	shift = 0;
	for(i = 1; i < 5; i++) {
	    buf[i] = (id >> shift) & 0xFF;
	    shift += 8;
	}
	strcpy(&buf[5], msg);
	break;

    case PQA_DBVERSION_95:
        strcpy(&buf[1], msg);
        msglen -= 4;
        break;
    }

    /************************************************************
     * Send it including the terminating nul
     ************************************************************/
    retcode = 0;
    if(Tcl_Write(conn->backend, buf, msglen) < 0) {
	strcpy(PQerrormsg, "pqatcl_send_simple(): Tcl_Write() failed");
	retcode = -1;
    }
    if(Tcl_Flush(conn->backend) < 0) {
	strcpy(PQerrormsg, "pqatcl_send_simple(): Tcl_Write() failed");
	retcode = -1;
    }
    if(msglen >= 4096) {
        ckfree(buf);
    }

    return retcode;
}


