/**********************************************************************
 * $Source: /home/wieck/src/tcltk/extensions/pqatcl/libpqa/RCS/lobj.c,v $
 * $Revision: 0.1 $ $State: Stab $
 * $Date: 1996/06/20 17:57:11 $
 * $Author: wieck $
 *
 * DESCRIPTION
 * 	lobj.c			- Large object interface
 *
 * HISTORY
 * $Log: lobj.c,v $
 * Revision 0.1  1996/06/20 17:57:11  wieck
 *   Initial revision
 *
 **********************************************************************/
char _RCSID_Pqa_lobj[] = "$Id: lobj.c,v 0.1 1996/06/20 17:57:11 wieck Stab $";

#include <tcl.h>
#include "pqatcl.h"


static int pqatcl_lobj_parsemode(Tcl_Interp *interp, char *arg);
static int pqatcl_lobj_creat_4_2(Tcl_Interp *interp,
		PQA_client_data *cd, PQA_large_obj *lobj, int mode);
static int pqatcl_lobj_creat_95(Tcl_Interp *interp,
		PQA_client_data *cd, PQA_large_obj *lobj, int mode);

static Tcl_ChannelType pqa_channeltype = {
	"pqalobj",
	pqatcl_lobj_blockmode,
	pqatcl_lobj_close,
	pqatcl_lobj_input,
	pqatcl_lobj_output,
	pqatcl_lobj_seek,
	NULL, /* pqatcl_lobj_setoption, */
	NULL  /* pqatcl_lobj_getoption  */
};


/**********************************************************************
 * Pqa_lo_creat()		- Create a large object
 **********************************************************************/
int Pqa_lo_creat(ClientData cdata, Tcl_Interp *interp, int argc, char *argv[])
{
    PQA_client_data *cd = (PQA_client_data *)cdata;
    PQA_connection *conn;
    PQA_result     *result;
    PQA_large_obj  *lobj;
    int		mode;
    int		status = 0;
    char	objid[32];

    /************************************************************
     * Check if we have at least the connection and the query
     ************************************************************/
    if(argc != 3) {
        Tcl_AppendResult(interp, argv[0], ": syntax error '",
        	argv[0], " conn mode '", NULL);
        return TCL_ERROR;
    }

    /************************************************************
     * Get the connection
     ************************************************************/
    if((conn = pqatcl_getconnbyid(interp, cd, argv[1])) == NULL) {
        return TCL_ERROR;
    }

    /************************************************************
     * The database connection must be alive and there must
     * not be any operation in progress.
     ************************************************************/
    while(conn->status != PQA_QSTATUS_IDLE) {
	if(conn->status == PQA_QSTATUS_LOST) {
	    Tcl_AppendResult(interp, argv[0], 
		    ": connection to database lost", NULL);
	    return TCL_ERROR;
	}
        Tcl_DoOneEvent(TCL_ALL_EVENTS);
        if(pqatcl_getconnbyid(interp, cd, argv[1]) == NULL) {
            return TCL_ERROR;
        }
    }
    if(conn->large_obj_current != NULL) {
        Tcl_AppendResult(interp, argv[0], 
        	": large object operation in progress", NULL);
        return TCL_ERROR;
    }

    /************************************************************
     * Get the mode
     ************************************************************/
    if((mode = pqatcl_lobj_parsemode(interp, argv[2])) < 0) {
        return TCL_ERROR;
    }

    /************************************************************
     * Make sure the large object interface is initialized
     ************************************************************/
    if(!(conn->lobj_initialized)) {
        if(pqatcl_lobj_init(interp, cd, conn) == TCL_ERROR) {
            return TCL_ERROR;
        }
    }

    /************************************************************
     * Allocate and initialize the control buffer
     ************************************************************/
    lobj = (PQA_large_obj *)ckalloc(sizeof(PQA_large_obj));
    memset(lobj, 0, sizeof(PQA_large_obj));
    lobj->struct_type = PQA_STRUCT_LARGE_OBJ;
    lobj->conn        = conn;
    lobj->cdata       = cd;
    lobj->status      = PQA_QSTATUS_BUSY;
    sprintf(lobj->handle, "pqalo%d", cd->large_obj_id++);

    /************************************************************
     * Large object operations require transaction control
     ************************************************************/
    if(!(conn->transaction_state)) {
        if(Tcl_VarEval(interp, "pqa_exec ", conn->handle, 
        	" begin", NULL) == TCL_ERROR) {
            ckfree((char *)lobj);
            return TCL_ERROR;
        }
        if((result = pqatcl_getresultbyid(interp, cd, interp->result)) == NULL) {
            ckfree((char *)lobj);
            return TCL_ERROR;
        }
        if(result->status != PQA_QSTATUS_DONE) {
            Tcl_VarEval(interp, "pqa_result ", result->handle,
            		" -errormsg -clear", NULL);
            ckfree((char *)lobj);
            return TCL_ERROR;
        }
	if(Tcl_VarEval(interp, "pqa_result ", result->handle,
            		" -clear", NULL) == TCL_ERROR) {
            ckfree((char *)lobj);
            return TCL_ERROR;
        }
        lobj->auto_end = 1;
    }

    /************************************************************
     * Now call the DB-Version specific routine to create
     * the large object
     ************************************************************/
    conn->large_obj_current = lobj;
    conn->status            = PQA_QSTATUS_LOGETID;
    Tcl_DStringInit(&(lobj->errormsg));
    if(conn->backend_version == PQA_DBVERSION_4_2) {
        status =  pqatcl_lobj_creat_4_2(interp, cd, lobj, mode);
    }
    if(conn->backend_version == PQA_DBVERSION_95) {
        status =  pqatcl_lobj_creat_95(interp, cd, lobj, mode);
    }
    conn->large_obj_current = NULL;


    /************************************************************
     * On errors be sure to put the backend into transaction
     * abort mode
     ************************************************************/
    if(status == TCL_ERROR) {
        Tcl_VarEval(interp, "pqa_exec ", conn->handle, " ERROR", NULL);
        Tcl_VarEval(interp, "pqa_result ", interp->result, " -clear", NULL);
    }

    /************************************************************
     * If we started the transaction, end it 
     ************************************************************/
    if(lobj->auto_end) {
        if(Tcl_VarEval(interp, "pqa_exec ", conn->handle, 
        	" end", NULL) == TCL_ERROR) {
            Tcl_DStringFree(&(lobj->errormsg));
            ckfree((char *)lobj);
            return TCL_ERROR;
        }
        if((result = pqatcl_getresultbyid(interp, cd, interp->result)) == NULL) {
            Tcl_DStringFree(&(lobj->errormsg));
            ckfree((char *)lobj);
            return TCL_ERROR;
        }
        if(result->status != PQA_QSTATUS_DONE) {
	    Tcl_DStringAppend(&(lobj->errormsg),
	    		Tcl_DStringValue(&(result->errormsg)), -1);
            status = TCL_ERROR;
        }
	if(Tcl_VarEval(interp, "pqa_result ", result->handle,
			" -clear", NULL) == TCL_ERROR) {
	    Tcl_DStringAppend(&(lobj->errormsg), interp->result, -1);
	    status =  TCL_ERROR;
	}
    }

    /************************************************************
     * Return the result
     ************************************************************/
    if(status == TCL_OK) {
        sprintf(objid, "%d", lobj->oid);
        Tcl_SetResult(interp, objid, TCL_VOLATILE);
    } else {
        Tcl_SetResult(interp, Tcl_DStringValue(&(lobj->errormsg)),
        	TCL_VOLATILE);
    }

    Tcl_DStringFree(&(lobj->errormsg));
    ckfree((char *)lobj);

    return status;
}


/**********************************************************************
 * Pqa_lo_unlink()		- Unlink a large object
 **********************************************************************/
int Pqa_lo_unlink(ClientData cdata, Tcl_Interp *interp, int argc, char *argv[])
{
    PQA_client_data *cd = (PQA_client_data *)cdata;
    PQA_connection *conn;
    PQA_result     *result;
    PQA_large_obj  *lobj;
    int		status = 0;
    char	objid[32];
    PQA_fnarg	unlink_arg[1];

    /************************************************************
     * Check if we have at least the connection and the query
     ************************************************************/
    if(argc != 3) {
        Tcl_AppendResult(interp, argv[0], ": syntax error '",
        	argv[0], " conn oid '", NULL);
        return TCL_ERROR;
    }

    /************************************************************
     * Get the connection
     ************************************************************/
    if((conn = pqatcl_getconnbyid(interp, cd, argv[1])) == NULL) {
        return TCL_ERROR;
    }

    /************************************************************
     * The database connection must be alive and there must
     * not be any operation in progress.
     ************************************************************/
    while(conn->status != PQA_QSTATUS_IDLE) {
	if(conn->status == PQA_QSTATUS_LOST) {
	    Tcl_AppendResult(interp, argv[0], 
		    ": connection to database lost", NULL);
	    return TCL_ERROR;
	}
        Tcl_DoOneEvent(TCL_ALL_EVENTS);
        if(pqatcl_getconnbyid(interp, cd, argv[1]) == NULL) {
            return TCL_ERROR;
        }
    }
    if(conn->large_obj_current != NULL) {
        Tcl_AppendResult(interp, argv[0], 
        	": large object operation in progress", NULL);
        return TCL_ERROR;
    }

    /************************************************************
     * Make sure the large object interface is initialized
     ************************************************************/
    if(!(conn->lobj_initialized)) {
        if(pqatcl_lobj_init(interp, cd, conn) == TCL_ERROR) {
            return TCL_ERROR;
        }
    }

    /************************************************************
     * Allocate and initialize the control buffer
     ************************************************************/
    lobj = (PQA_large_obj *)ckalloc(sizeof(PQA_large_obj));
    memset(lobj, 0, sizeof(PQA_large_obj));
    lobj->struct_type = PQA_STRUCT_LARGE_OBJ;
    lobj->conn        = conn;
    lobj->cdata       = cd;
    lobj->status      = PQA_QSTATUS_BUSY;
    sprintf(lobj->handle, "pqalo%d", cd->large_obj_id++);

    /************************************************************
     * Large object operations require transaction control
     ************************************************************/
    if(!(conn->transaction_state)) {
        if(Tcl_VarEval(interp, "pqa_exec ", conn->handle, 
        	" begin", NULL) == TCL_ERROR) {
            ckfree((char *)lobj);
            return TCL_ERROR;
        }
        if((result = pqatcl_getresultbyid(interp, cd, interp->result)) == NULL) {
            ckfree((char *)lobj);
            return TCL_ERROR;
        }
        if(result->status != PQA_QSTATUS_DONE) {
            Tcl_AppendResult(interp, Tcl_DStringValue(&(result->errormsg)), 
            		NULL);
            Tcl_VarEval(interp, "pqa_result ", result->handle,
            		" -clear", NULL);
            ckfree((char *)lobj);
            return TCL_ERROR;
        }
	if(Tcl_VarEval(interp, "pqa_result ", result->handle,
            		" -clear", NULL) == TCL_ERROR) {
            ckfree((char *)lobj);
            return TCL_ERROR;
        }
        lobj->auto_end = 1;
    }

    /************************************************************
     * Call the unlink function
     ************************************************************/
    conn->large_obj_current = lobj;
    conn->status            = PQA_QSTATUS_LOGETID;
    Tcl_DStringInit(&(lobj->errormsg));
    if(conn->backend_version == PQA_DBVERSION_4_2) {
        sprintf(objid, "/pqa%s", argv[2]);
        unlink_arg[0].len    = PQA_VAR_LENGTH_ARG_4_2;
        unlink_arg[0].isint  = 0;
        unlink_arg[0].u.parg = objid;

        status = pqatcl_lobj_sendfunc(conn, conn->lobj_fnid_unlink,
        	4, 1, unlink_arg);
    }
    if(conn->backend_version == PQA_DBVERSION_95) {
        unlink_arg[0].len    = 4;
        unlink_arg[0].isint  = 1;
        unlink_arg[0].u.iarg = atoi(argv[2]);

        status = pqatcl_lobj_sendfunc(conn, conn->lobj_fnid_unlink,
        	4, 1, unlink_arg);
    }
    /************************************************************
     * Check for error
     ************************************************************/
    if(status != 0) {
        Tcl_DStringAppend(&(lobj->errormsg), PQerrormsg, -1);
        status = TCL_ERROR;
    } else {
        status = TCL_OK;
    }

    /************************************************************
     * Wait for the function to complete
     ************************************************************/
    while(lobj->status == PQA_QSTATUS_BUSY) {
        Tcl_DoOneEvent(TCL_ALL_EVENTS);
    }
    conn->large_obj_current = NULL;
    conn->status = PQA_QSTATUS_IDLE;

    if(lobj->status != PQA_QSTATUS_DONE) {
        status = TCL_ERROR;
    }

    /************************************************************
     * Destroy the class under V4.2 (Postgres95 compatibility)
     ************************************************************/
    if(conn->backend_version == PQA_DBVERSION_4_2) {
        sprintf(objid, "Xinv%s", argv[2]);
        if(Tcl_VarEval(interp, "pqa_exec ", conn->handle,
        	" \"destroy ", objid, "\"", NULL) == TCL_ERROR) {
            Tcl_DStringFree(&(lobj->errormsg));
            ckfree((char *)lobj);
            return TCL_ERROR;
        }
        if((result = pqatcl_getresultbyid(interp, cd, interp->result)) == NULL) {
            Tcl_DStringFree(&(lobj->errormsg));
            ckfree((char *)lobj);
            return TCL_ERROR;
        }
        if(result->status != PQA_QSTATUS_DONE) {
            Tcl_DStringAppend(&(lobj->errormsg), 
            		Tcl_DStringValue(&(result->errormsg)), -1);
            status = TCL_ERROR;
        }
	if(Tcl_VarEval(interp, "pqa_result ", result->handle,
            		" -clear", NULL) == TCL_ERROR) {
            status =  TCL_ERROR;
        }
    }

    /************************************************************
     * If we started the transaction, end it 
     ************************************************************/
    if(lobj->auto_end) {
        if(Tcl_VarEval(interp, "pqa_exec ", conn->handle, 
        	" end", NULL) == TCL_ERROR) {
            Tcl_DStringFree(&(lobj->errormsg));
            ckfree((char *)lobj);
            return TCL_ERROR;
        }
        if((result = pqatcl_getresultbyid(interp, cd, interp->result)) == NULL) {
            Tcl_DStringFree(&(lobj->errormsg));
            ckfree((char *)lobj);
            return TCL_ERROR;
        }
        if(result->status != PQA_QSTATUS_DONE) {
            Tcl_DStringAppend(&(lobj->errormsg), 
            		Tcl_DStringValue(&(result->errormsg)), -1);
            status = TCL_ERROR;
        }
	if(Tcl_VarEval(interp, "pqa_result ", result->handle,
            		" -clear", NULL) == TCL_ERROR) {
            status =  TCL_ERROR;
        }
    }

    /************************************************************
     * Return the result
     ************************************************************/
    if(status == TCL_OK) {
        Tcl_SetResult(interp, "", TCL_VOLATILE);
    } else {
        Tcl_SetResult(interp, Tcl_DStringValue(&(lobj->errormsg)),
        	TCL_VOLATILE);
    }

    Tcl_DStringFree(&(lobj->errormsg));
    ckfree((char *)lobj);

    return status;
}


/**********************************************************************
 * Pqa_lo_open()		- Open a large object
 **********************************************************************/
int Pqa_lo_open(ClientData cdata, Tcl_Interp *interp, int argc, char *argv[])
{
    PQA_client_data *cd = (PQA_client_data *)cdata;
    PQA_connection *conn;
    PQA_result     *result;
    PQA_large_obj  *lobj;
    int		mode;
    int		status = 0;
    char	objid[32];
    PQA_fnarg	open_arg[2];
    int		i;
    int		shift;

    Tcl_Channel pqa_channel;
    Tcl_File	pqa_infile;
    Tcl_File	pqa_outfile;
    char	*channel_name;

    static int	f_disconnect;
    static PQA_argument args[] = {
        { "-disconnect", PQA_ARGTYPE_INTVAL1, &f_disconnect },
        { NULL, 0, NULL }
    };

    /************************************************************
     * Check if we have all required arguments
     ************************************************************/
    if(argc < 4) {
        Tcl_AppendResult(interp, argv[0], ": syntax error '",
        	argv[0], " conn oid mode'", NULL);
        return TCL_ERROR;
    }

    /************************************************************
     * Get the connection
     ************************************************************/
    if((conn = pqatcl_getconnbyid(interp, cd, argv[1])) == NULL) {
        return TCL_ERROR;
    }

    /************************************************************
     * The database connection must be alive and there must
     * not be any operation in progress.
     ************************************************************/
    while(conn->status != PQA_QSTATUS_IDLE) {
	if(conn->status == PQA_QSTATUS_LOST) {
	    Tcl_AppendResult(interp, argv[0], 
		    ": connection to database lost", NULL);
	    return TCL_ERROR;
	}
        Tcl_DoOneEvent(TCL_ALL_EVENTS);
        if(pqatcl_getconnbyid(interp, cd, argv[1]) == NULL) {
            return TCL_ERROR;
        }
    }
    if(conn->large_obj_current != NULL) {
        Tcl_AppendResult(interp, argv[0], 
        	": large object operation in progress", NULL);
        return TCL_ERROR;
    }

    /************************************************************
     * Get the open mode
     ************************************************************/
    if((mode = pqatcl_lobj_parsemode(interp, argv[3])) < 0) {
        return TCL_ERROR;
    }

    /************************************************************
     * Get the options
     ************************************************************/
    f_disconnect = 0;
    if(pqatcl_get_arguments(interp, args, 4, argc, argv) != TCL_OK) {
        return TCL_ERROR;
    }

    /************************************************************
     * Make sure the large object interface is initialized
     ************************************************************/
    if(!(conn->lobj_initialized)) {
        if(pqatcl_lobj_init(interp, cd, conn) == TCL_ERROR) {
            return TCL_ERROR;
        }
    }

    /************************************************************
     * Allocate and initialize the control buffer
     ************************************************************/
    lobj = (PQA_large_obj *)ckalloc(sizeof(PQA_large_obj));
    memset(lobj, 0, sizeof(PQA_large_obj));
    lobj->struct_type = PQA_STRUCT_LARGE_OBJ;
    lobj->conn        = conn;
    lobj->cdata       = cd;
    lobj->status      = PQA_QSTATUS_BUSY;
    lobj->auto_disconnect = f_disconnect;
    channel_name      = ckalloc(64);
    sprintf(channel_name, "pqalobj%d", cd->large_obj_id);
    sprintf(lobj->handle, "pqalo%d", cd->large_obj_id++);

    /************************************************************
     * Large object operations require transaction control
     ************************************************************/
    if(!(conn->transaction_state)) {
        if(Tcl_VarEval(interp, "pqa_exec ", conn->handle, 
        	" begin", NULL) == TCL_ERROR) {
            ckfree((char *)lobj);
            ckfree(channel_name);
            return TCL_ERROR;
        }
        if((result = pqatcl_getresultbyid(interp, cd, interp->result)) == NULL) {
            ckfree((char *)lobj);
            ckfree(channel_name);
            return TCL_ERROR;
        }
        if(result->status != PQA_QSTATUS_DONE) {
            Tcl_AppendResult(interp, Tcl_DStringValue(&(result->errormsg)), 
            		NULL);
            Tcl_VarEval(interp, "pqa_result ", result->handle,
            		" -clear", NULL);
            ckfree((char *)lobj);
            ckfree(channel_name);
            return TCL_ERROR;
        }
	if(Tcl_VarEval(interp, "pqa_result ", result->handle,
            		" -clear", NULL) == TCL_ERROR) {
            ckfree((char *)lobj);
            ckfree(channel_name);
            return TCL_ERROR;
        }
        lobj->auto_end = 1;
    }

    /************************************************************
     * Call the open function
     ************************************************************/
    conn->large_obj_current = lobj;
    conn->status            = PQA_QSTATUS_LOGETID;
    Tcl_DStringInit(&(lobj->errormsg));
    if(conn->backend_version == PQA_DBVERSION_4_2) {
        sprintf(objid, "/pqa%s", argv[2]);
        open_arg[0].len    = PQA_VAR_LENGTH_ARG_4_2;
        open_arg[0].isint  = 0;
        open_arg[0].u.parg = objid;

        open_arg[1].len    = 4;
        open_arg[1].isint  = 1;
        open_arg[1].u.iarg = mode;

        status = pqatcl_lobj_sendfunc(conn, conn->lobj_fnid_open,
        	4, 2, open_arg);
    }
    if(conn->backend_version == PQA_DBVERSION_95) {
        open_arg[0].len    = 4;
        open_arg[0].isint  = 1;
        open_arg[0].u.iarg = atoi(argv[2]);

        open_arg[1].len    = 4;
        open_arg[1].isint  = 1;
        open_arg[1].u.iarg = mode;

        status = pqatcl_lobj_sendfunc(conn, conn->lobj_fnid_open,
        	4, 2, open_arg);
    }
    /************************************************************
     * Check for error
     ************************************************************/
    if(status != 0) {
        Tcl_DStringAppend(&(lobj->errormsg), PQerrormsg, -1);
        status = TCL_ERROR;
    } else {
        status = TCL_OK;
    }

    /************************************************************
     * Wait for the function to complete
     ************************************************************/
    while(lobj->status == PQA_QSTATUS_BUSY) {
        Tcl_DoOneEvent(TCL_ALL_EVENTS);
    }
    conn->large_obj_current = NULL;
    conn->status = PQA_QSTATUS_IDLE;

    if(lobj->status != PQA_QSTATUS_DONE) {
        status = TCL_ERROR;
    } else {
	/************************************************************
	 * Remember the backend filedescriptor and object ID
	 ************************************************************/
	lobj->backend_fd = 0;
	for(i = 0, shift = 0; i < 4; i++, shift += 8) {
	    lobj->backend_fd |= ((lobj->buf[i]) & 0xFF) << shift;
	}
	lobj->oid = atoi(argv[2]);
	if(lobj->backend_fd < 0) {
	    Tcl_DStringAppend(&(lobj->errormsg),
	    	"Error opening large object - backend fd invalid",
	    	-1);
	    status = TCL_ERROR;
	} else {
	    /************************************************************
	     * Now create the channel to the large object
	     ************************************************************/
	    if(mode & PQA_INV_READ) {
		pqa_infile = Tcl_GetChannelFile(conn->backend, TCL_READABLE);
	    } else {
		pqa_infile = NULL;
	    }
	    if(mode & PQA_INV_WRITE) {
		pqa_outfile = Tcl_GetChannelFile(conn->backend, TCL_WRITABLE);
	    } else {
		pqa_outfile = NULL;
	    }

	    pqa_channel = Tcl_CreateChannel(&pqa_channeltype,
			    channel_name, pqa_infile, pqa_outfile,
			    (ClientData)lobj);
	    Tcl_RegisterChannel(interp, pqa_channel);
        }
    }

    /************************************************************
     * If we started the transaction, end it on errors
     ************************************************************/
    if(lobj->auto_end && status != TCL_OK) {
        if(Tcl_VarEval(interp, "pqa_exec ", conn->handle, 
        	" end", NULL) == TCL_ERROR) {
            Tcl_DStringFree(&(lobj->errormsg));
            ckfree((char *)lobj);
            ckfree(channel_name);
            return TCL_ERROR;
        }
        if((result = pqatcl_getresultbyid(interp, cd, interp->result)) == NULL) {
            Tcl_DStringFree(&(lobj->errormsg));
            ckfree((char *)lobj);
            ckfree(channel_name);
            return TCL_ERROR;
        }
        if(result->status != PQA_QSTATUS_DONE) {
            Tcl_DStringAppend(&(lobj->errormsg), 
            		Tcl_DStringValue(&(result->errormsg)), -1);
            status = TCL_ERROR;
        }
	if(Tcl_VarEval(interp, "pqa_result ", result->handle,
            		" -clear", NULL) == TCL_ERROR) {
            status =  TCL_ERROR;
        }
    }

    /************************************************************
     * Return the result and on success link this channel into
     * the list of large object channels for this connection.
     ************************************************************/
    if(status == TCL_OK) {
        if(conn->large_obj_list != NULL) {
            conn->large_obj_list->prev = lobj;
        }
        lobj->next = conn->large_obj_list;
        conn->large_obj_list = lobj;
        Tcl_SetResult(interp, channel_name, TCL_VOLATILE);
    } else {
        Tcl_SetResult(interp, Tcl_DStringValue(&(lobj->errormsg)),
        	TCL_VOLATILE);
	Tcl_DStringFree(&(lobj->errormsg));
	ckfree((char *)lobj);
	ckfree(channel_name);
    }

    return status;
}


/**********************************************************************
 * pqatcl_lobj_init()		- Initialize the large object interface
 **********************************************************************/
int pqatcl_lobj_init(Tcl_Interp *interp, PQA_client_data *cd,
		PQA_connection *conn)
{
    char	resid[64];
    int		status = 0;
    PQA_result	*result;
    char	**tuples;
    int		i;

    conn->lobj_fnid_creat  = -1;
    conn->lobj_fnid_open   = -1;
    conn->lobj_fnid_close  = -1;
    conn->lobj_fnid_lseek  = -1;
    conn->lobj_fnid_tell   = -1;
    conn->lobj_fnid_read   = -1;
    conn->lobj_fnid_write  = -1;
    conn->lobj_fnid_unlink = -1;

    /************************************************************
     * Execute the query to get the function ID's we need
     ************************************************************/
    if(conn->backend_version == PQA_DBVERSION_4_2) {
        status = Tcl_VarEval(interp, "pqa_exec ", conn->handle, " "
        	"\"retrieve (f.proname, f.oid) from f in pg_proc	\
        	      where f.proname = \\\"LOcreat\\\"		\
        	         or f.proname = \\\"LOopen\\\"		\
        	         or f.proname = \\\"LOclose\\\"		\
        	         or f.proname = \\\"LOlseek\\\"		\
        	         or f.proname = \\\"LOtell\\\"		\
        	         or f.proname = \\\"LOread\\\"		\
        	         or f.proname = \\\"LOwrite\\\"		\
        	         or f.proname = \\\"LOunlink\\\"\"", NULL);
    }
    if(conn->backend_version == PQA_DBVERSION_95) {
        status = Tcl_VarEval(interp, "pqa_exec ", conn->handle, " "
        	"\"select f.proname, f.oid from pg_proc f	\
        	    where f.proname = 'lo_creat'		\
        	       or f.proname = 'lo_open'			\
        	       or f.proname = 'lo_close'		\
        	       or f.proname = 'lo_lseek'		\
        	       or f.proname = 'lo_tell'			\
        	       or f.proname = 'LOread'			\
        	       or f.proname = 'LOwrite'			\
        	       or f.proname = 'lo_unlink'\"", NULL);
    }
    if(status == TCL_ERROR) return TCL_ERROR;
    strncpy(resid, interp->result, 63);
    resid[63] = 0;

    /************************************************************
     * Get the result buffer internal and check that the
     * query finished successful
     ************************************************************/
    if((result = pqatcl_getresultbyid(interp, cd, resid)) == NULL) {
        return TCL_ERROR;
    }

    if(result->status != PQA_QSTATUS_DONE) {
        Tcl_VarEval(interp, "pqa_result ", resid, " -errormsg -clear", NULL);
        Tcl_AppendResult(interp, 
        	"cannot initialize large object interface", NULL);
        return TCL_ERROR;
    }

    /************************************************************
     * Now get the values for the function ID's
     ************************************************************/
    tuples = result->tuple_strings;
    for(i = 0; i < result->num_tuples; i++, tuples++) {
        sscanf(*tuples, "LOcreat %d",   &(conn->lobj_fnid_creat));
        sscanf(*tuples, "lo_creat %d",  &(conn->lobj_fnid_creat));
        sscanf(*tuples, "LOopen %d",    &(conn->lobj_fnid_open));
        sscanf(*tuples, "lo_open %d",   &(conn->lobj_fnid_open));
        sscanf(*tuples, "LOclose %d",   &(conn->lobj_fnid_close));
        sscanf(*tuples, "lo_close %d",  &(conn->lobj_fnid_close));
        sscanf(*tuples, "LOlseek %d",   &(conn->lobj_fnid_lseek));
        sscanf(*tuples, "lo_lseek %d",  &(conn->lobj_fnid_lseek));
        sscanf(*tuples, "LOtell %d",    &(conn->lobj_fnid_tell));
        sscanf(*tuples, "lo_tell %d",   &(conn->lobj_fnid_tell));
        sscanf(*tuples, "LOread %d",    &(conn->lobj_fnid_read));
        sscanf(*tuples, "LOwrite %d",   &(conn->lobj_fnid_write));
        sscanf(*tuples, "LOunlink %d",  &(conn->lobj_fnid_unlink));
        sscanf(*tuples, "lo_unlink %d", &(conn->lobj_fnid_unlink));
    }

    /************************************************************
     * Check that all functions where found
     ************************************************************/
    if(conn->lobj_fnid_creat  < 0 ||
       conn->lobj_fnid_open   < 0 ||
       conn->lobj_fnid_close  < 0 ||
       conn->lobj_fnid_lseek  < 0 ||
       conn->lobj_fnid_tell   < 0 ||
       conn->lobj_fnid_read   < 0 ||
       conn->lobj_fnid_write  < 0 ||
       conn->lobj_fnid_unlink < 0) {
        Tcl_AppendResult(interp,
        	"cannot initialize large object interface\n",
        	"backend doesn't support all required functions",
        	NULL);
        Tcl_VarEval(interp, "pqa_result ", resid, " -clear", NULL);
        return TCL_ERROR;
    }

    /************************************************************
     * Clear the result buffer and remember initialization
     ************************************************************/
    if(Tcl_VarEval(interp, "pqa_result ", resid, " -clear", NULL) == TCL_ERROR) {
        return TCL_ERROR;
    }

    conn->lobj_initialized = 1;
    return TCL_OK;
}


/**********************************************************************
 * pqatcl_lobj_parsemode()	- Interpret the list of access modes
 **********************************************************************/
static int pqatcl_lobj_parsemode(Tcl_Interp *interp, char *arg)
{
    int		mode;
    int		mode_argc;
    char	**mode_argv;
    char	**val;
    int		i;

    if(Tcl_SplitList(interp, arg, &mode_argc, &mode_argv) == TCL_ERROR) {
        return -1;
    }

    if(mode_argc <= 0) {
        ckfree((char *)mode_argv);
        Tcl_AppendResult(interp, "invalid lobj access mode '",
        	arg, "'", NULL);
        return -1;
    }

    mode = 0;
    val  = mode_argv;
    for(i = 0; i < mode_argc; i++, val++) {
        if(!strcmp(*val, "INV_ARCHIVE")) {
            mode |= PQA_INV_ARCHIVE;
            continue;
        }
        if(!strcmp(*val, "INV_WRITE")) {
            mode |= PQA_INV_WRITE;
            continue;
        }
        if(!strcmp(*val, "INV_READ")) {
            mode |= PQA_INV_READ;
            continue;
        }

        ckfree((char *)mode_argv);
        Tcl_AppendResult(interp, "invalid lobj access mode '",
        	arg, "'", NULL);
        return -1;
    }

    ckfree((char *)mode_argv);
    return mode;
}


/**********************************************************************
 * pqatcl_lobj_creat_4_2()	- Create a large object in V4.2
 **********************************************************************/
static int pqatcl_lobj_creat_4_2(Tcl_Interp *interp,
		PQA_client_data *cd, PQA_large_obj *lobj, int mode)
{
    PQA_connection	*conn = lobj->conn;
    PQA_fnarg	creat_arg[3];
    int		ret;
    int		i;
    int		shift;
    PQA_result	*result;
    char	newname[32];

    /************************************************************
     * Call create function in backend
     ************************************************************/
    creat_arg[0].len    = PQA_VAR_LENGTH_ARG_4_2;
    creat_arg[0].isint  = 0;
    creat_arg[0].u.parg = "/pqa_lo_dummy";

    creat_arg[1].len    = 4;
    creat_arg[1].isint  = 1;
    creat_arg[1].u.iarg = mode;

    creat_arg[2].len    = 4;
    creat_arg[2].isint  = 1;
    creat_arg[2].u.parg = PQA_Inversion;

    if(pqatcl_lobj_sendfunc(conn, conn->lobj_fnid_creat,
    		4, 3, creat_arg) < 0) {
        conn->status = PQA_QSTATUS_LOST;
        Tcl_Close(interp, conn->backend);
        Tcl_DStringAppend(&(lobj->errormsg), PQerrormsg, -1);
        return TCL_ERROR;
    }

    while(lobj->status == PQA_QSTATUS_BUSY) {
        Tcl_DoOneEvent(TCL_ALL_EVENTS);
    }
    if(lobj->status != PQA_QSTATUS_DONE) {
        conn->status = PQA_QSTATUS_IDLE;
        return TCL_ERROR;
    }

    /************************************************************
     * Get the backend filedescriptor
     ************************************************************/
    lobj->backend_fd = 0;
    for(i = 0, shift = 0; i < 0; i++, shift += 8) {
        lobj->backend_fd |= ((lobj->buf[i]) & 0xFF) << shift;
    }

    /************************************************************
     * Close the new large object (Postgres95 compatibility)
     ************************************************************/
    creat_arg[0].len    = 4;
    creat_arg[0].isint  = 1;
    creat_arg[0].u.iarg = lobj->backend_fd;

    lobj->status = PQA_QSTATUS_BUSY;
    if(pqatcl_lobj_sendfunc(conn, conn->lobj_fnid_close,
    		4, 1, creat_arg) < 0) {
        conn->status = PQA_QSTATUS_LOST;
        Tcl_Close(interp, conn->backend);
        Tcl_DStringAppend(&(lobj->errormsg), PQerrormsg, -1);
        return TCL_ERROR;
    }

    while(lobj->status == PQA_QSTATUS_BUSY) {
        Tcl_DoOneEvent(TCL_ALL_EVENTS);
    }
    if(lobj->status != PQA_QSTATUS_DONE) {
        conn->status = PQA_QSTATUS_IDLE;
        return TCL_ERROR;
    }

    /************************************************************
     * Get the result
     ************************************************************/
    ret = 0;
    for(i = 0, shift = 0; i < 4; i++, shift += 8) {
        ret |= ((lobj->buf[i]) & 0xFF) << shift;
    }

    conn->status = PQA_QSTATUS_IDLE;

    /************************************************************
     * Now lookup for the object ID
     ************************************************************/
    if(Tcl_VarEval(interp, "pqa_exec ", conn->handle,
		" \"retrieve (pg_naming.ourid) \
    		where pg_naming.filename = \\\"pqa_lo_dummy\\\"\"",
    		NULL) == TCL_ERROR) {
        Tcl_DStringAppend(&(lobj->errormsg), interp->result, -1);
        return TCL_ERROR;
    }
    if((result = pqatcl_getresultbyid(interp, cd, interp->result)) == NULL) {
        Tcl_DStringAppend(&(lobj->errormsg), interp->result, -1);
        return TCL_ERROR;
    }
    if(result->status != PQA_QSTATUS_DONE) {
        Tcl_DStringAppend(&(lobj->errormsg), 
        	Tcl_DStringValue(&(result->errormsg)), -1);
        Tcl_VarEval(interp, "pqa_result ", result->handle,
        	" -clear", NULL);
        return TCL_ERROR;
    }
    if(result->num_tuples != 1) {
        Tcl_DStringAppend(&(lobj->errormsg), 
        	"lookup for OID of large object failed",
        	-1);
        Tcl_VarEval(interp, "pqa_result ", result->handle,
        	" -clear", NULL);
        return TCL_ERROR;
    }
    lobj->oid = atoi(*(result->tuple_strings));
    if(Tcl_VarEval(interp, "pqa_result ", result->handle, 
    		" -clear", NULL) == TCL_ERROR) {
        return TCL_ERROR;
    }

    /************************************************************
     * Rename the large object
     ************************************************************/
    sprintf(newname, "pqa%d", lobj->oid);
    if(Tcl_VarEval(interp, "pqa_exec ", conn->handle,
		" \"replace pg_naming (filename = \\\"",
		newname, "\\\") \
    		where pg_naming.filename = \\\"pqa_lo_dummy\\\"\"",
    		NULL) == TCL_ERROR) {
        Tcl_DStringAppend(&(lobj->errormsg), interp->result, -1);
        return TCL_ERROR;
    }
    if((result = pqatcl_getresultbyid(interp, cd, interp->result)) == NULL) {
        Tcl_DStringAppend(&(lobj->errormsg), interp->result, -1);
        return TCL_ERROR;
        return TCL_ERROR;
    }
    if(result->status != PQA_QSTATUS_DONE) {
        Tcl_DStringAppend(&(lobj->errormsg),
        	Tcl_DStringValue(&(result->errormsg)),
        	-1);
        Tcl_VarEval(interp, "pqa_result ", result->handle,
        	" -clear", NULL);
        return TCL_ERROR;
    }
    if(Tcl_VarEval(interp, "pqa_result ", result->handle, 
    		" -clear", NULL) == TCL_ERROR) {
        Tcl_DStringAppend(&(lobj->errormsg), interp->result, -1);
        return TCL_ERROR;
    }

    return TCL_OK;
}


/**********************************************************************
 * pqatcl_lobj_creat_95()	- Create a large object in pg95
 **********************************************************************/
static int pqatcl_lobj_creat_95(Tcl_Interp *interp,
		PQA_client_data *cd, PQA_large_obj *lobj, int mode)
{
    PQA_connection	*conn = lobj->conn;
    PQA_fnarg	creat_arg[1];
    int		i;
    int		shift;

    /************************************************************
     * Call create function in backend
     ************************************************************/
    creat_arg[0].len    = 4;
    creat_arg[0].isint  = 1;
    creat_arg[0].u.iarg = mode;

    if(pqatcl_lobj_sendfunc(conn, conn->lobj_fnid_creat,
    		4, 1, creat_arg) < 0) {
        conn->status = PQA_QSTATUS_LOST;
        Tcl_Close(interp, conn->backend);
        Tcl_DStringAppend(&(lobj->errormsg), PQerrormsg, -1);
        return TCL_ERROR;
    }

    while(lobj->status == PQA_QSTATUS_BUSY) {
        Tcl_DoOneEvent(TCL_ALL_EVENTS);
    }
    conn->status = PQA_QSTATUS_IDLE;
    if(lobj->status != PQA_QSTATUS_DONE) {
        return TCL_ERROR;
    }

    /************************************************************
     * Get the object ID
     ************************************************************/
    lobj->oid = 0;
    for(i = 0, shift = 0; i < 4; i++, shift += 8) {
        lobj->oid |= ((lobj->buf[i]) & 0xFF) << shift;
    }

    return TCL_OK;
}


/**********************************************************************
 * Inline function to check if we have a complete string
 **********************************************************************/
static inline int have_string(char *cp, int left)
{
    int n;

    for(n = 0; n < left; n++, cp++) {
        if(*cp == '\0') return n;
    }
    return -1;
}

#define need_nbytes_and_string(n)	if(left <= (n)) {	\
					    done++;	\
					    break;	\
					}		\
					if((i = have_string(cp + (n), left - (n))) < 0) { \
					    done++;	\
					    break;	\
					}		\
					cp += (n);	\
					left -= (n);


/**********************************************************************
 * pqatcl_lobj_getresult()	- Interpret the data from backend
 **********************************************************************/
int pqatcl_lobj_getresult(Tcl_Interp *interp, PQA_connection *conn, 
		char *buf, int len)
{
    char	*cp;
    int		left;
    int		id;
    int		i;
    int		shift;
    int		done;
    char	erridbuf[16];
    int		reslen;
    PQA_large_obj *lobj = conn->large_obj_current;

    /************************************************************
     * Gcc claimed these two might be used uninitialized. In
     * fact, they're not.
     ************************************************************/
    id = i = 0;

    /************************************************************
     * Collect the data in the temp buffer
     ************************************************************/
    if(len > 0) {
	memcpy(&(conn->recv_buf[conn->recv_have]), buf, len);
	conn->recv_have += len;
    }

    cp = conn->recv_buf;
    left = conn->recv_have;
    done = 0;
    /************************************************************
     * Process the messages as long as there is enough
     * data and we are happy with it
     ************************************************************/
    while(left > 0 && !done) {

	/************************************************************
	 * Get the next ID 
	 ************************************************************/
	id = *cp;

	/************************************************************
	 * Now process the message according to it's ID
	 ************************************************************/
	switch(id) {
	case 'R':	/* Collect remarks and notices	*/
	case 'N':
	    if(conn->backend_version == PQA_DBVERSION_4_2) {
		need_nbytes_and_string(5);
	    } else
	    if(conn->backend_version == PQA_DBVERSION_95) {
		need_nbytes_and_string(1);
	    }
	    Tcl_DStringAppend(&(lobj->errormsg), cp, -1);
	    cp += i + 1;
	    left -= i + 1;
	    break;

	case 'V':	/* The function ID reply	*/
	    if(conn->backend_version == PQA_DBVERSION_4_2) {
		cp += 5;
		left -= 5;
	    } else
	    if(conn->backend_version == PQA_DBVERSION_95) {
		cp++;
		left--;
	    }
	    break;

	case '0':	/* Function without return val	*/
	    cp++;
	    left--;
	    lobj->status = PQA_QSTATUS_DONE;
	    done++;
	    break;

	case 'G':	/* Function sends return value	*/
	    if(left < 5) {
	        done++;
	        break;
	    }
	    reslen = 0;
	    for(i = 1, shift = 0; i < 5; i++, shift += 8) {
	        reslen |= (cp[i] & 0xFF) << shift;
	    }
	    if(reslen == PQA_VAR_LENGTH_RES_4_2) {
	        need_nbytes_and_string(5);
	        strcpy(lobj->buf, cp);
	        lobj->inbuf = strlen(cp);
	        cp += i + 1;
	        left -= i + 1;
	    } else {
	        if(left < (reslen + 5)) {
	            done++;
	            break;
	        }
	        memcpy(lobj->buf, &cp[5], reslen);
	        lobj->inbuf = reslen;
	        cp += reslen + 5;
	        left -= reslen + 5;
	    }
	    break;

	case 'E':	/* Severe error from backend	*/
	    if(conn->backend_version == PQA_DBVERSION_4_2) {
		need_nbytes_and_string(5);
	    } else
	    if(conn->backend_version == PQA_DBVERSION_95) {
		need_nbytes_and_string(1);
	    }
	    lobj->status = PQA_QSTATUS_ERROR;
	    Tcl_DStringAppend(&(lobj->errormsg), cp, -1);
	    cp += i + 1;
	    left -= i + 1;
	    break;

	case 'A':	/* Asynchronous portal notification */
	    if(conn->backend_version == PQA_DBVERSION_4_2) {
		need_nbytes_and_string(5);
	    } else
	    if(conn->backend_version == PQA_DBVERSION_95) {
		need_nbytes_and_string(1);
	    }
	    conn->notify_pending++;
	    Tcl_DStringAppendElement(&(conn->notify_list), cp);
	    cp += i + 1;
	    left -= i + 1;
	    break;

	default:	/* Unsupported		*/
	    lobj->status = PQA_QSTATUS_FATAL;
	    Tcl_DStringAppend(&(lobj->errormsg),
		"Fatal error: Backend protocol error - got ID ", -1);
	    sprintf(erridbuf, "'0x%X'\n", id);
	    Tcl_DStringAppend(&(lobj->errormsg), erridbuf, -1);
	    conn->status = PQA_QSTATUS_LOST;
	    Tcl_Close(interp, conn->backend);
	    return 1;
	}
    }

    /************************************************************
     * Remove the complete interpreted messages from
     * our internal buffer
     ************************************************************/
    if(left > 0) {
        memcpy(conn->recv_buf, cp, left);
        conn->recv_have = left;
    } else {
        conn->recv_have = 0;
    }

    /************************************************************
     * In any case get back
     ************************************************************/
    return 1;
}


/**********************************************************************
 * pqatcl_lobj_sendfunc()	- Send a function call
 **********************************************************************/
int pqatcl_lobj_sendfunc(PQA_connection *conn,
	int fnid, int reslen, int argc, PQA_fnarg *argv)
{
    char	buf[8192];
    char	*cp;
    int		n;
    int		i;
    int		shift;

    cp = buf;

    /************************************************************
     * Create the function message header
     *    F
     *    xactid	- V4.2 only
     *    fnid
     *    result-length
     *    nargs
     ************************************************************/
    *cp++ = 'F';
    if(conn->backend_version == PQA_DBVERSION_95) {
        *cp++ = ' ';
        *cp++ = '\0';
    }
    if(conn->backend_version == PQA_DBVERSION_4_2) {
	for(i = 0, shift = 0; i < 4; i++, shift += 8) {
	    *cp++ = ((conn->xactid) >> shift) & 0xFF;
	}
    }
    for(i = 0, shift = 0; i < 4; i++, shift += 8) {
        *cp++ = (fnid >> shift) & 0xFF;
    }
    if(conn->backend_version == PQA_DBVERSION_4_2) {
	for(i = 0, shift = 0; i < 4; i++, shift += 8) {
	    *cp++ = (reslen >> shift) & 0xFF;
	}
    }
    for(i = 0, shift = 0; i < 4; i++, shift += 8) {
        *cp++ = (argc >> shift) & 0xFF;
    }

    /************************************************************
     * Append the arguments
     ************************************************************/
    for(n = 0; n < argc; n++) {
	for(i = 0, shift = 0; i < 4; i++, shift += 8) {
	    *cp++ = ((argv[n].len) >> shift) & 0xFF;
	}
	if(argv[n].isint) {
	    for(i = 0, shift = 0; i < 4; i++, shift += 8) {
		*cp++ = ((argv[n].u.iarg) >> shift) & 0xFF;
	    }
	} else {
	    if(argv[n].len == PQA_VAR_LENGTH_ARG_4_2) {
	        strcpy(cp, argv[n].u.parg);
	        cp += strlen(cp);
	        cp++;
	    } else {
	        memcpy(cp, argv[n].u.parg, argv[n].len);
	        cp += argv[n].len;
	    }
	}
    }

    /************************************************************
     * Send the message
     ************************************************************/
    if(Tcl_Write(conn->backend, buf, cp - buf) < 0) {
        strcpy(PQerrormsg, "pqatcl_lobj_sendfunc(): Tcl_Write() failed");
        return -1;
    }
    if(Tcl_Flush(conn->backend) < 0) {
        strcpy(PQerrormsg, "pqatcl_lobj_sendfunc(): Tcl_Flush() failed");
        return -1;
    }

    return 0;
}


