#---------------------------------------------------------------------
# Sqlpqa		- Package for 'embedded' SQL inside Tcl
#			  with the Pqatcl Postgres95 interface
#
# $Source: /home/wieck/src/tcltk/extensions/pqatcl/libpqa/RCS/sqlpqa.tcl,v $
# $Revision: 0.2 $ $State: Exp $
# $Date: 1996/10/09 06:08:59 $
# $Author: wieck $
#
# HISTORY
# $Log: sqlpqa.tcl,v $
# Revision 0.2  1996/10/09 06:08:59  wieck
# Set variable SQLnumtuples in the interpreter level
# of the original SQL statement before executing the
# command loop.
#
# Revision 0.1  1996/10/01 06:50:29  wieck
# Initial revision.
#
# Revision 0.1  1996/06/20 17:57:11  wieck
#   Initial revision
#
#---------------------------------------------------------------------


#---------------------------------------------------------------------
# SQLconnect		- Connect to a database
#---------------------------------------------------------------------
proc SQLconnect {args} {
    #-----------------------------------------------------------
    # This package requires Pqatcl
    #-----------------------------------------------------------
    package require Pqatcl

    #-----------------------------------------------------------
    # Establish the real database connection
    #-----------------------------------------------------------
    set dbh [eval pqa_connect $args]

    #-----------------------------------------------------------
    # Create the Tcl-command for this connection
    #-----------------------------------------------------------
    set wrap _SQL_cmdwrap_$dbh
    proc $wrap {cmd args} "return \[_SQL_command $dbh \$cmd \$args\]"

    #-----------------------------------------------------------
    # Return the command name
    #-----------------------------------------------------------
    return $wrap
}


#---------------------------------------------------------------------
# SQLdisconnect		- Disconnect from a database
#---------------------------------------------------------------------
proc SQLdisconnect {wrap} {
    #-----------------------------------------------------------
    # Check if this connection is valid
    #-----------------------------------------------------------
    if {[info procs $wrap] == ""} {
        return -code error "invalide SQL db handle '$wrap'"
    }

    #-----------------------------------------------------------
    # Extract the real db connection handle, remove the
    # Tcl-command for this connection and close
    #-----------------------------------------------------------
    regsub "^_SQL_cmdwrap_" $wrap "" dbh
    rename $wrap {}
    return [pqa_close $dbh]
}


#---------------------------------------------------------------------
# SQLquote		- Quote literals properly
#---------------------------------------------------------------------
proc SQLquote {val} {
    regsub -all {\\} $val {\\\\} val
    regsub -all {'}  $val {\\'} val
    regsub -all "\n" $val {\\n} val
    regsub -all "\t" $val {\\t} val
    return $val
}


#---------------------------------------------------------------------
# _SQL_command		- Internal wrapper for SQL execution
#---------------------------------------------------------------------
proc _SQL_command {dbh cmd query} {
    #-----------------------------------------------------------
    # Check if there's a special handling for this
    # SQL statement
    #-----------------------------------------------------------
    set xcmd _SQL_cmd_$cmd
    if {[info procs $xcmd] == ""} {
	#-----------------------------------------------------------
	# No special handling - execute directly
	#-----------------------------------------------------------
	set query [join $query]
	regsub -all "\n" $query " " query
	set res [pqa_exec $dbh "$cmd $query"]
	if {[pqa_result $res -status] != "ok"} {
	    return -code error [pqa_result $res -errormsg -clear]
	}
	return [pqa_result $res -cmdreturn -clear]
    }

    #-----------------------------------------------------------
    # Call the special handling procedure
    #-----------------------------------------------------------
    return [$xcmd $dbh $query]
}


#---------------------------------------------------------------------
# _SQL_cmd_select	- Special handling of select statement
#---------------------------------------------------------------------
proc _SQL_cmd_select {dbh query} {
    #-----------------------------------------------------------
    # Check for 'single' keyword and build the real query
    #-----------------------------------------------------------
    if {[lindex $query 0] == "single"} {
        set single 1
	set query [join [lrange $query 1 end]]
    } else {
        set single 0
	set cmd [lindex $query end]
	set query [join [lreplace $query end end]]
    }
    regsub -all "\n" $query " " query

    #-----------------------------------------------------------
    # Execute the query
    #-----------------------------------------------------------
    set res [pqa_exec $dbh "select $query"]
    if {[pqa_result $res -status] != "ok"} {
        return -code error [pqa_result $res -errormsg -clear]
    }

    #-----------------------------------------------------------
    # Create the attibute list from the result buffer.
    # Upvar the attributes up the the interpreter level
    # where the original 'select' came from and initialize
    # the variables to an empty string.
    #-----------------------------------------------------------
    set attrlist {}
    foreach attrd [pqa_result $res -attributes] {
        set attr [lindex $attrd 0]
	lappend attrlist $attr
	upvar 3 $attr var_$attr
	set var_$attr {}
    }

    #-----------------------------------------------------------
    # Set the number of tuples now so the loop can access it.
    #-----------------------------------------------------------
    upvar 3 SQLnumtuples ntuples
    set ntuples [pqa_result $res -numtuples]

    if {$single} {
	#-----------------------------------------------------------
	# On 'select single ...' set the attribute variables
	# to the values of the first tuple (if we got anything)
	#-----------------------------------------------------------
        if {[pqa_result $res -numtuples] > 0} {
	    set data [pqa_result $res -gettuple 0]
	} else {
	    set data {}
	}
	set i 0
	foreach attr $attrlist {
	    set var_$attr [lindex $data $i]
	    incr i
	}
    } else {
	#-----------------------------------------------------------
	# On normal 'select ...' set the attribute variables
	# inside a loop over all tuples and evaluate the command.
	#-----------------------------------------------------------
        if {[catch {pqa_loop $res data {
		set i 0
		foreach attr $attrlist {
		    set var_$attr [lindex $data $i]
		    incr i
		}
		uplevel 3 $cmd
	      }} msg]} {
	    pqa_result $res -clear
	    return -code error $msg
	} 
    }

    #-----------------------------------------------------------
    # Clear the result buffer and return the number of
    # tuples we got for this query.
    #-----------------------------------------------------------
    return [set ntuples [pqa_result $res -numtuples -clear]]
}


#---------------------------------------------------------------------
# _SQL_cmd_fetch	- Special handling of fetch statement
#---------------------------------------------------------------------
proc _SQL_cmd_fetch {dbh query} {
    #-----------------------------------------------------------
    # Check for special keyword 'single' and build the
    # real query.
    #-----------------------------------------------------------
    set how_much [lindex $query 0]
    set how_idx 0
    if {$how_much == "forward" || $how_much == "backward"} {
        set how_much [lindex $query 1]
	set how_idx 1
    }
    if {$how_much == "single"} {
        set single 1
	set query [join [lreplace $query $how_idx $how_idx "1"]]
    } else {
        set single 0
	set cmd [lindex $query end]
	set query [join [lreplace $query end end]]
    }
    regsub -all "\n" $query " " query

    #-----------------------------------------------------------
    # Execute the query
    #-----------------------------------------------------------
    set res [pqa_exec $dbh "fetch $query"]
    if {[pqa_result $res -status] != "ok"} {
        return -code error [pqa_result $res -errormsg -clear]
    }

    #-----------------------------------------------------------
    # Create the attibute list from the result buffer.
    # Upvar the attributes up the the interpreter level
    # where the original 'select' came from and initialize
    # the variables to an empty string.
    #-----------------------------------------------------------
    set attrlist {}
    foreach attrd [pqa_result $res -attributes] {
        set attr [lindex $attrd 0]
	lappend attrlist $attr
	upvar 3 $attr var_$attr
	set var_$attr {}
    }

    #-----------------------------------------------------------
    # Set the number of tuples now so the loop can access it.
    #-----------------------------------------------------------
    upvar 3 SQLnumtuples ntuples
    set ntuples [pqa_result $res -numtuples]

    if {$single} {
	#-----------------------------------------------------------
	# On 'select single ...' set the attribute variables
	# to the values of the first tuple (if we got anything)
	#-----------------------------------------------------------
        if {[pqa_result $res -numtuples] > 0} {
	    set data [pqa_result $res -gettuple 0]
	} else {
	    set data {}
	}
	set i 0
	foreach attr $attrlist {
	    set var_$attr [lindex $data $i]
	    incr i
	}
    } else {
	#-----------------------------------------------------------
	# On normal 'select ...' set the attribute variables
	# inside a loop over all tuples and evaluate the command.
	#-----------------------------------------------------------
        if {[catch {pqa_loop $res data {
		set i 0
		foreach attr $attrlist {
		    set var_$attr [lindex $data $i]
		    incr i
		}
		uplevel 3 $cmd
	      }} msg]} {
	    pqa_result $res -clear
	    return -code error $msg
	} 
    }

    #-----------------------------------------------------------
    # Clear the result buffer and return the number of
    # tuples we got for this query.
    #-----------------------------------------------------------
    return [set ntuples [pqa_result $res -numtuples -clear]]
}


#---------------------------------------------------------------------
# _SQL_cmd_describe	- Special handling of describe statement
#---------------------------------------------------------------------
proc _SQL_cmd_describe {dbh argv} {
    set what [lindex $argv 0]
    switch -- $what {
    table {
	    if {[llength $argv] != 2} {
	        return -code error "syntax error - describe table <name>"
	    }
	    set table [SQLquote [lindex $argv 1]]
	    set result {}
	    _SQL_cmdwrap_$dbh select				\
		    A.attname,					\
		    A.attnum,					\
		    T.typname,					\
		    T.typlen					\
		from pg_class C, pg_attribute A, pg_type T	\
		where C.relname = "'$table'"			\
		  and A.attrelid = C.oid			\
		  and A.attnum   > 0				\
		  and A.atttypid = T.oid			\
		order by attnum					\
	    {   if {$typlen < 0} {
		    set typlen "var"
	        }
	        lappend result [list $attname $typname $typlen]
	    }
	    return $result
	}
    default {
	    return -code error "don't know how to describe '$what'"
    	}
    }
}


#---------------------------------------------------------------------
# _SQL_cmd_create	- Special handling of create statement
#			  This implements the 'create large object'
#			  pseudo SQL syntax.
#---------------------------------------------------------------------
proc _SQL_cmd_create {dbh query} {
    #-----------------------------------------------------------
    # Check for large object operation
    #-----------------------------------------------------------
    if {[regexp "^\[ \t\n\]*large\[ \t\n+\]object" $query]} {
        if {[llength $query] < 4 || [lindex $query 2] != "mode"} {
	    return -code error \
	        {syntax error - 'create large object mode <mode>'}
	}
        set mode ""
	foreach m [lrange $query 3 end] {
	    switch -- $m {
		read	-
	        r	{ lappend mode "INV_READ"
			}
	        write	-
		w	{ lappend mode "INV_WRITE"
			}
		readwrite -
		rw	{ lappend mode "INV_READ" "INV_WRITE"
	    		}
	        archive	{ lappend mode "INV_ARCHIVE"
			}
	        default	{ return -code error \
      {invalid mode - must be a combination of 'read', 'write' and 'archive'}
			}
	    }
	}
	return [pqa_lo_creat $dbh $mode]
    }

    #-----------------------------------------------------------
    # No special handling - execute directly
    #-----------------------------------------------------------
    set query [join $query]
    regsub -all "\n" $query " " query
    set res [pqa_exec $dbh "create $query"]
    if {[pqa_result $res -status] != "ok"} {
	return -code error [pqa_result $res -errormsg -clear]
    }
    return [pqa_result $res -cmdreturn -clear]
}


#---------------------------------------------------------------------
# _SQL_cmd_open		- Special handling of open statement
#			  This implements the 'open large object'
#			  pseudo SQL syntax.
#---------------------------------------------------------------------
proc _SQL_cmd_open {dbh query} {
    #-----------------------------------------------------------
    # Check for large object operation
    #-----------------------------------------------------------
    if {[regexp "^\[ \t\n\]*large\[ \t\n+\]object" $query]} {
        if {[llength $query] < 5 || [lindex $query 3] != "mode"} {
	    return -code error \
	        {syntax error - 'open large object <OID> mode <mode>'}
	}
	set oid [lindex $query 2]
        set mode ""
	foreach m [lrange $query 4 end] {
	    switch -- $m {
		read	-
	        r	{ lappend mode "INV_READ"
			}
	        write	-
		w	{ lappend mode "INV_WRITE"
			}
		readwrite -
		rw	{ lappend mode "INV_READ" "INV_WRITE"
	    		}
	        archive	{ lappend mode "INV_ARCHIVE"
			}
	        default	{ return -code error \
      {invalid mode - must be a combination of 'read', 'write' and 'archive'}
			}
	    }
	}
	return [pqa_lo_open $dbh $oid $mode]
    }

    #-----------------------------------------------------------
    # No special handling - execute directly
    #-----------------------------------------------------------
    set query [join $query]
    regsub -all "\n" $query " " query
    set res [pqa_exec $dbh "open $query"]
    if {[pqa_result $res -status] != "ok"} {
	return -code error [pqa_result $res -errormsg -clear]
    }
    return [pqa_result $res -cmdreturn -clear]
}


#---------------------------------------------------------------------
# _SQL_cmd_drop		- Special handling of drop statement
#			  This implements the 'drop large object'
#			  pseudo SQL syntax.
#---------------------------------------------------------------------
proc _SQL_cmd_drop {dbh query} {
    #-----------------------------------------------------------
    # Check for large object operation
    #-----------------------------------------------------------
    if {[regexp "^\[ \t\n\]*large\[ \t\n+\]object" $query]} {
        if {[llength $query] != 3} {
	    return -code error \
	        {syntax error - 'drop large object <OID>'}
	}
	set oid [lindex $query 2]
	return [pqa_lo_unlink $dbh $oid]
    }

    #-----------------------------------------------------------
    # No special handling - execute directly
    #-----------------------------------------------------------
    set query [join $query]
    regsub -all "\n" $query " " query
    set res [pqa_exec $dbh "drop $query"]
    if {[pqa_result $res -status] != "ok"} {
	return -code error [pqa_result $res -errormsg -clear]
    }
    return [pqa_result $res -cmdreturn -clear]
}


#---------------------------------------------------------------------
# _SQL_cmd_info		- Implementation of info command
#---------------------------------------------------------------------
proc _SQL_cmd_info {dbh query} {
    set what [lindex $query 0]
    switch -- $what {
	dbhandle {
		  return $dbh
		}
        default { return -code error \
	  "unknown info '$what' - must be 'dbhandle'"
		}
    }
}


#---------------------------------------------------------------------
# For pkg_mkIndex
#---------------------------------------------------------------------
package provide Sqlpqa 1.0


# END OF PACKAGE Sqlpqa
