//go:build !windows
// +build !windows

/*
   Copyright The containerd Authors.

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*/

package fs

import (
	"errors"
	"os"
	"path/filepath"
	"testing"

	"github.com/containerd/continuity/fs/fstest"
)

type RootCheck struct {
	unresolved string
	expected   string
	scope      func(string) string
	cause      error
}

func TestRootPath(t *testing.T) {
	tests := []struct {
		name   string
		apply  fstest.Applier
		checks []RootCheck
	}{
		{
			name:   "SymlinkAbsolute",
			apply:  Symlink("/b", "fs/a/d"),
			checks: Check("fs/a/d/c/data", "b/c/data"),
		},
		{
			name:   "SymlinkRelativePath",
			apply:  Symlink("a", "fs/i"),
			checks: Check("fs/i", "fs/a"),
		},
		{
			name:   "SymlinkSkipSymlinksOutsideScope",
			apply:  Symlink("realdir", "linkdir"),
			checks: CheckWithScope("foo/bar", "foo/bar", "linkdir"),
		},
		{
			name:   "SymlinkLastLink",
			apply:  Symlink("/b", "fs/a/d"),
			checks: Check("fs/a/d", "b"),
		},
		{
			name:  "SymlinkRelativeLinkChangeScope",
			apply: Symlink("../b", "fs/a/e"),
			checks: CheckAll(
				Check("fs/a/e/c/data", "fs/b/c/data"),
				CheckWithScope("e", "b", "fs/a"), // Original return
			),
		},
		{
			name:  "SymlinkDeepRelativeLinkChangeScope",
			apply: Symlink("../../../../test", "fs/a/f"),
			checks: CheckAll(
				Check("fs/a/f", "test"),             // Original return
				CheckWithScope("a/f", "test", "fs"), // Original return
			),
		},
		{
			name: "SymlinkRelativeLinkChain",
			apply: fstest.Apply(
				Symlink("../g", "fs/b/h"),
				fstest.Symlink("../../../../../../../../../../../../root", "fs/g"),
			),
			checks: Check("fs/b/h", "root"),
		},
		{
			name:   "SymlinkBreakoutPath",
			apply:  Symlink("../i/a", "fs/j/k"),
			checks: CheckWithScope("k", "i/a", "fs/j"),
		},
		{
			name:   "SymlinkToRoot",
			apply:  Symlink("/", "foo"),
			checks: Check("foo", ""),
		},
		{
			name:   "SymlinkSlashDotdot",
			apply:  Symlink("/../../", "foo"),
			checks: Check("foo", ""),
		},
		{
			name:   "SymlinkDotdot",
			apply:  Symlink("../../", "foo"),
			checks: Check("foo", ""),
		},
		{
			name:   "SymlinkRelativePath2",
			apply:  Symlink("baz/target", "bar/foo"),
			checks: Check("bar/foo", "bar/baz/target"),
		},
		{
			name: "SymlinkScopeLink",
			apply: fstest.Apply(
				Symlink("root2", "root"),
				Symlink("../bar", "root2/foo"),
			),
			checks: CheckWithScope("foo", "bar", "root"),
		},
		{
			name: "SymlinkSelf",
			apply: fstest.Apply(
				Symlink("foo", "root/foo"),
			),
			checks: ErrorWithScope("foo", "root", errTooManyLinks),
		},
		{
			name: "SymlinkCircular",
			apply: fstest.Apply(
				Symlink("foo", "bar"),
				Symlink("bar", "foo"),
			),
			checks: ErrorWithScope("foo", "", errTooManyLinks), //TODO: Test for circular error
		},
		{
			name: "SymlinkCircularUnderRoot",
			apply: fstest.Apply(
				Symlink("baz", "root/bar"),
				Symlink("../bak", "root/baz"),
				Symlink("/bar", "root/bak"),
			),
			checks: ErrorWithScope("bar", "root", errTooManyLinks), // TODO: Test for circular error
		},
		{
			name: "SymlinkComplexChain",
			apply: fstest.Apply(
				fstest.CreateDir("root2", 0777),
				Symlink("root2", "root"),
				Symlink("r/s", "root/a"),
				Symlink("../root/t", "root/r"),
				Symlink("/../u", "root/root/t/s/b"),
				Symlink(".", "root/u/c"),
				Symlink("../v", "root/u/x/y"),
				Symlink("/../w", "root/u/v"),
			),
			checks: CheckWithScope("a/b/c/x/y/z", "w/z", "root"), // Original return
		},
		{
			name: "SymlinkBreakoutNonExistent",
			apply: fstest.Apply(
				Symlink("/", "root/slash"),
				Symlink("/idontexist/../slash", "root/sym"),
			),
			checks: CheckWithScope("sym/file", "file", "root"),
		},
		{
			name: "SymlinkNoLexicalCleaning",
			apply: fstest.Apply(
				Symlink("/foo/bar", "root/sym"),
				Symlink("/sym/../baz", "root/hello"),
			),
			checks: CheckWithScope("hello", "foo/baz", "root"),
		},
	}

	for _, test := range tests {
		t.Run(test.name, makeRootPathTest(t, test.apply, test.checks))
	}

	// Add related tests which are unable to follow same pattern
	t.Run("SymlinkRootScope", testRootPathSymlinkRootScope)
	t.Run("SymlinkEmpty", testRootPathSymlinkEmpty)
}

func TestDirectoryCompare(t *testing.T) {
	for i, tc := range []struct {
		p1 string
		p2 string
		r  int
	}{
		{"", "", 0},
		{"", "/", -1},
		{"/", "", 1},
		{"/", "/", 0},
		{"", "", 0},
		{"/dir1", "/dir1/", -1},
		{"/dir1", "/dir1", 0},
		{"/dir1/", "/dir1", 1},
		{"/dir1", "/dir2", -1},
		{"/dir2", "/dir1", 1},
		{"/dir1/1", "/dir1-1", -1},
		{"/dir1-1", "/dir1/1", 1},
		{"/dir1/dir2", "/dir1/dir2", 0},
		{"/dir1/dir2", "/dir1/dir2/", -1},
		{"/dir1/dir2", "/dir1/dir2/f1", -1},
		{"/dir1/dir2/", "/dir1/dir2", 1},
		{"/dir1/dir2/f1", "/dir1/dir2", 1},
		{"/dir1/dir2-f1", "/dir1/dir2", 1},
		{"/dir1/dir2-f1", "/dir1/dir2/", 1},
		{"/dir1/dir2/", "/dir1/dir2/", 0},
		{"/dir1/dir2你", "/dir1/dir2a", 1},
		{"/dir1/dir2你", "/dir1/dir2", 1},
		{"/dir1/dir2你", "/dir1/dir2/", 1},
		{"/dir1/dir2你/", "/dir1/dir2/", 1},
		{"/dir1/dir2你/", "/dir1/dir2你/", 0},
		{"/dir1/dir2你/", "/dir1/dir2你好/", -1},
		{"/dir1/dir2你/", "/dir1/dir2你-好/", -1},
		{"/dir1/dir2你/", "/dir1/dir2好/", -1},
		{"/dir1/dir2/f1", "/dir1/dir2/f1", 0},
		{"/d1/d2/d3/d4/d5/d6/d7/d8/d9/d10", "/d1/d2/d3/d4/d5/d6/d7/d8/d9/d10", 0},
		{"/d1/d2/d3/d4/d5/d6/d7/d8/d9/d10", "/d1/d2/d3/d4/d5/d6/d7/d8/d9/d11", -1},
	} {
		r := directoryCompare(tc.p1, tc.p2)
		if r != tc.r {
			t.Errorf("[%d] Test case failed, %q <> %q = %d, expected %d", i, tc.p1, tc.p2, r, tc.r)
		}
	}
}

func testRootPathSymlinkRootScope(t *testing.T) {
	tmpdir := t.TempDir()
	expected, err := filepath.EvalSymlinks(tmpdir)
	if err != nil {
		t.Fatal(err)
	}
	rewrite, err := RootPath("/", tmpdir)
	if err != nil {
		t.Fatal(err)
	}
	if rewrite != expected {
		t.Fatalf("expected %q got %q", expected, rewrite)
	}
}

func testRootPathSymlinkEmpty(t *testing.T) {
	wd, err := os.Getwd()
	if err != nil {
		t.Fatal(err)
	}
	res, err := RootPath(wd, "")
	if err != nil {
		t.Fatal(err)
	}
	if res != wd {
		t.Fatalf("expected %q got %q", wd, res)
	}
}

func makeRootPathTest(t *testing.T, apply fstest.Applier, checks []RootCheck) func(t *testing.T) {
	return func(t *testing.T) {
		applyDir := t.TempDir()
		if apply != nil {
			if err := apply.Apply(applyDir); err != nil {
				t.Fatalf("Apply failed: %+v", err)
			}
		}

		for i, check := range checks {
			root := applyDir
			if check.scope != nil {
				root = check.scope(root)
			}

			actual, err := RootPath(root, check.unresolved)
			if check.cause != nil {
				if err == nil {
					t.Errorf("(Check %d) Expected error %q, %q evaluated as %q", i+1, check.cause.Error(), check.unresolved, actual)
				}
				if !errors.Is(err, check.cause) {
					t.Fatalf("(Check %d) Failed to evaluate root path: %+v", i+1, err)
				}
			} else {
				expected := filepath.Join(root, check.expected)
				if err != nil {
					t.Fatalf("(Check %d) Failed to evaluate root path: %+v", i+1, err)
				}
				if actual != expected {
					t.Errorf("(Check %d) Unexpected evaluated path %q, expected %q", i+1, actual, expected)
				}
			}
		}
	}
}

func Check(unresolved, expected string) []RootCheck {
	return []RootCheck{
		{
			unresolved: unresolved,
			expected:   expected,
		},
	}
}

func CheckWithScope(unresolved, expected, scope string) []RootCheck {
	return []RootCheck{
		{
			unresolved: unresolved,
			expected:   expected,
			scope: func(root string) string {
				return filepath.Join(root, scope)
			},
		},
	}
}

func ErrorWithScope(unresolved, scope string, cause error) []RootCheck {
	return []RootCheck{
		{
			unresolved: unresolved,
			cause:      cause,
			scope: func(root string) string {
				return filepath.Join(root, scope)
			},
		},
	}
}

func CheckAll(checks ...[]RootCheck) []RootCheck {
	all := make([]RootCheck, 0, len(checks))
	for _, c := range checks {
		all = append(all, c...)
	}
	return all
}

func Symlink(oldname, newname string) fstest.Applier {
	dir := filepath.Dir(newname)
	if dir != "" {
		return fstest.Apply(
			fstest.CreateDir(dir, 0755),
			fstest.Symlink(oldname, newname),
		)
	}
	return fstest.Symlink(oldname, newname)
}
