/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var isnan = require( '@stdlib/math/base/assert/is-nan' );
var gamma = require( '@stdlib/math/base/special/gamma' );
var PINF = require( '@stdlib/constants/float64/pinf' );


// MAIN //

/**
* Returns the expected value for a Fréchet distribution with shape `alpha`, scale `s`, and location `m`.
*
* @param {PositiveNumber} alpha - shape parameter
* @param {PositiveNumber} s - scale parameter
* @param {number} m - location parameter
* @returns {number} mean
*
* @example
* var y = mean( 5.0, 2.0, 0.0 );
* // returns ~2.328
*
* @example
* var y = mean( 5.0, 2.0, -4.0 );
* // returns ~-1.672
*
* @example
* var y = mean( 1.0, 1.0, 0.0 );
* // returns Infinity
*
* @example
* var y = mean( NaN, 1.0, 0.0 );
* // returns NaN
*
* @example
* var y = mean( 1.0, NaN, 0.0 );
* // returns NaN
*
* @example
* var y = mean( 1.0, 1.0, NaN );
* // returns NaN
*/
function mean( alpha, s, m ) {
	if (
		isnan( alpha ) ||
		isnan( s ) ||
		isnan( m ) ||
		alpha <= 0.0 ||
		s <= 0.0
	) {
		return NaN;
	}
	if ( alpha <= 1.0 ) {
		// Case: 0 < alpha <= 1
		return PINF;
	}
	return m + ( s * gamma( 1.0 - (1.0/alpha) ) );
}


// EXPORTS //

module.exports = mean;
