import * as YAML from 'yaml'

test('basic', () => {
  const src = `- &a 1\n- *a\n`
  const doc = YAML.parseDocument(src)
  expect(doc.errors).toHaveLength(0)
  expect(doc.contents.items).toMatchObject([
    { anchor: 'a', value: 1 },
    { source: 'a' }
  ])
  expect(String(doc)).toBe(src)
})

test('re-defined anchor', () => {
  const src = '- &a 1\n- &a 2\n- *a\n'
  const doc = YAML.parseDocument(src)
  expect(doc.errors).toHaveLength(0)
  expect(doc.contents.items).toMatchObject([
    { anchor: 'a', value: 1 },
    { anchor: 'a', value: 2 },
    { source: 'a' }
  ])
  expect(doc.toJS()).toMatchObject([1, 2, 2])
  expect(String(doc)).toBe('- &a 1\n- &a 2\n- *a\n')
})

test('circular reference', () => {
  const src = '&a [ 1, *a ]\n'
  const doc = YAML.parseDocument(src)
  expect(doc.errors).toHaveLength(0)
  expect(doc.warnings).toHaveLength(0)
  expect(doc.contents).toMatchObject({
    anchor: 'a',
    items: [{ value: 1 }, { source: 'a' }]
  })
  const res = doc.toJS()
  expect(res[1]).toBe(res)
  expect(String(doc)).toBe(src)
})

describe('anchor on tagged collection', () => {
  test('!!set', () => {
    const res = YAML.parse('- &a !!set { 1, 2 }\n- *a\n')
    expect(Array.from(res[0])).toMatchObject([1, 2])
    expect(res[1]).toBe(res[0])
  })

  test('!!omap', () => {
    const res = YAML.parse('- &a !!omap [ 1: 1, 2: 2 ]\n- *a\n')
    expect(Array.from(res[0])).toMatchObject([
      [1, 1],
      [2, 2]
    ])
    expect(res[1]).toBe(res[0])
  })
})

describe('create', () => {
  test('node.anchor', () => {
    const doc = YAML.parseDocument('[{ a: A }, { b: B }]')
    doc.get(0).anchor = 'AA'
    doc.getIn([0, 'a'], true).anchor = 'a'
    doc.getIn([0, 'a'], true).anchor = ''
    doc.getIn([1, 'b'], true).anchor = 'BB'
    expect(String(doc)).toBe('[ &AA { a: A }, { b: &BB B } ]\n')
  })

  test('doc.createAlias', () => {
    const doc = YAML.parseDocument('[{ a: A }, { b: B }]')
    const alias = doc.createAlias(doc.get(0), 'AA')
    doc.contents.items.push(alias)
    expect(doc.toJS()).toMatchObject([{ a: 'A' }, { b: 'B' }, { a: 'A' }])
    const alias2 = doc.createAlias(doc.get(1), 'AA')
    expect(doc.get(1).anchor).toBe('AA1')
    expect(alias2.source).toBe('AA1')
    expect(String(doc)).toMatch('[ &AA { a: A }, &AA1 { b: B }, *AA ]\n')
  })

  test('repeated Array in createNode', () => {
    const doc = new YAML.Document()
    const ref = []
    const node = doc.createNode({ src: ref, ref })
    expect(node).toMatchObject({
      items: [
        { key: { value: 'src' }, value: { items: [], anchor: 'a1' } },
        { key: { value: 'ref' }, value: { source: 'a1' } }
      ]
    })
  })

  test('repeated Date in createNode', () => {
    const doc = new YAML.Document()
    const date = new Date()
    const value = date.toJSON()
    const node = doc.createNode({ src: date, ref: date })
    expect(node).toMatchObject({
      items: [
        { key: { value: 'src' }, value: { value, anchor: 'a1' } },
        { key: { value: 'ref' }, value: { source: 'a1' } }
      ]
    })
  })
})

describe('errors', () => {
  test('invalid anchor characters', () => {
    const doc = YAML.parseDocument('[{ a: A }, { b: B }]')
    doc.get(0).anchor = 'A A'
    expect(() => String(doc)).toThrow(
      'Anchor must not contain whitespace or control characters: "A A"'
    )
  })

  test('set tag on alias', () => {
    const doc = YAML.parseDocument('[{ a: A }, { b: B }]')
    const node = doc.contents.items[0]
    const alias = doc.createAlias(node, 'AA')
    expect(() => {
      alias.tag = 'tag:yaml.org,2002:alias'
    }).toThrow('Alias nodes cannot have tags')
  })

  test('alias before anchor', () => {
    const doc = YAML.parseDocument('[{ a: A }, { b: B }]')
    const alias = doc.createAlias(doc.get(0), 'AA')
    doc.contents.items.unshift(alias)
    expect(() => String(doc)).toThrow(
      'Unresolved alias (the anchor must be set before the alias): AA'
    )
  })

  test('empty alias', () => {
    const doc = YAML.parseDocument('- *\n')
    expect(doc.errors).toMatchObject([{ code: 'BAD_ALIAS' }])
  })

  test('empty anchor on value', () => {
    const doc = YAML.parseDocument('- & foo\n')
    expect(doc.errors).toMatchObject([{ code: 'BAD_ALIAS' }])
  })

  test('empty anchor at end', () => {
    const doc = YAML.parseDocument('- &\n')
    expect(doc.errors).toMatchObject([{ code: 'BAD_ALIAS' }])
  })
})

describe('warnings', () => {
  test('anchor ending in colon', () => {
    const doc = YAML.parseDocument('- &x:\n')
    expect(doc.errors).toHaveLength(0)
    expect(doc.warnings).toMatchObject([{ code: 'BAD_ALIAS' }])
  })

  test('alias ending in colon', () => {
    const doc = YAML.parseDocument('- *x:\n')
    expect(doc.errors).toHaveLength(0)
    expect(doc.warnings).toMatchObject([{ code: 'BAD_ALIAS' }])
  })
})

describe('__proto__ as anchor name', () => {
  test('parse', () => {
    const src = `- &__proto__ 1\n- *__proto__\n`
    const doc = YAML.parseDocument(src)
    expect(doc.errors).toHaveLength(0)
    expect(doc.contents.items).toMatchObject([
      { anchor: '__proto__', value: 1 },
      { source: '__proto__' }
    ])
    expect(doc.toJS()).toMatchObject([1, 1])
    expect(String(doc)).toBe(src)
  })

  test('create/stringify', () => {
    const doc = YAML.parseDocument('[{ a: A }, { b: B }]')
    const alias = doc.createAlias(doc.contents.items[0], '__proto__')
    doc.contents.items.push(alias)
    expect(doc.toJSON()).toMatchObject([{ a: 'A' }, { b: 'B' }, { a: 'A' }])
    expect(String(doc)).toMatch(
      '[ &__proto__ { a: A }, { b: B }, *__proto__ ]\n'
    )
  })
})

describe('merge <<', () => {
  const src = `---
- &CENTER { x: 1, y: 2 }
- &LEFT { x: 0, y: 2 }
- &BIG { r: 10 }
- &SMALL { r: 1 }

# All the following maps are equal:

- # Explicit keys
  x: 1
  y: 2
  r: 10
  label: center/big

- # Merge one map
  << : *CENTER
  r: 10
  label: center/big

- # Merge multiple maps
  << : [ *CENTER, *BIG ]
  label: center/big

- # Override
  << : [ *BIG, *LEFT, *SMALL ]
  x: 1
  label: center/big`

  test('YAML.parse', () => {
    const res = YAML.parse(src, { merge: true })
    expect(res).toHaveLength(8)
    for (let i = 4; i < res.length; ++i) {
      expect(res[i]).toMatchObject({ x: 1, y: 2, r: 10, label: 'center/big' })
    }
  })

  test('YAML.parse with merge:false', () => {
    const res = YAML.parse(src)
    expect(res).toHaveLength(8)
    for (let i = 5; i < res.length; ++i) {
      expect(res[i]).toHaveProperty('<<')
    }
  })

  test('YAML.parseDocument', () => {
    const doc = YAML.parseDocument(src, { merge: true })
    expect(
      doc.contents.items.slice(5).map(it => it.items[0].value)
    ).toMatchObject([
      { source: 'CENTER' },
      { items: [{ source: 'CENTER' }, { source: 'BIG' }] },
      { items: [{ source: 'BIG' }, { source: 'LEFT' }, { source: 'SMALL' }] }
    ])
  })

  test('alias is associated with a sequence', () => {
    const res = YAML.parse(
      '- &items [{ a: A }, { b: B }]\n- { <<: *items, c: C }',
      {
        merge: true
      }
    )
    expect(res[1]).toEqual({ a: 'A', b: 'B', c: 'C' })
  })

  describe('non-alias merges', () => {
    test('single map', () => {
      const res = YAML.parse('{ <<: { a: A }, b: B }', { merge: true })
      expect(res).toEqual({ a: 'A', b: 'B' })
    })

    test('multiple maps', () => {
      const res = YAML.parse('{ <<: [{ a: A }, { b: B }], c: C }', {
        merge: true
      })
      expect(res).toEqual({ a: 'A', b: 'B', c: 'C' })
    })

    test('mixed refs & maps', () => {
      const res = YAML.parse('{ <<: [{ a: &a { A: 1 } }, *a], b: B }', {
        merge: true
      })
      expect(res).toEqual({ a: { A: 1 }, A: 1, b: 'B' })
    })
  })

  describe('create', () => {
    test('simple case', () => {
      const doc = YAML.parseDocument('[{ a: A }, { b: B }]', { merge: true })
      const [a, b] = doc.contents.items
      const merge = doc.createPair('<<', doc.createAlias(a))
      b.items.push(merge)
      expect(doc.toJS()).toMatchObject([{ a: 'A' }, { a: 'A', b: 'B' }])
      expect(String(doc)).toBe('[ &a1 { a: A }, { b: B, <<: *a1 } ]\n')
    })

    test('merge pair of an alias', () => {
      const doc = YAML.parseDocument('[{ a: A }, { b: B }]', { merge: true })
      const [a, b] = doc.contents.items
      const alias = doc.createAlias(a, 'AA')
      const merge = doc.createPair('<<', alias)
      b.items.push(merge)
      expect(doc.toJS()).toMatchObject([{ a: 'A' }, { a: 'A', b: 'B' }])
      expect(String(doc)).toBe('[ &AA { a: A }, { b: B, <<: *AA } ]\n')
    })

    test('require map node', () => {
      const doc = YAML.parseDocument('[{ a: A }, { b: B }]', { merge: true })
      const alias = doc.createAlias(doc.getIn([0, 'a'], true))
      doc.addIn([1], doc.createPair('<<', alias))
      expect(String(doc)).toBe('[ { a: &a1 A }, { b: B, <<: *a1 } ]\n')
      expect(() => doc.toJS()).toThrow(
        'Merge sources must be maps or map aliases'
      )
    })
  })

  describe('merge multiple times', () => {
    const srcKeys = `
x:
  - &a
    k0: v1
    k1: v1
  - &b
    k1: v2
    k2: v2
y:
  k0: v0
  <<: *a
  <<: *b`

    const srcSeq = `
x:
  - &a
    k0: v1
    k1: v1
  - &b
    k1: v2
    k2: v2
y:
  k0: v0
  <<: [ *a, *b ]`

    const expObj = {
      x: [
        { k0: 'v1', k1: 'v1' },
        { k1: 'v2', k2: 'v2' }
      ],
      y: { k0: 'v0', k1: 'v1', k2: 'v2' }
    }

    const expMap = new Map([
      [
        'x',
        [
          new Map([
            ['k0', 'v1'],
            ['k1', 'v1']
          ]),
          new Map([
            ['k1', 'v2'],
            ['k2', 'v2']
          ])
        ]
      ],
      [
        'y',
        new Map([
          ['k0', 'v0'],
          ['k1', 'v1'],
          ['k2', 'v2']
        ])
      ]
    ])

    test('multiple merge keys, masAsMap: false', () => {
      const res = YAML.parse(srcKeys, { merge: true })
      expect(res).toEqual(expObj)
    })

    test('multiple merge keys, masAsMap: true', () => {
      const res = YAML.parse(srcKeys, { merge: true, mapAsMap: true })
      expect(res).toEqual(expMap)
    })

    test('sequence of anchors, masAsMap: false', () => {
      const res = YAML.parse(srcSeq, { merge: true })
      expect(res).toEqual(expObj)
    })

    test('sequence of anchors, masAsMap: true', () => {
      const res = YAML.parse(srcSeq, { merge: true, mapAsMap: true })
      expect(res).toEqual(expMap)
    })
  })

  test('do not throw error when key is null', () => {
    const src = ': 123'
    expect(() => YAML.parse(src, { merge: true })).not.toThrow()
  })

  describe('parse errors', () => {
    test('non-alias merge value', () => {
      const src = '{ <<: A, B: b }'
      expect(() => YAML.parse(src, { merge: true })).toThrow(
        'Merge sources must be maps or map aliases'
      )
    })

    test('non-map alias', () => {
      const src = '- &A a\n- { <<: *A, B: b }'
      expect(() => YAML.parse(src, { merge: true })).toThrow(
        'Merge sources must be maps or map aliases'
      )
    })

    test('immediate non-map merge value', () => {
      const src = '{ <<: [ [42] ], a: A }'
      expect(() => YAML.parse(src, { merge: true })).toThrow(
        'Merge sources must be maps or map aliases'
      )
    })

    test('circular reference', () => {
      const src = '&A { <<: *A, B: b }\n'
      const doc = YAML.parseDocument(src, { merge: true })
      expect(doc.errors).toHaveLength(0)
      expect(doc.warnings).toHaveLength(0)
      expect(() => doc.toJS()).toThrow('Maximum call stack size exceeded')
      expect(String(doc)).toBe(src)
    })

    test('missing whitespace', () => {
      expect(() => YAML.parse('&a{}')).toThrow(
        'Tags and anchors must be separated from the next token by white space'
      )
    })
  })

  describe('stringify', () => {
    test('example', () => {
      const doc = YAML.parseDocument(src, { merge: true })
      expect(YAML.parse(String(doc), { merge: true })).toMatchObject([
        { x: 1, y: 2 },
        { x: 0, y: 2 },
        { r: 10 },
        { r: 1 },
        { label: 'center/big', r: 10, x: 1, y: 2 },
        { label: 'center/big', r: 10, x: 1, y: 2 },
        { label: 'center/big', r: 10, x: 1, y: 2 },
        { label: 'center/big', r: 10, x: 1, y: 2 }
      ])
    })
  })
})
