/* Open/close functions.

Copyright (C) 1999 Politecnico di Torino

This file is part of the NDIS Packet capture driver.

The GNU C Library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public License as
published by the Free Software Foundation; either version 2 of the
License, or (at your option) any later version.

The GNU C Library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with the GNU C Library; see the file COPYING.LIB.  If
not, write to the Free Software Foundation, Inc., 675 Mass Ave,
Cambridge, MA 02139, USA.  */

#include <basedef.h>
#include <vmm.h>
#include <ndis.h>
#include <vwin32.h>
#include "debug.h"
#include "packet.h"
#pragma VxD_LOCKED_CODE_SEG
#pragma VxD_LOCKED_DATA_SEG

void YieldExecution( void )
{
	VMMCall(Release_Time_Slice);
	VMMCall(Begin_Nest_Exec);
	VMMCall(Resume_Exec);
	VMMCall(End_Nest_Exec);
}

/************************************************************
Function called when the user level application performs a open
IOCT. Initializes the driver
************************************************************/

DWORD PacketOpen(POPEN_INSTANCE	Open,DWORD dwDDB,DWORD hDevice,PDIOCPARAMETERS pDiocParms)
{

	LARGE_INTEGER		SystemTime;
	__int64				ltime1;

	SystemTime=GetDate();	
	
	/*initialize the timer variables for this session*/

	ltime1=((__int64)SystemTime.HighPart*86400);
	ltime1+=(__int64)(SystemTime.LowPart/1000);	//current time from 1980 in seconds
	ltime1+=(__int64)315525600;	//current time from 1970 (Unix format) in seconds
	ltime1*=1250000;
	ltime1+=(SystemTime.LowPart%1000)*1250000/1000; //current time from 1970 in ticks
	ltime1-=QuerySystemTime();	//boot time from 1970 in ticks
	Open->StartTime=ltime1;


	Open->Dropped=0;		//reset the dropped packets counter
	Open->Received=0;		//reset the received packets counter
	Open->bpfprogram=NULL;	//set a accept-all filter
	Open->bpfprogramlen=0;

	Open->BufSize=0;
	Open->Buffer=NULL;
	Open->Bhead=0;
	Open->Btail=0;
	Open->BLastByte=0;


	/*return happily*/
	*(DWORD *)(pDiocParms->lpcbBytesReturned) = InstHead->Open.BufSize;
	return NDIS_STATUS_SUCCESS;

}

DWORD PacketClose(POPEN_INSTANCE	Open,DWORD dwDDB,DWORD hDevice,PDIOCPARAMETERS pDiocParms)
{

NDIS_STATUS			Status;

	if(Open!=NULL){
		Open->BufSize=0;
		Open->Bhead=0;
		Open->Btail=0;
		Open->BLastByte=0;
		Open->Received=0;
		Open->Dropped=0;
		NdisFreeMemory(Open->Buffer,Open->BufSize,0);
		Open->Buffer=NULL;
		if(Open->bpfprogram!=NULL)NdisFreeMemory(Open->bpfprogram,Open->bpfprogramlen,0);
		Open->bpfprogram=NULL;
		Open->bpfprogramlen=0;
		*(DWORD *)(pDiocParms->lpcbBytesReturned) = 0;
		PacketCleanUp( &Status, Open );
	}

	return Status;
}

/************************************************************
Function used by NDIS to update the VXD when a new MAC driver
is added
************************************************************/
VOID NDIS_API PacketBindAdapter( OUT PNDIS_STATUS Status,
						 IN  NDIS_HANDLE  BindAdapterContext,
						 IN  PNDIS_STRING AdapterName,
						 IN  PVOID        SystemSpecific1,
						 IN  PVOID        SystemSpecific2 )
{
	PDEVICE_EXTENSION	pde;
	POPEN_INSTANCE 	oiNew;
	NDIS_STATUS			nsErrorStatus, nsOpenStatus;
	UINT           	uiMedium;
	NDIS_MEDIUM    	nmMediumArray=NdisMedium802_3;
	UINT           	i;
	PWRAPPER_PROTOCOL_BLOCK				pWPBlock;
	PNDIS_PROTOCOL_CHARACTERISTICS	pNPChar;

	TRACE_ENTER( "BindAdapter" );
	pde = GlobalDeviceExtension;
	/*Allocate an element that describe an adapter*/
	NdisAllocateMemory( (PVOID *)&oiNew, sizeof( OPEN_INSTANCE ), 0, -1 );
	if ( oiNew == NULL ) 
	{
		*Status = NDIS_STATUS_RESOURCES;
		return;
	}
	NdisZeroMemory( (PVOID)oiNew, sizeof( OPEN_INSTANCE ) );
	/*allocate a pool for the packet headers*/
	NdisAllocatePacketPool( &nsErrorStatus,
							&(oiNew->PacketPool),
							TRANSMIT_PACKETS,
							sizeof(PACKET_RESERVED) );
	IF_TRACE_MSG( "PACKET_RESERVED :%lx",sizeof(PACKET_RESERVED));
	if ( nsErrorStatus != NDIS_STATUS_SUCCESS ) 
	{
		IF_TRACE_MSG( "Failed to allocate packet pool AllocStatus=%x", nsErrorStatus );
		NdisFreeMemory( oiNew, sizeof( OPEN_INSTANCE ) ,  0 );
		*Status = NDIS_STATUS_RESOURCES;
		TRACE_LEAVE( "BindAdapter" );
		return;
	}
	/*allocate a pool for the packet data*/
	NdisAllocateBufferPool( &nsErrorStatus,
							&(oiNew->BufferPool),
							TRANSMIT_PACKETS );
	if ( nsErrorStatus != NDIS_STATUS_SUCCESS )
	{
		IF_TRACE_MSG( "Failed to allocate packet pool AllocStatus=%x", nsErrorStatus );
		NdisFreePacketPool( oiNew->PacketPool );
		NdisFreeMemory( oiNew, sizeof( OPEN_INSTANCE ) ,  0 );
		*Status = NDIS_STATUS_RESOURCES;
		TRACE_LEAVE( "BindAdapter" );
		return;
	}
	NdisAllocateSpinLock( &(oiNew->ResetSpinLock) );
	InitializeListHead( &(oiNew->ResetIrpList) );
	NdisAllocateSpinLock( &(oiNew->RcvQSpinLock) );
	InitializeListHead( &(oiNew->RcvList) );
	NdisAllocateSpinLock( &(oiNew->RequestSpinLock) );
	InitializeListHead( &(oiNew->RequestList) );

	for ( i=0;i<MAX_REQUESTS;i++ ) 
	{
		InsertTailList( &(oiNew->RequestList), &(oiNew->Requests[i].Reserved.ListElement) );
	}
	oiNew->Status = NDIS_STATUS_PENDING;
	oiNew->BindAdapterContext = BindAdapterContext;
	/*open the MAC driver calling NDIS*/
	NdisOpenAdapter( &nsOpenStatus,
					 &nsErrorStatus,
					 &oiNew->AdapterHandle,
					 &uiMedium,
					 &nmMediumArray,
					 1,
					 pde->NdisProtocolHandle,
					 oiNew,
					 AdapterName,
					 0,
					 NULL );
	IF_TRACE_MSG( "Open Status                   : %lx", nsOpenStatus );
	IF_TRACE_MSG( "Error Status                  : %lx", nsErrorStatus );
	IF_TRACE_MSG( "Completion Status             : %lx", oiNew->Status );
	if ( nsOpenStatus == NDIS_STATUS_PENDING )
	{
		while ( oiNew->Status == NDIS_STATUS_PENDING )
			YieldExecution();
	}
	else
	{
		PacketOpenAdapterComplete( oiNew, nsOpenStatus, nsErrorStatus );
	}
	pWPBlock = ((PWRAPPER_OPEN_BLOCK)(oiNew->AdapterHandle))->ProtocolHandle;
	pNPChar  = &pWPBlock->ProtocolCharacteristics;
	IF_TRACE_MSG( "Protocol                      : %s",  pNPChar->Name.Buffer );
	IF_TRACE_MSG( "Protocol Handle               : %lx", pde->NdisProtocolHandle );
	IF_TRACE_MSG( "PWRAPPER_OPEN_BLOCK           : %lx", oiNew->AdapterHandle );
	IF_TRACE_MSG( "PWRAPPER_PROTOCOL_BLOCK       : %lx", pWPBlock );
	IF_TRACE_MSG( "NDIS_PROTOCOL_CHARACTERISTICS : %lx", pNPChar );
	IF_TRACE_MSG( "Name                          : %lx", &pNPChar->Name );
	IF_TRACE_MSG( "Adapter Name                  : %s",  AdapterName->Buffer );
	*Status = oiNew->Status;

	if ( *Status != NDIS_STATUS_SUCCESS ) 
	{
		NdisFreeMemory( oiNew, sizeof( OPEN_INSTANCE ) ,  0 );
		IF_TRACE( "Bind Operation FAILED!" );
	}
	else
	{
		/*Here the programmer can put further initializations*/
	}
	TRACE_LEAVE( "BindAdapter" );
	return;
}

/************************************************************
Function called by NDIS to indicate the completion of a bind
************************************************************/
VOID NDIS_API
PacketOpenAdapterComplete(
   IN NDIS_HANDLE  ProtocolBindingContext,
   IN NDIS_STATUS  Status,
   IN NDIS_STATUS  OpenErrorStatus )
{
	POPEN_INSTANCE	Open;
	TRACE_ENTER( "BindAdapterComplete" );
	IF_TRACE_MSG2( "ErrorStatus=%x Status=%x", OpenErrorStatus, Status );
	Open = (POPEN_INSTANCE)ProtocolBindingContext;
	if ( Status == NDIS_STATUS_SUCCESS ) 
	{
		/*Inserisce il nuovo NIc aperto nella lista di quelli
			gi inizializzati*/
		InsertTailList( &GlobalDeviceExtension->OpenList, &Open->ListElement );
	}
	else
	{
		/*Libera le risorse occupate da driver.*/
		PacketFreeResources( Open );
	}
	Open->Status = Status;
	/*Completa segnalando all'NDIS l'avvenuto completamento 
		dell'operazione*/
	NdisCompleteBindAdapter( Open->BindAdapterContext, Status, OpenErrorStatus );
	TRACE_LEAVE( "BindAdapterComplete" );
	return;
}

/************************************************************
Start the unbind of a network driver from the protocol driver
************************************************************/
VOID NDIS_API
PacketUnbindAdapter( OUT PNDIS_STATUS	Status,
					 IN  NDIS_HANDLE	ProtocolBindingContext,
					 IN  NDIS_HANDLE	UnbindContext )
{
	POPEN_INSTANCE	Open;
	NDIS_STATUS		nsCloseStatus;
	TRACE_ENTER( "CloseAdapter" );
	Open = (POPEN_INSTANCE)ProtocolBindingContext;
	Open->BindAdapterContext = UnbindContext;
	/*clean the pending requests*/
	PacketCleanUp( Status, Open );
	Open->Status = NDIS_STATUS_PENDING;
	/*Calls NDIS to close the selected adapter*/
	NdisCloseAdapter( &nsCloseStatus, Open->AdapterHandle );
	if ( nsCloseStatus == NDIS_STATUS_PENDING )
	{
		while ( Open->Status == NDIS_STATUS_PENDING )
			YieldExecution();
	}
	else
	{
		PacketUnbindAdapterComplete( Open, nsCloseStatus );
	}
	*Status = Open->Status;
	if ( *Status == NDIS_STATUS_SUCCESS )
	{
		NdisFreeMemory( Open, sizeof( OPEN_INSTANCE ) ,  0 );
	}
	else
	{
		IF_TRACE( "Unbind Operation FAILED!" );
	}
	TRACE_LEAVE( "CloseAdapter" );
	return;
}

/************************************************************
Complete the unbind of a network driver from the protocol driver
************************************************************/
VOID NDIS_API 
PacketUnbindAdapterComplete( IN NDIS_HANDLE ProtocolBindingContext,
							 IN NDIS_STATUS Status	)
{
	POPEN_INSTANCE Open;
	TRACE_ENTER( "UnbindAdapterComplete" );
	Open = (POPEN_INSTANCE)ProtocolBindingContext;
	if ( Status == NDIS_STATUS_SUCCESS )
	{
		PacketFreeResources( Open );
	}
	Open->Status = Status;
	/*Invoca l'NDIS per completare l'unbind*/
	NdisCompleteUnbindAdapter( Open->BindAdapterContext, Status );
	TRACE_LEAVE( "UnbindAdapterComplete" );
	return;
}

/************************************************************
free the resources allocated by an adapter
************************************************************/
VOID PacketFreeResources( POPEN_INSTANCE Open )
{
	RemoveEntryList( &Open->ListElement );
	NdisFreeSpinLock( &Open->RequestSpinLock );
	NdisFreeSpinLock( &Open->RcvQSpinLock );
	NdisFreeSpinLock( &Open->ResetSpinLock );
	NdisFreeBufferPool( Open->BufferPool );
	NdisFreePacketPool( Open->PacketPool );
}

/************************************************************
Function that frees the pending requests  
************************************************************/
VOID
PacketCleanUp(	PNDIS_STATUS	Status,
				POPEN_INSTANCE	Open )
{
	PLIST_ENTRY			PacketListEntry;
	PNDIS_PACKET   	pPacket;
	PPACKET_RESERVED  Reserved;
	TRACE_ENTER( "Cleanup" );
	/*clean all the pending requests*/
	NdisAcquireSpinLock( &(Open->RcvQSpinLock) );
	while( (PacketListEntry = PacketRemoveHeadList( &(Open->RcvList) )) != NULL )
	{
		IF_VERY_LOUD( "CleanUp - Completing read" );
		Reserved = CONTAINING_RECORD( PacketListEntry, PACKET_RESERVED, ListElement );
		pPacket  = CONTAINING_RECORD( Reserved, NDIS_PACKET, ProtocolReserved );
		/*emulate the end of a transfer to wake the processes that 
		are waiting on a request */
		PacketTransferDataComplete( Open, pPacket, NDIS_STATUS_SUCCESS, 0 );
	}
	NdisReleaseSpinLock( &(Open->RcvQSpinLock) );
	//PacketReset( Status, Open );
	TRACE_LEAVE( "Cleanup" );
	return;
}

/************************************************************
Start the reset of a instance of the driver
************************************************************/
VOID
PacketReset( PNDIS_STATUS	pStatus,
			 POPEN_INSTANCE	pOpen )
{
	PLIST_ENTRY	ResetListEntry;
	TRACE_ENTER( "PacketReset" );

	NdisAcquireSpinLock( &pOpen->RequestSpinLock );
	ResetListEntry = PacketRemoveHeadList( &pOpen->RequestList );
	NdisReleaseSpinLock( &pOpen->RequestSpinLock );
	if ( ResetListEntry == NULL ) 
	{
		*pStatus = NDIS_STATUS_RESOURCES;
		TRACE_LEAVE( "PacketReset" );
		return;
	}
	NdisAcquireSpinLock( &pOpen->ResetSpinLock );
	InsertTailList( &pOpen->ResetIrpList, ResetListEntry );
	NdisReleaseSpinLock( &pOpen->ResetSpinLock );

	/*Call NDIS to reset the adapter*/
	NdisReset( pStatus, pOpen->AdapterHandle );
	if ( *pStatus != NDIS_STATUS_PENDING ) 
	{
		/*synchronous reset of the adapter*/
		PacketResetComplete( pOpen, *pStatus );
	}
	TRACE_LEAVE( "PacketReset" );
	return;
}
	
/************************************************************
complete the reset of a instance of the driver
************************************************************/
VOID NDIS_API
PacketResetComplete( IN NDIS_HANDLE	ProtocolBindingContext,
					 IN NDIS_STATUS	Status   )
{
	POPEN_INSTANCE		Open;
	PLIST_ENTRY			ResetListEntry;
	TRACE_ENTER( "PacketResetComplete" );
	Open = (POPEN_INSTANCE)ProtocolBindingContext;

	NdisAcquireSpinLock( &Open->ResetSpinLock );
	ResetListEntry = PacketRemoveHeadList( &Open->ResetIrpList );
	NdisReleaseSpinLock( &Open->ResetSpinLock );
	if ( ResetListEntry == NULL ) 
	{
		IF_VERY_LOUD( "Reset List Empty Error" );
		TRACE_LEAVE( "PacketResetComplete" );
		return;
	}

	NdisAcquireSpinLock( &Open->RequestSpinLock );
	InsertTailList( &Open->RequestList, ResetListEntry );
	NdisReleaseSpinLock( &Open->RequestSpinLock );
	TRACE_LEAVE( "PacketResetComplete" );
	return;
}
