// graph-tool -- a general graph modification and manipulation thingy
//
// Copyright (C) 2006-2025 Tiago de Paula Peixoto <tiago@skewed.de>
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU Lesser General Public License as published by the Free
// Software Foundation; either version 3 of the License, or (at your option) any
// later version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#ifndef GRAPH_BLOCKMODEL_MULTIFLIP_MCMC_HH
#define GRAPH_BLOCKMODEL_MULTIFLIP_MCMC_HH

#include "config.h"

#include <vector>
#include <algorithm>

#include "graph_tool.hh"
#include "../support/graph_state.hh"
#include "graph_blockmodel_util.hh"
#include <boost/mpl/vector.hpp>

#include "idx_map.hh"
#include "../loops/merge_split.hh"

namespace graph_tool
{
using namespace boost;
using namespace std;

#define MCMC_BLOCK_STATE_params(State)                                         \
    ((__class__,&, decltype(hana::tuple_t<python::object>), 1))                \
    ((state, &, State&, 0))                                                    \
    ((beta,, double, 0))                                                       \
    ((c,, double, 0))                                                          \
    ((d,, double, 0))                                                          \
    ((psingle,, double, 0))                                                    \
    ((psplit,, double, 0))                                                     \
    ((pmerge,, double, 0))                                                     \
    ((pmergesplit,, double, 0))                                                \
    ((nproposal, &, vector<size_t>&, 0))                                       \
    ((nacceptance, &, vector<size_t>&, 0))                                     \
    ((gibbs_sweeps,, size_t, 0))                                               \
    ((entropy_args,, typename State::_entropy_args_t, 0))                      \
    ((verbose,, int, 0))                                                       \
    ((force_move,, bool, 0))                                                   \
    ((niter,, double, 0))

template <class State>
struct MCMC
{
    GEN_STATE_BASE(MCMCBlockStateBase, MCMC_BLOCK_STATE_params(State))

    template <class... Ts>
    class MCMCBlockStateImp
        : public MCMCBlockStateBase<Ts...>,
          public MergeSplitStateBase
    {
    public:
        GET_PARAMS_USING(MCMCBlockStateBase<Ts...>,
                         MCMC_BLOCK_STATE_params(State))
        GET_PARAMS_TYPEDEF(Ts, MCMC_BLOCK_STATE_params(State))

        template <class... ATs,
                  typename std::enable_if_t<sizeof...(ATs) ==
                                            sizeof...(Ts)>* = nullptr>
        MCMCBlockStateImp(ATs&&... as)
           : MCMCBlockStateBase<Ts...>(as...)
        {
            GILRelease gil_release;
            _state.init_mcmc(*this);
        }

        constexpr static group_t _null_group = null_group;

        constexpr static double _pmovelabel = 0;

        constexpr static double _psrandom = 1;
        constexpr static double _psscatter = 1;
        constexpr static double _pscoalesce = 1;

        typename State::m_entries_t _m_entries;

        template <class F>
        void iter_nodes(F&& f)
        {
            for (auto v : vertices_range(_state._g))
            {
                if (_state.node_weight(v) == 0)
                    continue;
                f(v);
            }
        }

        template <class F>
        void iter_groups(F&& f)
        {
            for (auto r : vertices_range(_state._bg))
            {
                if (_state._wr[r] == 0)
                    continue;
                f(r);
            }
        }

        auto get_group(size_t v)
        {
            return _state._b[v];
        }

        template <bool sample_branch=true, class RNG, class VS = std::array<group_t,0>>
        group_t sample_new_group(size_t v, RNG& rng, VS&& except = VS())
        {
            for (auto r : except)
            {
                if (_state._empty_groups.find(r) != _state._empty_groups.end())
                    _state._empty_groups.erase(r);
            }

            _state.get_empty_block(v);

            auto t = uniform_sample(_state._empty_groups, rng);

            for (auto r : except)
            {
                if (r != _null_group &&_state._wr[r] == 0)
                    _state._empty_groups.insert(r);
            }

            auto r = _state._b[v];
            _state._bclabel[t] = _state._bclabel[r];

            auto cstate = _state._coupled_state;
            if (cstate != nullptr)
            {
                if constexpr (sample_branch)
                {
                    cstate->sample_branch(t, r, rng);
                }
                else
                {
                    auto& bh = cstate->get_b();
                    bh[t] = bh[r];
                }
                auto& hpclabel = cstate->get_pclabel();
                hpclabel[t] = _state._pclabel[v];
            }
            assert(_state._wr[t] == 0);
            return t;
        }

        void move_node(size_t v, group_t r, bool cache)
        {
            if (cache)
                _state.move_vertex(v, r, _m_entries);
            else
                _state.move_vertex(v, r);
        }

        void reserve_empty_groups(size_t nB)
        {
            if (_state._empty_groups.size() < nB)
                _state.add_block(nB - _state._empty_groups.size());
        }

        bool allow_move(group_t r, group_t s)
        {
            return _state.allow_move(r, s);
        }

        double virtual_move(size_t v, group_t r, group_t s)
        {
            return _state.virtual_move(v, r, s, _entropy_args, _m_entries);
        }

        template <class RNG>
        group_t sample_group(size_t v, bool allow_empty, RNG& rng)
        {
            return _state.sample_block(v, _c, allow_empty ? _d : 0, rng);
        }

        double get_move_prob(size_t v, group_t r, group_t s, bool allow_empty,
                             bool reverse)
        {
            return _state.get_move_prob(v, r, s, _c, allow_empty ? _d : 0, reverse);
        }

        bool can_swap(group_t r, group_t s)
        {
            auto cstate = _state._coupled_state;
            if (cstate == nullptr)
                return _state._bclabel[r] == _state._bclabel[s];
            auto& hb = cstate->get_b();
            auto rr = hb[r];
            auto ss = hb[s];
            return (rr == ss) && (_state._bclabel[r] == _state._bclabel[s]);
        }

        void relax_update(bool relax)
        {
            _state.relax_update(relax);
        }

        void store_next_state(group_t v)
        {
            _state.store_next_state(v);
        }

        void clear_next_state()
        {
            _state._egroups.check(_state._bg, _state._mrs);
            _state.clear_next_state();
        }

        template <class V>
        void push_state(V&& vs)
        {
            _state.push_state(vs);
        }

        void pop_state()
        {
            _state.pop_state();
        }
   };

    using gmap_t = idx_map<group_t, idx_set<size_t, true>, false, true, true>;

    template <class T>
    using iset = idx_set<T>;

    template <class T, class V>
    using imap = idx_map<T, V>;

    template <class... Ts>
    class MCMCBlockState:
        public MergeSplit<MCMCBlockStateImp<Ts...>,
                          size_t,
                          group_t,
                          iset,
                          imap,
                          iset,
                          gmap_t, false, false>
    {
    public:
        template <class... ATs,
                  typename std::enable_if_t<sizeof...(ATs) ==
                                            sizeof...(Ts)>* = nullptr>
        MCMCBlockState(ATs&&... as)
           : MergeSplit<MCMCBlockStateImp<Ts...>,
                        size_t,
                        group_t,
                        iset,
                        imap,
                        iset,
                        gmap_t, false, false>(as...)
        {}
    };
};

} // graph_tool namespace

#endif //GRAPH_BLOCKMODEL_MULTIFLIP_MCMC_HH
