	TITLE	'CDCIOS -- CDC/SMS CBIOS FOR CPM 2.2'

;	08-03-81




;   'CDCIOS' IS THE CPM 2.2 I/O DRIVER FOR A CDC PHOENIX (CMD) HARD DISK
;   DRIVE (OR ITS EQUIVALENT). AN XCOMP SMS/SMR CONTROLLER IS USED. THE
;   PHOENIX IS CONFIGURED AS FOUR, EIGHT, OR TWELVE 6.7 MEGABYTE LOGICAL
;   DRIVES. 'CDCIOS' ALSO SUPPORTS TWO FLOPPY DISK DRIVES USING A CCS
;   2422 F/D CONTROLLER IN SINGLE DENSITY MODE.
;
;
;   DRIVE CONFIGURATION IS AS FOLLOWS ---
;
;	1. 32 MEG PHOENIX
;	   A) CDC = A: THRU D:
;	   B) F/D = E: & F:
;
;	2. 64 MEG PHOENIX
;	   A) CDC = A: THRU H:
;	   B) F/D = I: & J:
;
;	3. 96 MEG PHOENIX
;	   A) CDC = A: THRU L:
;	   B) F/D = M: & N:




;	---PROGRAM CONFIGURATION---
;
;
;   THE FOLLOWING STATMENTS ARE USED TO CONFIGURE THE PROGRAM.
;
;	1. Z80 -- SET THIS TRUE IF YOU ARE USING A Z-80 MICROPROCESSOR.
;	   THIS WILL SPEED UP DATA TRANSFERS TO/FROM THE CONTROLLER
;	   BUFFER.
;
;	2. ENBINT - SET THIS 'TRUE' IF YOU NEED INTERRUPTS ENABLED
;	   WHILE RUNNING THE PROGRAM. SET THIS 'FALSE' IF YOU DO
;	   NOT USE INTERRUPTS.
;
;	3. S100 - SET THIS 'TRUE' IF YOU ARE USING A CONTROLLER WITH
;	   AN S100 BUS INTERFACE (SMS). SET THIS 'FALSE' IF YOU ARE
;	   USING A CONTROLLER WITH A RIBBON CABLE INTERFACE (SMR).
;
;	4. DTYPE - DEFINES DRIVE TYPE. SET THIS TO A VALUE OF '1' IF
;	   YOU ARE USING A 32 MEG DRIVE. SET THIS TO A VALUE OF '2'
;	   IF YOU ARE USING A 64 MEG DRIVE. SET THIS TO A VALUE OF
;	   OF '3' IF YOU ARE USING A 96 MEG DRIVE.
;
;
;
FALSE	EQU	0		;DEFINE FALSE
TRUE	EQU	NOT FALSE	;..TRUE
;
Z80	EQU	TRUE		;USING A Z-80
;
ENBINT	EQU	FALSE		;WE DON'T USE INTERRUPTS
;
S100	EQU	TRUE		;S-100 BUS CONTROLLER (SMS)
;
DTYPE	EQU	1		;32 MEG DRIVE




;	---EQUATES---
;
;
MSIZE	EQU	62		;MEMORY SIZE IN K-BYTES
;
BIAS	EQU	(MSIZE-20)*1024	;SYSTEM BIAS
CPM	EQU	BIAS+3400H  	;BASE ADR OF CPM
BDOS	EQU	CPM+806H  	;LINK TO BDOS
BIOS	EQU	CPM+1600H	;CBIOS START ADDRESS
NSECT	EQU	(BIOS-CPM)/128	;NUMBER OF SECTORS TO WARM BOOT
IOBYT	EQU	3		;I/O SYSTEM CONFIGURATION
CDISK	EQU	4		;CURRENT DISK ADDRESS (USED BY CPM)
BUFF	EQU	80H		;FCB BUFFER
RPT	EQU	80H		;REPEAT
INDEX	EQU	10H		;INDEX BIT
VOL1	EQU	10H		;VOLUME #1
;
MAXFD	EQU	2		;NUMBER OF FLOPPY DISK DRIVES
MAXSEC	EQU	64		;NUMBER OF SECTORS PER TRACK
MAXHD	EQU	DTYPE*2		;NUMBER OF HEADS
MAXCYL	EQU	823		;NUMBER OF CYLINDERS
LDS	EQU	822/2		;LOGICAL DISK SIZE (IN TRACKS)
FFD	EQU	MAXHD*2		;FIRST FLOPPY DISK
LLD	EQU	FFD+MAXFD	;LAST LOGICAL DRIVE
;
;   CRT I/O
;
CRTS	EQU	9		;STATUS
CRTD	EQU	8		;DATA
CRTBE	EQU	1		;BUFFER EMPTY
CRTDA	EQU	2		;DATA AVAILABLE
;
;   LINE PRINTER I/O
;
LPS	EQU	0BH		;STATUS
LPD	EQU	0AH		;DATA
LPXR	EQU	85H		;XMTR READY
;
;   HARD DISK/CPM INTERFACE
;
BLKSIZ	EQU	4096		;NUMBER OF BYTES PER BLOCK
CPMSPT	EQU	2*MAXSEC	;NUMBER OF CPM SECTORS PER TRACK
;
;   CONTROLLER COMMANDS
;
BANK0	EQU	0		;BANK 0 SELECT
BANK1	EQU	1		;BANK 1 SELECT
DBENB	EQU	2		;DATA BUFFER ENABLE
CBENB	EQU	4		;COMPARE BUFFER ENABLE
START	EQU	8		;START COMMAND
;
;   DRIVE COMMANDS
;
D@CLR	EQU	10H		;FAULT CLEAR
D@TKZ	EQU	40H		;REZERO
;
;   DRIVE TAGS
;
DSTAG	EQU	1		;DRIVE SELECT TAG
TAG1	EQU	2+1		;  TAG1
TAG2	EQU	4+1		;  TAG2
TAG3	EQU	8+1		;  TAG3
;
;   DRIVE/CONTROLLER I/O
;
CBASE	EQU	70H		;BASE ADR OF THE CONTROLLER
	IF	S100	;  ** IF S-100 INTERFACE **
DRCSR	EQU	CBASE		;DRIVE COMMAND/STATUS
CBL	EQU	CBASE+1		;COMMAND BUS, LSB
CBH	EQU	CBASE+2		;  * MSB
DRSB	EQU	CBASE+3		;DRIVE SELECT BITS
CTCSR	EQU	CBASE+4		;CONTROLLER COMMAND/STATUS
CTBFR	EQU	CBASE+5		;CONTROLLER BUFFER ADDRESS
CTDP	EQU	CBASE+6		;CONTROLLER DATA PORT
CLRSEK	EQU	CBASE+7		;CLEAR SEEK END F/F
	ELSE		;  ** IF RIBBON CABLE INTERFACE **
DRCSR	EQU	CBASE+7		;DRIVE COMMAND/STATUS
CBL	EQU	CBASE+6		;COMMAND BUS, LSB
CBH	EQU	CBASE+5		;  * MSB
DRSB	EQU	CBASE+4		;DRIVE SELECT BITS
CTCSR	EQU	CBASE+3		;CONTROLLER COMMAND/STATUS
CTBFR	EQU	CBASE+2		;CONTROLLER BUFFER ADDRESS
CTDP	EQU	CBASE+1		;CONTROLLER DATA PORT
CLRSEK	EQU	CBASE		;CLEAR SEEK END F/F
	ENDIF
;
;   ASCII CHARACTERS
;
CR	EQU	0DH		;CARRIAGE RETURN
LF	EQU	0AH		;LINE FEED
BELL	EQU	7		;DING!
EOF	EQU	1AH		;CPM END-OF-FILE
;
;   FLOPPY DISK EQUATES
;
FD0	EQU	11H		;DRIVE ZERO
MAXI	EQU	10H		;MAXI (8") DISK
AUTO	EQU	80H		;AUTO WAIT
;
;   1791 I/O
;
DISK	EQU	0F8H		;BASE ADR OF F/D CONTROLLER
;
D@CSR	EQU	DISK		;COMMAND/STATUS
D@TK	EQU	DISK+1		;TRACK ADDRESS
D@SEC	EQU	DISK+2		;SECTOR ADDRESS
D@DP	EQU	DISK+3		;DATA PORT
D@CP1	EQU	DISK+4		;CONTROL PORT #1
D@CP2	EQU	DISK+5		;CONTROL PORT #2
;
;   DISK COMMANDS
;
STEP	EQU	1		;STEP RATE = 6 MSEC
C@TKZ	EQU	8+STEP		;TRACK ZERO SEEK
C@SEEK	EQU	1CH+STEP	;TRACK SEEK
C@SUN	EQU	10H		;SEEK (UNLOAD HEAD)
C@RD	EQU	80H		;SECTOR READ
C@WRT	EQU	0A0H		;SECTOR WRITE
;
;   DISK STATUS  (AFTER COMMAND EXECUTION)
;
NRB	EQU	80H		;NOT READY BIT
WPROT	EQU	40H		;WRITE PROTECT BIT
S@TKZ	EQU	9CH		;TRACK ZERO SEEK
S@SEEK	EQU	98H		;SEEK
S@RD	EQU	9CH		;READ
S@WRT	EQU	0ECH		;WRITE


	PAGE


;	============
;	** MACROS **
;	============




;	---SET FLAG---
;
;
SETFLG	MACRO	FLAG
	MVI	A,1
	STA	FLAG		;; SET FLAG
	ENDM





;	---RESET FLAG---
;
;
CLRFLG	MACRO	FLAG
	XRA	A
	STA	FLAG		;; CLEAR THE FLAG
	ENDM





;	---TEST A FLAG---
;
;
TSTFLG	MACRO	FLAG
	LDA	FLAG
	ORA	A		;; SET 8080 FLAGS
	ENDM





;	---Z80 BLOCK INPUT---
;
;
INIR	MACRO
	DB	0EDH,0B2H
	ENDM





;	---Z80 BLOCK OUTPUT---
;
;
OUTIR	MACRO
	DB	0EDH,0B3H
	ENDM


	PAGE

	ORG	BIOS


;	---JUMP TABLE---
;
;
	JMP	BOOT		;  0 - COLD BOOT
WBX:	JMP	WBOOT		;  1 - WARM BOOT
	JMP	CSTS		;  2 - CONSOLE STATUS REQUEST
	JMP	CI		;  3 - CONSOLE INPUT
	JMP	CO		;  4 - CONSOLE OUTPUT
	JMP	LO		;  5 - LIST OUTPUT
	JMP	PO		;  6 - PUNCH OUTPUT
	JMP	RDR		;  7 - READER INPUT
	JMP	HOME		;  8 - TRACK ZERO SEEK
	JMP	SETDR		;  9 - SET DRIVE #
	JMP	SETTK		; 10 - SET TRACK ADR
	JMP	SETSEC		; 11 - SET SECTOR ADR
	JMP	SETDMA		; 12 - SET BUFFER ADDRESS
	JMP	READ		; 13 - READ A SECTOR
	JMP	WRITE		; 14 - WRITE A SECTOR
	JMP	LOTST		; 15 - LIST OUTPUT READY TEST
	JMP	SXR		; 16 - SECTOR XLATE ROUTINE




;	---SIGN ON MESSAGE---
;
;
;
HELLO:	DB	CR,LF,LF,LF,LF
	DB	MSIZE/10 + '0'
	DB	MSIZE MOD 10  + '0'
	DB	'K CDCIOS Vers 2.2'
	DB	CR,LF
	DB	'3-Aug-81'
	DB	CR,LF,0


	PAGE



;	======================
;	** DISK DEFINITIONS **
;	======================





;	---DISK PARAMETERS---
;
;
DPHDR	EQU	$
;
;   THE FIRST FOUR ENTRIES ARE ALWAYS DEFINED
;
	DW	0,0		;CDC LOGICAL DRIVE #0
	DW	0,0
	DW	DIRBFR,HDBLK
	DW	LAST,HALL0
;
	DW	0,0		;CDC LOGICAL DRIVE #1
	DW	0,0
	DW	DIRBFR,HDBLK
	DW	LAST,HALL1
;
	DW	0,0		;CDC LOGICAL DRIVE #2
	DW	0,0
	DW	DIRBFR,HDBLK
	DW	LAST,HALL2
;
	DW	0,0		;CDC LOGICAL DRIVE #3
	DW	0,0
	DW	DIRBFR,HDBLK
	DW	LAST,HALL3
;
;   THE FOLLOWING FOUR TABLE ENTRIES ARE DEFINED IF
;   THE SIZE OF THE PHOENIX IS 64 OR 96 MEGABYTES.
;
	IF	DTYPE GT 1
;
	DW	0,0		;CDC LOGICAL DRIVE #4
	DW	0,0
	DW	DIRBFR,HDBLK
	DW	LAST,HALL4
;
	DW	0,0		;CDC LOGICAL DRIVE #5
	DW	0,0
	DW	DIRBFR,HDBLK
	DW	LAST,HALL5
;
	DW	0,0		;CDC LOGICAL DRIVE #6
	DW	0,0
	DW	DIRBFR,HDBLK
	DW	LAST,HALL6
;
	DW	0,0		;CDC LOGICAL DRIVE #7
	DW	0,0
	DW	DIRBFR,HDBLK
	DW	LAST,HALL7
	ENDIF
;
;   THE FOLLOWING FOUR TABLE ENTRIES ARE DEFINED IF
;   THE SIZE OF THE PHOENIX IS 96 MEGABYTES.
;
	IF	DTYPE GT 2
;
	DW	0,0		;CDC LOGICAL DRIVE #8
	DW	0,0
	DW	DIRBFR,HDBLK
	DW	LAST,HALL8
;
	DW	0,0		;CDC LOGICAL DRIVE #9
	DW	0,0
	DW	DIRBFR,HDBLK
	DW	LAST,HALL9
;
	DW	0,0		;CDC LOGICAL DRIVE #10
	DW	0,0
	DW	DIRBFR,HDBLK
	DW	LAST,HALL10
;
	DW	0,0		;CDC LOGICAL DRIVE #11
	DW	0,0
	DW	DIRBFR,HDBLK
	DW	LAST,HALL11
	ENDIF
;
;   THE FOLLOWING TWO TABLE ENTRIES ARE FOR
;   THE FLOPPY DISK AND ARE ALWAYS DEFINED.
;
	DW	XLATE,0		;F/D DRIVE 0
	DW	0,0
	DW	DIRBFR,FDBLK
	DW	FCHK0,FALL0
;
	DW	XLATE,0		;F/D DRIVE 1
	DW	0,0
	DW	DIRBFR,FDBLK
	DW	FCHK1,FALL1





;	---HARD DISK PARAMETER BLOCK---
;
;
;   THIS TABLE DEFINES THE HARD DISK CONFIGURATION.
;
;	1. THE SINGLE PHOENIX DRIVE IS CONFIGURED AS FOUR
;	   EIGHT, OR TWELVE 6.7 MEGABYTE LOGICAL DRIVES.
;
;	2. THE CPM LOGICAL BLOCK SIZE IS 4096 BYTES.
;
;	3. THERE ARE 1024 DIRECTORY ENTRIES.
;
;	4. THE FIRST TRACK IS RESERVED FOR THE SYSTEM. THIS WILL
;	   ALLOW AN AGGREGATE BDOS/CBIOS SIZE OF 16 K-BYTES.
;
;
HDBLK:	DW	2*MAXSEC	;SPT: SECTORS PER TRACK
	DB	5		;BSH: BLOCK SHIFT FACTOR
	DB	31		;BLM: BLOCK MASK
	DB	1		;EXM: NULL MASK
	DW	1639		;DSM: (DISK SIZE) - 1
	DW	1023		;DRM: (DIR SIZE) - 1
	DB	0FFH		;AL0: DIRECTORY ALLOCATION BITS, MSB
	DB	0		;AL1:   * LSB
	DW	0		;CKS: CHECK SIZE
	DW	1		;OFF: TRACK OFFSET





;	---FLOPPY DISK PARAMETER BLOCK---
;
;
;   THIS TABLE DEFINES FLOPPY DISK CONFIGURATION. THE F/D IS
;   DEFINED AS A STANDARD SINGLE DENSITY CP/M DISK.
;
;
FDBLK:	DW	26		;SPT: SECTORS PER TRACK
	DB	3		;BSH: BLOCK SHIFT FACTOR
	DB	7		;BLM: BLOCK MASK
	DB	0		;EXM: NULL MASK
	DW	242		;DSM: (DISK SIZE) - 1
	DW	63		;DRM: (DIR SIZE) - 1
	DB	0C0H		;AL0: DIRECTORY ALLOCATION BITS, MSB
	DB	0		;AL1:   * LSB
	DW	16		;CKS: CHECK SIZE
	DW	2		;OFF: TRACK OFFSET





;	---FLOPPY DISK TRANSLATION TABLE---
;
;
;
XLATE:	DB	1,7,13,19	;  1...4
	DB	25,5,11,17	;  5...8
	DB	23,3,9,15	;  9...12
	DB	21,2,8,14	; 13...16
	DB	20,26,6,12	; 17...20
	DB	18,24,4,10	; 21...24
	DB	16,22		; 25 & 26


	PAGE

;	=====================
;	** CBIOS FUNCTIONS **
;	=====================




;	---COLD BOOT---
;
;
BOOT:	LXI	SP,BUFF		;USE LOW MEMORY
	LXI	H,HELLO
	CALL	PRNT		;SAY HELLO
	CALL	INIT		;INIT FLAGS & VARIABLES
	STA	CDISK		;FOR USE BY CPM
	STA	CFD		;SET CURR F/D = DRIVE 0
	MVI	A,FD0
	OUT	D@CP1		;SLCT F/D DRIVE 0
	STA	RFD		;..SAVE FOR FUTURE USE
	MVI	A,0D0H
	OUT	D@CP2		;8", SNGL DENS
	JMP	XIT		;GOTO CPM




;	---WARM BOOT---
;
;
WBOOT:	LXI	SP,BUFF
	CALL	INIT		;INIT FLAGS & VARIABLES
;
	CALL	RTZ		;RESTORE HARD DISK
	CALL	XSEL		;SLCT HEAD #0
	LXI	H,CPM		;CPM LOAD ADR
	SHLD	DMADR		;  SAVE IT
	MVI	C,2
	CALL	SETSEC		;SET SECTOR ADR
	MVI	A,NSECT		; # SECTORS TO WARM BOOT
;
WB1:	STA	SECNT		;SAVE THE SECTOR COUNT
	CALL	HDREAD		;READ A SECTOR FROM THE HARD DISK
	ORA	A
	JZ	WB2		;JIF READ OK
	LXI	H,MSG1		;  'CAN'T BOOT'
	CALL	PRNT		;PRINT ERROR MSG
	CALL	CI		;WAIT FOR OPER
	CALL	CRLF
	JMP	WBOOT		;TRY ANOTHER BOOT
;
WB2:	LHLD	DMADR
	LXI	D,128
	DAD	D		;BUMP DMA ADR
	SHLD	DMADR
	LXI	H,R@SEC
	MOV	C,M
	INR	C		;INCR SECT ADR
	CALL	SETSEC		;SET NEW SECT ADR
	LDA	SECNT		;A = SECTOR COUNT
	DCR	A
	JNZ	WB1		;JIF MORE TO READ




;	---EXIT TO CPM---
;
;
XIT:	LXI	H,BUFF
	SHLD	DMADR		;INIT DMA ADR
	MVI	A,0C3H		;JMP INST
	STA	0
	LXI	H,WBX
	SHLD	1		;SET WARM BOOT LINK
	LXI	H,BDOS
	STA	5
	SHLD	6		;SET BDOS LINK
	LDA	CDISK		;CURRENT LOGGED-IN DISK & USER
	MOV	C,A
	JMP	CPM		;GOTO CPM




;	---CONSOLE INPUT---
;
;
CI:	IN	CRTS		;CRT STATUS
	ANI	CRTDA		;DATA AVAIL
	JZ	CI		;WAIT FOR DATA
	IN	CRTD		;CRT DATA
	ANI	7FH		;KILL PARITY BIT
	RET




;	---CONSOLE OUTPUT---
;
;
CO:	IN	CRTS		;CRT STATUS
	ANI	CRTBE		;BFR EMPTY
	JZ	CO		;WAIT FOR EMPTY
	MOV	A,C
	OUT	CRTD		;OUTPUT DATA
	RET




;	---CONSOLE STATUS---
;
;
CSTS:	IN	CRTS		;CRT STATUS
	ANI	CRTDA		;DATA AVAIL
	RZ			;RIF NOT AVAIL
	ORI	-1		;SET AVAIL FLAG
	RET




;	---LIST OUTPUT---
;
;
LO:	IN	LPS		;L/P STATUS
	ANI	LPXR		;XMTR RDY
	XRI	LPXR		;..INVERT BITS
	JNZ	LO		;WAIT FOR RDY
	MOV	A,C
	OUT	LPD		;OUTPUT PRINT DATA
	RET




;	---READER INPUT---
;
;
RDR:	MVI	A,EOF		;NO RDR -- SET 'EOF' FLAG
	RET




;	---PUNCH OUT---
;
;
PO:	RET			;  * NOT IMPLEMENTED *




;	---LIST READY TEST---
;
;
LOTST:	IN	LPS		;L/P STATUS
	ANI	LPXR		;XMTR RDY
	XRI	LPXR		;..INVERT BITS
	MVI	A,0
	RNZ			;RIF NOT READY
	ORI	-1		;SET READY FLAG
	RET




;	---SECTOR TRANSLATE ROUTINE---
;
;
SXR:	XCHG			;H/L = ADR OF XLATE TABLE
	MOV	A,L
	ORA	H		;TEST TABLE ADR
	JNZ	SXR1		;JIF DO XLATE
	DAD	B		;DON'T XLATE -- USE SECT ADR AS IS
	RET
;
SXR1:	DAD	B		;INDEX INTO THE TABLE
	MOV	L,M		;L = XLATED SECTOR ADR
	MVI	H,0		;H = 0 FOR 16-BIT VALUE
	RET




;	---HOME---
;
;
HOME:	LDA	R@DSK		;DRIVE #
	CPI	FFD
	JC	XTKZ		;JIF REZERO HARD DISK
	CPI	LLD
	JC	FTKZ		;JIF REZERO FLOPPY
;
;   SET ERROR FLAG
;
SEF:	MVI	A,1		;SET ERROR FLAG
	ORA	A		;SET 8080/Z80 FLAGS
	RET




;	---READ---
;
;
READ:	LDA	R@DSK		;DRIVE #
	CPI	FFD
	JC	HDREAD		;JIF READ FROM H/D
	CPI	LLD
	JC	FREAD		;JIF READ FROM F/D
	JMP	SEF		;ABORT, SELECT ERROR




;	---WRITE---
;
;
WRITE:	LDA	R@DSK		;DRIVE #
	CPI	FFD
	JC	HDWRT		;JIF WRITE ONTO H/D
	CPI	LLD
	JC	FWRT		;JIF WRITE ONTO F/D
	JMP	SEF		;ABORT, SELECT ERROR




;	---SET DRIVE NUMBER---
;
;
SETDR:	MOV	A,C		;A = NEW DRIVE #
	STA	R@DSK		;  SAVE IT
	LXI	H,0
	CPI	LLD
	RNC			;RIF INVALID DRIVE #
	MOV	L,A
	DAD	H		;SHIFT H/L LEFT 4
	DAD	H		;   (HL = HL*16)
	DAD	H
	DAD	H
	LXI	D,DPHDR		;DISK PARM HDR
	DAD	D		;H/L = DPHDR FOR SELECTED DRIVE
	RET




;	---SET SECTOR ADR---
;
;
SETSEC:	MOV	A,C
	STA	R@SEC		;SAVE SECTOR ADR
; SET REAL SECTOR ADR FOR THE H/D DRIVER
	ANI	NOT 1		;CLEAR LOW ORDER BIT
	RRC			;THERE ARE 2 CPM SECTORS PER H/D SECT
	STA	RSA		;SAVE REAL SECTOR ADR
	RET




;	---SET TRACK ADDRESS---
;
;
SETTK:	MOV	H,B
	MOV	L,C
	SHLD	R@TRK		;SAVE TRACK ADR
	RET




;	---SET DMA ADDRESS---
;
;
SETDMA:	MOV	H,B
	MOV	L,C
	SHLD	DMADR		;SAVE DMA ADR
	RET

	PAGE


;	======================================
;	** HARD DISK BLOCK/DEBLOCK ROUTINES **
;	======================================




;	---HARD DISK READ---
;
;
HDREAD:	XRA	A
	STA	ERFLG		;CLEAR THE ERROR FLAG
	SETFLG	F@RDOP		;SET READ OPERATION FLAG
	TSTFLG	F@WIP		;WRITE IN PROGRESS ?
	CNZ	XWRT		;YES, WRITE DATA BEFORE READ
	CLRFLG	F@WIP		;RESET 'WIP' FLAG
	CALL	TSTHST		;HOST = REQ ?
	JNZ	HDRD1		;NO, READ A BLOCK
	TSTFLG	F@RIP		;PRIOR BLOCK READ ?
	JNZ	HDRD2		;YES, JUST EXTRACT DATA FROM BFR
HDRD1:	CALL	SETHST		;MAKE HOST=REQ
	CALL	XREAD		;READ A BLOCK
HDRD2:	CALL	GETDMA		;GET DMA ADR, SET POINTERS
	CALL	BFRIN		;MOVE DATA TO RAM
	SETFLG	F@RIP		;SET READ-IN-PROGRESS FLAG
	LDA	ERFLG		;ERROR FLAG
	RET




;	---HARD DISK WRITE---
;
;
HDWRT:	XRA	A
	STA	ERFLG		;CLEAR THE ERROR FLAG
	CLRFLG	F@RIP		;CLEAR READ-IN-PROGRESS FLAG
	MOV	A,C		;A = WRITE MODE
	DCR	A
	JZ	WDIR		;DO DIRECTORY WRITE
	JM	WNORM		;DO NORMAL WRITE
;
;   UNALLOCATED WRITE
;
WUN:	TSTFLG	F@WIP		;WRITE IN PROGRESS ?
	CNZ	XWRT		;YES, WRITE DATA IN BFR
; SET UNALLOC RECORD PARAMETERS
	MVI	A,BLKSIZ/128
	STA	U@CNT		;SET UNALLOC RECORD COUNT
	LHLD	R@DSK
	SHLD	U@DSK		;UPDATE DRIVE & SECTOR
	LHLD	R@TRK
	SHLD	U@TRK		;UPDATE TRACK ADR
	CALL	SETHST		;SET HOST = REQ
	CALL	BUMP		;BUMP UNALC PARMS FOR NEXT PASS

; XFER DATA TO CTLR BFR
WXFER:	CLRFLG	F@RDOP		;CLEAR READ OPER FLAG
	CALL	GETDMA		;GET DMA ADR, SET FOR WRITE
	CALL	BFROUT		;MOVE DATA TO THE BFR
	SETFLG	F@WIP		;SET WRITE-IN-PROGRESS FLAG
	LDA	ERFLG		;ERROR FLAG
	RET
;
;   NORMAL WRITE
;
WNORM:	LDA	U@CNT		;UNALC RECORD COUNT
	ORA	A
	JZ	WALC		;JIF DO ALLOC WRITE
	LXI	H,U@TRK
	CALL	DSKCMP		;UNALC DSK/TRK = REQ DSK/TRK ?
	JNZ	WALC		;NO, DO ALLOC WRITE
	LDA	R@SEC
	CMP	M		;UNALC SECT = REQ SECT ?
	JNZ	WALC		;NO, DO ALLOC WRITE
	CALL	BUMP		;BUMP UNALC PARMS FOR NEXT PASS
	CALL	TSTHST		;HOST = REQ ?
	JZ	WN1		;YES, CON'T TO FILL THE BFR
	TSTFLG	F@WIP		;WRITE IN PROGRESS ?
	CNZ	XWRT		;YES, WRITE OLD DATA ONTO DISK
	CALL	SETHST		;MAKE HOST = REQ
;
WN1:	TSTFLG	F@RDOP		;INTERVENING READ ?
	CNZ	XREAD		;YES, READ OLD UNALC DATA
	JMP	WXFER		;MOVE DATA TO BFR, EXIT
;
;   ALLOCATED WRITE
;
WALC:	XRA	A
	STA	U@CNT		;CLEAR UNALC RECORD COUNT
	CALL	TSTHST		;HOST = REQ ?
	JZ	WXFER		;YES -  MOVE DATA TO BFR, EXIT
	TSTFLG	F@WIP		;WRITE IN PROGRESS ?
	CNZ	XWRT		;YES, WRITE OLD DATA ONTO DISK
	CALL	SETHST		;MAKE HOST = REQ
	CALL	XREAD		;READ IN ALLOCATED DATA
	JMP	WXFER		;MOVE NEW DATA IN BFR, EXIT
;
;   DIRECTORY WRITE
;
WDIR:	XRA	A
	STA	U@CNT		;CLEAR UNALC RECORD COUNT
	CLRFLG	F@RDOP		;RESET 'RDOP' FLAG
	TSTFLG	F@WIP		;WRITE IN PROGRESS ?
	CNZ	XWRT		;YES, WRITE OLD DATA ONTO THE DISK
	CLRFLG	F@WIP		;CLEAR 'WIP' FLAG
	CALL	SETHST		;MAKE HOST = REQ
	CALL	XREAD		;READ DIR DATA
	RNZ			;RIF READ ERROR
	CALL	GETDMA		;GET DMA ADR, SET POINTERS
	CALL	BFROUT		;MOVE NEW DIR DATA INTO BFR
	JMP	XWRT		;WRITE DIR DATA, EXIT




;	---TEST HOST---
;
;   DETERMINES IF THE HOST DISK ADDRESS IS
;   THE SAME AS THE REQUESTED DISK ADDRESS.
;
;
TSTHST:	LXI	H,H@TRK
	CALL	DSKCMP		;TRACK & DRIVE THE SAME ?
	RNZ			;RIF NO
	LDA	RSA
	CMP	M		;SECTOR THE SAME ?
	RET			;IF A = 0 THEN THEY MATCH




;	---SET HOST---
;
;   SETS THE HOST DISK ADDRESS TO BE THE
;   SAME AS THE REQUESTED DISK ADDRESS.
;
;
SETHST:	LDA	R@DSK
	STA	H@DSK		;DRIVE #
	LHLD	R@TRK
	SHLD	H@TRK		;TRACK ADR
	LDA	RSA
	STA	H@SEC		;SECTOR ADR
	RET




;	---BUMP---
;
;   BUMPS PARAMETERS FOR UNALLOCATED WRITES.
;   PARMS ARE CHANGED FOR THE NEXT PASS THRU
;   THE CODE (NOT THE CURRENT PASS).
;
;
BUMP:	LXI	H,U@CNT		;UNALC RECORD COUNT
	DCR	M		;  DECR IT
	DCX	H		;H/L = U@SEC
	INR	M		;  INCR IT
	MOV	A,M
	CPI	CPMSPT		;CPM SECTORS PER TRACK
	RC			;RIF STAY ON SAME TRACK
; OVERFLOW TO NEXT TRACK
	MVI	M,0		;RESET SECTOR ADR
	LHLD	U@TRK
	INX	H		;INCR TRACK ADR
	SHLD	U@TRK
	RET




;	---GET DMA ADDRESS---
;
;   SETS THE CONTROLLER BUFFER ADDRESS TO THE CORRECT
;   STARTING POINT. ALSO SETS B=128 & H/L=DMADR.
;
;
GETDMA:	LHLD	DMADR		;DMA ADR
	IF	Z80
	MVI	C,CTDP		;FOR BLOCK I/O
	ENDIF
	MVI	B,128		;B = COUNT
	MVI	A,DBENB
	OUT	CTCSR		;ENB DATA BFR
	LDA	R@SEC		;REQUESTED SECTOR
	RRC
	MVI	A,0
	JNC	GET1		;JIF USE 1ST HALF OF BFR
	MOV	A,B		;   USE 2ND HALF OF BFR
GET1:	OUT	CTBFR		;SET CTLR DATA BFR ADR
	RET




;	---BUFFER INPUT---
;
;   READ 128 BYTES FROM THE CONTROLLER BUFFER
;
;
BFRIN	EQU	$
	IN	CTDP		;PRIME DATA INPUT
	IF	Z80		;IF USING A Z-80
	INIR			;BLOCK INPUT
	RET
	ELSE			;IF USING AN 8080
BFRIN1:	IN	CTDP		;GET DATA
	MOV	M,A		;PUT DATA INTO RAM
	INX	H
	DCR	B
	JNZ	BFRIN1
	RET
	ENDIF




;	---BUFFER OUTPUT---
;
;   OUTPUT 128 BYTES TO THE CONTROLLER BUFFER
;
;
BFROUT	EQU	$
	IF	Z80		;IF USING A Z-80
	OUTIR			;BLOCK OUTPUT
	RET
	ELSE			;IF USING AN 8080
	MOV	A,M		;GET DATA FROM RAM
	OUT	CTDP		;PUT DATA INTO BFR
	INX	H
	DCR	B
	JNZ	BFROUT
	RET
	ENDIF


	PAGE



;	======================================
;	** HARD DISK I/O & SUPPORT ROUTINES **
;	======================================




;	---READ A BLOCK---
;
;
XREAD:	LXI	H,RTBL		;READ CMD TBL
	CALL	DORW		;READ
;
XR1:	MVI	A,0
	RZ			;RIF READ/WRITE OK
	INR	A
	STA	ERFLG		;SET ERROR FLAG
	RET




;	---WRITE A BLOCK---
;
;
XWRT:	LXI	H,WTBL		;WRITE CMD TBL
	CALL	DORW		;WRITE A SECTOR
	JMP	XR1		;SET ERROR FLAG




;	---EXECUTE READ/WRITE COMMANDS---
;
;
DORW:	MOV	A,M
	STA	RETRY		;SET RETRY COUNT
	INX	H
	MOV	A,M
	OUT	CTCSR		;ENB CMP BFR
	INX	H
	MOV	A,M
	OUT	CTBFR		;SET CMP BFR ADR
	INX	H
	SHLD	CTA		;SAVE CMD TBL ADR
;
	CALL	XSEK		;SEEK TO NEW TRACK (IF REQUIRED)
	RNZ			;RIF SEEK FAILED
;
	LXI	H,RCA		;REAL CYL ADR
	MVI	B,3
DO1:	MOV	A,M
	OUT	CTDP		;PUT HDR INFO INTO CMP BFR
	INX	H
	DCR	B
	JNZ	DO1
	LDA	H@SEC		;HOST SECT ADR
	OUT	CTDP		;PUT INTO CMP BFR
;
DO2:	CALL	XRDY		;DRIVE READY ?
	RNZ			;  RIF NO
	XRA	A
	OUT	CBH
	OUT	CBL		;CLEAR CMD BUS
	MVI	A,TAG3
	OUT	DRCSR		;SET TAG3
	LHLD	CTA		;CMD TBL ADR
	MOV	A,M		;A = CNTL BANK
	INX	H
	MOV	B,A
	OUT	CTCSR		;SLCT CNTL BANK
	MOV	A,M
	OUT	CTBFR		;SET START ADR
	INX	H
	MOV	A,B
	ORI	START
	OUT	CTCSR		;START R/W CMD
;
DO3:	IN	CTCSR		;CTLR STATUS
	RRC
	JNC	DO3		;WAIT FOR DONE
	NOP			;DELAY FOR H/W
	IN	CTCSR		;GET NON-CHANGING STATUS
	MOV	B,A		;   SAVE IT
	XRA	A
	OUT	CTCSR		;STOP CTLR
	INR	A
	OUT	DRCSR		;CLEAR TAG3
	MOV	A,B
	ANA	M		;TEST CTLR STATUS (0=OK)
	MOV	B,A
	IN	DRCSR		;DRIVE STATUS
	ANI	10H
	CNZ	CLRDF		;CIF CLEAR DRIVE FAULT
	ORA	B		;SET/CLEAR ERROR FLAG (0=OK)
	RZ			;RIF READ/WRITE OK
	LXI	H,RETRY
	DCR	M		;DECR RETRY COUNT
	JNZ	DO2		;JIF RETRY READ/WRITE
	ORI	1		;SET ERROR FLAG
	RET			;TAKE ERROR EXIT





;	---HOME---
;
;
XTKZ:	TSTFLG	F@WIP
	RNZ			;RIF WRITE IN PROGRESS
	STA	F@RIP
	STA	F@RDOP		;MAKE HOST INACTIVE
	RET




;	---REZERO---
;
;
RTZ:	MVI	A,1
	OUT	CLRSEK		;CLEAR SEEK END F/F
	XRA	A
	OUT	CBH
	MVI	A,D@TKZ
	OUT	CBL		;SET CMD BUS = REZERO
	MVI	A,TAG3
	DI			;INHIBIT INTPS
	OUT	DRCSR		;SET TAG3
	MVI	A,DSTAG
	OUT	DRCSR		;CLEAR TAG3
	IF	ENBINT		;IF INTPS ENABLED
	EI
	ENDIF
	JMP	WSC		;WAIT FOR SEEK DONE, EXIT




;	---SEEK---
;
;
XSEK:	CALL	XRDY		;DRIVE READY ?
	RNZ			;  RIF NO
; COMPUTE REAL CYL ADR
	LHLD	H@TRK		;HOST (REQ) TRACK
	LDA	H@DSK		;HOST DISK
	ANI	1
	JZ	XSEK1		;JIF DRIVE = 0
	LXI	D,LDS		;LOGICAL DISK SIZE
	DAD	D
;
XSEK1:	XCHG			;D/E = REQ TK ADR
	LHLD	RCA		;CURRENT CYL ADR
	MOV	A,L		;TEST TO SEE IF THE NEW CYLINDER
	SUB	E		;   ADDRESS IS THE SAME AS THE
	MOV	B,A		;   CURRENT CYLINDER ADDRESS
	MOV	A,H
	SBB	D
	ORA	B
	MOV	B,A
	JZ	XSEK2		;JIF SAME CYL ADR
	XCHG
	SHLD	RCA		;SAVE NEW CYL ADR
XSEK2:	LDA	H@DSK		;REQ DRIVE #
	ORA	A		;CLEAR CY
	RAR			;DIVIDE BY 2 FOR REAL HEAD #
	MOV	C,A
	LXI	H,RHN
	SUB	M		;REQ HEAD # = CURR HEAD ?
	ORA	B		;IF CYL & HEAD SAME AS LAST
	RZ			;  TIME, DON'T SEEK
	MOV	M,C		;SET NEW REAL HEAD #
	CALL	XSEL		;SLCT PROPER HEAD
;
XSEK3:	CALL	XRDY		;DRIVE READY ?
	RNZ			;  RIF NO
	MVI	A,1
	OUT	CLRSEK		;CLEAR SEEK END F/F
	LHLD	RCA		;REAL CYL ADR
	MOV	A,H
	OUT	CBH		;SET CYL ADR, MSB
	MOV	A,L
	OUT	CBL		;  * LSB
	MVI	A,TAG1
	DI			;INHIBIT INTPS
	OUT	DRCSR		;SET TAG1
	MVI	A,DSTAG
	OUT	DRCSR		;CLEAR TAG1
	IF	ENBINT		;IF INTPS ENABLED
	EI
	ENDIF
	CALL	WSC		;WAIT FOR SEEK DONE
	RZ			;RIF SEEK OK
	CALL	RTZ		;RESTORE THE HEAD
	RNZ			;RIF RESTORE FAILED
	LXI	H,RETRY
	DCR	M		;DECR RETRY CNT
	JNZ	XSEK3		;RETRY SEEK
	ORI	1		;SET ERROR FLAG
	RET			;TAKE ERROR EXIT




;	---WAIT FOR SEEK COMPLETE---
;
;
WSC:	IN	CTCSR		;CTLR STATUS
	ANI	40H		;SEEK END INTP
	JZ	WSC		;WAIT FOR SEEK DONE
	IN	DRCSR		;DRIVE STATUS
	ANI	22H		; 'SEEK ERR'  &  'ON CYL'
	XRI	2		;INVERT +ONCYL BIT
	RET			;IF A=0, SEEK OK




;	---HEAD SELECT---
;
;
XSEL:	XRA	A
	OUT	CBH		;CLEAR CMD BUS, MSB
	LXI	H,HSTBL
	LDA	RHN		;REAL HEAD # (0...N)
	CALL	BINDX		;GET VOLUME/HEAD #
	OUT	CBL		;  SET BOTH
	MVI	A,TAG2
	DI			;INHIBIT INTPS
	OUT	DRCSR		;SET TAG2
	MVI	A,DSTAG
	OUT	DRCSR		;CLEAR TAG2
	IF	ENBINT		;IF INTPS ENABLED
	EI
	ENDIF
	RET
;
;   HEAD SELECT TABLE
;
HSTBL:	DB	0		; 0 = V0,H0
	DB	0+VOL1		; 1 = V1,H0
	DB	1+VOL1		; 2 = V1,H1
	DB	2+VOL1		; 3 = V1,H2
	DB	3+VOL1		; 4 = V1,H3
	DB	4+VOL1		; 5 = V1,H4




;	---DRIVE READY TEST---
;
;
XRDY:	IN	DRCSR		;DRIVE STATUS
	ANI	9		;'UNIT SLCT'  &  'DRIVE RDY'
	XRI	9		;INVERT BOTH
	RET			;IF A = 0, DRIVE RDY




;	---CLEAR DRIVE FAULT---
;
;
CLRDF:	XRA	A
	OUT	CBH
	MVI	A,D@CLR
	OUT	CBL		;SET CMD BUS = FAULT CLEAR
	MVI	A,TAG3
	DI			;INHIBIT INTPS
	OUT	DRCSR		;SET TAG3
	MVI	A,DSTAG
	OUT	DRCSR		;CLEAR TAG3
	IF	ENBINT		;IF INTPS ENABLED
	EI
	ENDIF
	RET




;	---COMMAND TABLES---
;
;
;   WRITE
;
WTBL:	DB	5		;RETRY COUNT
	DB	5		;CMP BFR ENB
	DB	0EAH		;CMP BFR ADR
	DB	BANK1		;CNTL BANK
	DB	0CCH		;START ADR
	DB	0EH		;STATUS MASK
;
;   READ
;
RTBL:	DB	10		;RETRY COUNT
	DB	4		;CMP BFR ENB
	DB	0EAH		;CMP BFR ADR
	DB	BANK0		;CNTL BANK
	DB	0CCH		;START ADDRESS
	DB	0EH		;STATUS MASK




;	---DISK ADDRESS COMPARE---
;
;   COMPARES EITHER THE HOST DISK ADDRES OR THE UNALLOCATED
;   DISK ADDRESS AGAINST THE REQUESTED DISK ADDRESS.
;
;
DSKCMP:	LXI	D,R@TRK		;REQUESTED TRACK
	MVI	B,3
DC1:	LDAX	D
	CMP	M
	RNZ			;RIF ADRS NOT EQUAL
	INX	H
	INX	D
	DCR	B
	JNZ	DC1
	RET


	PAGE


;	==========================
;	** FLOPPY DISK ROUTINES **
;	==========================




;	---SECTOR READ---
;
;
FREAD:	CALL	FSEK		;DO TRACK SEEK
	RNZ			;RIF HAD SEEK ERROR
	MVI	D,10		;C = RETRY COUNT
;
FRD1:	IN	D@DP		;CLEAR 'DRQ'
	LDA	RFD		;F/D DRIVE #
	ORI	AUTO		;TURN ON AUTO WAIT BIT
	OUT	D@CP1		;TURN ON AUTO WAIT
	LHLD	DMADR		;H/L = DMA ADR
	MVI	B,128
	MVI	A,C@RD
	OUT	D@CSR		;ISSUE READ CMD
FRD2:	IN	D@DP		;INPUT DATA FROM FDC
	MOV	M,A		;PUT DATA INTO RAM
	INX	H		;INCR ADR
	DCR	B
	JNZ	FRD2
	CALL	FDW		;WAIT FOR CMD TO FINISH
	ANI	S@RD		;TEST FOR  READ ERRORS
	RZ			;RIF READ OK
	ANI	NRB		;TEST FOR NOT READY
	RNZ			;RIF DRIVE NOT READY
	DCR	D		;DECR RETRY COUNT
	JNZ	FRD1		;RETRY READ
	JMP	SEF		;SET ERROR FLAG, EXIT



;	---SECTOR WRITE---
;
;
FWRT:	CALL	FSEK		;DO TRACK SEEK
	RNZ			;RIF HAD SEEK ERROR
	MVI	D,10		;C = RETRY COUNT
;
FWRT1:	IN	D@DP		;CLEAR 'DRQ'
	LDA	RFD		;DRIVE #
	ORI	AUTO		;TURN ON AUTO WAIT BIT
	OUT	D@CP1		;TURN ON AUTO WAIT
	LHLD	DMADR		;H/L = DMA ADR
	MVI	B,128
	MVI	A,C@WRT
	OUT	D@CSR		;ISSUE WRITE COMMAND
FWRT2:	MOV	A,M		;GET DATA FROM RAM
	OUT	D@DP		;OUTPUT TO FDC
	INX	H		;INCR ADR
	DCR	B
	JNZ	FWRT2
	CALL	FDW		;WAIT FOR CMD TO FINISH
	ANI	S@WRT		;FOR WRITE ERRORS
	RZ			;RIF WRITE OK
	ANI	NRB + WPROT	;NOT RDY & CAN'T WRITE
	RNZ			;RIF DRIVE NOT READY
	DCR	D		;DECR RETRY COUNT
	JNZ	FWRT1		;RETRY WRITE
	JMP	SEF		;SET ERROR FLAG, EXIT




;	---TRACK ZERO SEEK---
;
;
FTKZ:	CALL	FSEL		;SLCT NEW DRIVE (IF REQUIRED)
	RNZ			;RIF HAD ERROR
	CALL	FRDY		;ENSURE READY TO SEEK
	RNZ			;EXIT, DRIVE NOT READY
	MVI	A,C@TKZ
	CALL	DOCMD		;DO SEEK, WAIT FOR DONE
	ANI	S@TKZ		;TEST FOR ERRORS
	XRI	4		;INVERT +TKZ BIT
	RET



;	---TRACK SEEK---
;
;
FSEK:	CALL	FSEL		;SLCT NEW DRIVE (IF REQUIRED)
	RNZ			;RIF HAD ERROR
	CALL	FRDY		;ENSURE DRIVE RDY TO SEEK
	RNZ			;ABORT, DRIVE NOT RDY
	LDA	R@SEC
	OUT	D@SEC		;SET SECTOR ADR
	IN	D@TK		;CURR TK ADR
	LXI	H,R@TRK
	CMP	M		;SAME TRACK ?
	RZ			;YES, DON'T MOVE HEAD
	MVI	D,3		;SET RETRY COUNT
;
FSEK1:	MOV	A,M
	OUT	D@DP		;SET TK ADR FOR SEEK
	MVI	A,C@SEEK
	CALL	DOCMD		;DO SEEK, WAIT FOR DONE
	ANI	S@SEEK		;TEST FOR ERRORS
	RZ			;EXIT SEEK OK
	MOV	E,M		;SAVE CURRENT TRACK ADR
	CALL	FTKZ		;DO TRACK ZERO SEEK
	MOV	M,E		;RESTORE TRACK ADR
	RNZ			;EXIT IF TKZ ERROR
	DCR	D		;DECR RETRY COUNT
	JNZ	FSEK1		;RE-SEEK
	JMP	SEF		;SET ERROR FLAG, EXIT




;	---DRIVE SELECT---
;
;
FSEL:	LDA	R@DSK		;REQUESTED DISK
	SUI	FFD		;ADJUST FOR F/D
	CPI	MAXFD
	JNC	SEF		;JIF INVALID DRIVE
	LXI	H,CFD
	MOV	B,M		;B = CURR DRIVE #
	CMP	B
	RZ			;RIF SAME DRIVE
	MOV	M,A		;UPDATE CURR DRIVE #
	MOV	C,A		;C = NEW DR #
	MOV	L,A		;L = SHIFT COUNT (DRIVE #)
	INR	L
	XRA	A
	STC
FSEL1:	RAL			;SHIFT DRIVE SLCT BIT
	DCR	L
	JNZ	FSEL1
	ORI	MAXI		;FOR 8" DRIVES
	OUT	D@CP1		;SLCT NEW DRIVE
	STA	RFD		;  SAVE FOR FUTURE USE
	LXI	H,TKSAV		;TK ADR SAVE AREA
	MOV	E,B
	MVI	D,0
	DAD	D		;H/L = OLD DR SAVE
	IN	D@TK		;GET CURR TK ADR
	MOV	M,A		;SAVE TK ADR OF OLD DR
	LXI	H,TKSAV
	MOV	E,C
	DAD	D		;H/L = NEW DR SAVE
	MOV	A,M		;A = TK ADR OF NEW DRIVE
	OUT	D@TK		;SET 1791 TK REG
	OUT	D@DP		;PUT TK ADR INTO DATA REG
	CALL	FRDY		;DRIVE READY ?
	RNZ			;NO, EXIT
	MVI	A,C@SUN+STEP	;YES, DO SEEK TO SAME TRACK
	CALL	DOCMD		;   TO UNLOAD THE HEAD
	XRA	A		;NO ERRORS !
	RET




;	---EXECUTE CONTROLLER COMMANDS---
;
;
DOCMD:	OUT	D@CSR		;ISSUE 1791 CMD
				;  * FALL THRU TO 'FDW'




;	---WAIT FOR DONE---
;
;
FDW:	IN	D@CP1		;CNTL PORT #1
	RRC			;CY = INTRQ
	JNC	FDW		;WAIT FOR DONE
	IN	D@CSR		;GET 1791 STATUS
	RET




;	---READY TEST---
;
;
FRDY:	IN	D@CSR		;1791 STATUS
	ANI	80H		;TEST READY BIT
	RET


	PAGE


;	======================================
;	** SUBROUTINES, MESSAGES, & STORAGE **
;	======================================




;	---INITIALIZE FLAGS & VARIABLES---
;
;
INIT:	LXI	H,VARBGN	;START OF STORAGE
	MVI	B,VARSIZ	;  SIZE OF SAME
	XRA	A
INIT1:	MOV	M,A		;CLEAR FLAGS & VARIABLES
	INX	H
	DCR	B
	JNZ	INIT1
	RET




;	---PRINT ROUTINE---
;
;
PRNT:	MOV	A,M		;PRINT CHAR
	INX	H		;FOR NEXT CHAR
	ORA	A
	RZ			;RIF DONE
	MOV	C,A
	CALL	CO		;PRINT A CHAR
	JMP	PRNT		;DO NEXT CHAR




;	---CARRIAGE RETURN, LINE FEED----
;
;
CRLF:	MVI	C,CR
	CALL	CO		;PRINT A CR
	MVI	C,LF
	JMP	CO		;PRINT A LF, EXIT




;	---BYTE WIDE INDEX---
;
;
BINDX:	PUSH	D		;SAVE D/E
	MOV	E,A		;A = ENTRY #
	MVI	D,0
	DAD	D		;INDEX INTO TBL
	MOV	A,M		;A = TABLE ENTRY
	POP	D		;RESTORE D/E
	RET




;	---MESSAGES---
;
;
MSG1:	DB	CR,LF,BELL
	DB	'* Can''t Boot -- Hit any key to Retry ',0



CODEND	EQU	$		;DEFINES END OF CODE






;	---PROGRAM STORAGE---
;
;
;
VARBGN	EQU	$		;DEFINE BEGINNING OF VARIABLE STORAGE
;
R@TRK:	DS	2		;REQUESTED TRACK ADDRESS
R@DSK:	DS	1		;   DRIVE #
R@SEC:	DS	1		;   SECTOR ADDRESS
;
U@TRK:	DS	2		;UNALLOCATED TRACK ADDRESS
U@DSK:	DS	1		;   DRIVE #
U@SEC:	DS	1		;   SECTOR ADDRESS
U@CNT:	DS	1		;   RECORD COUNT
;
H@TRK:	DS	2		;HOST (CDC) TRACK ADDRESS
H@DSK:	DS	1		;   DRIVE #
H@SEC:	DS	1		;   SECTOR ADDRESS
;
SECNT:	DS	1		;SECTOR COUNT (FOR WARM BOOT)
RETRY:	DS	1		;RETRY COUNT
CTA:	DS	2		;COMMAND TABLE ADDRESS
ERFLG:	DS	1		;ERROR FLAG
TKSAV:	DS	4		;TRACK ADDRESS SAVE AREA (F/D)
DMADR:	DS	2		;BUFFER (DMA) ADDRESS
WRTMODE: DS	1		;WRITE MODE
CHDT:	DS	2		;CURRENT HARD DISK TRACK
;
RCA:	DS	2		;REAL CYLINDER ADDRESS
RHN:	DS	1		;REAL HEAD NUMBER
RSA:	DS	1		;REAL SECTOR ADDRESS
;
F@RDOP:	DS	1		;READ OPERATION
F@RIP:	DS	1		;READ IN PROGRESS
F@WIP:	DS	1		;WRITE IN PROGRESS
;
VARSIZ	EQU	$-VARBGN	;DEFINES SIZE OF VARIABLE STORAGE
;
CFD:	DS	1		;CURRENTLY SELECTED F/D DRIVE
RFD:	DS	1		;F/D SELECT INFO




;	---BDOS STORAGE---
;
;
DIRBFR:	DS	128		;DIRECTORY BUFFER
;
FALL0:	DS	31		;F/D ALLOCATION BUFFER, DRIVE 0
FALL1:	DS	31		;   DRIVE 1
FCHK0:	DS	16		;F/D CHECK BUFFER, DRIVE 0
FCHK1:	DS	16		;   DRIVE 1
;
HALL0:	DS	206		;CDC ALLOCATION BUFFER, LOGICAL DRIVE #0
HALL1:	DS	206		;   DRIVE #1
HALL2:	DS	206		;   DRIVE #2
HALL3:	DS	206		;   DRIVE #3
;
	IF	DTYPE GT 1
HALL4:	DS	206		;CDC LOGICAL DRIVE #4
HALL5:	DS	206		;   DRIVE #5
HALL6:	DS	206		;   DRIVE #6
HALL7:	DS	206		;   DRIVE #7
	ENDIF
;
	IF	DTYPE GT 2
HALL8:	DS	206		;CDC LOGICAL DRIVE #8
HALL9:	DS	206		;   DRIVE #9
HALL10:	DS	206		;   DRIVE #10
HALL11:	DS	206		;   DRIVE #11
	ENDIF
;
LAST	EQU	$



	END
