;*****************************************************************************
; MstrLph_p.asm
; Master Line Protocol Handler 
; This module contains code for protected mode masters.
;*****************************************************************************


%SET(Aug87JA,1)
%set(CTOSp,1)
%SET(Debug,0)
%SET(DebugChain,0)
%SET(LogTime,0)
%SET(LogErrors,0)
%SET(XBlockAudit,0)
%SET(logXb,0)
%SET(logIn,0)
%SET(logOut,0)

%SET(NGen,1)
%SET(MF,0)
%SET(Master,1)

%INCLUDE(:f1:MstrEqu.idf)
;INCLUDE(:f1:Cdtequ.edf)      ***obsoleted ****
; EXTERNAL PROCEDURES
EXTRN PaFromP:far

;*****************************************************************************
;  PUBLIC PROCEDURES
;*****************************************************************************
	PUBLIC SigIsr					;Bit IO Interrupt Service Routine
	PUBLIC SigIsr1					;Bit IO Interrupt Service Routine

;*****************************************************************************
;  LITERALS
;*****************************************************************************
apa 				EQU 80h		;All Parties Address
idle 				EQU 0		;Send Abort if Tx underrun
bop 				EQU 0		;Bit Oriented Protocol (as opposed to Byte)
burst2				EQU 1000h	;Cirrus chip dma burst 2 word - R&B
bEnableEarChan0 	EQU 11h		;ear enabled in word mode - GateArray
dcd  				EQU 040H
dcdIntEnb 			EQU 08h
dmarcv				EQU 02h		;flushing FIFO command to Cirrus - R&B
endxmit				EQU 05h		;Stop xmit, keep dmatxe on - R&B
exchClusterReceiver EQU 17
FifoEmptyBit		EQU 0100h	;Cirrus Cluster FIFO Status bit - R&B
hbe					EQU 01		;- R&B
maxWaitingTicks 	EQU 40		;1 second max between polls
NewGen 				EQU 9
rcvEnb  			EQU 022h
rEom  				EQU 02h
rErr  				EQU 80h
rOvrn  				EQU 08h
rAbort  			EQU 04h
rts  				EQU 080h
t1 					EQU 4
t2 					EQU 5
tErr  				EQU 80h
tSom  				EQU 01
tEom  				EQU 02
tUndrn  			EQU 010h
txwcen				EQU 0401h	;Cirrus chip transmit word count enable - R&B
statInt  			EQU 2
xmtDone  			EQU 020h
xmtEnb  			EQU 044h
xmtEnbNoDma 		EQU 40h

;*****************************************************************************
; Include common code
;*****************************************************************************
%INCLUDE(MstrLph_all.asm)

;*****************************************************************************
;  EXTERNAL DATA
;*****************************************************************************
DGroup GROUP DATA
Data SEGMENT PUBLIC 'DATA'

EXTRN DmaEarEnable:WORD
EXTRN ExtCtlReg:WORD
EXTRN OCW1_8259:WORD
EXTRN OCW2_8259:WORD
EXTRN processorType:BYTE
EXTRN hardwareType:BYTE
EXTRN shCPUSpeed:BYTE
EXTRN fHighSpeed:BYTE
EXTRN TeleclusterTiming: WORD
EXTRN vf: BYTE
%INCLUDE(:f1:vfequ.idf)

;*****************************************************************************
;  GLOBAL DATA
;*****************************************************************************
PUBLIC ccrRcvEnb, ccrXmtEnb, ccrStopDMA						;R&B
PUBLIC fMaskParallelDMA, parallelDmaMask, parallelDmaWrdCnt	;R&B
PUBLIC parallelMask, parallelUnmask							;R&B
PUBLIC SafeOutSbAddress
SafeOutSbAddress	DW 0

ccrRcvEnb			DW 0	;Set up by init code
ccrXmtEnb			DW 0	;Set up by init code
ccrStopDMA			DW 0	;Set up by init code
parallelDmaMask		DW 0	;Set up by init code (initClstr)	R&B ...
parallelDmaWrdCnt	DW 0	;Set up by init code (initClstr)
fMaskParallelDMA	DB 0	;Set up by init code (initClstr)
parallelMask		DB 0	;Set up by init code (initClstr)
parallelUnMask		DB 0	;Set up by init code (initClstr)	... R&B

PUBLIC lastByteRead
lastByteRead DB 0
PUBLIC rgLcb
EVEN
rgLcb		DB 2*sLcb DUP (0)			; define space for 2 lcb's

Data ENDS
MstrLph SEGMENT PUBLIC 'CODE'
	ASSUME CS: MstrLph, DS: Dgroup
SetupWrite:
PUBLIC SetupWrite
;*****************************************************************************
; Continuation of StartPoll following machine independent code
;*****************************************************************************
	MOV  DX,lcb_DmaMaskReg[BP]			;dmaMask
	MOV  AL,lcb_commDmaDis[BP]
	OUT  AL,DX							;disable Dma in case it's enabled
	JMP  $+2                            ;flush pipe

; We will write from the ackOutBuf or the XBlock. The lineStateTable tells 
; where we write from. If lstCbAckOut <> 0, then we write from the ackOutBuf. 
	XOR  CX,CX
	OR   CL,lstCbAckOut[SI]				;CX = count from ackOutBuf
	JZ   WriteFromXBlock
	JMP  TestNeedXBlock					;in MstrLph_all.asm

WriteFromAckoutBuf:
; Continue here from TestNeedXBlock in MstrLph_all.asm
; On return AX contains the frametype and station ID
	MOV  ES:[0],AX						;Store stationFrame in ackOutBuf
	CMP  AH,093h						;does type equal SNRM
	JNE  NotSNRM
;
; We must now initialize Line Speed on SNRMs in ACKOUTBUF
;
 	MOV AX,lcb_KHLineSpeed[BP]
 	MOV ES:[2],AX
	JMP WriteFromAckOutBufDMAAddr
NotSNRM:
;
; The length of the small frames is now 4 bytes 
;
	CMP CL,2 ;the length noticed in LineStateTable
	JNE WriteFromAckOutBufXID
	XOR AX,AX
 	MOV ES:[2],AX ;2 NULL bytes added
 	INC CX
 	INC CX
 	JMP WriteFromAckOutBufDMAAddr

WriteFromAckOutBufXID:
;
; We must now initialize again the XID Frame
;
 	MOV AL,dctOsType[BX]		; fix boot rom bug send back wstype
 	MOV ES:[XIDOsType],AL
 	MOV ES:[XIDRevisionLevel],91H

WriteFromAckOutBufDMAAddr:

; Our count to write is always in bytes (an even number).  We convert it to  
; words by shifting right one. The Dma will transfer one more than the count  
; programmed.  But the state machine will turn off the Dma when the count is  
; zero.  So we send the count that we really want to send.

	PUSH BX						; Save DCT index
	MOV  BX,lcb_ackOutDmaEar[BP]
	MOV  AX,lcb_ackOutDmaAddr[BP]
	TEST vf_f82380, 1			; set up byte count and addr for 82380
	JZ   OutDmaWrdCnt
	ADD  CX, 2					; inc count of bytes for fifo being trashed
	SUB  AX, 2					;  on Cirrus,  adjust address accordingly 
	JMP  SHORT OutDmaCount
OutDmaWrdCnt:
	SHR  CX,1					;word count on NGen
	TEST vf_f4MBit, 0FFh		;R&B ... If we're a cirrus, need to
	JZ   OutDmaCount			;  inc count of words to adjust for dec of
	INC  CX						;  DMA address because the first word put in
	DEC  AX						;  FIFO is going to get trashed by the 2652.
OutDmaCount:					;... R&B
	MOV  DL,Lcb_ModeSFromM[BP]
	CALL SetUpTxRxDMA
	POP  BX									; Retieve DCT index
	JMP  DMAProgrammed
	
WriteFromXBlock:
; We are writing from an XBlock. dct.saXBlockOut points to the XBlock that we 
; write from.  The XBlock contains the Dma word address and count.  See if we 
; can receive an IFrame in response.

	CMP  dctLineState[BX],stateSendIFrameMasterReady
	JNE  OrIFrameSeqBits
; We can receive an IFrame in response.  Check to see if we have an XBlock to 
; read into.
	CMP  lcb_saXBlockCurrent[BP],0
	JNE  OrIFrameSeqBits
; See if an XBlock is available.  If so, unchain it from the free list.
	CALL GetXBlock
	JNZ  OrIFrameSeqBits				
; If we don't have an XBlock available, change our state so we won't read into 
; an XBlock.
	MOV  SI,stateSendIFrame
	MOV  dctLineState[BX],stateSendIFrame

OrIFrameSeqBits:
PUBLIC OrIFrameSeqBits
;
; If this is an IFrame, we must OR in the sequence numbers. If this is not an 
; IFrame, dctNR and NS will be zero.
;
; Bug in error recovery where linestate says write from xblock but xblock
; already sent.  Reset state to workstationready.
;
	MOV  ES,dctSaXBlockOut[BX]			;ES points to XBlock
	CMP  dctSaXBlockOut[BX],0			;Xblock to write from?
	JNE	 OrIFrameOK
	MOV	 SI,stateWorkstationReady
	MOV  dctLineState[BX],stateWorkstationReady
	JMP  TestNeedXBlock					;in MstrLph_all.asm

OrIFrameOK:
	MOV  AX,dctNRNS[BX]					;AL = NR, AH = NS
	SHR  AH,4							;NS to be in bits 3 2 1
	OR   AH,AL							;NR to be in bits 7 6 5
	OR   AH,lstFrameType[SI]
	MOV  AL,dctStation[BX]				;AL = station we are polling
	MOV  ES:[xbStation],AX				;store frame type and sequence numbers

%IF (%Debug) THEN (;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
	MOV  DI,lcb_logBufferIndex[BP]
	MOV  logFrameOut[DI],AX
)FI ;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

	PUSH BX						;Save dct index
	MOV  AX,ES:[xbDmaAddr]		;AX = dma word address
	MOV  CX,ES:[xbCbData]		;CX = count of bytes - GWH
	TEST vf_f82380, 1			;
	JNZ  NoWordShift
	SHR  CX,1					;CX = count of words
	TEST vf_f4MBit, 0FFh		;R&B ... If we're a cirrus, need to dec DMA
	JZ   WrdShiftDone			;  address because the first word put in the
	DEC  AX						;  FIFO is going to get trashed by the 2652.
	INC  CX						;  inc the count to correspond to addr dec
								;... R&B
	JMP  Short WrdShiftDone		;  inc count of words to adjust for
NoWordShift:					;... R&B
	ADD  CX, 2					; Since 82380 is cirrus fix count and addr
	SUB  AX, 2					; for fifo smash
WrdShiftDone:
	MOV  BX,ES:[xbDmaEar]		;BX = Upper 16 bits of buffer address.
	MOV  DL,Lcb_ModeSFromM[BP]

	CALL SetUpTxRxDMA			; Program DMA controller.
	POP  BX						; Retrieve dct index

DMAProgrammed:	
	TEST vf_f82380, 1			;if 82380, already byte count
	JNZ  NoWordUnshift2
	SHL  CX,1					;Convert word count back to byte count
NoWordUnshift2:
	TEST vf_f4MBit, 0FFh		;R&B ... If we're a cirrus, need to
	JZ   NoWordUnshift			;  restore the value of count.
	DEC  CX
	DEC  CX
NoWordUnshift:					;... R&B
	MOV  lcb_cbWritten[BP],CX	;Save count of bytes written

; The Dma has been set up to write.  See if we need to wait for the dcd to 
; drop before we write.

StartWrite:
PUBLIC StartWrite
	MOV  AL,dctLineState[BX]			; see if booting or dumping
	CMP  AL,stateDumpingError			; highest dumping state
	JG   SL2
	CMP  AL,statePollingInactive 		; don't wait on snrms
	JE   SL2 
	PUSH CX
	MOV  AL,2
	MOV  CL,shCpuSpeed					; shcpuspeed set for processor type
	SHL  AL,CL
SL1:
	MOV  CX,1DH
	LOOP $
	DEC  AL
	JNZ  SL1
	POP  CX
SL2:
; Come here from interrupt routine when Dcd changes.
	MOV  DX,lcb_StatReg[BP]			; stat
	IN   AL,DX
	AND  AL,dcd
	JZ   NoWaitForDcdDrop

; We must wait for an interrupt for data carrier detect to change before we
; can continue. 
	MOV  AX,dcdIntEnb
	MOV  DX,lcb_ContReg[BP]			;cPort
	OUT  DX,AX
	JMP  $+2
	MOV  DX,lcb_StatReg[BP]			; stat
	IN   AX,DX
	AND  AL,dcd
	JZ   NoWaitForDcdDrop
	MOV  lcb_pollState[BP],stateWaitDcdDrop
%IF (%Debug) THEN (;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
	INC  nWaitDcd
)FI ;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
	RET

NoWaitForDcdDrop:
;	MOV  CX,TeleclusterTiming			; telecluster timing
;	LOOP $

; Set baud rate

	MOV  DX,lcb_TimerCtlReg[BP]			;8254 Control Register
	MOV  AL,lcb_timerMode[BP]			;Mode 3, Channel = 0 on SGEN
										;				   2 on NGEN
	OUT  DX,AL
	MOV  DX,lcb_timerCountReg[BP]		;8254 Count register
	MOV  AL,dctSioClock[BX]				;Divisor Low byte
	OUT  DX,AL
	MOV  AL,0							;Divisor High byte
	OUT  DX,AL

	MOV  DX,lcb_rdsr[BP]				;rdsrH
	IN   AX,DX    						;Reset Read Data Status Register
	JMP  $+2
	MOV  DX,lcb_DmaMaskReg[BP]			;Start Dma
	MOV  AL,lcb_commDmaEn[BP]
	OUT  DX,AL

	TEST fMaskParallelDma,0FFh			;R&B ... 
	JZ   TestForCirrus
	MOV  DX,parallelDmaMask				;Mask Parallel DMA
	MOV  AL,parallelMask
	OUT  DX,AL

TestForCirrus:							;... R&B
	TEST vf_f4MBit, 0FFh				;R&B ...
	JNZ  StartWriteCirrus

StartWriteIOGA:
PUBLIC StartWriteIOGA					;... R&B
	MOV  AL,tsom		
	MOV  DX,lcb_tdsr[BP]				;tdsrH
	OUT  DX,AX							;Send Start of Message to 2652
	JMP  $+2
	MOV  AX,rts							; rts  sends start of message
	MOV  DX,lcb_ContReg[BP]
	OUT  DX,AX
	JMP  $+2							;rts enables transmitter and
	PUSH CX								;Save CX 
	MOV  CL,dctSioClock[BX]				; timing loop to insure at least 8 ones
	MOV  CH,0
	LOOP $								;Wait for ones
	MOV  AX,rts+xmtEnbNoDma				;xmtEnbNoDma sets TxE in 2652
	OUT  DX,AX							;now send flags, DX still cPort
	
; The purpose of the loop$ is to allow time for the 2652 to send the flag
; characters. If we don't allow time to send flags, the workstation won't
; see the frame. 
	MOV  AL,dctSioClock[BX]
	SHR  AL,1					; timing loop to insure at least 4 flags
	MOV  CL,shCPUSpeed
	MOV  AH,0
	SHL  AX,CL
	MOV  CX,AX
	LOOP $						; Wait for flags
	POP  CX                     ; Restore CX

	MOV  AX,ccrXmtEnb			;Turn on the state machine
	OR   AL,rts
	MOV  DX,lcb_ContReg[BP]		;xmtEnb enables DMA
	OUT  DX,AX
	JMP  CalculatePollInterval	;in MstrLph_all.asm

; R&B ...This new code preloads the cirrus FIFO before enabling the transmit 
; clock.  This allows us to buffer up short frames and transmit them completely
; out of the FIFO.  This dramatically reduces the number of underruns and 
; timeouts.
StartWriteCirrus:
PUBLIC StartWriteCirrus
	MOV  AL,05				; Prepare ones - Get rid of first dummy byte
	MOV  DX,lcb_tdsr[BP]
	OUT  DX,AL
	MOV  AX,ccrXmtEnb		; Fill FIFO
	MOV  DX,lcb_ContReg[BP]	; Send start of message 
	OUT  DX,AX
	JMP  .+2			; Synchronize code execution to DMA transfers
	JMP  .+2			; Filling FIFO generates 1 or 2 bogus flags (TXC off)
	JMP  .+2			; Four Jumps due to 4-level FIFO (= 4 XFERS)
	JMP  .+2			; Now, the Cirrus chip has turned off TSOM
	PUSH CX				; Therefore, TXSO will become "1" within one flag time
	PUSH SI
;	XOR  SI,SI		; clear SI and
;Set SI so REP OUTSB won't fault - JF - 04/01/93
	MOV  SI,SafeOutSbAddress	
	CLD				; direction flag so REP OUTSB's don't fault - JM 05/21/92
	MOV  CH,0				; Filling FIFO generates 2 bogus flags
	MOV  CL,dctSioClock[BX]	;   so wait to discard them.
	SHL  CX,1
	MOV  DX,lcb_StatReg[BP]
	REP  OUTSB				; This OUT is harmless and provides a reliable,
	MOV  DX,lcb_ContReg[BP]	;   machine independent wait function.
	MOV  AX,ccrXmtEnb
	OR   AX,rts				; Send ones
	OUT  DX,AX
	MOV  CH,0				; Timming loop to insure at least 8 ones
	MOV  CL,dctSioClock[BX]
	SHL  CX,1
	MOV  DX,lcb_StatReg[BP]
	REP  OUTSB				; Use OUTSB for a delay again
	MOV  DX,lcb_tdsr[BP]
	MOV  AL,tsom
	OUT  DX,AL				; Send flags and get rid of second dummy byte
	MOV  CH,0				; Wait for flags
	MOV  CL,dctSioClock[BX]
	SHL  CX,3				; Telecluster fix - was 2
	MOV  DX,lcb_StatReg[BP]
	REP  OUTSB				; Use OUTSB for a delay again
	POP  SI					; Restore SI
	POP  CX					; Restore CX
	MOV  AL,0
	MOV  DX,lcb_tdsr[BP]	; tdsrH
	OUT  DX,AL				; Send data	
	JMP  CalculatePollInterval	;in MstrLph_all.asm
; ... R&B

WriteNotDone:
	RET						;Write has not finished.  Wait for next interrupt

Underrun:
; There isn't much we can do about underrun. It should not happen if the 
; hardware is working properly. If the workstation doesn't like the frame, it 
; will simply time out.  We need to stop the Cirrus chip DMA requests while 
; there is not request.
	PUSH AX							;R&B ...
	MOV  AX,ccrStopDMA				;If using a Cirrus chip, this enables the
	MOV  DX,lcb_ContReg[BP]			;  terminal counter, we load the counter
	OUT  AX,DX						;  with 1 to cause DMA to stop early.
	ADD  DX,2						;cPort + 2 is address of terminal count
									;  register on NGen with Cirrus chip
	MOV  AX,1						;Load 1 in counter
	OUT  AX,DX
	MOV  AX,rts OR endxmit OR txwcen;This turns off TXE. On the IOGA, DMA
									;  will finish after the stacker empties.
	MOV  DX,lcb_ContReg[BP]			;Restore Control Register port address
	OUT  DX,AX						;... R&B
	POP  AX
	INC  nUnderrun
%' This is a fix for Unisys UCF.
	MOV CL, AL						;Save error code
	MOV AX, tSom+tEom
	MOV DX, lcb_tdsr[BP]
	OUT DX, AX
	MOV AL, CL						;Restore error code
	JMP  TestXmitDone

WaitForWrite:	; R&B ...
; Come here if we have a short packet to send.
;
; 286i ethernet dma bug may cause write to never complete.
; Wait only for a reasonable time, 30us at high speed or 200us at low speed.

	MOV  CX,10
    MOV  DX,lcb_StatReg[BP]
WaitForWriteTimeout:
	IN   AX,DX
	TEST AL,xmtDone
	JNZ  WaitWriteCompleted
	LOOP WaitForWriteTimeout
; The write never completed.  Let StartRead figure it out.
	RET			; ...R&B

WaitWriteCompleted:
StartPoll ENDP
PUBLIC StartRead
StartRead PROC NEAR
;*****************************************************************************
; If we sent only four bytes, wait for the write to complete, and then
; start the read.
; Otherwise come here from the interrupt routine when the write has finished.
;******************************************************************************
	MOV  DX,lcb_StatReg[BP]	; stat
	IN   AL,DX
	TEST AL,tUndrn
	JNZ  Underrun
TestXmitDone:					; R&B ... Unmask parallel DMA if we masked 
	PUSH AX						;    it above.
	TEST fMaskParallelDma,0FFh
	JZ   ParallelDmaUnmasked
	MOV  DX,parallelDmaWrdCnt	; Is the word count not 0FFFFh?
	PUSHF
	CLI
	IN   AL,DX
	XCHG AL,AH
	JMP  $+2
	IN   AL,DX
	POPF
	CMP  AX,0FFFFh
	JZ   ParallelDmaUnmasked
	MOV  AL,parallelUnmask		; Unmask the channel
	MOV  DX,parallelDmaMask
	OUT  DX,AL
ParallelDmaUnmasked:
	POP  AX

	TEST AL,xmtDone
	JNZ  WaitForCRC
	JMP  WriteNotDone			; ... R&B

WaitForCRC:
	MOV  CX,10					; R&B ... Now wait for CRC & flags to be sent
	MOV  DX,lcb_StatReg[BP]
WaitingForCRC:
	IN   AX,DX
	TEST AX,4h
	JNZ  GotCRC
	LOOP WaitingForCRC
GotCRC:
	PUSH SI
	MOV  CH,0
	MOV  CL,dctSioClock[BX]
	SHL  CX,2
	MOV  DX,lcb_StatReg[BP]
;Set SI so REP OUTSB won't fault - JF - 04/01/93
	MOV  SI,SafeOutSbAddress	
	REP  OUTSB				; This OUTSB is harmless and provides a reliable,
	POP  SI					;   machine independent wait function.
							;... R&B
; Now that the 2652 has finished, we can turn off the state machine.  We must 
; leave the transmitter on so that the line will be busy until we are ready to 
; read.  The workstation will not attempt to answer until we turn the 
; transmitter off and turn the receiver on.
	MOV  AX,rts
	MOV  DX,lcb_ContReg[BP]				; cPort
	OUT  DX,AX
	JMP  $+2
	MOV  DX,Lcb_DmaMaskReg[BP]			; DmaMask
	MOV  AL,lcb_commDmaDis[BP]
	OUT  DX,AL							;disable Dma in case it's enabled
	JMP  $+2

	MOV  DL,lcb_modeSToM[BP]			; Set up 8237 DMA mode word value.

; Find out if we are reading into an xblock.  If so use appropiate buffer
; address.
	TEST BYTE PTR lstFlags[SI],fReadIntoXBlock
	JNZ  ReadToXBlock

; Here we are reading into AckInBuf.
	MOV  AX,ackInBufSize					;save max count to calculate count 
	MOV  lcb_cReadDataMax[BP],AX			;read when the read completes.
; Our max count to read is always in bytes (an even number).  We convert it 
; to words by shifting right one. The Dma will transfer one more than the 
; count programmed, so we then subtract one.
	TEST vf_f82380, 1						; if 82380, leave byte count
	JNZ  InDmaCount
	SHR  AX,1								; otherwise convert to words
InDmaCount:
	DEC  AX
	MOV  CX,AX								; CX holds the count.
	MOV  AX,lcb_ackInDmaAddr[BP]			; AX holds 16 bit base address
											; of ackinbuf.
	PUSH BX									; Save DCT index.
	MOV  BX,lcb_ackInDmaEar[BP]				; BX holds high order 16 bits
	CALL SetUpTxRxDMA						; Program DMA controller.
	POP  BX									; retrieve DCT index
	JMP  DMASetForRead						; Start programming cirrus chip.

ReadToXBlock:
	MOV  ES,lcb_saXBlockCurrent[BP]			;ES points to XBlock
	MOV  AX,sXBlk
	ADD  AX,2								;for (station,frame)
	MOV  lcb_cReadDataMax[BP],AX					
	TEST vf_f82380, 1						;if 82380, do not convert to words
	JNZ  InDmaCount2
	SHR  AX,1
InDmaCount2:
	DEC  AX
	MOV  CX,AX								; CX holds transfer count
	PUSH BX									; Save DCT index
	MOV  AX,ES:[xbDmaAddr]					; AX holds 16 bit base address of
											; XBlock to read into.
	MOV  BX,ES:[xbDmaEar]					; BX holds upper 16 bit address of
											; XBlock to read into.
	CALL SetUpTxRxDMA						; Program DMA Controller.
	POP  BX									; Retrieve DCT index.

; The Dma has been set up to read.

DMASetForRead:
	MOV	 DX,lcb_rdsr[BP]				;rdsrH
	IN   AX,DX    						;Reset Read Data Status Register

; Set up the parameter control register. The 2652 will not work unless we do 
; this each time.
	MOV	AL,apa + bop + idle
%IF (%Debug) THEN (;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
; For testing, we can send a bad CRC every n frames.
	CMP  nSendBadCrc,0
	JE   SendGoodCrc
	DEC  nSendBadCrc
	JNZ  SendGoodCrc
	MOV  DX,nSendBadCrcReload
	MOV  nSendBadCrc,DX
	OR   AL,1					;CRC Preset to zero -- normally preset to ones
	MOV  DI,lcb_logBufferIndex[BP]
	MOV  logBadCrc[DI],0BADCh
SendGoodCrc:
)FI ;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
	MOV	DX,lcb_pcSarH[BP]					;pcsarH
	OUT	AL,DX
	JMP  $+2

	MOV  AL,lcb_commDmaEn[BP]				;R&B ...
	MOV  DX,lcb_DmaMaskReg[BP]
	OUT  AL,DX								;Turn on the Dma
	JMP  $+2

	MOV  AX,ccrRcvEnb
	MOV  DX,lcb_ContReg[BP]					;Turn on the receiver
	OUT  DX,AX
	JMP  $+2								;... R&B

	MOV  lcb_pollState[BP],stateReading
	RET
StartRead ENDP
SigIsr PROC FAR
;*****************************************************************************
; This is the interrupt service routine for the 2652
; Each poll can have from one to three interrupts:
; (1) When we are ready to start the poll, we may have to wait for 
;     data carrier detect to drop.  If so, we wait for the dcd interrupt,
;     then start the write.
; (2) If the write is for more than two bytes, we will get an interrupt when
;     the write completes.  If the write is only two bytes, we will wait for it
;     to complete, then start the next read.
; (3) We always get an interrupt when the read completes.
;*****************************************************************************

; removed below because it causes stack fault.  MediateIntHandler doesn't do 
; anything anyhow (except cause a stack fault here), STI would suffice.
%IF (0) THEN (
;We mediate the cluster comm interrupt here so that other higher priority
;interrupts can be serviced to reduce latency.
	MOV  AX,0FFh						;Mediate will handle EOI for device
	PUSH AX
	CALL MediateIntHandler				;Returns enabled
)FI

    XOR  BP,BP      		           	;load rgLcb(0) Index
	JE   BPsetup						; always jump index first Lcb
SigIsr1	PROC FAR
	MOV  BP,sLcb						; address second Lcb
SigIsr1  ENDP
BPsetup:
	LEA	 BP,OFFSET DGROUP: rgLcb[BP]	; BP indexes Lcb	
SigLoop:
; BP is always set up when jumped to from bottom of interrupt loop
	MOV  BX,lcb_dctCurrent[BP]			;BX points to Dct being polled
	XOR  AX,AX
	MOV  AL,dctLineState[BX]									
	XCHG SI,AX							;SI points to state table
%IF (%Debug) THEN (;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
	CMP  BX,OFFSET DGROUP: listActiveHead
	JNE  DctCurrentOk
	MOV  AX,ercInternalConsistencyCheck
	PUSH AX
	CALL Crash
DctCurrentOk:
	MOV  DI,lcb_logBufferIndex[BP]
)FI ;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
	CMP  lcb_pollState[BP],stateWriting
	JL   IsrWaitDcd
	JG   IsrReading

IsrWriting:
PUBLIC IsrWriting
; Come here if a write just completed.  Go start the read.
%IF (%Debug) THEN (;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
	MOV  logIntWrite[DI],0FFh
)FI ;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
	CALL StartRead
	JMP  IsrDone

IsrWaitDcd:
PUBLIC IsrWaitDcd
; We have a possible race condition with StopRead.  If the read has already 
; been stopped due to a timeout, pollState will be stateIdle.
	CMP  lcb_pollState[BP],stateNoPoll
	JBE  IsrIdle

; Come here if we were waiting for data carrier detect to change.  The Dma has 
; already been set up for the next write.  Go start the write.
%IF (%Debug) THEN (;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
	MOV  logIntDcd[DI],0FFh
)FI ;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

; If the station is booting, we must not be too hasty about starting the write 
; when we get the data carrier detect interrupt.  The workstation may not be 
; ready to read yet, and we will get a timeout because the workstation never 
; got the message.  Why this is so is a mystery to me.
;	CMP  BYTE PTR dctLineState[BX],stateMasterNotReady
;	JAE  NoDelay
WaitBeforeWrite:
PUBLIC WaitBeforeWrite
;	MOV  CX,50
;	LOOP $
NoDelay:
	MOV  lcb_pollState[BP],stateWriting				;set state to Writing
	CALL StartWrite
	JMP  IsrDone
IsrIdle:
%IF (%Debug) THEN (;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
	MOV  logIntIdle[DI],0FFh
)FI ;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
	JMP  IsrDone

; It's a mystery to me why we get this interrupt. 
; We get an interrupt while reading @307Kb, but stat.eom is not yet set.
IsrError:
	XOR  AH, AH
	MOV  AL, dctSioClock[BX]
	MOV  CL, shCpuSpeed					; shcpuspeed set for processor type
	SHL  AX, CL
	XCHG AX, CX
IELoop:
	IN   AL, DX
	TEST AL, rEom
	JNZ  IREom
	LOOP IELoop
	MOV  CX,ercUnexpectedInterrupt
	JMP  SHORT HandleReadComplete

Overrun:
	INC  dctNOverrun[BX]
	INC  nOverrunError
	MOV  CX,ercOverrun
	JMP  SHORT HandleReadComplete
CrcError:
	INC  dctNCrcError[BX]
	INC  nCrcError
	MOV  CX, ercCrc
	JMP  SHORT HandleReadComplete

IsrReading:
PUBLIC IsrReading
;*****************************************************************************
; Come here when a read just completed.
; This is the heart of the interrupt routine.  Based on the frame received
; and the line state, decide what to do next.
;*****************************************************************************
%IF (%Debug) THEN (;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
	MOV  logIntRead[DI],0FFh
)FI ;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
	MOV  DX,lcb_StatReg[BP]			;stat
	IN   AL,DX							
	TEST AL,rEom
	JZ   IsrError
IREom:
	MOV  DX,lcb_rdsr[BP]			;rdsrH
	IN   AL,DX						;AL = Receiver Data Status Register High
	TEST AL,rOvrn					;R&B - Count overruns only (not rAbort)
	JNZ  Overrun

	TEST AL,rErr
	JNZ  CrcError

; Save the last byte read.  If we read an odd number of bytes, the last byte  
; will not have been transferred.  This only occurs when we read an old style  
; XID.  The workstation sends only one byte of data -- the OS type.
	MOV  DX,lcb_rdb[BP]					; rDsrL
	IN   AL,DX
	JMP  $+2
	MOV  lastByteRead,AL

;We need to disable all interrupts during this section of DMA programming.
;Else, the DMA First/Last Flip Flop will be in an inconsistent state.
	PUSHF
	CLI

; Calculate the number of bytes that were read and save the count in DI.
	MOV  DX,lcb_DmaClrFFReg[BP]			;DX points to DmaBytePtrClr
	OUT  AL,DX							;reset first/last flop-flop
	MOV  DX,lcb_DmaByteCount[BP]		; commDmaWrdCnt
	IN   AL,DX
	JMP  $+2
	MOV  CL,AL
	IN   AL,DX

	POPF

	MOV  CH,AL
	MOV  DI,lcb_cReadDataMax[BP]
	TEST vf_f82380, 1		; byte count on 82380
	JZ   NoInc
	INC  CX				;Inc byte count because saved count was count - 1.
	SUB  DI,CX
	JMP  GotBytesRead
NoInc:	
	CMP  CH,0FFH
	JE   GotBytesRead
	SHL  CX,1
	SUB  DI,CX
	DEC  DI
	DEC  DI
GotBytesRead:
	XOR  CX,CX							;Assume no error
HandleReadComplete:						;R&B ...
	PUSH CX								;save error code
	MOV  AX,dmarcv OR burst2 OR hbe		;Disable RXE, but leave RXDMA on
	MOV  DX,lcb_ContReg[BP]
	OUT  AX,DX							;Turn off the state machine
; Wait for DMA to complete.  For IOGA we just delay, for Cirrus we 
; wait for the FIFO to indicate empty.
	MOV  AH,0
	MOV  AL,dctSioClock[BX]				; Max number of loops is based
	MOV  CL,shCPUSpeed					;   on SioClock and CPU speed.
	SHL  AX,CL
	MOV  CX,AX
	MOV  DX,lcb_StatReg[BP]				; stat - Cirrus status register
	TEST vf_f4MBit, 0FFh
	JNZ  SHORT CheckFIFOReadError
	LOOP $								; IOGA: idle loop
	JMP  SHORT EndReadError
CheckFIFOReadError:						; Cirrus: Loop a maximum number of
	IN   AX,DX							;   times until the FIFO is empty.
	TEST AX,FifoEmptyBit				; Check FIFO empty bit
	JNZ  SHORT EndReadError
	LOOP CheckFIFOReadError
EndReadError:							; ... R&B
	POP  CX								;restore error code

	MOV  AX,0							;HBE is set
	MOV  DX,lcb_ContReg[BP]
	OUT  DX,AX							;Turn off the state machine

	MOV  AL,lcb_commDmaDis[BP]
	MOV  DX,lcb_dmaMaskReg[BP]			;Disable the Dma
	OUT  AL,DX
	JMP  $+2

	TEST BYTE PTR lstFlags[SI],fReadIntoXBlock
	JNZ  WsReplyInXBlock
	MOV  ES,lcb_saAckInBuf[BP]
	MOV  AX,ES:[0]							;AX = stationFrame
	CMP  saMeterBuffer,0
	JE   SaveErc
	MOV  DL,2								;2 = Read finished
	MOV  DH,dctLineState[BX]
	PUSH DI									;Save count of bytes read
	CALL MeterFrame
	POP  DI									;Restore count of bytes read
	JMP  SHORT SaveErc
WsReplyInXBlock:
	MOV  ES,lcb_saXBlockCurrent[BP]
	MOV  ES:[xbCbData],DI
	MOV  ES:[xbODct],BX						;XBlock points to dct
	CMP  saMeterBuffer,0
	JE   GetXBlockFrame
	MOV  DL,2								;2 = Read finished
	MOV  DH,dctLineState[BX]
	PUSH DI									;Save count of bytes read
	CALL MeterXBlock
	POP  DI									;Restore count of bytes read
GetXBlockFrame:
	MOV  AX,ES:[xbStation]
SaveErc:
	OR   CX,CX								;Test for error completion
	JNZ  JumpToState						;Don't check station if error already
; The read completed without error. See if the station address is correct.
	CMP  AL,dctStation[BX]
	JNE  StationAddressError

JumpToState:
PUBLIC JumpToState
;*****************************************************************************
; When we jump to handle the read depending on the state:
; AL = station address (already checked)
; AH = frameType
; BX points to dct
; CX = error code 
; ES point to XBlock if reply was read into XBlock
; SI points to lineStateTable entry
; conditionCode is set if there was an error
; DI and xbCbData (if XBlock used) contain number of bytes read 
; pollState = idle
;*****************************************************************************

%IF (%Debug) THEN (;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
	CMP  SI,lastLineState
	JBE  StateOk
	MOV  AX,ercInternalConsistencyCheck
	PUSH AX
	CALL Crash
StateOk:
	PUSH DI
	MOV  DI,lcb_logBufferIndex[BP]
	MOV  logFrameIn[DI],AX
	MOV  logErc[DI],CX
	POP  DI
	OR   CX,CX							;Set condition code again
)FI ;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

	MOV  dctErc[BX],CX					;Save erc in dct
	JMP  lstState[SI]
StationAddressError:
PUBLIC StationAddressError
	INC  dctNAddressError[BX]
	INC  nAddressError
	OR   CX,ercAddressError
	JMP  SHORT JumpToState

;*****************************************************************************
; This is the end of interrupt processing. 
; Cluster interrupt is now mediated.  Just RET and interrupt controller EOI
; automatically done.
;*****************************************************************************
IsrDone:
	RET
SigIsr ENDP


SetUpTxRxDMA PROC NEAR
PUBLIC SetUpTxRxDMA
;****************************************************************************
;
;	This Routine Programs the system DMA controller ( 8237 etc. ) for
;   Both Transmitt and Receive DMA operations.  Most of this code
;	runs with Interrupts disabled.
;
;	AX =	Low Order 16 bits of buffer address
;	BX =	High order 16 bits of buffer address
;				BH = High Page 8 bits
;				BL = Low Page 8 bits
;	CX =	Byte Count ( Converted to Word count here if necessary ).
;	DX =	8237 Mode Word ( to or from memory ).
;
;*****************************************************************************

	PUSHF								; Save flags to enable ints later.
	PUSH DX								; Save Mode Word.

; Diable interrupts around programming dma registers that rely on the byte
; pointer flip flop to be accurate.
	CLI									
										
	MOV	 DX,Lcb_DmaClrFFReg[BP]			; Clear byte pointer flip-flop
	OUT	 DX,AL							; AL can be anything.
	JMP  $+2

	MOV	 DX,Lcb_DMAAddr[BP]				; Program low byte of base address
	OUT  DX,AL
	JMP  $+2

	XCHG AL,AH
	OUT  DX,AL
	JMP  $+2							; Program high byte of base address

	POP  AX								; AX = Mode Word.
	MOV  DX,Lcb_DmaModeReg[BP]			; 8237 Mode register.
	OUT  DX,AL

	MOV  AX,CX							; AX = count of bytes to transfer.
	MOV  DX,lcb_DmaByteCount[BP]		;DX points to commDmaWrdCnt
	OUT  AL,DX							;send low byte of word count
	JMP  $+2
	XCHG AL,AH		
	OUT  AL,DX							;send high byte of word count
	JMP  $+2

	POPF								; Enable Interrrupts.

	MOV  DX,lcb_DMAEar[BP]				; Program EAR with bits 16-23 of buffer
										; address

	MOV  AX,BX							; AL holds low page address
	OUT  DX,AL
	JMP  $+2

	
; On machines that have > 20 MB RAM, we must program the High Page register.
	MOV  DX,Lcb_DMAEarH[BP]				; Program EAR with bits 24-32 of buffer
										; address.
	TEST vf_f386,1
	JZ	 HighPageExit

;	CMP	 ProcessorType,bGPType			; bGPType is of obsolete CdtEqu.edf
;	JAE	 DoGPEar						;
	TEST vf_fMF, 1
	JNZ  DoGPEar

	TEST vf_fNewGen,1
	JNZ	 HiPage386i
	TEST vf_fSGEN,1						; All other machines exit.
	JZ	 HighPageExit
	XCHG AL,AH							; AL holds High Page address.
	OUT  DX,AL
	JMP  $+2
	JMP	 HighPageExit

HiPage386i:
; Calculate and program the 386i Hi Page Register. - this should be
; done in initclstr_all.plm
	MOV	 AH,0
	SHL	 AX,1
	XCHG AL,AH
	OUT  DX,AL
	JMP  $+2
; On 386i there is an high page enable bit - set it here.
	MOV  DX, DmaEarEnable
	IN   DX,AL
	JMP  $+2
	OR   AL, bEnableEarChan0
	OUT  DX,AL
	JMP  HighPageExit

DoGPEar:
; The GP uses the 82380 DMA controller
	PUSHF
	CLI									; Disable interrupts
	MOV  DX,lcb_DmaClrFFReg[BP]			; DX points to DmaBytePtrClr
	OUT  AL,DX							; reset first/last flop-flop
	MOV  DX,lcb_DmaEarH[BP]	     		; needs 0 flipflop
	XCHG AL,AH		
	OUT  AL,DX							; send high byte of EAR
	JMP  $+2
	XCHG AL,AH		
	MOV  DX,lcb_DMAEar[BP]				; DX is now commDmaEar
	OUT  AL, DX							; send Dma Extended Address Register
	JMP  $+2
	POPF

HighPageExit:
	RET

SetUpTxRxDMA ENDP

CalculateDmaAddr PROC NEAR
;*****************************************************************************
; ES:DI contains address within XBlock.
; Convert to Dma word address and store in XBlock.
;*****************************************************************************
	TEST vf_f82380, 1 		       	;is this a cluster iop?
	JZ   SetAddr		            ;jump if not.
	SUB  DI,xbStation
	MOV  AX,ES:[xbsaveAddr]			;if GP addr/ear saved in xblock header.
	ADD	 AX,DI
	MOV  ES:[xbDmaAddr],AX
	MOV  AX,ES:[xbsaveEar]
	ADC	 AX,0
	MOV  ES:[xbDmaEar],AX
	JMP  SHORT CalcRet
	
SetAddr:
	SUB  DI,xbStation
	MOV  AX,ES:[xbsaveAddr]			;if GP addr/ear saved in xblock header.
	SHR  DI,1
	ADD	 AX,DI
	MOV  ES:[xbDmaAddr],AX
	MOV  AX,ES:[xbsaveEar]
	MOV  ES:[xbDmaEar],AX
CalcRet:
	RET
CalculateDmaAddr ENDP

StopRead PROC NEAR
PUBLIC StopRead
;*****************************************************************************
; Stop the read operation when a poll times out.
; On return, BX points to dct that was stopped.
;*****************************************************************************
; R&B ...
	CMP  lcb_pollState[BP],stateWriting
	JNE  NotWriting
	TEST vf_f4MBit, 0FFh			;R&B ... If we're a cirrus, need to  *FW*
	JZ   SkipTCUse					; use the terminal counter to stop DMA *FW*
	MOV  AX, ccrStopDMA				;With a Cirrus chip, this enables the
	MOV  DX, lcb_ContReg[BP]		;  terminal counter, we load the counter
	OUT  AX, DX						;  with 1 to cause DMA to stop early.
	ADD  DX, 2						;cPort + 2 is address of terminal count
									;  register on NGen with Cirrus chip
	MOV  AX, 1						;Load 1 in counter
	OUT  AX, DX
SkipTCUse: 							;  *FW*
	MOV  AX,rts OR endxmit OR txwcen;This turns off TXE. DMA will finish
									;  after the stacker empties.
	MOV  DX, lcb_ContReg[BP]		;Restore Control Register port address
	OUT  DX, AX
	JMP  Short DoDelay              ;Fifo may not be empty but we do not care
NotWriting:
	MOV  AX,dmarcv+burst2
Disable2652:
	MOV  DX,lcb_ContReg[BP]
	OUT  DX,AX              	    ;Turn off the state machine
DoDelay:
; Wait for DMA to complete.  For IOGA we just delay, for Cirrus we 
; wait for the FIFO to indicate empty.
	MOV  AH,0
	MOV  AL,dctSioClock[BX]			;Max number of loops is based
	MOV  CL,shCPUSpeed				;   on SioClock and CPU speed.
	SHL  AX,CL
	MOV  CX,AX
	MOV  DX,lcb_StatReg[BP]			;Cirrus status register
	TEST vf_f4Mbit, 0FFh
	JNZ  SHORT CheckFIFOStopRead
	LOOP $							; IOGA: idle loop
	JMP  SHORT EndStopRead
CheckFIFOStopRead:					; Cirrus: Loop a max number of time until 
	IN   AX,DX						;   the FIFO is empty.
	TEST AX,FifoEmptyBit			; Check FIFO empty bit
	JNZ  EndStopRead
	LOOP CheckFIFOStopRead
EndStopRead:
	MOV  DX,lcb_ContReg[BP]
	XOR  AX,AX						;disable state machine before dma
	OUT  AX,DX						;Turn off the state machine
	MOV  DX,lcb_DmaMaskReg[BP]		;Disable the Dma
	MOV  AL,lcb_commDmaDis[BP]
	OUT  DX,AL
	JMP  $+2
; ... R&B
	MOV  lcb_pollState[BP],stateIdle	;Set pollState to stateIdle
	MOV  BX,lcb_dctCurrent[BP]			;BX points to dct that timed out
;P1 ...
; If no current DCT then we shouldn't be in StopRead
; Turn this off before release.
%IF (0) THEN (;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
	LEA  DX,lcb_listActiveHead[BP]
	CMP  BX,DX
	JNE  DctCurrentOk1
	MOV  AX,ercInternalConsistencyCheck
	PUSH AX
	CALL Crash
DctCurrentOk1:
)FI ;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
; ... P1
	RET
StopRead ENDP


MstrLph ENDS

; LOG
; 3/28/84 by Jim Frandeen: Created file
;  10/15/86 by GVW : Fix for CP003
;  10/17/86 by FW: 286i and many 'JMP $+2's
;  11/06/86 by RLM: wait before write when booting and dumping for old boot
;  roms (IWS/AWS/CWS).
; 02/25/87 by MS : the length of the small frames is now 4 bytes (minimum)
; 03/05/87 by MS management of the new frame XID with DAI number
;  04/17/87 by RLM: Write from xblock check if really have xblock to write
; 					from.  If not change state.
; 09/18/87 by JA add WaitForWrite timeout because of 286i ethernet dma bug.
; 11/10/87 by RLM/JA add stateNoPoll logic to stop reentrant problems in 
; pollcurrent.
; 01/20/88 by RLM fix bootid bug
; 02/17/88 by RLM change nop's for flag timming to shcpu loop$
; 04/12/88 by RLM 32 byte xblock header changes
; 2.3 changes
; 08/16/88 by RLM make 386 srp use NGen protected mode code.
; 09/26/88 by JA use vf_f386.
; 11/25/88 by JA/RLM use lcb_commDmaEn,lcb_commDmaDis. RET don't IRET.
;				SNRM send lcb_KHLineSpeed.  SetDmaEar clear FF.
; 01/09/90 by AT, Don't include header size in DMA transfer count.
; 01/09/90 by JA SysTimeTicksDiff, TimerIsr1&2 to MstrLph_All.Asm.
; 03/08/90 by RLM Make nDctXXX errors INC (word) not (byte).
; 04/16/90 by AT, Merged CTOS/XE 3.0 and CTOS/VM 3.3.
; 05/21/90 by JA dct gives sioClock for write.
; 08/15/90 by JC, Added BTOS RS232 latency reduction improvements
; 08/24/90 by JM, back out MediateInterrupt which causes stack fault.
; 09/06/90 by JA, Underrun clear tdsr.
; 10/09/90 by JA, SetDmaEar keep CLI around all DmaFF OUTs.
; 12/01/90 by GWH, Use lcb_TimerCtrReg and lcb_timerMode to set baud rate.
; 12/07/90 by JA, add 2 to dma count on read for (station,frame)
; 01/09/91 by AT: Use 2-word burst demand mode DMA on SG5000.
; 01/18/91 by JA, index through BP to get lcb_timerMode when setting baud rate.
; 01/28/91 by AT, Check for NIL dctCurrent in StopRead, P1 debugging only.
; 02/21/91 by GWH, Added SetUPTxRxDMA proc. Turned off p1 debug code.
; 03/18/91 by BA, Merged in rotate & burst changes from s2.4.3.  See R&B 
;                 comments.
; 03/29/91 by BA, Merged additional R&B enhancements.
; 04/26/91 by BA, Fixed bug in StartWriteIOGA.
; 05/07/91 by BA: As a consequence of rotating priority, a busy parallel port 
;				  can cause underruns on the cluster.  To get around this, 
;				  we mask the parallel DMA before transmitting and unmask
;				  it after (but only if the fMaskParallelDma flag was set at
;				  init time (in InitClstr)).
; 05/09/91 by BA: Corrected TSOM logic.
; 05/14/91 by BA: Don't count aborts when counting overruns in StateReading.  
;				  Removed obsolete commented out code.
; 05/16/91 by BA: Reduce TSOM delay (temporary latency fix) and reverse the 
;				  sense.
; 05/20/91 by BA: Forget the above TSOM fixes, this is the real one.
; 05/21/91 by BA: Send more flags for telecluster.
; 05/30/91 by GWH:Abort read operation in stopread when in statewaitdcddrop.
; 05/30/91 by GWH:Removed bogus delay loops in StartWrite, WaitForDCDDrop,
;                 WaitBeforeWrite.
; 05/31/91 by BA: Put delay loop back in StartWrite.
; 05/21/92 by JM: clear SI and direction flag in StartWriteCirrus so REP 
;				OUTSB's don't fault
; 03/24/93 by sg: add support for cluster IOP (vf.f82380 )
; 04/01/93by JF: Set SI addressable non-zero so OUTSB won't fault when
;			soft bus version uses DS allocation.  Change CalculateDmaAddr
;			to use saved DmaAddr for soft bus.
; 06/22/93 by JM: obsolete Cdtequ.edf.
; 07/16/93 by FW: In StopRead, don't poke the terminal count register if NOT
;                 a Cirrus chip.  Causes DMA hang on old CP-003 processors.
	END
