;*****************************************************************************
; WsLph_all.asm
; Workstation Line Protocol Handler.
; This module contains code that is common to all hardware types.
;*****************************************************************************

;*****************************************************************************
;  PUBLIC PROCEDURES
;*****************************************************************************
PUBLIC PrepareIdSearch
PUBLIC Timeout

;*****************************************************************************
;  LITERALS
;*****************************************************************************
ackInBufSize EQU 14
; NOTE: The Aws and Iws must always read more than the number of characters sent by the master. If the master send 12 bytes of XID and we read 12 bytes, we will never see the frame. However, if we read 14 bytes, and the master sends 12 bytes, we read the frame just fine. It's a mystery to me.

exchAgent 			EQU 12
maxErrorCount 		EQU 10
maxTimeoutCount 	EQU 20
pageSize 			EQU 512
nStationMax 		EQU 64
rcWrite  			EQU 36
rcResetAgent 		EQU 72
trbTimeoutValue 	EQU 20

;*****************************************************************************
; Request block definition
;*****************************************************************************
rq              EQU 0
rqSCntlInfo     EQU rq
rqRtInfo        EQU rq+1
rqNReqPbCb      EQU rq+2
rqNRespPbCb     EQU rq+3
rqUserNum       EQU rq+4
rqRespExch      EQU rq+6
rqErcRet        EQU rq+8
rqCode          EQU rq+10
rqFh            EQU rq+12
rqLfa           EQU rq+14
rqpBuf          EQU rq+18
rqsBuf          EQU rq+22
rqpsDataRet     EQU rq+24
rqssDataRet     EQU rq+28
rqReadWriteData EQU rq+30

;*****************************************************************************
;  EXTERNAL PROCEDURES
;*****************************************************************************
EXTRN Crash:FAR
;EXTRN MediateIntHandler:FAR
EXTRN PSend:FAR
EXTRN MasterWentDown:FAR

;*****************************************************************************
;  EXTERNAL DATA
;*****************************************************************************
DGroup GROUP DATA
Data SEGMENT PUBLIC 'DATA'

EXTRN fOldStyle:BYTE  			; TRUE if workstation has fixed ID
EXTRN nTryNewMaster:BYTE
EXTRN nWaitingRequests:WORD		;maintained by Agent
EXTRN nXBlocksToSend:WORD		;maintained by Agent
EXTRN oRcbFirst:WORD
EXTRN oRcbLast:WORD
EXTRN rcbWaitingHead:Word
EXTRN sioClock:BYTE
EXTRN sXBData:WORD
EXTRN sXBlk:WORD
EXTRN sXBlkRqMax:WORD
EXTRN timerRq:WORD
EXTRN clusterTimeout:WORD
trbInterval EQU timerRq			;2 sec timeout
trbResetValue EQU timerRq+2
trbCEvents EQU timerRq+4
trbRespExch EQU timerRq+6
trbRqCode EQU timerRq+8


;*****************************************************************************
; AckOutBuf and AckInBuf definition
; buffer pointers pAckOutBuf and pAckInBuf are allocated in Sysgen.mdf and 
; initialized by Init to be on a paragraph boundary.
; saAckInBuf and saAckOutBuf are used to address them.
;*****************************************************************************
EXTRN pAckInBuf:DWORD
EXTRN pAckOutBuf:DWORD
saAckInBuf EQU WORD PTR pAckInBuf+2
saAckOutBuf EQU WORD PTR pAckOutBuf+2
;*****************************************************************************
;  GLOBAL DATA
;*****************************************************************************
PUBLIC masterRevisionLevel, fOldMaster, idTable, snrmsToWaitFor, stationAddress, bitIoErc, cbReadMax, lineState, NR, NS, pollSequenceNumber, pollState, saXBlock, saFreeList, retryCount, saXBlockCurrent, saXBlockIn, saXBlockOut, oAgentMessage, msgLph, stationAddressIn, ercMasterDown, wsStats, findIdRetryCount, rgKHLineSpeed, rgSioClock

fOldMaster 			DB 0
findIdRetryCount 	DB 0
idTable 			DB nStationMax DUP (0)
masterRevisionLevel DB 0
snrmsToWaitFor 		DB 3
stationAddress 		DB 0
stationAddressIn 	DB 0

   EVEN
bitIoErc 		DW 0
cbReadMax 		DW 0
lineState 		DW 0

ercMasterDown 	DW 0
msgLph 			DW 0
; We use bits in msgLph to send a message to the Agent:

msgMasterDown    EQU 1
msgMasterUp      EQU 2
msgXBlockIn      EQU 4
msgXBlockFill    EQU 8
msgReinitXBlocks EQU 10H

PUBLIC fMsgLphQueued
fMsgLphQueued DB 0


NR DB 0  ;last IFrame sequence number received
         ;number from 0 to 0E0, gets incremented by 20, ANDed by 0E0
         ;We send these bits with RR and RNR
NS DB 0  ;last IFrame sequence number sent
         ;number from 0 to 0E0, gets incremented by 20, ANDed by 0E0
         ;We expect RR or RNR with bits that match.
         ;We shift right 4 to send IFrame
oAgentMessage 		DW 0
pollSequenceNumber 	DW 0
pollState 			DW OFFSET StateIdle

saXBlock 			DW 0			;XBlocks for the Agent to process

saFreeList 			DW 0			;These must be together

retryCount 			DW 0
saXBlockCurrent 	DW 0
saXBlockIn 			DW 0			;List of XBlocks for Agent 
saXBlockOut 		DW 0

;  <= 2.x masters mistakenly put OsRevision (9100h) from XID frame in 
;  snrm frame at snrm.linespeed.  Run at 307kb.
rgKHLineSpeed		DW 9100h,4000,3680,1800,307, 0;KHLineSpeeds, 0 last,
rgSioClock			DB    24,   2,   2,   4, 24,24; ... sioClock values.
											; May change at boot in InitClstr.
; Next table maps xid.maxLineSpeed to khLineSpeed.  Only values we ever used
;  are 7=307Kb, 11=1.8Mb, 12=4Mb.
khFromXidMaxSpeed	DW 307,307,307,307,307,307,307,307,307,307,307,1800,3680
maxLineSpeedFromSioClock DB ?, 12, 11, ?,?,?,?,?,?,?,?,?, 7
;*****************************************************************************
; STATISTICS
;*****************************************************************************
PUBLIC nCrcError, nOverrunError, nSequenceError, nProtocolError, nAddressError, nLengthError, nTimeout, nBadIFrameError
PUBLIC sumSimpleRqTimeLow, sumSimpleRqTimeHigh, nSimpleRqLow, nSimpleRqHigh, maxSimpleRqTime, sumGetDtRqTimeLow, sumGetDtRqTimeHigh, nGetDtRq, maxGetDtRqTime, sumBlockRqTimeLow, sumBlockRqTimeHigh, nBlockRqLow, nBlockRqHigh, maxBlockRqTime, nTicksPerSecond, nUnderrun, nAbortError, nBadRcbError

wsStats LABEL WORD
; The following statistics are sent to the master on the end of each GetDateTime request:

nCrcError 			DW 0
nOverrunError 		DW 0
nSequenceError 		DW 0
nProtocolError 		DW 0
nAddressError 		DW 0
nLengthError 		DW 0
nTimeout 			DW 0

sumSimpleRqTimeLow 	DW 0
sumSimpleRqTimeHigh DW 0
nSimpleRqLow 		DW 0
nSimpleRqHigh 		DW 0
maxSimpleRqTime 	DW 0

sumGetDtRqTimeLow 	DW 0
sumGetDtRqTimeHigh 	DW 0
nGetDtRq 			DW 0
maxGetDtRqTime 		DW 0

sumBlockRqTimeLow 	DW 0
sumBlockRqTimeHigh 	DW 0
nBlockRqLow 		DW 0
nBlockRqHigh 		DW 0
maxBlockRqTime 		DW 0

nTicksPerSecond 	DW 0    ;Set by init code
sizeWsStats         EQU 44  ;known to WsAgent
; end of stats sent to master

; nUnderrun is not yet sent to the master
nUnderrun 			DW 0
nAbortError			DW 0
nBadRcbError		DW 0

nBadIFrameError 	DW 0

%IF (%Debug) THEN (;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
;*****************************************************************************
; VARIABLES FOR DEBUGGING
;*****************************************************************************

PUBLIC nTimeoutCalled, nWaitDcd, fWaitingToAckSnrm, fCrashIfOldMaster, lengthCalculated
fWaitingToAckSnrm 	DB 0
fCrashIfOldMaster 	DB 0
EVEN
lengthCalculated 	DW 0
nTimeoutCalled 		DW 0
nWaitDcd 			DW 0

; For testing, we can send a bad CRC every n frames
PUBLIC nSendBadCrc, nSendBadCrcReload
nSendBadCrc 		DW 0
nSendBadCrcReload 	DW 500

PUBLIC logBuffer, logBufferIndex, logFrameOut, logFrameIn, logErc, logLineState, logBufferIndexMax, logFrameNumber
PUBLIC logPollState, logBadCrc
PUBLIC logIntIdle, logIntDcd, logIntWrite, logIntRead
	EVEN
logEntrySize 		EQU 20
logBufferSize 		EQU 100
logBufferIndexMax 	EQU (logBufferSize-1)*logEntrySize
logBufferIndex 		DW 0
logBuffer 			LABEL WORD
logFrameNumber 		DW 0
logIntRead 			DW 0	;incremented by 1 if int on read
logIntDcd 			DB 0	;incremented by 1 if int on dcd change
logIntIdle 			DB 0	;incremented by 1 if int on idle
           			DB 0					
logIntWrite 		DB 0	;incremented by 1 if int on write
logFrameOut 		DW 0
logFrameIn 			DW 0
logErc 				DW 0
logLineState 		DW 0
logPollState 		DB 0
 					DB 0
logBadCrc 			DW 0	; = BADC when bad crc sent
	DB (logBufferSize-1)*logEntrySize DUP (0)

PUBLIC logMsgBuffer, logMsgBufferIndex
logMsgBufferSize 	EQU 20
logMsgEntrySize 	EQU 2
logMsgBufferIndexMax 	EQU (logMsgBufferSize-1)*logMsgEntrySize
logMsgBufferIndex 	DW logMsgBufferIndexMax
logMsgBuffer 		LABEL 	WORD
logMsgRa 			DW 0
	DB (logMsgBufferSize-1)*logMsgEntrySize DUP (0)

PUBLIC logTimeoutBuffer, logTimeoutBufferIndex, logTimeoutSequence, logTimeoutLineState, logTimeoutPollState
logTimeoutBufferSize	EQU 20
logTimeoutEntrySize 	EQU 6
logTimeoutBufferIndexMax EQU (logTimeoutBufferSize-1)*logTimeoutEntrySize
logTimeoutBufferIndex 	DW logTimeoutBufferIndexMax
logTimeoutBuffer 		LABEL 	WORD
logTimeoutSequence 		DW 0
logTimeoutLineState 	DW 0
logTimeoutPollState 	DW 0
	DB (logTimeoutBufferSize-1)*logTimeoutEntrySize DUP (0)

PUBLIC logSnrmBuffer, logSnrmBufferIndex, logSnrmLineState, logSnrmErc, logSnrmFrameIn
logSnrmBufferSize 		EQU 10
logSnrmEntrySize 		EQU 6
logSnrmBufferIndexMax 	EQU (logSnrmBufferSize-1)*logSnrmEntrySize
logSnrmBufferIndex 		DW logSnrmBufferIndexMax
logSnrmBuffer 			LABEL WORD
logSnrmLineState 		DW 0
logSnrmErc 				DW 0
logSnrmFrameIn 			DW 0
	DB (logSnrmBufferSize-1)*logSnrmEntrySize DUP (0)

PUBLIC logAckSnrmBuffer, logAckSnrmIndex, logAckSnrmState
logAckSnrmBufferSize 	EQU 10
logAckSnrmEntrySize 	EQU 6
logAckSnrmIndexMax 		EQU (logAckSnrmBufferSize-1)*logAckSnrmEntrySize
logAckSnrmIndex 		DW logAckSnrmIndexMax
logAckSnrmBuffer 		LABEL WORD
logAckSnrmStation 		DW 0
logAckSnrmWait 			DB 0
logAckSnrmFail 			DB 0
logAckSnrmState 		DW 0
	DB (logAckSnrmBufferSize-1)*logAckSnrmEntrySize DUP (0)

PUBLIC logErrorBuffer, logErrorErc, logErrorFrame, logErrorIndex
logErrorEntrySize 		EQU 4
logErrorSize 			EQU 50
logErrorIndexMax 		EQU (logErrorSize-1)*logErrorEntrySize
logErrorIndex 			DW logErrorIndexMax	;so we will start at zero
logErrorBuffer 			LABEL WORD
logErrorErc 			DW 0
logErrorFrame 			DW 0
	DB (logErrorSize-1)*logErrorEntrySize DUP (0)

PUBLIC logXBlockInBuffer, logXBlockInIndex
logXBlockInEntrySize 	EQU 34
logXBlockInSize 		EQU 20
logXBlockInIndexMax 	EQU (logXBlockInSize-1)*logXBlockInEntrySize
logXBlockInIndex 		DW logXBlockInIndexMax	;so we will start at zero
logXBlockInBuffer 		LABEL WORD
	DB (logXBlockInSize)*logXBlockInEntrySize DUP (0)

logRqEntrySize 			EQU 8	;known to WsAgent_all
logRqBufferSize 		EQU 100 ;known to WsAgent_all
logRqBufferIndexMax EQU (logRqBufferSize-1)*logRqEntrySize;known to WsAgent
EXTRN logRqBufferIndex:WORD
EXTRN logRqPoint:WORD
EXTRN logRqCode:WORD
EXTRN logRqErcRet:WORD
EXTRN logRqoRcb:WORD

)FI ;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
lineStateTable LABEL WORD
PUBLIC lineStateTable
;*****************************************************************************
; Line State Table Definition
; The lineStateTable determines what frame to send next and what to do when a 
; frame has been read from the master.
;*****************************************************************************
stateSearchId 	EQU 0
lstState     	DW RxSearchId  ;JMP here when read finished
lstFrameType 	DB 0           ;type of frame to send -- none
lstCbAckOut  	DB 0           ;number of bytes to ack out
lstFlags     	DB 0
lstSize      	EQU 5

; Values of lstFlags:

fReadIntoXBlock EQU 80h
fElseWsNotReady EQU 3Fh

stateAckSnrmWithXID EQU 1*lstSize					;5
    DW RxAckSnrmWithXID  ;JMP here when read finished
	DB frameTypeXID  ;type of frame to send
	DB 14            ;number of bytes to ack out
	DB 0

stateAckXID EQU 2*lstSize							;0Ah
stateAckSnrmWithUA EQU 2*lstSize
	DW RxAckXID      ;JMP here when read finished
	DB frameTypeUA   ;type of frame to send
	DB 2             ;number of bytes to ack out
	DB 0

stateWsReady EQU 3*lstSize							;0Fh
	DW RxWsReady    ;JMP here when read finished
	DB frameTypeRR  ;type of frame to send
	DB 2            ;number of bytes to ack out
	DB fReadIntoXBlock+stateWsNotReady

stateWsNotReady EQU 4*lstSize						;14h
	DW RxWsNotReady ;JMP here when read finished
	DB frameTypeRNR ;type of frame to send
	DB 2            ;number of bytes to ack out
	DB 0

stateSendIFrameWsNotReady EQU 5*lstSize				;19h
	DW RxSendIFrameWsNotReady     ;JMP here when read finished
	DB frameTypeI   ;type of frame to send
	DB 0            ;number of bytes to ack out
	DB 0

stateSendIFrameWsReady EQU 6*lstSize				;1Eh
	DW RxSendIFrameWsReady     ;JMP here when read finished
	DB frameTypeIWsReady   ;type of frame to send
	DB 0            ;number of bytes to ack out
	DB fReadIntoXBlock+stateSendIFrameWsNotReady

stateAckIFrameWithIFrame EQU 7*lstSize				;23h
	DW RxAckIFrameWithIFrame     ;JMP here when read finished
	DB frameTypeIWsReady   ;type of frame to send
	DB 0            ;number of bytes to ack out
	DB fReadIntoXBlock+stateSendIFrameWsNotReady

stateResendIFrame EQU 8*lstSize						;28h
	DW RxResendIFrame     ;JMP here when read finished
	DB frameTypeRNR   ;type of frame to send
	DB 2            ;number of bytes to ack out
	DB 0

stateBadIFrame EQU 9*lstSize						;2Dh
	DW RxBadIFrame     ;JMP here when read finished
	DB frameTypeREJ   ;type of frame to send
	DB 2            ;number of bytes to ack out
	DB fReadIntoXBlock

;stateResendIFrameNewMaster EQU 10*lstSize			;32h
;	DW RxResendIFrame     ;JMP here when read finished
;	DB frameTypeRNR   ;type of frame to send
;	DB 4            ;number of bytes to ack out
;	DB 0
;
;lastLineState EQU stateResendIFrameNewMaster

; R&B ... Replaced above state with the following two states

stateResendIFrameNewMasterNotReady EQU 10*lstSize			;32h
	DW RxResendIFrame		;JMP here when read finished
	DB frameTypeI			;type of frame to send
	DB 0					;number of bytes to ack out
	DB 0

stateResendIFrameNewMasterReady EQU 11*lstSize				;37h
	DW RxResendIFrame		;JMP here when read finished
	DB frameTypeIWsReady	;type of frame to send
	DB 0					;number of bytes to ack out
	DB fReadIntoXBlock

lastLineState EQU stateResendIFrameNewMasterReady

; ... R&B

endOfLineStateTable DW 0
PUBLIC endOfLineStateTable
Data ENDS

WsLph SEGMENT PUBLIC 'CODE'
	ASSUME CS: WsLph, DS: Dgroup
PrepareIdSearch PROC FAR
;*****************************************************************************
; Initialize our id table in preparation for the id search.
; This procedure is called by the Agent once to start things up. Thereafter 
; it is called from the interrupt level whenever the master goes down.
;*****************************************************************************
; Zero the idTable
	CLD

%IF(0) THEN (%' obsolete, speed set after SNRM rcvd.
	; Reset the linespeed while searching for a new ID
	MOV  BL, 24				; 307K
	MOV  sioClock,BL
	CALL SetLineSpeed
)FI

	MOV  CX,nStationMax
	PUSH DS
	POP  ES
	LEA  DI,idTable
	XOR  AX,AX
	REPNZ STOSB
	MOV  retryCount,AX
	MOV  NR,AL					;last sequence number received = 0
	MOV  NS,AL					;last sequence number sent = 0
; Set up parameterControl for promiscuous mode. We will read every frame that 
; goes by.
%IF (%Ngen) THEN (
	MOV  parameterControl,bop+idle
)FI

%IF (%Aws) THEN (
	MOV  comm5Control,comm5Dtr+comm5Tx8+comm5CrcCcitt
	MOV  comm3Control,comm3Rx8+comm3RxCrcEnb+comm3Hunt+comm3RxEnb
)FI
%IF (%Iws) THEN (
	MOV  comm3Control,comm3Rx8+comm3RxCrcEnb+comm3Hunt+comm3RxEnb
)FI
	MOV  SI,stateSearchId
	MOV  lineState,SI
	PUSHF						;Save flags and enable/disble status
	CLI							;Disable
; Generate a random number of SNRMs to wait for.
	CALL GenerateRandomNumber
	AND  CL,0Fh
	INC  CL						;Gives a number from 1 to 16
	CMP  fOldStyle,0			;Fixed ID?
	JE   SetSnrmsToWaitFor		;No
	MOV  CL,3					;Wait for 3 SNRMs if fixed ID
SetSnrmsToWaitFor:
	MOV  snrmsToWaitFor,CL
	CALL FAR PTR BitStartRead
	POPF						;Restore flags and enable if enabled before
 	RET
PrepareIdSearch ENDP
Timeout PROC FAR
;*****************************************************************************
; This routine gets called by the Agent process when we get a timeout. This 
; means our read did not complete before the timer went off (2 seconds).
;*****************************************************************************
%IF (%Debug) THEN (;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
	INC  nTimeoutCalled
)FI ;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
	PUSH BP
	CLI  						;Disable
; Check for race condition with completion of read operation. When the read 
; completes, it sets pollState to stateIdle. When the next write starts, the 
; time interval gets set again.
	CMP  pollState,OFFSET StateIdle
	JE   TimeoutRet1
	CMP  WORD PTR trbInterval,0
	JNE  TimeoutRet1
	INC  nTimeout
	CALL StopRead
	MOV  SI,lineState
%IF (%Debug) THEN (;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
	MOV  DI,logTimeoutBufferIndex
	ADD  DI,logTimeoutEntrySize
	CMP  DI,logTimeoutBufferIndexMax
	JBE  logTimeoutBufferIndexOk
	MOV  DI,0
logTimeoutBufferIndexOk:
	MOV  logTimeoutBufferIndex,DI
	MOV  AX,pollSequenceNumber
	MOV  logTimeoutSequence[DI],AX
	MOV  AX,lineState
	MOV  logTimeoutLineState[DI],AX
	MOV  AX,pollState
	MOV  logTimeoutPollState[DI],AX
)FI ;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
	CMP  SI,stateSearchId
	JNE  TimeoutNotSearchId
RetryAfterTimeout:
	CALL FAR PTR BitStartRead		;SI = lineState
TimeoutRet1:
	JMP  SHORT TimeoutRet

TimeoutNotSearchId:
%IF (%Debug) THEN (;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
	MOV  DI,logBufferindex
	MOV  logErc[DI],ercTimeout
)FI ;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

	INC  retryCount
	MOV  AX, clusterTimeout
	CMP  retryCount, AX
	JB   RetryAfterTimeout
	STI								;Enable
	CMP  SI,stateAckSnrmWithXID
	JE   TryOldMaster
; If 2 seconds * maxTimeoutCount have gone by, the master has dropped us. Tell
; the Agent the master is down and prepare to search for a new id.
	MOV  ercMasterDown,ercSrpDown
	CALL MasterWentDown
	JMP  SHORT TimeoutRet

TryOldMaster:
	INC  findIdRetryCount
	MOV  AL,findIdRetryCount
	CMP  AL,nTryNewMaster
	JBE  Restart
; The master did not reply to our XID. Change the value of fOldMaster and call
; PrepareIdSearch again. The next time we will respond to a SNRM with UA 
; instead of XID.
	MOV  fOldMaster,0FFh
%IF (%Debug) THEN (;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
	CMP  fCrashIfOldMaster,0FFh
	JNE  OldMasterOk1
	MOV  AX,3
	PUSH AX
	CALL Crash
OldMasterOk1:
)FI ;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
Restart:
	CALL PrepareIdSearch
TimeoutRet:
	STI							;Enable
	POP  BP
	RET
Timeout ENDP
ReadInterrupt PROC FAR
;*****************************************************************************
; This is the beginning of the read interrupt handling. This is a FAR PROC 
; so that we will return FAR to the interrupted process.
;*****************************************************************************

RxSearchId:
PUBLIC RxSearchId
;*****************************************************************************
; In this state we are looking for SNRMs. We read every frame that goes by. 
; When we find enough SNRMs, go to state SendXID if this is a new master. 
; Otherwise go to state SendUA.
;*****************************************************************************
	JNZ  ReadNextId				;Ignore frame if error
	CMP  AH,frameTypeSNRM
	JNE  ReadNextId				;Ignore frame if not SNRM
	CMP  AL,0
	JE   ReadNextId				;Ignore frame if zero stationID
	CMP  AL,nStationMax
	JA   ReadNextId				;Ignore frame if stationID out of range
	CMP  fOldStyle,0			;Fixed ID?
	JE   IncrementSnrmCount		;No
	CMP  AL,stationAddress		;Yes -- is this our fixed ID?
	JNE  ReadNextId				;No
IncrementSnrmCount:
; We read a SNRM that is in the proper range of station ids. Increment the 
; count in our idTable.
	MOV  BL,AL
	MOV  BH,0					;BX = station address
	INC  BYTE PTR idTable-1[BX]
	MOV  CL,BYTE PTR idTable-1[BX]
	CMP  CL,snrmsToWaitFor
	JAE  IdFound
ReadNextId:
	MOV  SI,stateSearchId
	MOV  lineState,SI
	CALL FAR PTR BitStartRead
	JMP  IsrDone

IdFound:
PUBLIC IdFound
; AL enters and exits with the stationId
; We have found a stationAddress. Now set the sioClock.  The second word of a 
; SNRM contains the maximum line speed in Khz.
; The master may have a larger max speed than this workstation.  Calculate our
; max speed.
	CMP  DI, 4					;Is SNRM long enough to have line speed?
	JB   KHzDone
	MOV  BX, ES:[2]				;get line speed from SNRM frame
	MOV  SI, 0-2				; initial loop value
KHzNext:
	ADD  SI, 2
	CMP  BX, rgKHLineSpeed[SI]
	JB   KHzNext
	SHR  SI, 1
	MOV  CL, rgSioClock[SI]
	MOV  sioClock,CL			;save new sioClock
	MOV  BX, saAckOutBuf
	MOV  ES, BX
	SHL  SI, 1
	MOV  BX, rgKHLineSpeed[SI]
	MOV  ES:[xidKHLineSpeed], BX
	MOV  BL, CL		; find xidMaxLineSpeed from sioClock
	XOR  BH, BH
	SHR  BX, 1
	MOV  BL, maxLineSpeedFromSioClock[BX]
	MOV  ES:[xidMaxLineSpeed], BL
KHzDone:

%IF (%Debug) THEN (;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
	MOV  SI,logAckSnrmIndex
	ADD  SI,logAckSnrmEntrySize
	CMP  SI,logAckSnrmIndexMax
	JBE  logAckSnrmIndexOk
	MOV  SI,0
logAckSnrmIndexOk:
	MOV  logAckSnrmIndex,SI
	MOV  logAckSnrmStation[SI],BX
	MOV  logAckSnrmWait[SI],0
	MOV  logAckSnrmFail[SI],0
	PUSH AX
	MOV  AX,lineState
	MOV  logAckSnrmState[SI],AX
	POP  AX
)FI ;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

%IF (%Aws) THEN (
	MOV  stationAddress,AL
	MOV  DX,ioCommCtl
	MOV  AL,wr1
	OUT  AL,DX
	MOV  AL,wr1ExIntEnable
	OUT  AL,DX
	IN   AL,DX
	TEST AL,comm0Dcd
	JZ   StateWaitDcdDropToAckSnrm
%IF (%Debug) THEN (;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
	MOV  logAckSnrmWait[SI],0FFh
)FI ;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
	MOV  pollState,OFFSET StateWaitDcdDropToAckSnrm
	JMP  IsrDone
)FI
%IF (%Iws) THEN (
	MOV  stationAddress,AL
	MOV  DX,ioCommCtl
	MOV  AL, commWr1
	%ZOUT(DX,AL)
	MOV  AL,comm1ExIntEnb
	%ZOUT(DX,AL)
	IN   AL,ioCommExtCtl
	AND  AL,01H
	OR   AL,commExCtlDcdEnb
	OUT  ioCommExtCtl,AL
)FI
StateWaitDcdDropToAckSnrm:
PUBLIC StateWaitDcdDropToAckSnrm
	CALL DelayBeforeAckSNRM
	JE   RespondToSnrm			;if data carrier detect = 0
; Continue if dcd is not zero after our delay. Some other station must be 
; responding. BX still contains the stationId. Zero this entry in the idTable
; and search for another SRNM.
%IF (%Debug) THEN (;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
	MOV  SI,logAckSnrmIndex
	MOV  logAckSnrmFail[SI],0FFh
)FI ;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
	MOV  BH,0
	MOV  BL,stationAddress
	MOV  BYTE PTR idTable-1[BX],0
; Generate a random number of SNRMs to wait for.
	CALL GenerateRandomNumber
	AND  CL,0Fh
	INC  CL						;Gives a number from 1 to 16
	MOV  snrmsToWaitFor,CL
	MOV  pollState,OFFSET StateIdle
	JMP  ReadNextId

RespondToSnrm:
PUBLIC RespondToSnrm
; We are ready to respond to a SNRM. If this is an old master, respond with
; UA. If this is a new master, respond with an XID. We assume we have a new
; master until we find out otherwise. 

%IF (%Ngen) THEN (
; Set the line speed to the value specified by sioClock.  We do change the
; line speed if we receive an XID from the master.
	MOV  BL,sioClock
	CALL SetLineSpeed
)FI
	MOV  lineState,stateAckSnrmWithXID		;Assume new master
	CMP  fOldMaster,0FFh
	JNE  AckSnrm
	MOV  lineState,stateAckSnrmWithUA
%IF (%Debug) THEN (;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
	CMP  fCrashIfOldMaster,0FFh
	JNE  OldMasterOk
	MOV  AX,3
	PUSH AX
	CALL Crash
OldMasterOk:
)FI ;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

AckSnrm:
%IF (%Aws) THEN (
	MOV  comm5Control,comm5Dtr+comm5Tx8+comm5CrcCcitt+comm5Rts
)FI
	JMP  SendPollResponse

RxAckSnrmWithXID:
PUBLIC RxAckSnrmWithXID
;*****************************************************************************
; We wish to lock onto an id. We got a reply after sending an XID to the 
; master in response to a SNRM. The next frame that we see must be an XID. If
; the master wants to change station address, the XID will have the new
; stationAddress in the AField, and our old address will be in xidOldAddress.
; ES points to the ackInBuf.
; AL = stationAddress
;*****************************************************************************
	JZ   RxAckSnrmWithXID2
	JMP	 RetrySnrm
RxAckSnrmWithXID2:	
	CMP  AH,frameTypeXID
	JE   CheckXid
	JMP  RetryXID
CheckXid:
	MOV  AH,ES:[xidRevisionLevel]
	CMP  AH,90h
	JLE  NoStationChange
; This is a new master, so the stationAddress could be different. Check to be
; sure this XID was addressed to us. It is ours if xidOldAddress contains our
; stationAddress. 
	MOV  DL,stationAddress
	CMP  DL,ES:[xidOldAddress]
	JMP  SHORT CheckStationAddress
NoStationChange:
; This is an old master, so check to see if the stationAddress of the XID is
; our old stationAddress.
	CMP  AL,stationAddress
CheckStationAddress:
	JNE  RetryXID
SaveMasterRevisionLevel:
; If the software revision level of the master > 90h, the master can exchange
; IFrames.
	MOV  masterRevisionLevel,AH
	MOV  stationAddress,AL
%IF (%Ngen) THEN (
	OR   parameterControl,sam		;Set to read only frames addressed to us
	MOV  DX,pcSarL
	OUT  AL,DX						;Send station address to the 2652
)FI

%IF (%Aws) THEN (
	MOV comm3Control,comm3Rx8+comm3RxCrcEnb+comm3Hunt+comm3RxEnb+comm3Srch
	MOV  DX,ioCommCtl
	MOV  AL,commWr6
	OUT  AL,DX
	MOV  AL,stationAddress
	OUT  AL,DX						;Send station address to chip
)FI

%IF (%Iws) THEN (
	MOV  comm3Control,comm3Rx8+comm3RxCrcEnb+comm3Hunt+comm3RxEnb+comm3Srch
	MOV  DX,ioCommCtl
	MOV  AL,commWr6
	%ZOUT(DX,AL)
	MOV  AL,stationAddress
	%ZOUT(DX,AL)
)FI

	MOV  lineState,stateWsReady

; The XID says what speed the master is running at.  Set up sioClock.
	MOV  SI,0-2						; loop index for FindSioClock
	MOV  BX,ES:[xidKHLineSpeed]
	CMP  DI, cbXidKHLineSpeed
	JAE  FindSioClock

; xidKhLineSpeed field not defined.  Use old xidMaxLineSpeed index, find kh.
	MOV  BL,ES:[xidMaxLineSpeed]	; Has value 7, 11 or 12.
	CMP  BL, 12
	JB   MaxOk
	MOV  BL, 12
MaxOk:
	XOR  BH,BH
    SHL  BX, 1
    MOV  BX, khFromXidMaxSpeed[BX]	; Translate to khLineSpeed.

FindSioClock:
	ADD  SI, 2
	CMP  BX, rgKHLineSpeed[SI]
	JB   FindSioClock
	SHR  SI, 1
	MOV  BL, rgSioClock[SI]

	MOV  sioClock,BL				;Set sioClock for timing
	CALL SetLineSpeed

CheckXBlockSize:
; See if we have different size XBlocks than the master. If so, change ours.
; CTOS/SRP 3.0 - The best thing would be to have the master send its sXBlk
; and just use it on the workstation.  Because of the need for compatibility
; with old workstations, the master must send (sXBlk-sXBlkRqMax)/512 in
; place of its old nSectorPerXBlock.
	MOV  AX, sXblk
	SUB  AX, sXBlkRqMax
	SHR  AX, 9
; AX = nSectorPerXBlock
	MOV  AH,ES:[xidNSectorPerXBlock] ;AH = nSectorPerXBlock of master
	CMP  AL,AH
	JE   XBlockSizeOk
	MOV  AL,0
	SHL  AX,1						;AX = nSectorPerXBlock * 512
; Prior to CTOS/SRP 3.0 the following was done:
;	ADD  AX,xbHeaderSize
;	MOV  sXbMax,AX
; That is clearly wrong, and disagrees with the old sysgen (also wrong) which
; added both the header and request size.  We really just want to add the size
; of the request, because we never dma the header.  The only reason this old
; code ever worked is because few people ever change the size of XBlocks, and
; the few that do probably don't ever run into the limit conditions.  The old
; sysgen erred on the side of making the XBlocks too big on the workstation, so
; it was never a problem.
	MOV  sXBData,AX
	ADD  AX,sXBlkRqMax
	MOV  sXBlk,AX
	CALL SendReinitMsg
XBlockSizeOk:
	MOV  lineState,stateAckXID
	JMP  SendPollResponse	
RetryXID:
	INC  findIdRetryCount
; Our reply of XID could fail for two reasons: (1) many workstations are all
; trying to respond at the same time and collisions occur, and (2) we are
; responding to an old master that does not recognize XIDs. In the case of an
; NGen master where we can have many workstations, we want to retry lots of
; times. For other workstations, it's not as important. nTryNewMaster is a
; Sysgen parameter. It should be around 10 for an NGen workstation on a 
; cluster with 40 workstations.
	MOV  AL,findIdRetryCount
	CMP  AL,nTryNewMaster
	JBE  RetrySnrm
; The master did not reply to our XID. Change the value of fOldMaster and call
; PrepareIdSearch again. The next time we will respond to a SNRM with UA 
; instead of XID. Since no XID to tell us xblock size, set to minimum.
	MOV  fOldMaster,0FFh
	MOV  AX, 2560
	MOV  sXBData,AX
	ADD  AX,sXBlkRqMax
	MOV  sXBlk,AX
	CALL SendReinitMsg
RetrySnrm:
%IF (%Debug) THEN (;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
	MOV  SI,logSnrmBufferIndex
	ADD  SI,logSnrmEntrySize
	CMP  SI,logSnrmBufferIndexMax
	JBE  logSnrmBufferIndexOk
	MOV  SI,0
logSnrmBufferIndexOk:
	MOV  logSnrmBufferIndex,SI
	MOV  logSnrmFrameIn[SI],AX
	MOV  AX,lineState
	MOV  logSnrmLineState[SI],AX
	MOV  logSnrmErc[SI],CX
)FI ;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
	CALL PrepareIdSearch
	JMP  IsrDone

RxAckXID:
;*****************************************************************************
; We sent UA to the master in response to the XID received. We expect
; to receive RR or RNR from the master. 
;*****************************************************************************

RxAckSnrmWithUA:
PUBLIC RxAckSnrmWithUA
;*****************************************************************************
; We wish to lock onto an id. We sent UA to the master in response to a SNRM. 
; The next frame that we see must be RR or RNR addressed to us.
; If we get a good response from the Master, mediate this interrupt and
; send a message to the Agent to tell him we are up.
;*****************************************************************************
	JNZ  RetrySnrm

%IF (%Ngen) THEN (
	OR   parameterControl,sam		;Set to read only frames addressed to us
)FI

%IF (%Aws OR %Iws) THEN (
	MOV comm3Control,comm3Rx8+comm3RxCrcEnb+comm3Hunt+comm3RxEnb+comm3Srch
)FI

	MOV  lineState,stateWsReady
	CMP  AH,frameTypeRR
	JE   MasterIsUp
	CMP  AH,frameTypeRNR
	JNE  RetrySnrm
MasterIsUp:
	CALL FAR PTR BitStartWrite		;Send RR to master
	OR   msgLph,msgMasterUp
	JMP  SendMessageToAgent								

RxWsNotReady:
;*****************************************************************************
; We sent RNR to the master
; We expect RR or RNR in reply.
; stateWsNotReady is set by BitStartWrite when we are in stateWsReady
; and we do not have a free XBlock.
;*****************************************************************************
	JNZ  PollError
	JMP  SHORT NotIFrame

RxBadIFrame:
PUBLIC RxBadIFrame
;*****************************************************************************
; We received a bad IFrame from the master and send a REJ frame back.
; The IFrame did not have a good oRcb field in the response field.
;*****************************************************************************
	MOV  lineState,stateWsReady

RxWsReady:
PUBLIC RxWsReady
;*****************************************************************************
; We are waiting for the master to poll us. We are reading into an XBlock. We 
; could receive an IFrame, RR, or RNR from the master.
;*****************************************************************************
	JNZ  PollError
	TEST AH,1
	JNZ  NotIFrame
	JMP  IFrameReceived
NotIFrame:
	MOV  DH,AH
	AND  DH,frameTypeU
	CMP  DH,frameTypeU
	JE   FrameTypeUReceived
	MOV  DH,AH
	AND  DH,frameMaskOffNR
	CMP  DH,frameTypeRR
	JE   MasterReady
	CMP  DH,frameTypeRNR
	JE   MasterNotReady
	MOV  CX,ercProtocolError
PollError:
	JMP  HandleErrorAndIsrDone

MasterNotReady:
MasterReadyNoIFrameToSend:
;*****************************************************************************
; We are still ready, so reply with RR. Our state could get changed to 
; stateNotReady when we answer the poll if we don't have a free XBlock.
;*****************************************************************************
	MOV  lineState,stateWsReady
	CALL FAR PTR BitStartWrite
	JMP  SendMessageToAgent

FrameTypeUReceived:
PUBLIC FrameTypeUReceived
;*****************************************************************************
; Come here because we received a U type frame addressed to our station.
; This probably means the master went down, came back up again, and is sending
; a SNRM to our station address.
; Mediate this interrupt and send a message to the Agent to tell
; the Agent the master went down.
; The Agent will call PrepareIdSearch.
;*****************************************************************************
	MOV  ercMasterDown,ercMasterProtocolFailure
SendMasterDownMessage:
; Come here from HandleError if we got too many errors.
	OR   msgLph,msgMasterDown
	JMP  SendMessageToAgent			;Returns enabled

MasterReady:
PUBLIC MasterReady
;*****************************************************************************
; We received RR from the master.  See if we have an IFrame to send. If not,
; send RR.
;*****************************************************************************
	MOV  lineState,stateSendIFrameWsNotReady
	CMP  masterRevisionLevel,90h
	JBE  SendNextFrame			;Master cannot exchange IFrames
; The master can ack an IFrame with another IFrame. 
	MOV  lineState,stateSendIFrameWsReady

SendNextFrame:
	CMP  nXBlocksToSend,0
	JE   MasterReadyNoIFrameToSend

; Find an rcb with a filled XBlock attached.
	MOV  BX,OFFSET DGROUP:rcbWaitingHead
TestNextRcb:
	MOV  BX,rcbNext[BX]
	CMP  BX,OFFSET DGROUP:rcbWaitingHead
	JE   MasterReadyNoIFrameToSend	; No XBlock found, list end.
	CMP  rcbsaXBlock, 0
    JE   TestNextRcb
	MOV  ES, rcbsaXBlock
	MOV  saXBlockOut, ES
	MOV  AX, ES:[xbSaLink]			; Unlink XBlock from rcb.
	MOV  rcbsaXBlock,AX
	DEC  nXBlocksToSend

; saXBlockOut points to the XBlock to send.
; We must keep this XBlock until we receive a positive ack from the master. 
	CLI								;Disable
	CALL FAR PTR BitStartWrite		;Send the filled XBlock
	JMP  SendMessageToAgent

IFrameReceived:
PUBLIC IFrameReceived
;*****************************************************************************
; Come here from RxWsReady
; We received an IFrame from the master. 
; We are in state WsReady
; ES and saXBlockCurrent point to the IFrame received.
; AX contains station frame
; DI contains number of characters read
; Test the sequence number to make sure we did not lose a frame. 
; If sequence is ok, see if the master can receive an IFrame in reply.
;*****************************************************************************
; DI contains the number of bytes transferred. Check to be sure this is the
; same as the number of bytes sent by the WS agent.
	SUB  DI,2						;DI = xbCbData-2
%IF (%Iws) THEN (
	MOV  DL,ES:[xbCbDataSent]		;DL = cbDataSent low
	MOV  DH,ES:[xbCbDataSent+2]		;DH = cbDataSent high
	CMP  DI,DX
) ELSE (
	CMP  DI,ES:[xbCbDataSent]
)FI
	JE   CheckSequenceNumber
LengthError:
	INC  nLengthError
	MOV  CX,ercLengthError
	JMP  HandleErrorAndIsrDone
CheckSequenceNumber:
%IF (%Debug) THEN (;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
	PUSH AX							;Save AX
	PUSH DI
	MOV  DI,logXBlockInIndex
	ADD  DI,logXBlockInEntrySize
	CMP  DI,logXBlockInIndexMax
	JBE  logXBlockInIndexOk
	MOV  DI,0
logXBlockInIndexOk:
	MOV  logXBlockInIndex,DI
	MOV  AX,DGroup
	MOV  ES,AX						;ES points to DGroup
	LEA  DI,logXBlockInBuffer[DI]	;ES:DI point to log entry
	MOV  DS,saXBlockCurrent			;DS points to XBlock
	MOV  SI,xbStation				;DS:SI point to xbStation
	MOV  CX,logXBlockInEntrySize/2
	CLD
	REPNZ MOVSW
	MOV  AX,DGroup
	MOV  DS,AX						;DS points to DGroup
	POP  DI
	POP  AX							;Restore AX
	MOV  ES,saXBlockCurrent
)FI ;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

%IF (%Debug) THEN (;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
	PUSH AX							;Save AX
	PUSH DI
	MOV  DI,logRqBufferIndex
	ADD  DI,logRqEntrySize
	CMP  DI,logRqBufferIndexMax
	JBE  logRqBufferIndexOk
	MOV  DI,0
logRqBufferIndexOk:
	MOV  logRqBufferIndex,DI
	MOV  logRqPoint[DI],5   		;Request comes back from master
%IF (%Iws) THEN (
	MOV  AL,ES:[xbRqCode]							
	MOV  AH,ES:[xbRqCode+2]							
) ELSE (
	MOV  AX,ES:[xbRqCode]
)FI
	MOV  logRqCode[DI],AX
%IF (%Iws) THEN (
	MOV  BL,ES:[xbRqRespExch]
	MOV  BH,ES:[xbRqRespExch+2]
) ELSE (
	MOV  BX,ES:[xbRqRespExch]
)FI
	MOV  logRqoRcb[DI],BX
%IF (%Iws) THEN (
	MOV  AL,ES:[xbRqErcRet]							
	MOV  AH,ES:[xbRqErcRet+2]							
) ELSE (
	MOV  AX,ES:[xbRqErcRet]
)FI
	MOV  logRqErcRet[DI],AX
	POP  DI
	POP  AX							;Restore AX
)FI ;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

	MOV  DH,AH						;Sequence no. comes in bits 3 2 1
	SHL  DH,1
	SHL  DH,1
	SHL  DH,1
	SHL  DH,1						;DH has sequence no in bits 7 6 5
	CMP  DH,NR
	JE   SequenceNumberOk

IFrameSequenceError:
; We did not receive the IFrame sequence number expected. Resend the last
; frame. The master will then resend the last IFrame.
	INC  nSequenceError
	MOV  CX,ercSequenceError
	JMP  HandleErrorAndIsrDone

SequenceNumberOk:
; We can't verify the length if we have an old master. Old masters do not
; align response data on an even boundary. If we have an odd number of bytes
; of control info and an odd number of bytes of response data, the response
; size will not come out right for old masters.
;	CMP  masterRevisionLevel,90h
;	JBE  VerifyoRcb
; Some comm servers in an attempt to improve line effiency change the response
; cb from max count to exact count.  This worked for old code but not for new.
; Don't check so old servers still work.
	JMP  VerifyoRcb
; Calculate the length of the request block header and the response data and
; check to see this is the same as the number of bytes received. If we are
; having line problems, or especially if a terminator is missing either at
; the master or at the end of the line, it is possible to receive an echo. 
; That is, we send a request block and it comes right back to us. This is a
; way of checking that we did not receive an echo, thinking it is our 
; response.

%IF (%Iws) THEN (
; If the request came back with erc 33, then the length will not be correct.
; This is an iodiosyncrasy of 9.1 non IOP masters. If we got back erc 33, don't ; check the length.
	MOV  DL,ES:[xbRqErcRet]						
	MOV  DH,ES:[xbRqErcRet+2]						
	CMP  DX,33
	JE   VerifyoRcb
	PUSH AX						;Save stationFrame
	MOV  DL,ES:[xbRqNReqPbCb]						
	MOV  BX,12
	ADD  BL,ES:[xbRq]			;BX = sCntlInfo + 12
	MOV  AL,6
	MUL  DL						;AX = 6*nReqPbCb
	ADD  BX,AX					;BX = sCntlInfo + 12 + 6*nReqPbCb
	MOV  AX,BX					;AX = sCntlInfo + 12 + 6*nReqPbCb
	SHL  BX,1
; BX points to first response pb/cb pair.
; AX contains length of request block header + response data.
	XOR  CX,CX
	MOV  CL,ES:[xbRqNRespPbCb]	;CX = nRespPbCb
	JCXZ TestLengthReceived
CalculateSize:
	MOV  DL,ES:[BX+xbRq+8]					
	MOV  DH,ES:[BX+xbRq+10]		;DX = next cb
	INC  DX						;Round to word boundary
	AND  DL,0FEh
	ADD  AX,DX					;Add next cb to total size
	ADD  AX,6					;Add 6 for next pb/cb pair to total
	ADD  BX,12					;Add 6 to index next pb/cb pair
) ELSE (
; If the request came back with erc 33, then the length will not be correct.
; This is an iodiosyncrasy of 9.1 non IOP masters. If we got back erc 33, don't ; check the length.
	MOV  DX,ES:[xbRqErcRet]						
	CMP  DX,33
	JE   VerifyoRcb
	PUSH AX						;Save stationFrame
	MOV  DX,ES:[xbRqNReqPbCb]	;DL = nReqPbCb, DH = nRespPbCb
	MOV  BX,12
	ADD  BL,ES:[xbRq]			;BX = sCntlInfo + 12
	MOV  AL,6
	MUL  DL						;AX = 6*nReqPbCb
	ADD  BX,AX					;BX = sCntlInfo + 12 + 6*nReqPbCb
	MOV  AX,BX
; BX points to first response pb/cb pair.
; AX contains length of request block header + response data.
	XOR  CX,CX
	MOV  CL,DH					;CX = nRespPbCb
	JCXZ TestLengthReceived
CalculateSize:
	MOV  DX,ES:[BX+xbRq+4]		;DX = next cb
	INC  DX						;Round to word boundary
	AND  DL,0FEh
	ADD  AX,DX					;Add next cb to total size
	ADD  AX,6					;Add 6 for next pb/cb pair to total
	ADD  BX,6					;Add 6 to index next pb/cb pair
)FI
	LOOP CalculateSize
TestLengthReceived:
	INC  AX
	INC  AX						;Add 2 for xbCbData
	INC  AX
	AND  AL,0FEh				;And round to even number
	CMP  DI,AX
%IF (%Debug) THEN (;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
	MOV  lengthCalculated,AX
)FI
	POP  AX						;Restore stationFrame
	JE   VerifyoRcb
	JMP  LengthError

VerifyoRcb:
; Increment number of requests received.
	ADD  NR,20h
; Get oRcb from response exchange of request so we can verify this IFrame as
; one of ours.
%IF (%Iws) THEN (
	MOV  BL,ES:[xbRqRespExch]
	MOV  BH,ES:[xbRqRespExch+2]
) ELSE (
	MOV  BX,ES:[xbRqRespExch]
)FI
; Check validity range of rcb.
	MOV  DX,rcbWaitingHead
	CMP  DX,OFFSET DGROUP:rcbWaitingHead
	JE   InvalidRcb
	CMP  BX,oRcbFirst
	JB   InvalidRcb
	CMP  BX,oRcbLast
	JA   InvalidRcb
; Check to see if we have responded to this rcb.  
; If so, the prq is nil.
	CMP  rcbState,stateReset
	JE   TestFrameBit
	CMP  rcbState,stateWaitingForXmit
	JLE  InvalidRcb
	CMP  rcbState,stateWaitingForClusterTerm
	JLE  CheckRqCode
InvalidRcb:
PUBLIC InvalidRcb
; We got an IFrame that either does not belong to us or has garbage in it. 
; This can happen when the master comes up and many workstations are trying to
; respond to a SNRM at the same time. It is possible that we have the same ID
; as someone else.
;	MOV  ercMasterDown,ercInvalidRcb
;	JMP  SendMasterDownMessage
; When a IOGA server xmt underruns, there is a chance that a packet can get 
; through with the correct CRC and Length, but the oRCB is trashed.  This used
; to be fatal. Now we treat it like a length error.  Our response back to the
; server will cause a sequence error and subsequent re-transmission of the
; offending I-Frame packet.
	SUB	 NR,20h						; Bad Packet to adjust sequence back.
	MOV  CX,ercLengthError
	INC  nBadRcbError				; Track as invalidrcb.
	JMP  HandleErrorAndIsrDone
	
CheckRqCode:
; Check the rqCode in this request to assure that we got the request that we
; were waiting for.
	MOV  DX,ES:[xbRqCode]		;DX = rqCode from XBlock
	PUSH ES						;Save pointer to XBlock
	LES  DI,rcbpRq				;ES:DI point to client rq
	CMP  WORD PTR ES:[DI+rqCode],rcResetAgent
	JE   oRcbOk					;ResetAgent comes back as CloseAllFiles
	CMP  WORD PTR ES:[DI+rqCode],rcLockInCache
	JE   oRcbOk					;rcLockInCache comes back as Read
	CMP  DX,ES:[DI+rqCode]
oRcbOk:
	POP  ES						;Restore pointer to XBlock
	JNE  InvalidRcb
TestFrameBit:
	TEST AH,frameFBit
	JZ   IframeReceivedMasterReady
NoIFrameToSend:
;*****************************************************************************
; Either we don't have an IFrame to send, or the master is not ready to
; receive an IFrame. We must respond with RR or RNR. If we DO have another
; XBlock, send RR to the master and send the IFrame to the Agent. The next
; read will read into the other XBlock. If we do NOT have another XBlock, send
; RNR to the master and send the IFrame to the Agent. The next read will read
; into ackInBuf. 
;*****************************************************************************
; Zero saXBlockCurrent since we don't own it any more and set the line state
; to stateWsReady. This will get changed to stateWsNotReady if we don't have a
; free XBlock.
	MOV  lineState,stateWsReady
	XOR  AX,AX
	XCHG AX,saXBlockCurrent
	MOV  ES,AX					;Save address of XBlock received
	CALL ChainXBlockIn
	CALL FAR PTR BitStartWrite	;Send RR or RNR (saXBlockCurrent must be 0)
; Send IFrame to the Agent. This is a response to a request sent out by the
; Agent. 
	OR   msgLph,msgXBlockIn		;Tell Agent this is an input IFrame
	JMP  SendMessageToAgent			

IframeReceivedMasterReady:
PUBLIC IframeReceivedMasterReady
;*****************************************************************************
; We received an IFrame, and the master is ready to receive an IFrame. If we
; don't have an IFrame ready, send RR or RNR. If we do have an IFrame but we
; don't have another XBlock, send RNR.
; DI contains number of characters read
; ES contains pointer to XBlock
;*****************************************************************************
	CALL ChainXBlockIn
; Zero saXBlockCurrent because we don't own it.
	MOV  saXBlockCurrent,0
	OR   msgLph,msgXBlockIn		;Tell Agent this is an input IFrame
	MOV  lineState,stateAckIFrameWithIFrame
	JMP  SendNextFrame

SendReinitMsg PROC NEAR
PUBLIC SendReinitMsg
;*****************************************************************************
; Send a message to the Agent to reinitialize the XBlocks.  We cannot do them
; here because the Agent could be using an XBlock.  
; As long as the msgReinitXBlocks bit is set in msgLph, we will not
; allocate an XBlock.
;*****************************************************************************
	OR   msgLph,msgReinitXBlocks
	CALL FAR PTR SendMessageToAgent
	RET
SendReinitMsg ENDP

SendMessageToAgent:
PUBLIC SendMessageToAgent
;*****************************************************************************
; msgLph contains bits to tell the Agent what to do:
; msgMasterDown    EQU 1
; msgMasterUp      EQU 2
; msgXBlockIn      EQU 4
; msgXBlockFill    EQU 8
; msgReinitXBlocks EQU 10h
;*****************************************************************************
	MOV  AX,WORD PTR msgLph
	OR   AX,AX
	JNZ  SendMsgLph
	JMP  IsrDone
$MOD386
SendMsgLph:
	BTS  fMsgLphQueued, 0		; has WsAgent processed last msg?
	JC   SendMsgDone			; no, don't send again
$MOD286
	MOV  DX,exchAgent
	PUSH DX
	PUSH DS						;sa of message
	LEA  AX,msgLph
	PUSH AX						;ra of message
%IF (%Debug) THEN (;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
	MOV  SI,logMsgBufferIndex
	ADD  SI,logMsgEntrySize
	CMP  SI,logMsgBufferIndexMax
	JBE  logMsgBufferIndexOk
	MOV  SI,0
logMsgBufferIndexOk:
	MOV  logMsgBufferIndex,SI
	MOV  AX,oAgentMessage
	MOV  logMsgRa[SI],AX
)FI ;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
	CALL PSend
SendMsgDone:
	RET								;Return to interrupted process

RxSendIFrameWsNotReady:
PUBLIC RxSendIFrameWsNotReady
;*****************************************************************************
; We sent an IFrame to the master and our state is not Ready. 
; We expect RR or RNR in reply.
; Check the sequence number of the ack to be sure the master got our IFrame.
;*****************************************************************************
;	JNZ  SendIFrameError
	JNZ  SendIFrameErrorWsNotReady	; R&B
	MOV  DH,AH
	AND  DH,frameTypeU
	CMP  DH,frameTypeU
	JE   FrameTypeUReceived1
	CALL CheckAckIFrame
;	JNZ  SendIFrameError
	JNZ  SendIFrameErrorWsNotReady	; R&B
	JMP  NotIFrame
SendIFrameErrorWsNotReady:			; R&B ...
PUBLIC SendIFrameErrorWsNotReady
; If we get any kind of error on the ack of our IFrame, we must resend the
; last IFrame at the next opportunity.
	MOV  lineState,stateResendIFrame
	CMP  masterRevisionLevel,90h
	JBE  HandleErrorAndIsrDone1
;	MOV  lineState,stateResendIFrameNewMaster
	MOV  lineState,stateResendIFrameNewMasterNotReady
	JMP  HandleErrorAndIsrDone1		; ... R&B
FrameTypeUReceived1:
	JMP  FrameTypeUReceived

RxAckIFrameWithIFrame:
PUBLIC RxAckIFrameWithIFrame
;*****************************************************************************
; We sent an IFrame to the master in reply to an IFrame and our state is 
; Ready. We expect RR, RNR or an IFrame in reply.
;*****************************************************************************

RxSendIFrameWsReady:
;*****************************************************************************
; We sent an IFrame to the master and our state is Ready. 
; We expect RR, RNR or an IFrame in reply.
; Check the sequence number of the ack to be sure the master got our IFrame.
; DI contains number of bytes read
;*****************************************************************************
;	JNZ  SendIFrameError
	JNZ  SendIFrameErrorWsReady		; R&B
	MOV  DH,AH
	AND  DH,frameTypeU
	CMP  DH,frameTypeU
	JE   FrameTypeUReceived1
	CALL CheckAckIFrame
;	JNZ  SendIFrameError
	JNZ  SendIFrameErrorWsReady		; R&B
AckIFrameOk:
	MOV  ES,saXBlockCurrent			; ES points to frame just read
	JMP  RxWsReady
;SendIFrameError:
SendIFrameErrorWsReady:				; R&B
PUBLIC SendIFrameErrorWsReady
; If we get any kind of error on the ack of our IFrame, we must resend the
; last IFrame at the next opportunity.
	MOV  lineState,stateResendIFrame
	CMP  masterRevisionLevel,90h
	JBE  HandleErrorAndIsrDone1
;	MOV  lineState,stateResendIFrameNewMaster
	MOV  lineState,stateResendIFrameNewMasterReady	; R&B 
; If we are sending RNR to the master, the master may never respond RR if we
; have too many requests outstanding. stateResendIFrameNewMaster is just like
; stateResendIFrame except that we send 2 extra bytes. When the master sees a
; 4-byte RNR, it knows that we need a RR to avoid a deadlock, and it will skip
; the check of outstanding requests.
HandleErrorAndIsrDone1:
	JMP  HandleErrorAndIsrDone

RxResendIFrame:
;*****************************************************************************
; We did not get a good response to the last IFrame we sent.
; We sent RNR in response to the ack of our last IFrame.
; Resend the IFrame at the next opportunity.
; It is possible that the master did receive the IFrame, and we got an
; error on the ack of the IFrame, so keep looking for a good ack.
;*****************************************************************************
	JNE  ResendError
	MOV  DH,AH
	AND  DH,frameTypeU
	CMP  DH,frameTypeU
	JNE  NotFrameTypeU
	JMP  FrameTypeUReceived
NotFrameTypeU:
	CALL CheckAckIFrame
	JZ   AckIFrameOk
	MOV  lineState,stateResendIFrame
	MOV  DH,AH
	AND  DH,frameMaskOffNR
	CMP  DH,frameTypeRR
	JNE  CantResendYet
; Now we can resend the last IFrame
	MOV  lineState,stateSendIFrameWsNotReady
	JMP  SHORT SendPollResponse
CantResendYet:
	CMP  DH,frameTypeRNR
	JE   MasterNotReadyYet
	MOV  CX,ercProtocolError
ResendError:
HandleErrorAndIsrDone:
	CALL HandleError
	JMP  IsrDone

MasterNotReadyYet:
; Keep sending RNR until the master is ready for our IFrame

SendPollResponse:
	CALL FAR PTR BitStartWrite
	JMP  IsrDone
ReadInterrupt ENDP
ChainXBlockIn PROC NEAR
PUBLIC ChainXBlockIn
;*****************************************************************************
; ES points to XBlock received. Chain this to the list pointed to by
; saXBLockIn. It is possible that the Agent is processing one XBlock when we
; are ready to send it another.
;*****************************************************************************
	CMP  saXBlockIn,0
	JNZ  ListXBlockInNotEmpty
	MOV  saXBlockIn,ES
	RET

ListXBlockInNotEmpty:
; The Agent is already processing one XBlock. Chain the one that just came in to the end of the list.
	MOV  AX,ES						;AX has new saXBlockIn
	MOV  ES,saXBlockIn				;ES points to list
FindEndOfXBlockIn:
	CMP  WORD PTR ES:[xbSaLink],0
	JE   ChainToEnd  
	MOV  ES,ES:[xbSaLink]
	JMP  FindEndOfXBlockIn
 ChainToEnd:
	MOV  ES:[xbSaLink],AX			;End of list points to new XBlockIn
	RET
ChainXBlockIn ENDP
CheckAckIFrame PROC NEAR
PUBLIC CheckAckIFrame
;*****************************************************************************
; We just received an ack to the IFrame we sent to the master. 
; saXBlockOut points to the IFrame just sent. 
; AH contains the frameCField received. Make sure
; the sequence number of the ack is correct. If correct, return CC zero.
; If not correct, return CC <> zero.
; If correct, return the XBlock to the free list.
; Return with lineState set to stateWsReady. It will get changed to
; stateWsNotReady if we don't have a free XBlock.
; DI still contains number of bytes read -- do not disturb!
;*****************************************************************************
	MOV  lineState,stateWsReady	;Ready if we have a free XBlock
	MOV  DH,AH						;Preserve AX in case of error
	AND  DH,0E0h
	MOV  DL,NS
	ADD  DL,20h						;Increment sequence number sent
	CMP  DH,DL
	MOV  CX,ercSequenceError
	JNE  CheckAckError
; The sequence number was correct, so update Number Sent. 
	MOV  NS,DL
FreeXBlock:
; Now return the XBlock to the free list.
; This entry is called if the master goes down while we are sending an XBlock.
	MOV  ES,saXBlockOut				;ES points to XBlock just sent
	MOV  saXBlockOut,0
	MOV  DX,saFreeList
	MOV  ES:[xbSaLink],DX			;XBlock just sent points old free list
	MOV  saFreeList,ES				;saFreeList points to XBlock just sent
	OR   msgLph,msgXBlockFill		;Agent has one more XBlock to fill
	CMP  AX,AX						;Zero flag set -> ack ok
	RET
CheckAckError:
	INC  nSequenceError
	RET
CheckAckIFrame ENDP

GetXBlock PROC NEAR
PUBLIC GetXBlock
;*****************************************************************************
; See if an XBlock is available. If so, unchain it from the free list.
; On return, ES and saXBlockCurrent point to XBlock.
; Condition code is zero if no XBlock is available.
;*****************************************************************************
	TEST msgLph,msgReinitXBlocks	;Do XBlocks need to be reinitialized?
	JZ   GetXTestFreeList			;No
	XOR  AX,AX						;Set condition code -- no XBlocks avail
	JMP  SHORT GetXBlockRet
GetXTestFreeList:
	MOV  AX,saFreeList
	MOV  saXBlockCurrent,AX			;Save pointer to XBlock to read into
	MOV  ES,AX						;ES points to XBlock
	OR   AX,AX
	JZ   GetXBlockRet				;no XBlock is free
	MOV  AX,0						;AX = 0
	MOV  ES:[xbFlags],AL			;Initialize xbFlags
	XCHG AX,ES:[xbSaLink]			;AX = next pointer, next pointer = 0
	MOV  saFreeList,AX				;saFreeList points to next XBlock
GetXBlockRet:
	RET
GetXBlock ENDP
HandleError PROC NEAR
PUBLIC HandleError
;*****************************************************************************		
; If the maximum error count has not been reached, resend the last frame.
; If the maximum error count has been reached, tell the Agent the master is
; down and begin searching for the next SNRM. 
; On entry, CX = error code.
; AX = first word received from workstation -- stationID and frameType
;*****************************************************************************		

%IF (%Debug) THEN (;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
	MOV  DI,logErrorIndex
	ADD  DI,logErrorEntrySize
	CMP  DI,logErrorIndexMax
	JBE  logErrorIndexOk
	MOV  DI,0
logErrorIndexOk:
	MOV  logErrorIndex,DI
	MOV  logErrorErc[DI],CX
	MOV  logErrorFrame[DI],AX
)FI ;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

	MOV  bitIoErc,CX
	CMP  CX,ercProtocolError
	JNE  NotProtocolError
	INC  nProtocolError
NotProtocolError:
	INC  retryCount
	CMP  retryCount,maxErrorCount
	JAE  TooManyErrors
	MOV  AL,stationAddressIn
	CMP  AL,stationAddress
	JNE  RetryRead
	CALL FAR PTR BitStartWrite		;Send last message again.
	RET
RetryRead:
; If we got an address error, we don't want to try to reply because the frame
; is not addressed to us. Restart the read.
	MOV  SI,lineState
	CALL FAR PTR BitStartRead
	RET
HandleError ENDP

TooManyErrors:
; If we get too many errors, we must tell the Agent the master is down. We
; must not have any state in our stack when we mediate the interrupt in order
; to send a message, so discard the return address.
	MOV  ercMasterDown,ercCommHardwareFailure
	POP  AX
	JMP  SendMasterDownMessage

ClearStatsAtMidnight PROC FAR
PUBLIC ClearStatsAtMidnight
;*****************************************************************************	
; Called by FilterPros at midnight.
;*****************************************************************************	
	ENTER	0,0
	PUSH	DS
	POP		ES
	LEA		DI,WsStats
; sizeWsStats - 2 so we don't clear nTicksPerSecond 
	MOV		CX,sizeWsStats-2
	CLD
	XOR		AX,AX
	REPNZ	STOSB
	LEAVE
	RET
ClearStatsAtMidnight ENDP


WsLph ENDS

; LOG
; 3/28/84 by Jim Frandeen: Created file
; 3/1/85 by Jim Frandeen: Correct calculation of request size to account for
;           odd number of bytes of control info.
; 2/23/87 by RLM if no RCB's active the this RCB must be invalid.
; 3/12/87 erc 8126
; 9/11/87 JM/RLM, clusterTimeout configurable.
;12/14/89 by GWH, reset sioclock and linespeed after master goes
;				  and snrm is received.
; 01/09/90 by AT, use nXBlk, sXBlk, etc. instead of old variables.
; 01/25/90 by AT, Set sXBData for WsAgent.
; 05/21/90 by JA, Line speed from master SNRM or XID.  Send khLineSpeed in XID.
; 07/09/90 by JA, fix khFromXidMaxSpeed index.
; 07/12/90 by JM, if SNRM line speed is bogus, respond at 307kb.
; 07/30/90 by JM, obsolete nSectorPerXblock.
; 08/01/90 by JM, if old master (no XID), use 2.5k xblocks.
; 08/15/90 by JC, Added BTOS RS232 latency improvements, definition for
;				  ercIdSearchFailure (collision detection)
; 08/22/90 by JA, call ReInitXBlocks when sXBlk known.
; 08/24/90 by JA, Respond to poll immediately; fill/empty xblock between polls.
; 08/27/90 by JA, don't STI before SendMessageToAgent.
; 11/12/90 by JM, change max khFromXidMaxSpeed from 4000 to 3680.
; 01/05/91 by JA, Tolerate LockInCache returning as Read (no ercInvalidRcb).
; 03/13/91 by BA, Merged Rotate & Burst changes from s2.4.3 - see R&B comments.
; 04/23/91 by JA, Tolerate SNRM.khLineSpeed=9100h, treat as 307Kb.
; 04/29/91 by JF, Make sure all the XBlocks are free before ReInitXBlocks.
; 05/14/91 by JF, Move ReInitXBlocks to Agent process.
; 05/24/91 by JF, If state is reset don't check rcbprq because it was
;					set to nil.
; 12/10/91 by GWH, VerifyRcb is no longer fatal.  Now we treat an invalid rcb
;				  check as a length error and respond to the server with a
;				  sequence error.  This fix is needed as the IOGA (T2&T3)
;				  modules have lots of tx underruns some of which can pass
;				  CRC and Length tests but finally fail the RCB check.
; 6/17/93  by JM  synchronize with WsAgent to not queue multiple msgLph's - 
;				  cures ercNoLinkBlock crash on busy systems.
; 06/23/93 by JF, Create procedure ClearStatsAtMidnight to replace SETB
;				in FilterPros so that FilterPros will work as well with
;				a different LAN.
