/*****************************  MODULE HEADER  *******************************/
/*                                                                           */
/*                                                                           */
/*  MACHINE:                LANGUAGE:  Metaware C            OS: CTOS        */
/*                                                                           */
/*  seq_service_buffers.c                                                    */
/*                                                                           */
/*  Buffer manipulation/copy procedures for CTOS Sequential Access service.  */
/*                                                                           */
/*  HISTORY:                                                                 */
/*  --------                                                                 */
/*                                                                           */
/*  MM/DD/YY  VVVV/MM  PROGRAMMER    /  DESCRIPTION                          */
/*                                                                           */
/*  05/20/91  121J.05  P. Johansson  /  Correct errors in variable length    */
/*                                      record support.                      */
/*  05/17/91  121J.04  P. Johansson  /  Enhancements for unbuffered support. */
/*  04/19/91  121J.03  P. Johansson  /  Buffer sizes other than multiples    */
/*                                      of 512 are permitted.                */
/*  04/02/91  121H.02  P. Johansson  /  Bus address for remote slot copies   */
/*                                      was not updated correctly.           */
/*  01/02/91  121F.01  P. Johansson  /  Enhance buffer pool initialization.  */
/*  12/09/90  121E.00  P. Johansson  /  Created.                             */
/*                                                                           */
/*                    PROPRIETARY  PROGRAM  MATERIAL                         */
/*                                                                           */
/*  THIS MATERIAL IS PROPRIETARY TO UNISYS CORPORATION AND IS NOT TO         */
/*  BE REPRODUCED, USED OR DISCLOSED EXCEPT IN ACCORDANCE WITH PROGRAM       */
/*  LICENSE OR UPON WRITTEN AUTHORIZATION OF THE PATENT DIVISION OF          */
/*  UNISYS CORPORATION, DETROIT, MICHIGAN 48232, USA.                        */
/*                                                                           */
/*  COPYRIGHT (C) 1990 UNISYS CORPORATION. ALL RIGHTS RESERVED               */
/*                                                                           */
/*****************************************************************************/
/*                                                                           */
/*  UNISYS BELIEVES THAT THE SOFTWARE FURNISHED HEREWITH IS ACCURATE         */
/*  AND RELIABLE, AND MUCH CARE HAS BEEN TAKEN IN ITS PREPARATION. HOWEVER,  */
/*  NO RESPONSIBILITY, FINANCIAL OR OTHERWISE, CAN BE ACCEPTED FOR ANY       */
/*  CONSEQUENCES ARISING OUT OF THE USE OF THIS MATERIAL, INCLUDING LOSS     */
/*  OF PROFIT, INDIRECT, SPECIAL, OR CONSEQUENTIAL DAMAGES, THERE ARE NO     */
/*  WARRANTIES WHICH EXTEND BEYOND THE PROGRAM SPECIFICATION.                */
/*                                                                           */
/*  THE CUSTOMER SHOULD EXERCISE CARE TO ASSURE THAT USE OF THE SOFTWARE     */
/*  WILL BE IN FULL COMPLIANCE WITH LAWS, RULES AND REGULATIONS OF THE       */
/*  JURISDICTIONS WITH RESPECT TO WHICH IT IS USED.                          */
/*                                                                           */
/**************************  END OF MODULE HEADER  ***************************/

#ifdef debug
#define private
#else
#define private static
#endif

/* Standard C library macros and functions invoked by this module */

pragma Off(List);
#include <intel80X86.h>
#include <string.h>
pragma Pop(List);

/* There are no procedures in the Sequential Access service that can cope with
   a variable number of arguments, so this pragma makes everything much more
   efficient.  However, it has to be established AFTER any standard C library
   functions are defined because it reverses the normal C convention. */

pragma Calling_convention(_CALLEE_POPS_STACK);

/* External CTOS and CTOS Toolkit functions invoked by this module */

#define AllocMemorySL
#define CheckErc
#define DmaTransfer
#define ErrorExit
#define ExpandAreaSL
#define FProtectedMode
#define FSrp
#define Wait

pragma Off(List);
#include <ctosLib.h>
pragma Pop(List);

#if defined(debug) && defined(breakpoint)
#undef breakpoint
extern void breakpoint(unsigned debug_value_for_AX);
#endif

/* Type definitions used by this module */

#define last(array) (sizeof(array) / sizeof(*array) - 1)

#define iob_dma_type

pragma Off(List);
#include <ctosTypes.h>
#include <ext_ctos_types.h>
#include "seq_service.h"
pragma Pop(List);

/* Other external functions in this application invoked by this module */

extern char input(unsigned io_address);
extern void output(unsigned io_address, char value);
extern void outword(unsigned io_address, unsigned value);

/* Error return codes used by this module */

pragma Off(List);
#include <erc.h>
pragma Pop(List);

/* External variables imported by this module */

extern unsigned default_resp_exch;

/* Static variables global within this manuscript */

private buffer_descriptor_type *avail_descriptor = (void *) &avail_descriptor;
private unsigned buffer_segments = 0;
private Boolean dma_service_available;
private unsigned log2[16] = {0x0001, 0x0002, 0x0004, 0x0008,
                             0x0010, 0x0020, 0x0040, 0x0080,
                             0x0100, 0x0200, 0x0400, 0x0800,
                             0x1000, 0x2000, 0x4000, 0x8000};
private segment_descriptor_type *segment_descriptor =
                                                  (void *) &segment_descriptor;

/* Function prototypes defined before the functions themselves are declared */

void reserve_buffer_page(segment_descriptor_type *segment_descriptor);

pragma Page(1);
/*-----------------------------------------------------------------------------
 The size of the buffer pool allocated is limited only by the amount of memory
 available, but because of Intel segmented addressing limitations it is
 managed as a collection of buffer areas, each of which may be up to one
 segment (64 Kb) long.  The granularity for the allocation of buffers is 512
 byte chunks; if the final buffer segment allocated is smaller than 64 Kb, the
 descriptor must be modified so that the nonexistent portions of the segment
 are marked "in use." */

void initialize_buffer_pool(unsigned buffer_pages) {

   unsigned alloc_size, i, j;

   dma_service_available = (FSrp(NULL) && FProtectedMode());
   buffer_segments = ((((unsigned long) buffer_pages) << LOG2_PAGE_SIZE)
                      + (SEGMENT_SIZE - 1)) >> LOG2_SEGMENT_SIZE;
   alloc_size = sizeof(buffer_descriptor_type) * (buffer_pages + 1);
   CheckErc(ExpandAreaSL(alloc_size, selector_of(avail_descriptor),
                         &offset_of(avail_descriptor)));
   memset(avail_descriptor, 0, alloc_size);
   for (i = 0; i < buffer_pages; i++)
      avail_descriptor[i].next = &avail_descriptor[i + 1];
   alloc_size = buffer_segments * sizeof(segment_descriptor_type);
   CheckErc(ExpandAreaSL(alloc_size, selector_of(segment_descriptor),
                         &offset_of(segment_descriptor)));
   memset(segment_descriptor, 0, alloc_size);
   i = 0;
   while (buffer_pages >= PAGES_PER_SEGMENT) {
      CheckErc(AllocMemorySL(SEGMENT_SIZE - 1,	/* Allocate a segment */
                             &segment_descriptor[i].raw_buffer));
      segment_descriptor[i++].bit_map[0] = 2;	/* One 64K buffer available */
      buffer_pages -= PAGES_PER_SEGMENT;
   }
   if (buffer_pages > 0) {			/* Need a partial segment? */
      CheckErc(AllocMemorySL(buffer_pages * PAGE_SIZE,	/* Allocate it... */
                             &segment_descriptor[i].raw_buffer));
      segment_descriptor[i].bit_map[0] = 2;	/* As if 64 Kb available... */
      for (j = 0; j < PAGES_PER_SEGMENT - buffer_pages; j++)
         reserve_buffer_page(&segment_descriptor[i]);	/* ...reserve unused */
   }

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 If a buffer pool segment less than 64 Kb is allocated, the unavailable buffer
 pages must be marked "in use" in the segment descriptor.  This procedure uses
 the same algorithm as 'allocate_buffer()' but it does not calculate any
 buffer addresses or allocate a buffer descriptor---it only updates the free
 and available "buddy" bits in the segment descriptor. */

private void reserve_buffer_page(segment_descriptor_type *segment_descriptor) {

   unsigned i, j, log2_avail;

   for (log2_avail = LOG2_PAGE_SIZE; log2_avail <= LOG2_SEGMENT_SIZE;
        log2_avail++) {
      j = 2 * (1 << (16 - log2_avail));
      for (i = j / 2; i < j; i++)
         if (segment_descriptor->bit_map[i / 16] & log2[i % 16]) {
            segment_descriptor->bit_map[i / 16] ^= log2[i % 16];
            while (log2_avail != LOG2_PAGE_SIZE) {
               i *= 2;
               segment_descriptor->bit_map[i / 16] |= log2[i % 16];
               i++;
               log2_avail--;
            }
            return;
         }
   }
   ErrorExit(ercInconsistency);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 All of the buffers for each 64 Kb buffer segment are organized and managed
 according to the "buddy" system.  To allocate a buffer of size n bytes, the
 smallest k where 2 ** k is greater than or equal to n is determined and the
 buffer pool is searched for a buffer of exactly this size.  If no such buffer
 is found, a search for a buffer of size 2 ** (k + 1) is performed and the
 buffer found will be split into two (with one of the two new buffers being
 used to satisfy the allocation request.  The buffer search proceeds within
 each segment to larger buffer sizes until a buffer is found and otherwise
 proceeds from segment to segment.  If no buffer can be located, a null
 pointer is returned to the user. */

buffer_descriptor_type *allocate_buffer(unsigned buffer_size) {

   buffer_descriptor_type *buffer_descriptor;
   unsigned i, j, k, log2_avail, log2_target = 0, rounded_buffer_size;

   if (avail_descriptor->next == NULL)
      return(NULL);	/* If it can't be described, it can't be returned */
   if (buffer_size > 0x8000)
      log2_target = 16;		/* Allocate a 64 Kb buffer */
   else {
      rounded_buffer_size = buffer_size;
      while (rounded_buffer_size >>= 1)
         log2_target++;		/* Calculate log2(buffer_size) once... */
      rounded_buffer_size = buffer_size + (log2[log2_target] - 1);
      log2_target = 0;
      while (rounded_buffer_size >>= 1)
         log2_target++;		/* ...so we can recalculate to 2**k properly */
   }
   for (i = 0; i < buffer_segments; i++)
      for (log2_avail = log2_target; log2_avail <= LOG2_SEGMENT_SIZE;
           log2_avail++) {
         k = 2 * (1 << (16 - log2_avail));
         for (j = k / 2; j < k; j++)
            if (segment_descriptor[i].bit_map[j / 16] & log2[j % 16]) {
               segment_descriptor[i].bit_map[j / 16] ^= log2[j % 16];
               while (log2_avail != log2_target) {
                  j *= 2;
                  segment_descriptor[i].bit_map[j / 16] |= log2[j % 16];
                  j++;
                  log2_avail--;
               }
               buffer_descriptor = avail_descriptor->next;
               avail_descriptor->next = buffer_descriptor->next;
               buffer_descriptor->next = NULL;
               buffer_descriptor->base = segment_descriptor[i].raw_buffer;
               offset_of(buffer_descriptor->base) = (1 << log2_target)
                              * (j - (1 << (LOG2_SEGMENT_SIZE - log2_target)));
               buffer_descriptor->data = buffer_descriptor->base;
               buffer_descriptor->bus_address = 0;
               buffer_descriptor->size = buffer_size;
               buffer_descriptor->available = buffer_size;
               buffer_descriptor->index = (i << 8) + j;
               return(buffer_descriptor);
            }
      }
   return(NULL);		/* No buffer available */
}

pragma Page(1);
/*-----------------------------------------------------------------------------
 When a buffer is returned by a user, a check is made to see if its "buddy" is
 free, also.  If both buffers are free, they are coalesced into one larger
 buffer and this same check is applied recursively up to the limit of the 64
 Kb segment size.  An entirely free segment is denoted by a single available
 bit in the bit map for buffer size 64 Kb. */
 
void deallocate_buffer(buffer_descriptor_type *buffer_descriptor) {

   unsigned long buffer_size;
   unsigned i, j, log2_free = 0;

   i = buffer_descriptor->index >> 8;
   j = buffer_descriptor->index & 0xFF;
   buffer_size = buffer_descriptor->size + PAGE_SIZE - 1;
   while (buffer_size >>= 1)
      log2_free++;
   while (log2_free < LOG2_SEGMENT_SIZE
       && (segment_descriptor[i].bit_map[(j ^ 1) / 16] & log2[(j ^ 1) % 16])) {
      segment_descriptor[i].bit_map[(j ^ 1) / 16] ^= log2[(j ^ 1) % 16];
      j /= 2;
      log2_free++;
   }
   segment_descriptor[i].bit_map[j / 16] |= log2[j % 16];
   memset(buffer_descriptor, 0, sizeof(buffer_descriptor_type));
   buffer_descriptor->next = avail_descriptor->next;
   avail_descriptor->next = buffer_descriptor;

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 This function copies data between the client's buffer and a buffer on the Dcb
 'client_buffer' queue.  The 'available' field in the buffer descriptor
 governs how much may be copied:  for reads (copies from the buffer to the
 client data space) 'available' indicates how much valid data is still left in
 the buffer, for writes (copies from the client data space to the buffer)
 'available' indicates how much empty space is left to receive data.  When
 'available' goes to zero, the buffer is transferred from the 'client_buffer'
 queue to the end of the 'device_buffer' queue where it may be consumed by a
 future device data transfer operation.  Note that the sense of 'available' is
 reset when this transfer takes place (e.g. a buffer just emptied of read data
 has 'available' reset to the full size of the buffer to indicate how much
 space is ready for a read from the device. */

void copy_client_buffer(dcb_type *dcb) {

   buffer_descriptor_type *client_buffer, *device_buffer;
   iob_dma_type iob;
   void *msg, *remote_data;
   char save_slot_id;
   unsigned xfer_length;

   client_buffer = dcb->client_buffer;	/* Optimize pointer dereferences */
   if ((xfer_length = _min(dcb->rq_residual, client_buffer->available)) == 0)
      return;
   if (dma_service_available) {
      memset(&iob, 0, sizeof(iob));
      if (!dcb->state.inbound)
         iob.ctrl_flags =  TARGET_WRITE;
      if ((iob.slot_id = dcb->rq_slot_id) == 0)
         iob.requester = dcb->rq_data;
      else {
         iob.ctrl_flags |= REQUESTER_MAPPED;
         iob.bus_address_requester = (unsigned long) dcb->rq_data;
      }
      iob.target = client_buffer->data;
      iob.completion_exch = default_resp_exch;
      iob.completion_msg = &iob;
      iob.xfer_length = xfer_length;
      if (DmaTransfer(offset_of(iob.completion_msg)) == ercOK)
         Wait(iob.completion_exch, &msg);
   } else if (dcb->rq_slot_id == 0)
      if (dcb->state.inbound)
         memcpy(dcb->rq_data, client_buffer->data, xfer_length);
      else
         memcpy(client_buffer->data, dcb->rq_data, xfer_length);
   else {
      save_slot_id = input(REMOTE_SLOT_REG);
      output(REMOTE_SLOT_REG, dcb->rq_slot_id);
      outword(BASE_REG0, selector_of(dcb->rq_data));
      outword(BASE_REG1, selector_of(dcb->rq_data) + 1);
      selector_of(remote_data) = REMOTE_ADDRESS_SELECTOR
                                  + (offset_of(dcb->rq_data) >> 4);
      offset_of(remote_data) = offset_of(dcb->rq_data) & 0x000F;
      if (dcb->state.inbound)
         memcpy(remote_data, client_buffer->data, xfer_length);
      else
         memcpy(client_buffer->data, remote_data, xfer_length);
      output(REMOTE_SLOT_REG, save_slot_id);
   }
   if (dcb->rq_slot_id == 0)
      offset_of(dcb->rq_data) += xfer_length;
   else
      *((unsigned long *) &dcb->rq_data) += xfer_length;
   dcb->rq_residual -= xfer_length;
   offset_of(client_buffer->data) += xfer_length;
   if (     (client_buffer->available -= xfer_length) == 0
         || dcb->operating_mode.variable_length) {
      client_buffer->data = client_buffer->base;
      client_buffer->bus_address = 0;
      if (dcb->operating_mode.variable_length)
         if (dcb->state.inbound) {
            dcb->rq_residual -= client_buffer->available;
            client_buffer->available = client_buffer->size;
         } else
            client_buffer->available = client_buffer->size
                                        - client_buffer->available;
      else
         client_buffer->available = client_buffer->size;
      dcb->client_buffer = client_buffer->next;
      client_buffer->next = NULL;
      device_buffer = (buffer_descriptor_type *) &dcb->device_buffer;
      while (device_buffer->next != NULL)
         device_buffer = device_buffer->next;
      device_buffer->next = client_buffer;
   }

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 When a checkpoint (or other I/O operation such as writing a filemark that
 implies a checkpoint) operation occurs, partially full buffers on the client
 queue must be transferred to the device queue so they may be written. */

void flush_client_buffer(dcb_type *dcb) {

   buffer_descriptor_type *client_buffer, *device_buffer;

   client_buffer = dcb->client_buffer;	/* Optimize pointer dereferences */
   if (client_buffer->available != client_buffer->size) {
      client_buffer->data = client_buffer->base;
      client_buffer->bus_address = 0;
      client_buffer->available = client_buffer->size
                                  - client_buffer->available;
      dcb->client_buffer = client_buffer->next;
      client_buffer->next = NULL;
      device_buffer = (buffer_descriptor_type *) &dcb->device_buffer;
      while (device_buffer->next != NULL)
         device_buffer = device_buffer->next;
      device_buffer->next = client_buffer;
   }

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 This function copies data from a buffer on the Dcb 'device_buffer' queue to
 the client's request buffer.  The 'available' field in the buffer descriptor
 governs how much may be copied from the buffer to the client data space.  When
 'available' goes to zero, the buffer is transferred from the 'device_buffer'
 queue to the end of the 'client_buffer' queue where it may be consumed by a
 future client data transfer operation.  Note that the sense of 'available' is
 reset when this transfer takes place (e.g. a buffer just emptied of unwritten
 data has 'available' reset to the full size of the buffer to indicate how
 much space is ready to accept data from a client request. */

void recover_device_buffer(dcb_type *dcb) {

   buffer_descriptor_type *client_buffer, *device_buffer;
   iob_dma_type iob;
   void *msg, *remote_data;
   char save_slot_id;
   unsigned xfer_length;

   device_buffer = dcb->device_buffer;	/* Optimize pointer dereferences */
   xfer_length = _min(dcb->rq_residual, device_buffer->available);
   if (dma_service_available) {
      memset(&iob, 0, sizeof(iob));
      if ((iob.slot_id = dcb->rq_slot_id) == 0)
         iob.requester = dcb->rq_data;
      else {
         iob.ctrl_flags |= REQUESTER_MAPPED;
         iob.bus_address_requester = (unsigned long) dcb->rq_data;
      }
      iob.target = device_buffer->data;
      iob.completion_exch = default_resp_exch;
      iob.completion_msg = &iob;
      iob.xfer_length = xfer_length;
      if (DmaTransfer(offset_of(iob.completion_msg)) == ercOK)
         Wait(iob.completion_exch, &msg);
   } else if (dcb->rq_slot_id == 0)
      memcpy(dcb->rq_data, device_buffer->data, xfer_length);
   else {
      save_slot_id = input(REMOTE_SLOT_REG);
      output(REMOTE_SLOT_REG, dcb->rq_slot_id);
      output(REMOTE_SLOT_REG, dcb->rq_slot_id);
      outword(BASE_REG0, selector_of(dcb->rq_data));
      outword(BASE_REG1, selector_of(dcb->rq_data) + 1);
      selector_of(remote_data) = REMOTE_ADDRESS_SELECTOR
                                  + (offset_of(dcb->rq_data) >> 4);
      offset_of(remote_data) = offset_of(dcb->rq_data) & 0x000F;
      if (dcb->state.inbound)
         memcpy(remote_data, device_buffer->data, xfer_length);
      else
         memcpy(device_buffer->data, remote_data, xfer_length);
      output(REMOTE_SLOT_REG, save_slot_id);
   }
   if (dcb->rq_slot_id == 0)
      offset_of(dcb->rq_data) += xfer_length;
   else
      *((unsigned long *) &dcb->rq_data) += xfer_length;
   dcb->rq_residual -= xfer_length;
   offset_of(device_buffer->data) += xfer_length;
   if ((device_buffer->available -= xfer_length) == 0) {
      device_buffer->data = device_buffer->base;
      device_buffer->bus_address = 0;
      device_buffer->available = device_buffer->size;
      dcb->device_buffer = device_buffer->next;
      device_buffer->next = NULL;
      client_buffer = (buffer_descriptor_type *) &dcb->client_buffer;
      while (client_buffer->next != NULL)
         client_buffer = client_buffer->next;
      client_buffer->next = device_buffer;
   }

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 After data has been transferred between the peripheral device and the buffer,
 this function may be called to update the buffer pointers and available
 counters.  When no more buffer is available (either full buffer for reads or
 empty buffer for writes), the buffer is transferred from the 'device_buffer'
 queue in the Dcb to the 'client_buffer' queue. */

void update_device_buffer(dcb_type *dcb) {

   buffer_descriptor_type *client_buffer, *device_buffer;

   if (dcb->operating_mode.buffered) {
      if (dcb->io_xfer_actual != 0) {
         device_buffer = dcb->device_buffer;	/* Optimize dereferences */
         offset_of(device_buffer->data) += dcb->io_xfer_actual;
         device_buffer->bus_address += dcb->io_xfer_actual;
         if (     (device_buffer->available -= dcb->io_xfer_actual) == 0
               || dcb->operating_mode.variable_length) {
            device_buffer->data = device_buffer->base;
            device_buffer->bus_address = 0;
            device_buffer->available = device_buffer->size
                                        - device_buffer->available;
            dcb->device_buffer = device_buffer->next;
            device_buffer->next = NULL;
            client_buffer = (buffer_descriptor_type *) &dcb->client_buffer;
            while (client_buffer->next != NULL)
               client_buffer = client_buffer->next;
            client_buffer->next = device_buffer;
         }
      }
   } else {
      dcb->rq_residual -= dcb->io_xfer_actual;
      if ((dcb->unbuffered.available -= dcb->io_xfer_actual) == 0) {
         memset(&dcb->unbuffered, 0, sizeof(dcb->unbuffered));
         dcb->device_buffer = NULL;
      } else {
         offset_of(dcb->unbuffered.data) += dcb->io_xfer_actual;
         dcb->unbuffered.bus_address += dcb->io_xfer_actual;
      }
   }

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 When an exception condition has occurred (even a "normally" expected one such
 as encountering a filemark during read-ahead operations) any partially full
 buffer(s) in the device queue must be transferred to the client queue so the
 data in them is accessible. */

void flush_device_buffer(dcb_type *dcb) {

   buffer_descriptor_type *client_buffer, *device_buffer;

   if (dcb->operating_mode.buffered) {
      device_buffer = dcb->device_buffer;	/* Optimize dereferences */
      if (device_buffer->available != device_buffer->size) {
         device_buffer->data = device_buffer->base;
         device_buffer->bus_address = 0;
         device_buffer->available = device_buffer->size
                                     - device_buffer->available;
         dcb->device_buffer = device_buffer->next;
         device_buffer->next = NULL;
         client_buffer = (buffer_descriptor_type *) &dcb->client_buffer;
         while (client_buffer->next != NULL)
            client_buffer = client_buffer->next;
         client_buffer->next = device_buffer;
      }
   } else {
      memset(&dcb->unbuffered, 0, sizeof(dcb->unbuffered));
      dcb->device_buffer = NULL;
   }

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Return all the buffer queues to their initial (empty or available) state.
 This may occur if the user rewinds a tape opened in read mode---the
 look-ahead data already in the buffers is not needed and must be discarded. */

void purge_buffers(dcb_type *dcb) {

   buffer_descriptor_type *buffer;

   if (dcb->state.inbound) {
      if (dcb->device_buffer != NULL) {
         dcb->device_buffer->data = dcb->device_buffer->base;
         dcb->device_buffer->bus_address = 0;
         dcb->device_buffer->available = dcb->device_buffer->size;
      }
      while ((buffer = dcb->client_buffer) != NULL) {
         dcb->client_buffer = buffer->next;
         buffer->next = dcb->device_buffer;
         buffer->data = buffer->base;
         buffer->bus_address = 0;
         buffer->available = buffer->size;
         dcb->device_buffer = buffer;
      }
   } else {
      if (dcb->client_buffer != NULL) {
         dcb->client_buffer->data = dcb->client_buffer->base;
         dcb->client_buffer->bus_address = 0;
         dcb->client_buffer->available = dcb->client_buffer->size;
      }
      while ((buffer = dcb->device_buffer) != NULL) {
         dcb->device_buffer = buffer->next;
         buffer->next = dcb->client_buffer;
         buffer->data = buffer->base;
         buffer->bus_address = 0;
         buffer->available = buffer->size;
         dcb->client_buffer = buffer;
      }
   }

}
