/*****************************  MODULE HEADER  *******************************/
/*                                                                           */
/*                                                                           */
/*  MACHINE:                LANGUAGE:  Metaware C            OS: CTOS        */
/*                                                                           */
/*  seq_service_pertec.c                                                     */
/*                                                                           */
/*  SRP/XE Pertec device procedures for CTOS Sequential Access service.      */
/*                                                                           */
/*  HISTORY:                                                                 */
/*  --------                                                                 */
/*                                                                           */
/*  MM/DD/YY  VVVV/MM  PROGRAMMER    /  DESCRIPTION                          */
/*                                                                           */
/*  05/17/91  121J.04  P. Johansson  /  Don't allow buffered mode to change  */
/*                                      in SeqAccessModeSet; use request     */
/*                                      code value 0xFFFE to signal wake-up  */
/*                                      in round-robin algorithm.            */
/*  04/29/91  121J.03  P. Johansson  /  Fixes to checkpoint logic at EOM;    */
/*                                      round-robin "fairness" algorithm     */
/*                                      corrected.                           */
/*  04/10/91  121H.02  P. Johansson  /  Filter 'ercTapeNotReady' to 'ercOK'  */
/*                                      if command was UNLOAD.               */
/*  04/05/91  121H.01  P. Johansson  /  Set 'cant_deinstall' flag in Dcb.    */
/*  03/12/91  121H.00  P. Johansson  /  Created.                             */
/*                                                                           */
/*                    PROPRIETARY  PROGRAM  MATERIAL                         */
/*                                                                           */
/*  THIS MATERIAL IS PROPRIETARY TO UNISYS CORPORATION AND IS NOT TO         */
/*  BE REPRODUCED, USED OR DISCLOSED EXCEPT IN ACCORDANCE WITH PROGRAM       */
/*  LICENSE OR UPON WRITTEN AUTHORIZATION OF THE PATENT DIVISION OF          */
/*  UNISYS CORPORATION, DETROIT, MICHIGAN 48232, USA.                        */
/*                                                                           */
/*  COPYRIGHT (C) 1990 UNISYS CORPORATION. ALL RIGHTS RESERVED               */
/*                                                                           */
/*****************************************************************************/
/*                                                                           */
/*  UNISYS BELIEVES THAT THE SOFTWARE FURNISHED HEREWITH IS ACCURATE         */
/*  AND RELIABLE, AND MUCH CARE HAS BEEN TAKEN IN ITS PREPARATION. HOWEVER,  */
/*  NO RESPONSIBILITY, FINANCIAL OR OTHERWISE, CAN BE ACCEPTED FOR ANY       */
/*  CONSEQUENCES ARISING OUT OF THE USE OF THIS MATERIAL, INCLUDING LOSS     */
/*  OF PROFIT, INDIRECT, SPECIAL, OR CONSEQUENTIAL DAMAGES, THERE ARE NO     */
/*  WARRANTIES WHICH EXTEND BEYOND THE PROGRAM SPECIFICATION.                */
/*                                                                           */
/*  THE CUSTOMER SHOULD EXERCISE CARE TO ASSURE THAT USE OF THE SOFTWARE     */
/*  WILL BE IN FULL COMPLIANCE WITH LAWS, RULES AND REGULATIONS OF THE       */
/*  JURISDICTIONS WITH RESPECT TO WHICH IT IS USED.                          */
/*                                                                           */
/**************************  END OF MODULE HEADER  ***************************/

#define debug

#ifdef debug
#define private
#else
#define private static
#endif

/* Standard C library macros and functions invoked by this module */

pragma Off(List);
#include <intel80X86.h>
#include <string.h>
pragma Pop(List);

/* There are no procedures in the Sequential Access service that can cope with
   a variable number of arguments, so this pragma makes everything much more
   efficient.  However, it has to be established AFTER any standard C library
   functions are defined because it reverses the normal C convention. */

pragma Calling_convention(_CALLEE_POPS_STACK);

/* External CTOS and CTOS Toolkit functions invoked by this module */

#define CloseRTClock
#define MapBusAddress
#define OpenRTClock
#define PaFromP
#define Send
#define SetIntHandler

pragma Off(List);
#include <ctoslib.h>
pragma Pop(List);

#if defined(debug) && defined(breakpoint)
#undef breakpoint
extern void breakpoint(unsigned debug_value_for_AX);
#endif

/* Type definitions used by this module */

#define last(array) (sizeof(array) / sizeof(*array) - 1)

#define SysConfigType
#define tpib_type

pragma Off(List);
#include <ctosTypes.h>
#include <ext_ctos_types.h>
#include "pertec.h"
#include "seq_service.h"
pragma Pop(List);

/* Other external functions in this application invoked by this module */

extern void disable(void);
extern void enable(void);
extern void exit_with_msg(unsigned erc, unsigned msg_num);
extern unsigned inword(unsigned io_address);
extern void outword(unsigned io_address, unsigned value);
extern void update_device_buffer(dcb_pertec_type *dcb);

/* Error return codes used by this module */

#define TapeErc

pragma Off(List);
#include <erc.h>
pragma Pop(List);

/* External variables imported by this module */

extern unsigned own_user_num;
extern unsigned serv_exch;
extern SysConfigType *sysConfig;

/* Static variables global within this manuscript */

private dcb_pertec_type *active_dcb = NULL;
private unsigned n_units = 0;
private unsigned unit_map[MAX_UNITS] = {0, 2, 1, 3, 4, 6, 5, 7};
private watchdog_type watchdog = {0, 10, 0, 0, 0, 0xFFFF};

/* Function prototypes defined before the functions themselves are declared */

unsigned issue_pertec_command(dcb_pertec_type *dcb, unsigned command);
unsigned map_pertec_status();
void pertec_interrupt(void);
void pertec_round_robin(dcb_pertec_type *dcb, Boolean poll_rewinds);
unsigned pertec_status(dcb_pertec_type *dcb);

pragma Page(1);
/*-----------------------------------------------------------------------------
 This procedure is invoked when the Sequential Access service is installed.
 Note that we must be executing on an SP or DP processor board in an SRP (or
 else there is no Pertec tape controller hardware!).  If we are in the right
 place at the right time, reset the Pertec controller, establish the interrupt
 vector and configure the interrupt controller to accept interrupts.  Any
 other initializations necessary are performed when the device is opened. */

void pertec_init(dcb_pertec_type *dcb, unsigned unit) {

   unsigned erc;
   void *data_segment = &data_segment;

   if (sysConfig->HardwareType != SP && sysConfig->HardwareType != DP)
      exit_with_msg(ercNoTapeHardware, 0);
   n_units++;				/* Need to know how many to poll */
   dcb->cant_deinstall= TRUE;		/* There is no ResetIntHandler... */
   dcb->unit = unit;
   if (watchdog.dcb == NULL)		/* Build linked list for polling */
      watchdog.dcb = dcb->chain = dcb;
   else {
      dcb->chain = ((dcb_pertec_type *) watchdog.dcb)->chain;
      ((dcb_pertec_type *) watchdog.dcb)->chain = dcb;
   }
   watchdog.exch_resp = serv_exch;
   outword(TAPE_COMMAND_REG, RESET_FIRMWARE);
   outword(TAPE_GO_REG, 0xFFFF);
   if ((erc = SetIntHandler(14, (void *) pertec_interrupt,
                            selector_of(data_segment), TRUE, FALSE)) != ercOK)
      exit_with_msg(erc, 0);
   disable();
   outword(INT2_CTRL_REG, inword(INT2_CTRL_REG) & ~LEVEL_TRIGGERED);
   outword(INT_MASK_REG, inword(INT_MASK_REG) & ~MASK_INT2);
   enable();

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 This function provides device specific information about buffer capacity so
 that SeqAccessOpen may allocate buffers intelligently. */

unsigned pertec_defaults(default_info_type *default_info) {

   default_info->device_buffer_size = 0;
   default_info->recommended_buffers = 2;
   default_info->recommended_buffer_size = 8192;
   default_info->recommended_buffer_pool_size = 16384;
   default_info->recommended_write_buffer_full_threshold = 8192;
   default_info->recommended_read_buffer_empty_threshold = 8192;
   return(ercOK);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Information that is in the "erasable" part of the Dcb must be reestablished
 each time the device is opened (the information in the device dependent
 portion, which is set up by the initialization procedure, is safe from
 erasure).  In addition, a real-time clock for the gross timing of 1/2" tape
 operations must be established. */

unsigned pertec_open(dcb_pertec_type *dcb) {

   unsigned erc;

   dcb->own_rq.s_cnt_info = 6;
   dcb->own_rq.n_req_pb_cb = 1;
   dcb->own_rq.exch_resp = serv_exch;
   dcb->own_rq.user_num = own_user_num;
   dcb->own_rq.param[0].pb = dcb;
   dcb->min_record_size = dcb->max_record_size = dcb->block_size = PAGE_SIZE;
   if (pertec_status(dcb) == REQUEST_IN_PROGRESS)
      erc = REQUEST_IN_PROGRESS;
   else if (watchdog.user_count++ == 0) {
      watchdog.counter = watchdog.counter_reload;
      watchdog.events = 0;
      if ((erc = OpenRTClock(&watchdog)) != ercOK)
         watchdog.user_count--;
   } else
      erc = ercOK;
   return(erc);

}

/*-----------------------------------------------------------------------------
 If we are the last user of the real-time clock, close it.  Otherwise, just
 return. */

unsigned pertec_close(dcb_pertec_type *dcb) {

   if (watchdog.user_count == 0)
      return(ercOK);
   if (--watchdog.user_count == 0)
      return(CloseRTClock(&watchdog));
   return(ercOK);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Return information about the current operating configuration of the half-inch
 tape.  The sequential parameters block is supplied zero-filled and/or with
 generic information already completed, so all we have to do here is flesh out
 the device-specific details. */

unsigned pertec_mode_query(dcb_pertec_type *dcb,
                           seq_parameters_type *seq_parameters) {

   seq_parameters->write_protected = dcb->unit_status.write_protected;
   seq_parameters->variable_length = dcb->operating_mode.variable_length;
   seq_parameters->unbuffered = !dcb->operating_mode.buffered;
   seq_parameters->speed = (dcb->high_speed) ? 1 : 0;
   seq_parameters->density = dcb->density;
   seq_parameters->block_size = dcb->block_size;
   seq_parameters->min_record_size = dcb->min_record_size;
   seq_parameters->max_record_size = dcb->max_record_size;
   seq_parameters->data_buffer_nonrecoverable = TRUE;
   seq_parameters->disable_error_correction = TRUE;
   seq_parameters->disable_read_retries = (dcb->read_retry_limit == 0); 
   seq_parameters->disable_write_retries = (dcb->write_retry_limit == 0); 
   seq_parameters->read_retry_limit = dcb->read_retry_limit;
   seq_parameters->write_retry_limit = dcb->write_retry_limit;
   return(ercOK);

}

/*-----------------------------------------------------------------------------
 Complementary to the inquiry function just preceding, establish some of the
 device-specific characteristics requested by the user. */

unsigned pertec_mode_set(dcb_pertec_type *dcb,
                         seq_parameters_type *seq_parameters) {

   if (     (   seq_parameters->variable_length
             && seq_parameters->max_record_size
                 <= seq_parameters->min_record_size
             && seq_parameters->block_size != 0)
         || (   !seq_parameters->variable_length
             && seq_parameters->min_record_size
                 != seq_parameters->max_record_size
             && seq_parameters->block_size % seq_parameters->block_size != 0))
      return(ercInvalidTapeParams);
   dcb->operating_mode.variable_length = seq_parameters->variable_length;
   dcb->high_speed = (seq_parameters->speed != 0);
   dcb->density = (seq_parameters->density == PE_3200) ? PE_3200 : PE_1600;
   if (seq_parameters->min_record_size != 0)
      dcb->min_record_size = seq_parameters->min_record_size;
   if (seq_parameters->max_record_size != 0)
      dcb->max_record_size = seq_parameters->max_record_size;
   if (seq_parameters->block_size != 0)
      dcb->block_size = seq_parameters->block_size;
   if (seq_parameters->disable_read_retries)
      dcb->read_retry_limit = 0;
   else
      dcb->read_retry_limit = (seq_parameters->read_retry_limit == 0)
                                        ? 8 : seq_parameters->read_retry_limit;
   if (seq_parameters->disable_write_retries)
      dcb->write_retry_limit = 0;
   else
      dcb->write_retry_limit = (seq_parameters->write_retry_limit == 0)
                                      ? 16 : seq_parameters->write_retry_limit;
   return(ercOK);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Status for one of the eight tape drives cannot be obtained unless the
 controller is idle.  If it is BUSY, return an in progress indication---the
 request will be rescheduled later.  Otherwise, read the tape status register
 (if necessary) and map the status bits into an error code. */

unsigned pertec_status(dcb_pertec_type *dcb) {

   unsigned i;

   disable();				/* Critical section with ISR */
   if (active_dcb != NULL) {
      dcb->state.awaiting_controller = TRUE;
      enable();
      return(REQUEST_IN_PROGRESS);
   }
   enable();
   if (!dcb->state.status_available) {
      outword(TAPE_COMMAND_REG, (dcb->command = NOP) | unit_map[dcb->unit]);
      outword(TAPE_GO_REG, 0xFFFF);
      for (i = 1; i <= 5; i++)			/* Delay a few microseconds */
         ;
      dcb->tape_status = inword(TAPE_STATUS_REG);
   }
   pertec_round_robin(dcb->chain, FALSE);	/* Give other units a break */
   dcb->state.pending_status = dcb->state.status_available = FALSE;
   return(map_pertec_status(dcb));

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 For each of the different SeqAccessControl functions, select the appropriate
 Pertec command and establish a timeout limit (in tenths of seconds) long
 enough for the operation to complete.  Note that record spacing has to be
 implemented by Pertec read data operations---in this case, the fact that
 'data_transfer' is not set in the Dcb state indicates that data read is to be
 discarded, not transferred to a (nonexistent!) user buffer. */

unsigned pertec_ctrl(dcb_pertec_type *dcb) {

   unsigned command;
   seq_access_rq_type *rq = dcb->rq;

   disable();				/* Critical section with ISR */
   if (active_dcb != NULL) {
      dcb->state.awaiting_controller = TRUE;
      enable();
      return(REQUEST_IN_PROGRESS);
   }
   enable();
   switch (dcb->ctrl_function = rq->ctrl_function) {
      case CTRL_REWIND:
      case CTRL_RETENSION:
         command = REWIND;
         dcb->timeout = 300;
         break;

      case CTRL_UNLOAD:
         command = UNLOAD;
         dcb->timeout = 300;
         break;

      case CTRL_ERASE_MEDIUM:
         command = ERASE_MEDIUM;
         dcb->timeout = 1500;
         break;

      case CTRL_WRITE_FILEMARK:
         if (rq->ctrl_qualifier < 0)
            return(ercTapeCommandIllegal);
         command = WRITE_FILEMARK;
         dcb->ctrl_qualifier = rq->ctrl_qualifier;
         dcb->timeout = 60;
         break;

      case CTRL_SCAN_FILEMARK:
         command = SEARCH_FILEMARK;
         if (rq->ctrl_qualifier < 0) {
            command |= REV;
            dcb->ctrl_qualifier = -rq->ctrl_qualifier;
         } else
            dcb->ctrl_qualifier = rq->ctrl_qualifier;
         dcb->timeout = 1500;
         break;

      case CTRL_SPACE_RECORD:
         command = READ;
         if (rq->ctrl_qualifier < 0) {
            command |= REV;
            dcb->ctrl_qualifier = -rq->ctrl_qualifier;
         } else
            dcb->ctrl_qualifier = rq->ctrl_qualifier;
         dcb->timeout = 1500;
         break;

      case CTRL_ERASE_GAP:
         if (rq->ctrl_qualifier < 0)
            return(ercTapeCommandIllegal);
         command = ERASE_GAP;
         dcb->ctrl_qualifier = rq->ctrl_qualifier;
         dcb->timeout = 60;
         break;

      default:
         return(ercTapeCommandIllegal);
   }
   if (     (dcb->position.beginning_of_medium && (command & REV))
         || (   dcb->position.end_of_medium
             && !(   command == REWIND || command == UNLOAD
                  || command == WRITE_FILEMARK || (command & REV) != 0)))
      return(ercTapeEomWarning);
   return(issue_pertec_command(dcb, command));

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Data transfer operations with the Pertec controller are inherently simple:
 load the DMA address, load the two's complement of the transfer count in
 bytes and then call the Pertec command procedure to start the transfer.
 However, because up to eight tape drives may share one controller we have to
 check if the controller is free before we start.  If not, simply return
 REQUEST_IN_PROGRESS.  When the controller is eventually free (see the end of
 the interrupt service routine), a poll is made of all tape Dcb's to see if
 any are in the data transfer state and awaiting I/O commencement.  If one
 such is found, I/O is resumed by the (somewhat sneaky) method of sending a
 "completion" Iob to the service exchange---the actual transfer count is zero,
 so the end result is that this procedure is invoked once again but this time
 the data transfer may be started because the controller is idle. */

unsigned pertec_io(dcb_pertec_type *dcb) {

   unsigned erc, garbage, i;
   unsigned long bus_address;

   if (      dcb->device_buffer->bus_address == 0
          && (erc = MapBusAddress(dcb->io_data = dcb->device_buffer->data,
                                  dcb->device_buffer->size, NO_DMA_FENCE,
                                  &dcb->device_buffer->bus_address,
                                  &garbage)) != ercOK)
      return(erc);
   disable();				/* Critical section with ISR */
   if (active_dcb != NULL) {
      dcb->state.awaiting_controller = TRUE;
      enable();
      dcb->io_xfer_actual = 0;
      return(REQUEST_IN_PROGRESS);
   }
   enable();
   outword(TAPE_COMMAND_REG, unit_map[dcb->unit]);
   for (i = 1; i <= 5; i++)	/* Delay a few microseconds... */
      ;
   if (!((dcb->tape_status = inword(TAPE_STATUS_REG)) & READY))
      return(ercTapeNotReady);
   dcb->ctrl_function = 0xFFFF;	/* Make sure it's not a real function */
   dcb->retry_count = 0;	/* Reset for each new I/O attempt */
   dcb->timeout = 60;
   outword(DMA_ADDR_LOW_REG, bus_address = dcb->device_buffer->bus_address);
   outword(DMA_ADDR_HIGH_REG, bus_address >> 16);
   outword(DMA_COUNT_REG, (dcb->io_xfer_count =
                        (dcb->operating_mode.variable_length)
                      ? dcb->device_buffer->available
                      : _min(dcb->block_size, dcb->device_buffer->available)));
   return(issue_pertec_command(dcb, (dcb->state.inbound) ? READ : WRITE));

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Commands to the tape drive are issued by writing them to the command register
 and then strobing the GO signal to the firmware.  Some commands (e.g. REWIND
 when the medium is already at BOT) may be done immediately, so check the tape
 status before returning an in-progress indicator.  Note that for all commands
 except REWIND and UNLOAD, it is an unexpected hardware error if the
 formatter is NOT busy after the command has been issued! */

private unsigned issue_pertec_command(dcb_pertec_type *dcb, unsigned command) {

   unsigned i;

   dcb->state.status_available = FALSE;	/* Status cleared by new command */
   *((char *) &dcb->command_status) = 0;	/* Unlatch prior results */
   dcb->io_xfer_actual = 0;		/* ISR determines how much */
   dcb->own_rq.erc_ret = REQUEST_IN_PROGRESS;
   outword(TAPE_COMMAND_REG, (dcb->command = command)
                              | ((dcb->high_speed) ? (HS | unit_map[dcb->unit])
                                                   : unit_map[dcb->unit]));
   outword(TAPE_GO_REG, 0xFFFF);
   for (i = 1; i <= 5; i++)		/* Delay a few microseconds... */
      ;
   dcb->tape_status = inword(TAPE_STATUS_REG);
   if ((dcb->tape_status & (BUSY | REWINDING | DATA_BUSY))) {
      if (command & (RW | RU)) {
         dcb->state.rewinding = TRUE;
         active_dcb = NULL;
         pertec_round_robin(dcb->chain, FALSE);	/* Controller is idle... */
      } else
         active_dcb = dcb;
      return(REQUEST_IN_PROGRESS);	/* The operation has commenced */
   }
   dcb->timeout = 0;		/* Either it's done or it never started */
   return((command & (RW | RU)) ? map_pertec_status(dcb) : ercTapeNotReady);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 The Pertec interrupt service implements a fairly complicated state machine
 that depends both upon the last command issued to the interface and the
 nature of the request that is underway.  In particular, SPACE and
 SPACE_REVERSE commands may be generated by a user request, may be the result
 of retry logic following (potentially) recoverable errors during a READ or
 WRITE_FILEMARK operation, or may be issued when looking for the 2nd through
 Nth filemarks in a SCAN_FILEMARK sequence (this is to make sure there is no
 intervening data between the consecutive filemarks).  If the operation is
 completed or else in error (and no retries are possible), send the
 information to the background service loop so that a Respond may be generated
 for the request.  Last, poll the other tape Dcb's to see if any other request
 may be started. */

void pertec_interrupt(void) {

   unsigned long bus_address;
   dcb_pertec_type *dcb;
   unsigned erc;

   if ((dcb = active_dcb) == NULL)	/* Check for bogus interrupts... */
      return;
   dcb->tape_status = inword(TAPE_STATUS_REG);
   dcb->state.status_available = TRUE;
   erc = map_pertec_status(dcb);
   switch (dcb->command) {
      case READ:
         if (erc == ercTapeIoError) {
            if (dcb->retry_count < dcb->read_retry_limit) {
               dcb->retry_count++;
               erc = issue_pertec_command(dcb, SPACE_REVERSE);
            }
         } else {
            dcb->io_xfer_actual = dcb->io_xfer_count - inword(DMA_COUNT_REG);
            if (erc == ercOK)
               if (dcb->io_xfer_actual != dcb->io_xfer_count)
                  if (dcb->operating_mode.variable_length)
                     erc = (   dcb->io_xfer_actual >= dcb->min_record_size
                            && dcb->io_xfer_actual <= dcb->max_record_size)
                           ? ercOK : ercInvalidTapeRecordSize;
                  else
                     erc = (dcb->io_xfer_actual % dcb->min_record_size == 0)
                           ? ercTapeRecordTruncated : ercInvalidTapeRecordSize;
               else if (dcb->io_xfer_actual < dcb->device_buffer->available) {
                  update_device_buffer(dcb);
                  active_dcb = NULL;		/* Sleight of hand... */
                  erc = pertec_io(dcb);		/* ...is reestablished */
               }
         }
         break;

      case WRITE:
      case WRITE_EDIT:
         if (erc == ercTapeIoError) {
            if (dcb->retry_count < dcb->write_retry_limit) {
               dcb->retry_count++;
               erc = issue_pertec_command(dcb, READ_EDIT_REVERSE);
            }
         } else {
            dcb->io_xfer_actual = dcb->io_xfer_count - inword(DMA_COUNT_REG);
            if (erc == ercOK)
               if (dcb->io_xfer_actual != dcb->io_xfer_count)
                  erc = ercTapeHardwareError;
               else if (dcb->io_xfer_actual < dcb->device_buffer->available) {
                  update_device_buffer(dcb);
                  active_dcb = NULL;		/* Sleight of hand... */
                  erc = pertec_io(dcb);		/* ...is reestablished */
               }
         }
         break;

      case READ_EDIT_REVERSE:
         if (erc == ercOK || erc == ercTapeIoError)
            if (dcb->retry_count == dcb->write_retry_limit / 2)
               erc = issue_pertec_command(dcb, ERASE_GAP);
            else {
               outword(DMA_ADDR_LOW_REG,
                       bus_address = dcb->device_buffer->bus_address);
               outword(DMA_ADDR_HIGH_REG, bus_address >> 16);
               outword(DMA_COUNT_REG, dcb->io_xfer_count);
               erc = issue_pertec_command(dcb, WRITE_EDIT);
            }
         break;

      case WRITE_FILEMARK:
         if (erc == ercTapeFileMark || erc == ercTapeEomWarning)
            if (--dcb->ctrl_qualifier == 0)
               erc = ercOK;
            else {
               dcb->retry_count = 0;
               erc = issue_pertec_command(dcb, WRITE_FILEMARK);
            }
         else if (erc == ercOK && dcb->retry_count < dcb->write_retry_limit) {
            dcb->retry_count++;
            erc = issue_pertec_command(dcb, SPACE_REVERSE);
         }
         break;

      case SEARCH_FILEMARK:
      case SEARCH_FILEMARK_REVERSE:
         if (erc == ercTapeFileMark)
            erc = (--dcb->ctrl_qualifier == 0)
                     ? ercOK
                     : issue_pertec_command(dcb, SPACE | (dcb->command & REV));
         else if (erc == ercOK)
            erc = ercTapeHardwareError;
         break;

      case ERASE_GAP:
      case SPACE:
      case SPACE_REVERSE:
         switch (dcb->ctrl_function) {
            case CTRL_ERASE_GAP:
            case CTRL_SPACE_RECORD:
               if (erc == ercOK || erc == ercTapeIoError)
                  erc = (--dcb->ctrl_qualifier == 0)
                             ? ercOK : issue_pertec_command(dcb, dcb->command);
               break;

            case CTRL_SCAN_FILEMARK:
               if (erc == ercTapeFileMark)
                  erc = (--dcb->ctrl_qualifier == 0)
                     ? ercOK
                     : issue_pertec_command(dcb, SPACE | (dcb->command & REV));
               else if (erc != ercTapeEomWarning){	/* Continue search.. */
                  dcb->ctrl_qualifier = dcb->rq->ctrl_qualifier;
                  erc = issue_pertec_command(dcb, SEARCH_FILEMARK
                                                   | (dcb->command & REV));
               }
               break;

            case CTRL_WRITE_FILEMARK:
               if (erc == ercOK || erc == ercTapeIoError)
                  erc = issue_pertec_command(dcb, WRITE_FILEMARK);
               break;

            default:			/* These are READ or WRITE retries */
               if (erc == ercOK || erc == ercTapeIoError) {
                  outword(DMA_ADDR_LOW_REG,
                          bus_address = dcb->device_buffer->bus_address);
                  outword(DMA_ADDR_HIGH_REG, bus_address >> 16);
                  outword(DMA_COUNT_REG, dcb->io_xfer_count);
                  erc = issue_pertec_command(dcb, (dcb->state.inbound)
                                                               ? READ : WRITE);
               }
         }
         break;

      case ERASE_MEDIUM:
         if (erc == ercTapeEomWarning)	/* Uncertain of actual Pertec status */
            erc = ercOK;
         break;
   }
   if (erc != REQUEST_IN_PROGRESS) {	/* Has I/O or a retry been started? */
      active_dcb = NULL;		/* No, the controller is free */
      pertec_round_robin(dcb->chain, TRUE);	/* "Fairness" algorithm */
      dcb->timeout = 0;			/* OK, this unit is done */
      dcb->own_rq.erc_ret = erc;
      dcb->own_rq.rq_code = 0;
      Send(dcb->own_rq.exch_resp, &dcb->own_rq);
   }

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 */

private void pertec_round_robin(dcb_pertec_type *dcb, Boolean poll_rewinds) {

   unsigned erc, i;

   for (i = 1; i < n_units; i++) {	/* Poll other units for a starter... */
      if (     poll_rewinds
            && dcb->state.rewinding
            && (erc = pertec_status(dcb)) != ercTapeBusy) {
         dcb->state.rewinding = FALSE;		/* Rewinding has finished */
         dcb->timeout = 0;
         dcb->own_rq.erc_ret = (   erc == ercTapeNotReady
                                && dcb->ctrl_function == CTRL_UNLOAD) ? ercOK
                                                                      : erc;
         Send(dcb->own_rq.exch_resp, &dcb->own_rq);
      } else if (dcb->state.awaiting_controller) {
         dcb->state.awaiting_controller = FALSE;
         dcb->own_rq.erc_ret = ercOK;
         dcb->own_rq.rq_code = (dcb->state.data_transfer) ? 0 : 0xFFFE;
         Send(dcb->own_rq.exch_resp, &dcb->own_rq);
         break;
      }
      dcb = dcb->chain;			/* Nothing to do, on to next Dcb */
   }

}

/*-----------------------------------------------------------------------------
 Although most of the completion and/or error logic is contained in the
 interrupt service itself, this stub of a procedure is necessary to return the
 error code determined by the interrupt service to the generic
 'seq_access_rq_done' which invoked this code. */

unsigned pertec_rq_done(dcb_pertec_type *dcb) {

   dcb->timeout = 0;			/* OK or in error, it's still done */
   if (dcb->own_rq.erc_ret != ercOK)
      dcb->state.pending_status = TRUE;
   return(dcb->own_rq.erc_ret);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 The watchdog procedure is periodically activated by a timer request block.
 Because the counter reload field is set to 10 and the timer has a granularity
 of 100 milliseconds, the 'events' field holds the number of seconds that have
 elapsed since the last invocation.  This value is used to decrement the
 timeout field in the Dcb, which is also expressed in one second units.
 If time has run out, interrupts are disabled to guard the critical section
 with the interrupt service routine and the HB-001 interface firmware is
 reset.  After this reset, the timeout error code is inserted into the Iob
 which is then sent to the service exchange for the normal completion
 processing.  While the tape device is open by a user, the watchdog is always
 running (even if no tape operations are in progress). */

void pertec_watchdog(watchdog_type *watchdog) {

   dcb_pertec_type *dcb = (dcb_pertec_type *) watchdog->dcb;
   unsigned erc, i;

   for (i = 1; i <= n_units; i++) {
      dcb = dcb->chain;
      if (     dcb->state.rewinding && active_dcb == NULL
            && (erc = pertec_status(dcb)) != ercTapeBusy) {
         dcb->state.rewinding = FALSE;		/* Rewinding has finished */
         dcb->timeout = 0;
         dcb->own_rq.erc_ret = (   erc == ercTapeNotReady
                                && dcb->ctrl_function == CTRL_UNLOAD) ? ercOK
                                                                      : erc;
         Send(dcb->own_rq.exch_resp, &dcb->own_rq);
      } else {
         disable();				/* Critical section with ISR */
         if (dcb->timeout > watchdog->events) {	/* Time still left? */
            dcb->timeout -= watchdog->events;	/* Yes, just decrement it */
            enable();				/* Exit critical section */
         } else if (dcb->timeout == 0)		/* Is the timer enabled? */
            enable();
         else {
            enable();
            dcb->timeout = 0;			/* Not any more! */
            outword(TAPE_COMMAND_REG, FE | dcb->unit);	/* Reset runaway... */
            outword(TAPE_GO_REG, 0xFFFF);
            if (active_dcb == dcb)
               active_dcb = NULL;
            dcb->own_rq.erc_ret = ercTapeTimeout;	/* Error completion */
            Send(dcb->own_rq.exch_resp, &dcb->own_rq);
         }
      }
   }
   watchdog->events = 0;		/* Ensure a subsequent reactivation */

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Synthesize the Pertec status information into the virtual status information
 maintained in the Dcb.  After that is done, see if an error return code has
 to be generated unique to the status condition. */

private unsigned map_pertec_status(dcb_pertec_type *dcb) {

   unsigned erc = ercOK;

   dcb->unit_status.on_line = ((dcb->tape_status & ONLINE) != 0);
   dcb->unit_status.write_protected =
                                     ((dcb->tape_status & WRITE_PROTECT) != 0);
   dcb->unit_status.ready = ((dcb->tape_status & READY) != 0);
   dcb->unit_status.busy = ((dcb->tape_status & (REWINDING | DATA_BUSY)) != 0);
   dcb->position.beginning_of_medium = ((dcb->tape_status & BOT) != 0);
   dcb->position.end_of_medium = ((dcb->tape_status & EOT) != 0);
   dcb->command_status.file_mark = ((dcb->tape_status & FILEMARK) != 0);
   dcb->command_status.recovered_error =
                                   ((dcb->tape_status & CORRECTED_ERROR) != 0);
   dcb->command_status.hard_error = ((dcb->tape_status & HARD_ERROR) != 0);
   if (!dcb->unit_status.on_line)
      erc = (dcb->unit_status.ready) ? ercTapeHardwareError : ercTapeNotReady;
   else if (dcb->unit_status.busy)
      erc = ercTapeBusy;
   else if (!dcb->unit_status.ready)
      erc = ercTapeNotReady;
   else if (   dcb->position.end_of_medium && !dcb->state.inbound
            && !dcb->state.ignore_EOM)
      erc = ercTapeEomWarning;
   else if (dcb->command_status.file_mark)
      erc = ercTapeFileMark;
   else if (dcb->command_status.hard_error) {
      dcb->hard_errors++;
      erc = ercTapeIoError;
   } else if (dcb->unit_status.write_protected && dcb->mode == MODE_MODIFY)
      erc = ercTapeReadOnly;
   else if (dcb->command_status.recovered_error)
      dcb->soft_errors++;
   return(erc);

}
