(* Copyright (C) 1992, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)

(* File: Dec.m3                                                *)
(* Last Modified On Tue Nov  3 17:12:25 PST 1992 By kalsow     *)
(*      Modified On Tue Apr  2 03:46:13 1991 By muller         *)

MODULE Dec;

IMPORT CallExpr, Expr, Type, Procedure, Emit, Error, Int, Module;
IMPORT Addr, Void, Temp, Target, IntegerExpr, Host, Frame, Fault;

VAR Z: CallExpr.MethodList;

PROCEDURE Check (<*UNUSED*> proc: Expr.T; VAR args: Expr.List;  <*UNUSED*> VAR cs: Expr.CheckState): Type.T =
  BEGIN
    DoCheck ("DEC", args);
    RETURN Void.T;
  END Check;

PROCEDURE DoCheck (name: TEXT; args: Expr.List) =
  VAR t: Type.T; e: Expr.T;
  BEGIN
    e := args[0];
    t := Type.Base (Expr.TypeOf (e));
    IF (Type.Number (t) < 0) THEN
      IF Type.IsSubtype (t, Addr.T) THEN
        IF Module.IsSafe () THEN Error.ID (name, "unsafe operation") END;
      ELSE
        Error.ID (name, "first argument must be of an ordinal type");
      END;
    ELSIF (NOT Expr.IsDesignator (e)) THEN
      Error.ID (name, "first argument must be a variable");
    ELSIF (NOT Expr.IsWritable (e)) THEN
      Error.ID (name, "first argument must be writable");
    END;
    IF (NUMBER (args^) > 1) THEN
      t := Type.Base (Expr.TypeOf (args[1]));
      IF (t # Int.T) THEN
        Error.ID (name, "second argument must be an integer");
      END;
    END;
  END DoCheck;

PROCEDURE Compile (<*UNUSED*> proc: Expr.T;  args: Expr.List): Temp.T =
  VAR
    t1, t2: Temp.T;
    bmin, bmax, imin, imax: INTEGER;
    dec: Expr.T;
    check: [0..4] := 0;
    block: INTEGER;
  BEGIN
    IF (NUMBER (args^) > 1)
      THEN dec := args[1];
      ELSE dec := IntegerExpr.New (1);
    END;
    Expr.GetBounds (args[0], bmin, bmax);
    Expr.GetBounds (dec, imin, imax);

    IF Host.doRangeChk THEN
      IF (bmin # Target.MININT) AND (imax > 0) THEN INC (check) END;
      IF (bmax # Target.MAXINT) AND (imin < 0) THEN INC (check, 2) END;
    END;
    IF Type.IsSubtype (Expr.TypeOf (args[0]), Addr.T) THEN check := 4; END;

    t1 := Expr.CompileLValue (args[0]); 
    t2 := Expr.Compile (dec);

    CASE check OF
    | 0 => (* no range checking *)
           Emit.OpTT ("@ -= @;\n", t1, t2);
    | 1 => (* check lower bound only *)
           Frame.PushBlock (block, 1);
           Emit.Op   ("register int _r;\n");
           Emit.OpTT ("_r = @ - @;\n", t1, t2);
           Emit.OpI  ("if (_r < @) ", bmin);
           Fault.Range ();
           Emit.OpT  ("@ = _r;\n", t1);
           Frame.PopBlock (block);
    | 2 => (* check upper bound only *)
           Frame.PushBlock (block, 1);
           Emit.Op   ("register int _r;\n");
           Emit.OpTT ("_r = @ - @;\n", t1, t2);
           Emit.OpI  ("if (@ < _r) ", bmax);
           Fault.Range ();
           Emit.OpT  ("@ = _r;\n", t1);
           Frame.PopBlock (block);
    | 3 => (* check both bounds *)
           Frame.PushBlock (block, 1);
           Emit.Op   ("register int _r;\n");
           Emit.OpTT ("_r = @ - @;\n", t1, t2);
           IF (bmin = 0) AND (bmax >= 0)
             THEN Emit.OpI ("if (@ < (unsigned)_r) ", bmax);
             ELSE Emit.OpII ("if ((_r < @) || (@ < _r)) ", bmin, bmax);
           END;
           Fault.Range ();
           Emit.OpT  ("@ = _r;\n", t1);
           Frame.PopBlock (block);
    | 4 => (* address *)
	   Emit.OpTT ("*((char**)&@) -= @; \n", t1, t2);
    END;

    Expr.NoteWrite (args[0]);
    Temp.Free (t2);
    Temp.Free (t1);
    RETURN t1; (*DUMMY*)
  END Compile;

PROCEDURE Initialize () =
  BEGIN
    Z := CallExpr.NewMethodList (1, 2, FALSE, FALSE, Void.T,
                                 NIL, Check, Compile, CallExpr.NoValue, 
                                 CallExpr.IsNever, (* writable *)
                                 CallExpr.IsNever, (* designator *)
                                 CallExpr.NotWritable (* noteWriter *));
    Procedure.Define ("DEC", Z, TRUE);
  END Initialize;

BEGIN
END Dec.
