(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)
(*                                                                           *)
(* Last modified on Wed Oct 28 11:11:50 PST 1992 by steveg                       *)
(*      modified on Sat Oct 17 18:23:19 PDT 1992 by mhb                          *)
(*      modified on Tue Sep 22 16:21:12 PDT 1992 by meehan                       *)
(*      modified on Tue Jun 16 23:04:34 PDT 1992 by muller                   *)
(*      modified on Mon Jan 15 16:48:10 PST 1990 by brooks                   *)
(*      modified on Sun May 21 17:17:10 PDT 1989 by gidi                     *)

MODULE FormsVBT EXPORTS FVRuntime, FormsVBT, FVTypes;

(* This module contains the code to construct (parse) FV-expressions. *)

IMPORT FVRuntime;

IMPORT Axis, BiFeedbackVBT, BorderedFeedbackVBT, BooleanVBT, ButtonVBT,
       ChoiceVBT, CIE, ColorName, Cursor, Feedback, FileBrowserVBT, Filter,
       FlexShape, Fmt, HSV, HVSplit, Image, List, ListVBT, Macro,
       MarginFeedbackVBT, MenuSwitchVBT, Multi, NumericVBT,
       NumericScrollerVBT, PaintOp, PaintOpCache, Pixmap, PixmapFromAscii,
       Pts, Radio, Rd, RdUtils, ReactivityVBT, RGB, Rsrc, ScaleFilter,
       ScrnPixmap, Shadow, ShadowedFeedbackVBT, ShadowedVBT, SourceVBT,
       Split, SplitterVBT, SwitchVBT, Sx, SxSymbol, Text, TextEditVBT,
       TextPort, TextureVBT, TextVBT, TextWr, Thread, TSplit, TxtIntTbl,
       VBT, ViewportVBT, Wr, ZChildVBT;

<* FATAL Multi.NotAChild, Sx.PrintError, Thread.Alerted *>

<* PRAGMA LL *>

TYPE
  ParseClosure = Thread.SizedClosure OBJECT
                   description: S_exp;
                   fv         : T;
                   fixupList  : FixupLink := NIL;
                   formstack  : List.T    := NIL;
                   state      : State
                 OVERRIDES
                   apply := Apply
                 END;
  FixupLink = REF RECORD
                    targetName: TEXT;
                    sourceVBT : VBT.T;
                    next      : FixupLink
                  END;

PROCEDURE Parse (t: T; description: S_exp; READONLY state: State): VBT.T
  RAISES {Error} =
  BEGIN
    formstack := NIL;
    TYPECASE
        Thread.Join (Thread.Fork (NEW (ParseClosure, stackSize := 10000,
                                       description := description, fv := t,
                                       state := state))) OF
    | TEXT (msg) => RAISE Error (msg)
    | VBT.T (ch) => RETURN ch
    END
  END Parse;

PROCEDURE Apply (cl: ParseClosure): REFANY =
  <* LL = 0 *>
  VAR ch: VBT.T;
  BEGIN
    TRY
      ch := Item (cl, cl.description, cl.state);
      Pass2 (cl);
      RETURN ch
    EXCEPT
    | Error (msg) => RETURN msg
    END
  END Apply;

EXCEPTION Narrow;
(* NARROW-faults are checked runtime errors, but implementations
   are not required to map them into exceptions, so you can't
   catch them with TRY EXCEPT ELSE. That would have been handy
   in the following procedure.  (So would multi-methods!) *)

PROCEDURE Pass2 (cl: ParseClosure) RAISES {Error} =
  (* Find targets of (For xxx) forms. *)
  BEGIN
    WHILE cl.fixupList # NIL DO
      TRY
        WITH target = GetVBT (cl.fv, cl.fixupList.targetName),
             source = cl.fixupList.sourceVBT                   DO
          TYPECASE source OF
          | FVHelper (fbh) =>
              TYPECASE target OF
              | FVFileBrowser (x) =>
                  FileBrowserVBT.SetHelper (x, fbh)
              ELSE
                RAISE Narrow
              END
          | FVDirMenu (dm) =>
              TYPECASE target OF
              | FVFileBrowser (x) =>
                  FileBrowserVBT.SetDirMenu (x, dm)
              ELSE
                RAISE Narrow
              END
          | FVPageButton (pagebutton) =>
              TYPECASE target OF
              | FVTSplit (x) => pagebutton.target := x
              ELSE
                RAISE Narrow
              END
          | FVLinkButton (lb) =>
              FindTparentAndChild (
                lb, target, cl.fixupList.targetName)
          | FVCloseButton (cb) =>
              TYPECASE target OF
              | ZChildVBT.T (x) => cb.target := x
              ELSE
                RAISE Narrow
              END
          | FVPopButton, FVPopMButton =>
              TYPECASE target OF
              | ZChildVBT.T (x) =>
                  FVRuntime.SetPopTarget (source, x)
              ELSE
                RAISE Narrow
              END
          ELSE
            Gripe ("Internal error [Pass2]: ", source)
          END                   (* TYPECASE source *)
        END                     (* WITH *)
      EXCEPT
      | FileBrowserVBT.Error (e) =>
          Gripe (Fmt.F ("Error in FileBrowser %s: %s %s",
                        cl.fixupList.targetName, e.path, e.text))
      ELSE                      (* NARROW fault, NIL, etc. *)
        Gripe (Fmt.F ("The form named %s is of the wrong type",
                      cl.fixupList.targetName))
      END;
      cl.fixupList := cl.fixupList.next
    END                         (* WHILE *)
  END Pass2;

PROCEDURE FindTparentAndChild (lb     : FVLinkButton;
                               vbt    : VBT.T;
                               vbtName: TEXT                  )
  RAISES {Error} =
  BEGIN
    (* "The named component must be either a TSplit child, or a descendant
       of something that is. In the latter case the TSplit child is the
       true target." *)
    LOOP
      TYPECASE VBT.Parent (vbt) OF
      | NULL => RAISE Error (vbtName & " is not in a TSplit")
      | FVTSplit (t) => lb.Tparent := t; lb.Tchild := vbt; RETURN
      | VBT.T (parent) => vbt := parent
      END
    END
  END FindTparentAndChild;

(*************************** Parser *******************************)

TYPE
  ComponentProc =
    PROCEDURE (cl: ParseClosure; VAR list: List.T; state: State): VBT.T
      RAISES {Error};
  RealizeProc = PROCEDURE (): VBT.T RAISES {Error};
  StateProc = PROCEDURE (list: List.T; VAR state: State) RAISES {Error};
  MetricsProc = PROCEDURE (name: TEXT; arglist: List.T; VAR metrics: List.T)
                  RAISES {Error};

PROCEDURE Item (cl: ParseClosure; exp: S_exp; state: State):
  VBT.T RAISES {Error} =
  (*
    This routine interprets an S-expression as a component (VBT).
     - NIL is illegal.
     - The symbol Fill is parsed as (Fill).
     - The symbol Bar is parsed as (Bar 2).
     - The symbol Chisel is parsed as (Chisel 2).
     - The symbol Ridge is parsed as (Ridge 2).
     - The symbol Glue is parsed as (Glue 2).
     - A text "abc" is parsed as (Text "abc").
     - Lists whose first element names a component are parsed by
       specific routines, stored in "componentProcTable".
    Nothing else is legal. *)
  VAR
    list: List.T;
    res : VBT.T;
  BEGIN
    List.Push (cl.formstack, exp); (* For debugging *)
    formstack := cl.formstack;  (* The exported, global copy *)
    TYPECASE exp OF
    | NULL =>
    | SxSymbol.T (s) =>
        res := ParseSymbolComponent (cl, s, state);
        cl.formstack := cl.formstack.tail;
        formstack := cl.formstack;
        RETURN res
    | TEXT (text) =>
        list := List.List1 (text);
        res := pText (cl, list, state);
        cl.formstack := cl.formstack.tail;
        formstack := cl.formstack;
        RETURN res
    | List.T (list) =>
        TYPECASE list.first OF
        | NULL =>
        | SxSymbol.T (sym) =>
            list := list.tail;
            WITH p = FindComponentProc (sym.name) DO
              IF p # NIL THEN
                res := p (cl, list, state)
              ELSE
                WITH m = MacroFunction (sym, state) DO
                  IF m # NIL THEN
                    res := Item (cl, m.apply (list), state)
                  ELSIF Text.Equal (sym.name, "Insert") THEN
                    res :=
                      OneChild (
                        cl, InsertFile (OneText (list), cl.fv.path), state)
                  ELSE
                    Gripe ("Unknown component: ", sym.name)
                  END
                END
              END
            END;
            cl.formstack := cl.formstack.tail;
            formstack := cl.formstack;
            RETURN res
        ELSE
        END
    ELSE
    END;
    Gripe ("Syntax error: ", exp)
  END Item;

PROCEDURE MacroFunction (sym: SxSymbol.T; state: State): Macro.T =
  BEGIN
    WITH pair = List.AssocQ (state.macros, sym) DO
      IF pair # NIL THEN RETURN pair.tail.first ELSE RETURN NIL END
    END
  END MacroFunction;
      
PROCEDURE ParseSymbolComponent (cl   : ParseClosure;
                                sym  : SxSymbol.T;
                                state: State         ): VBT.T
  RAISES {Error} =
  BEGIN
    IF Text.Equal (sym.name, "Bar")
         OR Text.Equal (sym.name, "Glue")
         OR Text.Equal (sym.name, "Ridge")
         OR Text.Equal (sym.name, "Chisel") THEN
      RETURN
        Item (cl, List.List2 (sym, Sx.NewInteger (2)), state)
    ELSIF Text.Equal (sym.name, "Fill") THEN
      RETURN Item (cl, List.List1 (sym), state)
    ELSE
      Gripe ("Unknown Symbol-component: ", sym)
    END
  END ParseSymbolComponent;

PROCEDURE Gripe (msg: TEXT; form: REFANY := NIL) RAISES {Error} =
  BEGIN
    IF form # NIL THEN msg := msg & Sx.ToText (form, syntax := FVSyntax) END;
    RAISE Error (msg)
  END Gripe;

(* ====================================================================== *)
(* Parsing routines for components *)
(* ====================================================================== *)

VAR Unnamed := SxSymbol.FromName ("");

PROCEDURE NamePP (): SymbolPP =
  BEGIN
    RETURN NEW (SymbolPP, val := Unnamed, name := "Name")
  END NamePP;

PROCEDURE Named (n: SymbolPP): BOOLEAN =
  BEGIN
    RETURN n.val # Unnamed
  END Named;

(* ======================= Realizing VBTs ========================== *)

REVEAL
  T <: Private;
  Private = SemiPublic BRANDED OBJECT OVERRIDES realize := Realize END;

PROCEDURE Realize (<* UNUSED *> fv  : Private;
                                type: TEXT;
                   <* UNUSED *> name: TEXT     ): VBT.T RAISES {Error} =
  BEGIN
    RETURN FindRealizeProc (type) ()
  END Realize;

PROCEDURE rBar (): VBT.T =
  BEGIN
    RETURN NEW (FVBar)
  END rBar;

PROCEDURE rBoolean (): VBT.T =
  BEGIN
    RETURN NEW (FVBoolean)
  END rBoolean;

PROCEDURE rBorder (): VBT.T =
  BEGIN
    RETURN NEW (FVBorder)
  END rBorder;

PROCEDURE rBrowser (): VBT.T =
  BEGIN
    RETURN NEW (FVBrowser)
  END rBrowser;

PROCEDURE rButton (): VBT.T =
  BEGIN
    RETURN NEW (FVButton)
  END rButton;

PROCEDURE rChisel (): VBT.T =
  BEGIN
    RETURN NEW (FVChisel)
  END rChisel;

PROCEDURE rChoice (): VBT.T =
  BEGIN
    RETURN NEW (FVChoice)
  END rChoice;

PROCEDURE rCloseButton (): VBT.T =
  BEGIN
    RETURN NEW (FVCloseButton)
  END rCloseButton;

PROCEDURE rDirMenu (): VBT.T =
  BEGIN
    RETURN NEW (FVDirMenu)
  END rDirMenu;

PROCEDURE rFileBrowser (): VBT.T =
  BEGIN
    RETURN NEW (FVFileBrowser)
  END rFileBrowser;

PROCEDURE rFill (): VBT.T =
  BEGIN
    RETURN NEW (FVFill)
  END rFill;

PROCEDURE rFilter (): VBT.T =
  BEGIN
    RETURN NEW (FVFilter)
  END rFilter;

PROCEDURE rFrame (): VBT.T =
  BEGIN
    RETURN NEW (FVFrame)
  END rFrame;

PROCEDURE rGeneric (): VBT.T =
  BEGIN
    RETURN NEW (FVGeneric)
  END rGeneric;

PROCEDURE rGlue (): VBT.T =
  BEGIN
    RETURN NEW (FVGlue)
  END rGlue;

PROCEDURE rGuard (): VBT.T =
  BEGIN
    RETURN NEW (FVGuard)
  END rGuard;

PROCEDURE rHBox (): VBT.T =
  BEGIN
    RETURN NEW (FVHBox)
  END rHBox;

PROCEDURE rHPackSplit (): VBT.T =
  BEGIN
    RETURN NEW (FVHPackSplit)
  END rHPackSplit;

PROCEDURE rHTile (): VBT.T =
  BEGIN
    RETURN NEW (FVHTile)
  END rHTile;

PROCEDURE rHelper (): VBT.T =
  BEGIN
    RETURN NEW (FVHelper)
  END rHelper;

PROCEDURE rLinkButton (): VBT.T =
  BEGIN
    RETURN NEW (FVLinkButton)
  END rLinkButton;

PROCEDURE rMButton (): VBT.T =
  BEGIN
    RETURN NEW (FVMButton)
  END rMButton;

PROCEDURE rMenu (): VBT.T =
  BEGIN
    RETURN NEW (FVMenu)
  END rMenu;

PROCEDURE rMenuBar (): VBT.T =
  BEGIN
    RETURN NEW (FVMenuBar)
  END rMenuBar;

PROCEDURE rMultiBrowser (): VBT.T =
  BEGIN
    RETURN NEW (FVMultiBrowser)
  END rMultiBrowser;

PROCEDURE rNumeric (): VBT.T =
  BEGIN
    RETURN NEW (FVNumeric)
  END rNumeric;

PROCEDURE rPageButton (): VBT.T =
  BEGIN
    RETURN NEW (FVPageButton)
  END rPageButton;

PROCEDURE rPixmap (): VBT.T =
  BEGIN
    RETURN NEW (FVPixmap)
  END rPixmap;

PROCEDURE rPopButton (): VBT.T =
  BEGIN
    RETURN NEW (FVPopButton)
  END rPopButton;

PROCEDURE rPopMButton (): VBT.T =
  BEGIN
    RETURN NEW (FVPopMButton)
  END rPopMButton;

PROCEDURE rRadio (): VBT.T =
  BEGIN
    RETURN NEW (FVRadio)
  END rRadio;

PROCEDURE rRidge (): VBT.T =
  BEGIN
    RETURN NEW (FVRidge)
  END rRidge;

PROCEDURE rRim (): VBT.T =
  BEGIN
    RETURN NEW (FVRim)
  END rRim;

PROCEDURE rScale (): VBT.T =
  BEGIN
    RETURN NEW (FVScale)
  END rScale;

PROCEDURE rScroller (): VBT.T =
  BEGIN
    RETURN NEW (FVScroller)
  END rScroller;

PROCEDURE rShape (): VBT.T =
  BEGIN
    RETURN NEW (FVShape)
  END rShape;

PROCEDURE rSource (): VBT.T =
  BEGIN
    RETURN NEW (FVSource)
  END rSource;

PROCEDURE rTSplit (): VBT.T =
  BEGIN
    RETURN NEW (FVTSplit)
  END rTSplit;

PROCEDURE rTarget (): VBT.T =
  BEGIN
    RETURN NEW (FVTarget)
  END rTarget;

PROCEDURE rText (): VBT.T =
  BEGIN
    RETURN NEW (FVText)
  END rText;

PROCEDURE rTextArea (): VBT.T =
  BEGIN
    RETURN NEW (FVTextArea)
  END rTextArea;

PROCEDURE rTextEdit (): VBT.T =
  BEGIN
    RETURN NEW (FVTextEdit)
  END rTextEdit;

PROCEDURE rTexture (): VBT.T =
  BEGIN
    RETURN NEW (FVTexture)
  END rTexture;

PROCEDURE rTrillButton (): VBT.T =
  BEGIN
    RETURN NEW (FVTrillButton)
  END rTrillButton;

PROCEDURE rTypeIn (): VBT.T =
  BEGIN
    RETURN NEW (FVTypeIn)
  END rTypeIn;

PROCEDURE rTypescript (): VBT.T =
  BEGIN
    RETURN NEW (FVTypescript)
  END rTypescript;

PROCEDURE rVBox (): VBT.T =
  BEGIN
    RETURN NEW (FVVBox)
  END rVBox;

PROCEDURE rVTile (): VBT.T =
  BEGIN
    RETURN NEW (FVVTile)
  END rVTile;

PROCEDURE rViewport (): VBT.T =
  BEGIN
    RETURN NEW (FVViewport)
  END rViewport;

PROCEDURE rVPackSplit (): VBT.T =
  BEGIN
    RETURN NEW (FVHPackSplit)
  END rVPackSplit;

PROCEDURE rZBackground (): VBT.T =
  BEGIN
    RETURN NEW (FVZBackground)
  END rZBackground;

PROCEDURE rZChassis (): VBT.T =
  BEGIN
    RETURN NEW (FVZChassis)
  END rZChassis;

PROCEDURE rZChild (): VBT.T =
  BEGIN
    RETURN NEW (FVZChild)
  END rZChild;

PROCEDURE rZGrow (): VBT.T =
  BEGIN
    RETURN NEW (FVZGrow)
  END rZGrow;

PROCEDURE rZMove (): VBT.T =
  BEGIN
    RETURN NEW (FVZMove)
  END rZMove;

PROCEDURE rZSplit (): VBT.T =
  BEGIN
    RETURN NEW (FVZSplit)
  END rZSplit;


(* ========================= Bar & Glue ============================= *)

(* Bar uses the current Color.  Glue uses the current BgColor. *)

CONST
  PtsToMM = 25.4 / 72.0;
  Flex2   = FlexShape.SizeRange {2.0, 0.0, 0.0};

PROCEDURE pBar (cl: ParseClosure; VAR list: List.T; state: State):
  VBT.T RAISES {Error} =
  VAR
    name                := NamePP ();
    main                := NEW (SizeRangePP, val := Flex2, found := TRUE);
    res : FVBar;
  BEGIN
    IF state.hvsplit = NIL THEN
      RAISE Error ("Bar must appear inside an HBox or VBox.")
    END;
    ParseProps (cl, list, state, PP2 {name, main}, main := main);
    res := cl.fv.realize ("Bar", name.val.name);
    EVAL res.init (TextureVBT.New (state.fgOp),
                   ShapefromSpec (main.val, state));
    AddNameProp (cl, res, name, state);
    RETURN res
  END pBar;

PROCEDURE pGlue (cl: ParseClosure; VAR list: List.T; state: State):
  VBT.T RAISES {Error} =
  VAR
    name := NamePP ();
    main := NEW (SizeRangePP, val := Flex2, found := TRUE);
    res: FVGlue;
  BEGIN
    IF state.hvsplit = NIL THEN
      RAISE Error ("Glue must appear inside an HBox or VBox.")
    END;
    ParseProps (cl, list, state, PP2 {name, main}, main := main);
    res := cl.fv.realize ("Glue", name.val.name);
    EVAL res.init (TextureVBT.New (state.bgOp),
                   ShapefromSpec (main.val, state));
    AddNameProp (cl, res, name, state);
    RETURN res
  END pGlue;

PROCEDURE ShapefromSpec (         f    : FlexShape.SizeRange;
                         READONLY state: State                ):
  FlexShape.Shape RAISES {Error} =
  VAR sh := FlexShape.DefaultShape;
  BEGIN
    sh [state.glueAxis] := f;
    RETURN sh
  END ShapefromSpec;

(* ========================= Border & Rim ============================= *)

PROCEDURE pBorder (cl: ParseClosure; VAR list: List.T; state: State): VBT.T
  RAISES {Error} =
  VAR
    name                      := NamePP ();
    pen                       := NEW (RealPP, name := "Pen", val := 1.0);
    texture                   := NEW (TextPP, name := "Pattern");
    txt                       := Pixmap.Solid;
    res    : FVBorder;
    ch     : VBT.T;
  BEGIN
    ParseProps (cl, list, state, PP3 {name, pen, texture});
    ch := OneChild (cl, list, state);
    res := cl.fv.realize ("Border", name.val.name);
    IF texture.val # NIL THEN txt := GetPixmap (texture.val, cl) END;
    EVAL res.init (ch, pen.val * PtsToMM, state.shadow.bgFg, txt);
    AddNameProp (cl, res, name, state);
    RETURN res
  END pBorder;

PROCEDURE pRim (cl: ParseClosure; VAR list: List.T; state: State): VBT.T
  RAISES {Error} =
  VAR
    name                   := NamePP ();
    pen                    := NEW (RealPP, name := "Pen", val := 1.0);
    texture                := NEW (TextPP, name := "Pattern");
    txt                    := Pixmap.Solid;
    res    : FVRim;
    ch     : VBT.T;
  BEGIN
    ParseProps (cl, list, state, PP3 {name, pen, texture});
    ch := OneChild (cl, list, state);
    res := cl.fv.realize ("Rim", name.val.name);
    IF texture.val # NIL THEN txt := GetPixmap (texture.val, cl) END;
    EVAL res.init (ch, pen.val * PtsToMM, state.shadow.fgBg, txt);
    AddNameProp (cl, res, name, state);
    RETURN res
  END pRim;

PROCEDURE GetPixmap (name: TEXT; cl: ParseClosure): Pixmap.T RAISES {Error} =
  VAR rd: Rd.T;
  BEGIN
    TRY
      rd := Rsrc.Open (name, cl.fv.path);
      TRY RETURN PixmapFromAscii.Read (rd) FINALLY Rd.Close (rd) END
    EXCEPT
    | PixmapFromAscii.Error =>
        RAISE Error ("Format error in pixmap for " & name)
    | Rsrc.NotFound => RAISE Error ("No such resource: " & name)
    | Rd.Failure (ref) => RAISE Error (RdUtils.FailureText (ref))
    END
  END GetPixmap;

PROCEDURE GetImage (              name : TEXT;
                                  cl   : ParseClosure;
                    VAR (* OUT *) depth: INTEGER       ): Image.T
  RAISES {Error} =
  VAR rd: Rd.T;
  BEGIN
    TRY
      rd := Rsrc.Open(name, cl.fv.path);
      TRY
        WITH image = NEW(Image.Raw, bits := PixmapFromAscii.ReadRaw(rd),
                         colors := Image.BitmapColors) DO
          depth := image.bits.depth;
          RETURN Image.Scaled(ARRAY OF Image.Raw{image})
        END
      FINALLY
        Rd.Close(rd)
      END;
    EXCEPT
    | PixmapFromAscii.Error =>
        RAISE Error("Format error in pixmap for " & name)
    | Rsrc.NotFound => RAISE Error("No such resource: " & name)
    | Rd.Failure (ref) => RAISE Error(RdUtils.FailureText(ref))
    END
  END GetImage;


(* ========================= Frame & Ridge ============================= *)

PROCEDURE pFrame (cl: ParseClosure; VAR list: List.T; state: State): VBT.T
  RAISES {Error} =
  VAR
    name        := NamePP ();
    shadowStyle := NewShadowStyle (Shadow.Style.Raised);
  VAR
    res: FVFrame;
    ch : VBT.T;
  BEGIN
    ParseProps (cl, list, state, PP1 {name}, enums := EP1 {shadowStyle});
    ch := OneChild (cl, list, state);
    res := cl.fv.realize ("Frame", name.val.name);
    EVAL res.init (ch, state.shadow, VAL (shadowStyle.chosen, Shadow.Style));
    AddNameProp (cl, res, name, state);
    RETURN res
  END pFrame;

PROCEDURE pRidge (cl: ParseClosure; VAR list: List.T; state: State): VBT.T
  RAISES {Error} =
  VAR
    name := NamePP ();
    main := NEW (RealPP, val := DefaultShadowSize, found := TRUE);
  VAR
    res   : FVRidge;
    shadow: Shadow.T;
  BEGIN
    ParseProps (cl, list, state, PP2 {name, main}, main := main);
    res := cl.fv.realize ("Ridge", name.val.name);
    shadow :=
      Shadow.New (
        main.val, state.bgOp, state.fgOp, state.lightOp, state.darkOp);
    EVAL res.init (state.glueAxis, shadow, Shadow.Style.Ridged);
    AddNameProp (cl, res, name, state);
    RETURN res
  END pRidge;

PROCEDURE pChisel (cl: ParseClosure; VAR list: List.T; state: State): VBT.T
  RAISES {Error} =
  VAR
    name := NamePP ();
    main := NEW (RealPP, val := DefaultShadowSize, found := TRUE);
  VAR
    res   : FVChisel;
    shadow: Shadow.T;
  BEGIN
    ParseProps (cl, list, state, PP2 {name, main}, main := main);
    res := cl.fv.realize ("Chisel", name.val.name);
    shadow :=
      Shadow.New (
        main.val, state.bgOp, state.fgOp, state.lightOp, state.darkOp);
    EVAL res.init (state.glueAxis, shadow, Shadow.Style.Chiseled);
    AddNameProp (cl, res, name, state);
    RETURN res
  END pChisel;

(* =========================== Fill & Shape =========================== *)

PROCEDURE pFill (cl: ParseClosure; VAR list: List.T; state: State): VBT.T
  RAISES {Error} =
  CONST
    INFINITESTRETCH = FlexShape.SizeRange {natural := 0.0, shrink := 0.0,
                                           stretch := FlexShape.Infinity};
  VAR
    name                  := NamePP ();
    shape                 := FlexShape.DefaultShape;
    res  : FVFill;
  BEGIN
    IF state.hvsplit = NIL THEN
      RAISE Error ("Fill must appear inside an HBox or VBox.")
    END;
    ParseProps (cl, list, state, PP1 {name});
    AssertEmpty (list);
    shape [state.glueAxis] := INFINITESTRETCH;
    res := cl.fv.realize ("Fill", name.val.name);
    EVAL res.init (TextureVBT.New (state.bgOp), shape);
    AddNameProp (cl, res, name, state);
    RETURN res
  END pFill;

PROCEDURE pShape (cl: ParseClosure; VAR list: List.T; state: State): VBT.T
  RAISES {Error} =
  VAR
    name                    := NamePP ();
    height                  := NEW (SizeRangePP, name := "Height");
    width                   := NEW (SizeRangePP, name := "Width");
    res   : FVShape;
    ch    : VBT.T;
  BEGIN
    ParseProps (cl, list, state, PP3 {name, height, width});
    ch := OneChild (cl, list, state);
    res := cl.fv.realize ("Shape", name.val.name);
    EVAL res.init (ch, FlexShape.Shape {width.val, height.val});
    AddNameProp (cl, res, name, state);
    RETURN res
  END pShape;


(* =========================== Buttons =============================== *)

PROCEDURE pButton (cl: ParseClosure; VAR list: List.T; state: State): VBT.T
  RAISES {Error} =
  VAR
    name                   := NamePP ();
    res : FVButton;
    ch  : VBT.T;
  BEGIN
    ParseProps (cl, list, state, PP1 {name});
    ch := OneChild (cl, list, state);
    res := cl.fv.realize ("Button", name.val.name);
    EVAL res.init (NEW (ShadowedFeedbackVBT.T).init (ch, state.shadow));
    AddNameProp (cl, res, name, state);
    RETURN res
  END pButton;

PROCEDURE pMButton (cl: ParseClosure; VAR list: List.T; state: State): VBT.T
  RAISES {Error} =
  VAR
    name                    := NamePP ();
    res : FVMButton;
    ch  : VBT.T;
  BEGIN
    ParseProps (cl, list, state, PP1 {name});
    ch := OneChild (cl, list, state);
    res := cl.fv.realize ("MButton", name.val.name);
    EVAL res.init (ShadowedFeedbackVBT.NewMenu (ch, state.shadow));
    AddNameProp (cl, res, name, state);
    RETURN res
  END pMButton;

PROCEDURE pPopButton (cl: ParseClosure; VAR list: List.T; state: State): VBT.T
  RAISES {Error} =
  VAR
    name                         := NamePP ();
    forName                      := NEW (SymbolPP, name := "For");
    res    : FVPopButton;
    ch     : VBT.T;
  BEGIN
    ParseProps (cl, list, state, PP2 {name, forName});
    ch := OneChild (cl, list, state);
    res := cl.fv.realize ("PopButton", name.val.name);
    EVAL res.init (NEW (ShadowedFeedbackVBT.T).init (ch, state.shadow));
    AddForProp (cl, res, forName);
    AddNameProp (cl, res, name, state);
    RETURN res
  END pPopButton;

PROCEDURE pPopMButton (cl: ParseClosure; VAR list: List.T; state: State):
  VBT.T RAISES {Error} =
  VAR
    name                          := NamePP ();
    forName                       := NEW (SymbolPP, name := "For");
    res    : FVPopMButton;
    ch     : VBT.T;
  BEGIN
    ParseProps (cl, list, state, PP2 {name, forName});
    ch := OneChild (cl, list, state);
    res := cl.fv.realize ("PopMButton", name.val.name);
    EVAL res.init (ShadowedFeedbackVBT.NewMenu (ch, state.shadow));
    AddForProp (cl, res, forName);
    AddNameProp (cl, res, name, state);
    RETURN res
  END pPopMButton;

PROCEDURE pGuard (cl: ParseClosure; VAR list: List.T; state: State): VBT.T
  RAISES {Error} =
  VAR
    name                  := NamePP ();
    res : FVGuard;
    ch  : VBT.T;
  BEGIN
    ParseProps (cl, list, state, PP1 {name});
    ch := OneChild (cl, list, state);
    res := cl.fv.realize ("Guard", name.val.name);
    EVAL res.init (ch, state.shadow);
    AddNameProp (cl, res, name, state);
    RETURN res
  END pGuard;

PROCEDURE pTrillButton (cl: ParseClosure; VAR list: List.T; state: State):
  VBT.T RAISES {Error} =
  VAR
    name                        := NamePP ();
    res : FVTrillButton;
    ch  : VBT.T;
  BEGIN
    ParseProps (cl, list, state, PP1 {name});
    ch := OneChild (cl, list, state);
    res := cl.fv.realize ("TrillButton", name.val.name);
    EVAL res.init (NEW (ShadowedFeedbackVBT.T).init (ch, state.shadow));
    AddNameProp (cl, res, name, state);
    RETURN res
  END pTrillButton;

PROCEDURE pPageButton (cl: ParseClosure; VAR list: List.T; state: State):
  VBT.T RAISES {Error} =
  VAR
    name                            := NamePP ();
    forName                         := NEW (SymbolPP, name := "For");
    backwards                       := NEW (BooleanPP, name := "Back");
    res      : FVPageButton;
    ch       : VBT.T;
  BEGIN
    ParseProps (cl, list, state, PP2 {name, forName}, KP1 {backwards});
    ch := OneChild (cl, list, state);
    res := cl.fv.realize ("PageButton", name.val.name);
    IF forName.val # NIL THEN
      AddForProp (cl, res, forName)
    ELSIF state.tsplit # NIL THEN
      res.target := state.tsplit
    ELSE
      RAISE Error ("This PageButton is not included in a TSplit and "
                     & "it has no (For ...) property.")
    END;
    EVAL res.init (ch, state.shadow, backwards.val);
    AddNameProp (cl, res, name, state);
    RETURN res
  END pPageButton;

PROCEDURE pLinkButton (cl: ParseClosure; VAR list: List.T; state: State):
  VBT.T RAISES {Error} =
  VAR
    name                         := NamePP ();
    toName                       := NEW (SymbolPP, name := "To");
    res   : FVLinkButton;
    ch    : VBT.T;
  BEGIN
    ParseProps (cl, list, state, PP2 {name, toName});
    ch := OneChild (cl, list, state);
    res := cl.fv.realize ("LinkButton", name.val.name);
    IF toName.val = NIL THEN
      RAISE Error ("LinkButton must include (To <name>)")
    END;
    AddForProp (cl, res, toName);
    EVAL res.init (ch, state.shadow);
    AddNameProp (cl, res, name, state);
    RETURN res
  END pLinkButton;

PROCEDURE pCloseButton (cl: ParseClosure; VAR list: List.T; state: State):
  VBT.T RAISES {Error} =
  VAR
    name                           := NamePP ();
    forName                        := NEW (SymbolPP, name := "For");
    res    : FVCloseButton;
    ch     : VBT.T;
  BEGIN
    ParseProps (cl, list, state, PP2 {name, forName});
    ch := OneChild (cl, list, state);
    res := cl.fv.realize ("CloseButton", name.val.name);
    IF forName.val # NIL THEN
      AddForProp (cl, res, forName)
    ELSIF state.zchild # NIL THEN
      res.target := state.zchild
    ELSE
      RAISE Error ("This CloseButton is not included in a ZChild or ZChassis "
                     & "and it has no (For ...) property.")
    END;
    EVAL res.init (ch, state.shadow);
    AddNameProp (cl, res, name, state);
    RETURN res
  END pCloseButton;


(* ====================== Boolean, Choice, Radio ======================= *)

PROCEDURE pBoolean (    cl   : ParseClosure;
                    VAR list : List.T;
                        state: State         ): VBT.T
  RAISES {Error} =
  VAR
    name      := NamePP ();
    value     := NEW (BooleanPP, name := "Value");
    checkmark := NEW (BooleanPP, name := "CheckMark");
    inverting := NEW (BooleanPP, name := "Inverting");
    checkbox  := NEW (BooleanPP, name := "CheckBox");
    menustyle := NEW (BooleanPP, name := "MenuStyle");
    enum := NEW (EnumPP).init (
              KP3 {checkbox, checkmark, inverting}, 0);
    child, feedback: VBT.T;
    switch         : ButtonVBT.T;
    res            : FVBoolean;
  BEGIN
    ParseProps (cl, list, state, PP2 {name, value}, KP1 {menustyle},
                enums := EP1 {enum});
    child := OneChild (cl, list, state);
    IF inverting.val THEN
      feedback := NEW (BorderedFeedbackVBT.T).init (
                    child, state.shadow.size, state.shadow.bgFg)
    ELSIF checkmark.val THEN
      feedback :=
        MarginFeedbackVBT.NewCheck (child, state.shadow)
    ELSE
      feedback := MarginFeedbackVBT.NewBox (child, state.shadow)
    END;
    IF menustyle.val THEN
      switch := NEW (MenuSwitchVBT.T).init (
                  MenuStyle (feedback, state.shadow))
    ELSE
      switch := NEW (SwitchVBT.T).init (feedback)
    END;
    res := cl.fv.realize ("Boolean", name.val.name);
    EVAL res.init (switch);
    BooleanVBT.Put (res, value.val);
    AddNameProp (cl, res, name, state);
    RETURN res
  END pBoolean;

PROCEDURE pChoice (    cl   : ParseClosure;
                   VAR list : List.T;
                       state: State         ): VBT.T
  RAISES {Error} =
  VAR
    name      := NamePP ();
    value     := NEW (BooleanPP, name := "Value");
    checkmark := NEW (BooleanPP, name := "CheckMark");
    checkbox  := NEW (BooleanPP, name := "CheckBox");
    inverting := NEW (BooleanPP, name := "Inverting");
    enum := NEW (EnumPP).init (KP3 {checkbox, checkmark, inverting}, 0);
    menustyle := NEW (BooleanPP, name := "MenuStyle");
    child, feedback: VBT.T;
    switch         : ButtonVBT.T;
    res            : FVChoice;
  BEGIN
    IF state.radio = NIL THEN
      RAISE Error ("Choice must be contained within Radio")
    END;
    ParseProps (cl, list, state, PP2 {name, value}, KP1 {menustyle},
                enums := EP1 {enum});
    IF name.val = NIL THEN
      RAISE Error ("Choices must be named.")
    END;
    child := OneChild (cl, list, state);
    IF inverting.val THEN
      feedback := NEW (BorderedFeedbackVBT.T).init (
                    child, DefaultShadowSize, state.shadow.bgFg)
    ELSIF checkmark.val THEN
      feedback :=
        MarginFeedbackVBT.NewCheck (child, state.shadow)
    ELSE
      feedback :=
        MarginFeedbackVBT.NewBullet (child, state.shadow)
    END;
    IF menustyle.val THEN
      switch := NEW (MenuSwitchVBT.T).init (
                  MenuStyle (feedback, state.shadow))
    ELSE
      switch := NEW (SwitchVBT.T).init (feedback)
    END;
    res := cl.fv.realize ("Choice", name.val.name);
    EVAL res.init (switch, state.radio.radio);
    res.radio := state.radio;
    IF value.val THEN ChoiceVBT.Put (res) END;
    AddNameProp (cl, res, name, state);
    VBT.PutProp (
      res, NEW (FVRuntime.ChoiceName, name := name.val.name));
    RETURN res
  END pChoice;

PROCEDURE pRadio (cl: ParseClosure; VAR list: List.T; state: State): VBT.T
  RAISES {Error} =
  VAR
    name                   := NamePP ();
    value                  := NEW (SymbolPP, name := "Value");
    res  : FVRadio;
    ch   : VBT.T;
  BEGIN
    ParseProps (cl, list, state, PP2 {name, value});
    res := cl.fv.realize ("Radio", name.val.name);
    res.radio := Radio.New ();
    state.radio := res;
    ch := OneChild (cl, list, state);
    EVAL Filter.T.init (res, ch);
    (* Did the client select a choice via (Radio ...  =<symbol> ...)? *)
    IF value.val # NIL THEN
      ChoiceVBT.Put (GetVBT (cl.fv, value.val.name))
    END;
    AddNameProp (cl, res, name, state);
    RETURN res
  END pRadio;

PROCEDURE MenuStyle (feedback: Feedback.T; shadow: Shadow.T): Feedback.T =
  VAR res: BiFeedbackVBT.T;
  BEGIN
    res := NEW (BiFeedbackVBT.T);
    EVAL BiFeedbackVBT.T.init (res, ShadowedFeedbackVBT.NewMenu (feedback, 
                                                                 shadow));
    RETURN res
  END MenuStyle;


(* =========================== Splits =============================== *)

PROCEDURE pHBox (cl: ParseClosure; VAR list: List.T; state: State): VBT.T
  RAISES {Error} =
  BEGIN
    RETURN pHVBox (cl, list, state, Axis.T.Hor)
  END pHBox;

PROCEDURE pVBox (cl: ParseClosure; VAR list: List.T; state: State): VBT.T
  RAISES {Error} =
  BEGIN
    RETURN pHVBox (cl, list, state, Axis.T.Ver)
  END pVBox;

PROCEDURE pHVBox (    cl   : ParseClosure;
                  VAR list : List.T;
                      state: State;
                      axis : Axis.T        ): VBT.T RAISES {Error} =
  CONST TypeNames = ARRAY Axis.T OF TEXT {"HBox", "VBox"};
  VAR
    name            := NamePP ();
    res : HVSplit.T;
  BEGIN
    ParseProps (cl, list, state, PP1 {name});
    res := cl.fv.realize (TypeNames [axis], name.val.name);
    EVAL res.init (axis, adjustable := FALSE);
    state.glueAxis := axis;
    state.hvsplit := res;
    AddChildren (cl, res, list, state);
    AddNameProp (cl, res, name, state);
    RETURN res
  END pHVBox;

PROCEDURE pHTile (cl: ParseClosure; VAR list: List.T; state: State): VBT.T
  RAISES {Error} =
  BEGIN
    RETURN pHVTile (cl, list, state, Axis.T.Hor)
  END pHTile;

PROCEDURE pVTile (cl: ParseClosure; VAR list: List.T; state: State): VBT.T
  RAISES {Error} =
  BEGIN
    RETURN pHVTile (cl, list, state, Axis.T.Ver)
  END pVTile;

PROCEDURE pHVTile (    cl   : ParseClosure;
                   VAR list : List.T;
                       state: State;
                       axis : Axis.T        ): VBT.T
  RAISES {Error} =
  CONST TypeNames = ARRAY Axis.T OF TEXT {"HTile", "VTile"};
  VAR
    name      := NamePP ();
    asTargets := NEW (BooleanPP, name := "Targets");
    res: SplitterVBT.T;
  BEGIN
    ParseProps (cl, list, state, PP1 {name}, KP1 {asTargets});
    res := cl.fv.realize (TypeNames [axis], name.val.name);
    EVAL res.init (axis, asTargets.val, SplitterVBT.DefaultSize,
                   state.shadow.bgFg, Pixmap.Gray);
    state.glueAxis := axis;
    AddChildren (cl, res, list, state);
    AddNameProp (cl, res, name, state);
    RETURN res
  END pHVTile;

PROCEDURE pHPackSplit (cl: ParseClosure; VAR list: List.T; state: State):
  VBT.T RAISES {Error} =
  BEGIN
    RETURN pHVPackSplit (cl, list, state, Axis.T.Hor)
  END pHPackSplit;

PROCEDURE pVPackSplit (cl: ParseClosure; VAR list: List.T; state: State):
  VBT.T RAISES {Error} =
  BEGIN
    RETURN pHVPackSplit (cl, list, state, Axis.T.Ver)
  END pVPackSplit;

PROCEDURE pHVPackSplit (    cl   : ParseClosure;
                        VAR list : List.T;
                            state: State;
                            axis : Axis.T        ): VBT.T RAISES {Error} =
  CONST TypeNames = ARRAY Axis.T OF TEXT {"HPackSplit", "VPackSplit"};
  VAR
    name       := NamePP ();
    hgap       := NEW (RealPP, name := "HGap", val := 2.0);
    vgap       := NEW (RealPP, name := "VGap", val := 2.0);
    background := NEW (TextPP, name := "Background");
    res: FVHPackSplit;
    txt                       := Pixmap.Solid;
  BEGIN
    ParseProps (cl, list, state, PP4 {name, hgap, vgap, background});
    res := cl.fv.realize (TypeNames [axis], name.val.name);
    IF background.val # NIL THEN txt := GetPixmap (background.val, cl) END;
    EVAL res.init (hv := axis, hgap := Pts.ToMM (hgap.val),
                   vgap := Pts.ToMM (vgap.val), txt := txt,
                   op := state.bgOp);
    AddChildren (cl, res, list, state);
    AddNameProp (cl, res, name, state);
    RETURN res
  END pHVPackSplit;


(* ========================== TSplits ============================ *)

PROCEDURE pTSplit (cl: ParseClosure; VAR list: List.T; state: State): VBT.T
  RAISES {Error} =
  VAR
    name     := NamePP ();
    value    := NEW (CardinalPP, name := "Value", val := LAST (CARDINAL));
    which    := NEW (SymbolPP, name := "Which");
    circular := NEW (BooleanPP, name := "Circular");
    flexible := NEW (BooleanPP, name := "Flex");
    res                      : FVTSplit;
    n                        : CARDINAL;
    namedChild, numberedChild: VBT.T            := NIL;
  BEGIN
    ParseProps (
      cl, list, state, PP3 {name, value, which}, KP2 {circular, flexible});
    res := cl.fv.realize ("TSplit", name.val.name);
    EVAL res.init (fickle := flexible.val);
    res.circular := circular.val;
    state.tsplit := res;
    AddChildren (cl, res, list, state);

    (* Check validity and consistency of (Which n) and (Value name). *)
    n := Split.NumChildren (res);

    IF which.val # NIL THEN
      namedChild := GetVBT (cl.fv, which.val.name)
    END;

    TRY
      IF value.val = LAST (CARDINAL) THEN
        IF namedChild # NIL THEN
          TSplit.SetCurrent (res, namedChild)
        ELSE
          TSplit.SetCurrent (res, Split.Nth (res, 0))
        END
      ELSIF value.val < n THEN
        numberedChild := Split.Nth (res, value.val);
        IF namedChild = NIL OR namedChild = numberedChild THEN
          TSplit.SetCurrent (res, numberedChild)
        ELSE
          RAISE
            Error (Fmt.F ("(Which %s) is not the same child as (Value %s)",
                          which.val.name, Fmt.Int (value.val)))
        END
      ELSIF value.val = 1 THEN
        RAISE Error ("TSplit has no children.")
      ELSE
        RAISE Error (Fmt.F ("TSplit has only %s children.", Fmt.Int (n)))
      END
    EXCEPT
      Split.NotAChild =>
        RAISE Error (which.val.name
                       & " is not the name of a child of this TSplit.")
    END;
    AddNameProp (cl, res, name, state);
    RETURN res
  END pTSplit;


(* ===================== FileBrowser & Helper ==================== *)

PROCEDURE pFileBrowser (    cl   : ParseClosure;
                        VAR list : List.T;
                            state: State         ): VBT.T
  RAISES {Error} =
  VAR
    name     := NamePP();
    value    := NEW(TextPP, name := "Value", val := ".");
    suffixes := NEW(TextListPP, name := "Suffixes");
    readOnly := NEW(BooleanPP, name := "ReadOnly");
    res: FVFileBrowser;
  BEGIN
    ParseProps(
      cl, list, state, PP3{name, value, suffixes}, KP1{readOnly});
    AssertEmpty(list);
    res := cl.fv.realize("FileBrowser", name.val.name);
    EVAL res.init(state.font, state.shadow);
    TRY
      FileBrowserVBT.Set(res, value.val);
      FileBrowserVBT.SetReadOnly(res, readOnly.val);
      IF suffixes.val # NIL THEN
        FileBrowserVBT.SetSuffixes(
          res, SuffixesFromList(suffixes.val))
      END
    EXCEPT
    | FileBrowserVBT.Error (e) =>
        RAISE Error(Fmt.F("Error for %s: %s", e.path, e.text))
    END;
    AddNameProp(cl, res, name, state);
    RETURN res
  END pFileBrowser;

PROCEDURE SuffixesFromList (list: List.T): TEXT =
  VAR wr := TextWr.New ();
  <* FATAL Wr.Failure, Thread.Alerted *>
  BEGIN
    LOOP
      IF Text.Empty (list.first) THEN
        Wr.PutChar (wr, '$')
      ELSE
        Wr.PutText (wr, list.first)
      END;
      list := list.tail;
      IF list = NIL THEN RETURN TextWr.ToText (wr) END;
      Wr.PutChar (wr, ' ')
    END
  END SuffixesFromList;
      
PROCEDURE pHelper (cl: ParseClosure; VAR list: List.T; state: State): VBT.T
  RAISES {Error} =
  VAR
    name                      := NamePP ();
    forName                   := NEW (SymbolPP, name := "For");
    res    : FVHelper;
  BEGIN
    ParseProps (cl, list, state, PP2 {name, forName});
    IF forName.val = NIL THEN
      RAISE Error ("Helper must include (For <name>)")
    END;
    AssertEmpty (list);
    res := cl.fv.realize ("Helper", name.val.name);
    EVAL res.init (font := state.font, shadow := state.shadow);
    AddForProp (cl, res, forName);
    AddNameProp (cl, res, name, state);
    RETURN res
  END pHelper;

PROCEDURE pDirMenu (cl: ParseClosure; VAR list: List.T; state: State): VBT.T
  RAISES {Error} =
  VAR
    name                      := NamePP ();
    forName                   := NEW (SymbolPP, name := "For");
    res    : FVDirMenu;
  BEGIN
    ParseProps (cl, list, state, PP2 {name, forName});
    IF forName.val = NIL THEN
      RAISE Error ("DirMenu must include (For <name>)")
    END;
    AssertEmpty (list);
    res := cl.fv.realize ("DirMenu", name.val.name);
    EVAL res.init (font := state.font, shadow := state.shadow);
    AddForProp (cl, res, forName);
    AddNameProp (cl, res, name, state);
    RETURN res
  END pDirMenu;

PROCEDURE pBrowser (cl: ParseClosure; VAR list: List.T; state: State): VBT.T
  RAISES {Error} =
  VAR
    name                := NamePP ();
    value               := NEW (IntegerPP, name := "Value", val := -1);
    select              := NEW (TextPP, name := "Select");
    items               := NEW (TextListPP, name := "Items");
    from                := NEW (TextPP, name := "From");
    quick               := NEW (BooleanPP, name := "Quick");
    colors              := state.shadow;
    res   : FVBrowser;
    s     : UniSelector;
  BEGIN
    ParseProps (
      cl, list, state, PP5 {name, value, select, items, from}, KP1 {quick});
    AssertEmpty (list);
    res := cl.fv.realize ("Browser", name.val.name);
    TYPECASE res.painter OF
    | NULL =>
        res.painter :=
          NEW (ListVBT.TextPainter).init (
            colors.bg, colors.fg, colors.fg, colors.bg, state.font)
    | ListVBT.TextPainter (tp) =>
        EVAL tp.init (colors.bg, colors.fg, colors.fg, colors.bg, state.font)
    ELSE
    END;
    TYPECASE res.selector OF
    | NULL => s := NEW (UniSelector).init (res); res.selector := s
    | UniSelector (sel) => s := sel
    ELSE
      RAISE
        Error ("Browser has a selector that is not a subtype of UniSelector")
    END;
    s.browser := res;
    s.quick := quick.val;
    EVAL res.init (colors := state.shadow);
    IF items.val # NIL THEN
      SetValues (res, items.val)
    ELSIF from.val # NIL THEN
      SetValues (res, ItemsFromFile (from.val, cl))
    END;
    IF value.val # -1 THEN
      res.selectOnly (value.val)
    ELSIF select.val # NIL THEN
      res.selectOnly (ListVBTPosition (res, select.val))
    END;
    AddNameProp (cl, res, name, state);
    RETURN res
  END pBrowser;

PROCEDURE pMultiBrowser (cl: ParseClosure; VAR list: List.T; state: State):
  VBT.T RAISES {Error} =
  VAR
    name                   := NamePP ();
    value                  := NEW (CardinalListPP, name := "Value");
    select                 := NEW (TextListPP, name := "Select");
    items                  := NEW (TextListPP, name := "Items");
    from                   := NEW (TextPP, name := "From");
    quick                  := NEW (BooleanPP, name := "Quick");
    res   : FVMultiBrowser;
    s     : MultiSelector;
    colors                 := state.shadow;
  BEGIN
    ParseProps (
      cl, list, state, PP5 {name, value, select, items, from}, KP1 {quick});
    AssertEmpty (list);
    res := cl.fv.realize ("MultiBrowser", name.val.name);
    TYPECASE res.painter OF
    | NULL =>
        res.painter :=
          NEW (ListVBT.TextPainter).init (
            colors.bg, colors.fg, colors.fg, colors.bg, state.font)
    | ListVBT.TextPainter (tp) =>
        EVAL tp.init (colors.bg, colors.fg, colors.fg, colors.bg, state.font)
    ELSE
    END;
    TYPECASE res.selector OF
    | NULL => s := NEW (MultiSelector).init (res); res.selector := s
    | MultiSelector (sel) => s := sel
    ELSE
      RAISE Error ("Browser has a selector that is not a subtype "
                     & "of MultiSelector")
    END;
    s.quick := quick.val;
    s.browser := res;
    EVAL res.init (colors := state.shadow);
    IF items.val # NIL THEN
      SetValues (res, items.val)
    ELSIF from.val # NIL THEN
      SetValues (res, ItemsFromFile (from.val, cl))
    END;
    IF value.val # NIL THEN
      REPEAT
        res.select (NARROW (List.Pop (value.val), REF INTEGER)^, TRUE)
      UNTIL value.val = NIL
    ELSIF select.val # NIL THEN
      REPEAT
        res.select (ListVBTPosition (res, List.Pop (select.val)), TRUE)
      UNTIL select.val = NIL
    END;
    AddNameProp (cl, res, name, state);
    RETURN res
  END pMultiBrowser;

PROCEDURE SetValues (v: ListVBT.T; new: List.T) =
  VAR
    oldCount := v.count ();
    newCount := List.Length (new);
    delta    := oldCount - newCount;
  BEGIN
    IF delta < 0 THEN
      v.insertCells (oldCount, -delta)
    ELSIF delta > 0 THEN
      v.removeCells (newCount, delta)
    END;
    FOR j := 0 TO newCount - 1 DO v.setValue (j, List.Pop (new)) END
  END SetValues;
  
PROCEDURE ListVBTPosition (v: ListVBT.T; item: TEXT):
  [-1 .. LAST (CARDINAL)] =
  BEGIN
    FOR i := v.count () - 1 TO 0 BY -1 DO
      IF Text.Equal (v.getValue (i), item) THEN RETURN i END
    END;
    RETURN -1
  END ListVBTPosition;
  
PROCEDURE ItemsFromFile (name: TEXT; cl: ParseClosure): List.T RAISES {Error} =
  VAR tl: List.T := NIL;
  BEGIN
    TRY                         (* EXCEPT *)
      WITH in = Rsrc.Open (name, cl.fv.path) DO
        TRY                     (* FINALLY *)
          TRY                   (* EXCEPT *)
            LOOP List.Push (tl, Rd.GetLine (in)) END
          EXCEPT
          | Rd.EndOfFile => RETURN List.ReverseD (tl)
          END                   (* TRY *)
        FINALLY
          Rd.Close (in)
        END                     (* TRY *)
      END                       (* WITH *)
    EXCEPT
    | Rd.Failure (ref) => RAISE Error (RdUtils.FailureText (ref))
    | Rsrc.NotFound => RAISE Error ("No such resource: " & name)
    END                         (* TRY *)
  END ItemsFromFile;


(* =========================== Insert =========================== *)

PROCEDURE InsertFile (pathname: TEXT; path: Rsrc.Path): List.T
  RAISES {Error} =
  VAR
    res: List.T := NIL;
    rd : Rd.T;
  BEGIN
    TRY
      rd := Rsrc.Open (pathname, path);
      TRY
        LOOP List.Push (res, Sx.Read (rd, syntax := FVSyntax)) END
      FINALLY
        Rd.Close (rd)
      END
    EXCEPT
    | Sx.ReadError (txt) => RAISE Error ("Sx.ReadError: " & txt)
    | Rd.EndOfFile => RETURN List.ReverseD (res)
    | Rd.Failure (ref) => RAISE Error (RdUtils.FailureText (ref))
    | Rsrc.NotFound => RAISE Error ("No such resource: " & pathname)
    END
  END InsertFile;

(* =========================== Menus =============================== *)

PROCEDURE pMenu (cl: ParseClosure; VAR list: List.T; state: State):
  VBT.T RAISES {Error} =
  VAR
    name  := NamePP ();
    local := NEW (BooleanPP, name := "NotInTrestle");
    res  : FVMenu;
    count: CARDINAL;
  BEGIN
    ParseProps (cl, list, state, PP1 {name}, KP1 {local});
    WITH feedback = NEW (ShadowedFeedbackVBT.T).init (
                      NIL, state.shadow),
         menuFrame = NEW (ShadowedVBT.T).init (
                       NIL, state.shadow,
                       Shadow.Style.Raised) DO
      res := cl.fv.realize ("Menu", name.val.name);
      IF local.val THEN
        count := 0
      ELSE
        count := LAST (CARDINAL)
      END;
      EVAL res.init (feedback, menuFrame, count, state.menubar);
      AddChildren (cl, res, list, state);
      AddNameProp (cl, res, name, state);
      RETURN res
    END
  END pMenu;

PROCEDURE pMenuBar (cl: ParseClosure; VAR list: List.T; state: State): VBT.T
  RAISES {Error} =
  VAR
    name                    := NamePP ();
    res : FVMenuBar;
    ch  : VBT.T;
  BEGIN
    ParseProps (cl, list, state, PP1 {name});
    ch := OneChild (cl, list, state);
    res := cl.fv.realize ("MenuBar", name.val.name);
    state.menubar := res;
    EVAL res.init (ch);
    AddNameProp (cl, res, name, state);
    RETURN res
  END pMenuBar;


(* =========================== Numeric =============================== *)

PROCEDURE pNumeric (cl: ParseClosure; VAR list: List.T; state: State): VBT.T
  RAISES {Error} =
  VAR
    name        := NamePP ();
    allowEmpty  := NEW (BooleanPP, name := "AllowEmpty");
    hideButtons := NEW (BooleanPP, name := "HideButtons");
    value       := NEW (IntegerPP, name := "Value");
    min         := NEW (IntegerPP, name := "Min", val := FIRST (INTEGER));
    max         := NEW (IntegerPP, name := "Max", val := LAST (INTEGER));
    res: FVNumeric;
  BEGIN
    ParseProps (cl, list, state, PP4 {min, max, value, name},
                KP2 {allowEmpty, hideButtons});
    AssertEmpty (list);
    IF max.val < min.val THEN
      RAISE Error (Fmt.F ("Numeric max (%s) is less than min (%s)",
                          Fmt.Int (max.val), Fmt.Int (min.val)))
    ELSIF NOT value.found THEN
      value.val := MIN (MAX (0, min.val), max.val)
    ELSIF min.val <= value.val AND value.val <= max.val THEN (* skip *)
    ELSE
      RAISE
        Error (
          Fmt.F ("Initial Numeric value (%s) is not between %s and %s",
                 Fmt.Int (value.val), Fmt.Int (min.val), Fmt.Int (max.val)))
    END;
    res := cl.fv.realize ("Numeric", name.val.name);
    EVAL res.init (min.val, max.val, allowEmpty.val, hideButtons.val,
                   state.font, state.shadow);
    NumericVBT.Put (res, value.val);
    AddNameProp (cl, res, name, state);
    RETURN res
  END pNumeric;


(* ======================= Pixmap & Texture =========================== *)

PROCEDURE pPixmap (cl: ParseClosure; VAR list: List.T; state: State): VBT.T
  RAISES {Error} =
  VAR
    name             := NamePP();
    main             := NEW(TextPP);
    res  : FVPixmap;
    image: Image.T;
    depth: INTEGER;
    op   : PaintOp.T;
  BEGIN
    ParseProps(cl, list, state, PP2{name, main}, main := main);
    res := cl.fv.realize("Pixmap", name.val.name);
    image := GetImage(main.val, cl, depth);
    IF depth > 1 THEN op := PaintOp.Copy ELSE op := state.shadow.bgFg END;
    EVAL res.init(image, op, state.shadow.bg);
    AddNameProp(cl, res, name, state);
    RETURN res
  END pPixmap;

PROCEDURE pTexture (    cl   : ParseClosure;
                    VAR list : List.T;
                        state: State         ): VBT.T
  RAISES {Error} =
  VAR
    name       := NamePP ();
    main       := NEW (TextPP, found := TRUE);
    localalign := NEW (BooleanPP, name := "LocalAlign");
    res: FVTexture;
    txt                    := Pixmap.Solid;
  BEGIN
    ParseProps (cl, list, state, PP2 {name, main}, KP1 {localalign}, main);
    res := cl.fv.realize ("Texture", name.val.name);
    IF main.val # NIL THEN txt := GetPixmap (main.val, cl) END;
    EVAL res.init (state.shadow.bgFg, txt, localalign.val);
    AddNameProp (cl, res, name, state);
    RETURN res
  END pTexture;

(* =========================== Scroller =============================== *)

PROCEDURE pScroller (cl: ParseClosure; VAR list: List.T; state: State): VBT.T
  RAISES {Error} =
  VAR
    name              := NamePP ();
    value             := NEW (IntegerPP, name := "Value", val := 50);
    min               := NEW (IntegerPP, name := "Min", val := 0);
    max               := NEW (IntegerPP, name := "Max", val := 100);
    v                 := NEW (BooleanPP, name := "Vertical");
    thumb             := NEW (CardinalPP, name := "Thumb", val := 0);
    step              := NEW (CardinalPP, name := "Step", val := 1);
    axis              := Axis.T.Hor;
    res  : FVScroller;
  BEGIN
    ParseProps (
      cl, list, state, PP6 {name, value, min, max, thumb, step}, KP1 {v});
    AssertEmpty (list);
    IF v.val THEN axis := Axis.T.Ver END;
    thumb.val := MIN (thumb.val, max.val - min.val);
    res := cl.fv.realize ("Scroller", name.val.name);
    EVAL res.init (axis, min.val, max.val, state.shadow, step.val, thumb.val);
    NumericScrollerVBT.Put (res, value.val);
    AddNameProp (cl, res, name, state);
    RETURN res
  END pScroller;


(* ======================== Source & Target ======================= *)

PROCEDURE pSource (cl: ParseClosure; VAR list: List.T; state: State): VBT.T
  RAISES {Error} =
  VAR
    name                   := NamePP ();
    res : FVSource;
    ch  : VBT.T;
  BEGIN
    ParseProps (cl, list, state, PP1 {name});
    ch := OneChild (cl, list, state);
    res := cl.fv.realize ("Source", name.val.name);
    EVAL res.init (NEW (ShadowedFeedbackVBT.T).init (ch, state.shadow));
    AddNameProp (cl, res, name, state);
    RETURN res
  END pSource;

PROCEDURE pTarget (cl: ParseClosure; VAR list: List.T; state: State): VBT.T
  RAISES {Error} =
  VAR
    name                   := NamePP ();
    res : FVTarget;
    ch  : VBT.T;
  BEGIN
    ParseProps (cl, list, state, PP1 {name});
    ch := OneChild (cl, list, state);
    res := cl.fv.realize ("Target", name.val.name);
    EVAL res.init (ch);
    SourceVBT.BeTarget (res, SourceVBT.NewTarget ());
    AddNameProp (cl, res, name, state);
    RETURN res
  END pTarget;

(* ==================== Filter, Generic, Viewport ===================== *)

PROCEDURE pFilter (cl: ParseClosure; VAR list: List.T; state: State): VBT.T
  RAISES {Error} =
  VAR
    name    := NamePP ();
    active  := NEW (BooleanPP, name := "Active");
    passive := NEW (BooleanPP, name := "Passive");
    dormant := NEW (BooleanPP, name := "Dormant");
    vanish  := NEW (BooleanPP, name := "Vanish");
    enum    := NEW (EnumPP).init (KP4 {active, passive, dormant, vanish}, 0);
    res: FVFilter;
    ch : VBT.T;
  BEGIN
    ParseProps (cl, list, state, PP1 {name}, enums := EP1 {enum});
    ch := OneChild (cl, list, state);
    res := cl.fv.realize ("Filter", name.val.name);
    EVAL res.init (ch, state.shadow);
    ReactivityVBT.Set (res, VAL (enum.chosen, ReactivityVBT.State));
    AddNameProp (cl, res, name, state);
    RETURN res
  END pFilter;

PROCEDURE pScale (cl: ParseClosure; VAR list: List.T; state: State): VBT.T
  RAISES {Error} =
  VAR
    name   := NamePP ();
    hscale := NEW (RealPP, name := "HScale", val := 1.0);
    vscale := NEW (RealPP, name := "VScale", val := 1.0);
  VAR
    res: FVScale;
    ch : VBT.T;
  BEGIN
    ParseProps (cl, list, state, PP3 {name, hscale, vscale});
    ch := OneChild (cl, list, state);
    IF hscale.val < 1.0E-6 THEN RAISE Error ("HScale is too small") END;
    IF vscale.val < 1.0E-6 THEN RAISE Error ("VScale is too small") END;
    res := cl.fv.realize ("Scale", name.val.name);
    EVAL res.init (ch);
    ScaleFilter.Scale (res, hscale.val, vscale.val);
    AddNameProp (cl, res, name, state);
    RETURN res
  END pScale;

PROCEDURE pGeneric (cl: ParseClosure; VAR list: List.T; state: State):
  VBT.T RAISES {Error} =
  VAR
    name                    := NamePP ();
    res : FVGeneric;
  BEGIN
    ParseProps (cl, list, state, PP1 {name});
    AssertEmpty (list);
    res := cl.fv.realize ("Generic", name.val.name);
    EVAL res.init (NEW (TextureVBT.T).init (txt := Pixmap.Gray),
                   FVRuntime.EMPTYSHAPE);
    AddNameProp (cl, res, name, state);
    RETURN res
  END pGeneric;

PROCEDURE pViewport (cl: ParseClosure; VAR list: List.T; state: State): VBT.T
  RAISES {Error} =
  VAR
    name        := NamePP ();
    fixed       := NEW (BooleanPP, name := "Fixed");
    h           := NEW (BooleanPP, name := "Horizontal");
    v           := NEW (BooleanPP, name := "Vertical");
    enum1       := NEW (EnumPP).init (KP2 {h, v}, 1);
    step        := NEW (CardinalPP, name := "Step", val := 10);
    horandver   := NEW (BooleanPP, name := "HorAndVer");
    horonly     := NEW (BooleanPP, name := "HorOnly");
    veronly     := NEW (BooleanPP, name := "VerOnly");
    noscroll    := NEW (BooleanPP, name := "NoScroll");
    alaviewport := NEW (BooleanPP, name := "AlaViewport");
    auto        := NEW (BooleanPP, name := "Auto");
    enum2 := NEW (EnumPP).init (KP6 {horandver, horonly, veronly, noscroll,
                                     alaviewport, auto}, 2);
    unrelated := NEW (BooleanPP, name := "UnrelatedShape");
    axis      := Axis.T.Ver;
    res: FVViewport;
    ch : VBT.T;
  BEGIN
    ParseProps (cl, list, state, PP2 {name, step}, KP2 {fixed, unrelated},
                enums := EP2 {enum1, enum2});
    ch := OneChild (cl, list, state);
    IF h.val THEN axis := Axis.T.Hor END;
    res := cl.fv.realize ("Viewport", name.val.name);
    (**** TEMPORARY PATCH: "AUTO" DEADLOCKS TRESTLE ****)
    (* Steveg Says it's now fixed. 8/19/92
    IF enum2.chosen = 5 (* Auto *) THEN
      enum2.chosen := 2 (* VerOnly *)
    END;
    *)
    EVAL res.init (ch := ch, axis := axis, shadow := state.shadow,
                   step := step.val, adjustableViews := NOT fixed.val,
                   scrollStyle := VAL (enum2.chosen, ViewportVBT.ScrollStyle),
                   shapeStyle :=
                     VAL (ORD (NOT unrelated.val), ViewportVBT.ShapeStyle));
    AddNameProp (cl, res, name, state);
    RETURN res
  END pViewport;


(* ============================= Text ================================= *)

PROCEDURE pText (cl: ParseClosure; VAR list: List.T; state: State):
  VBT.T RAISES {Error} =
  VAR
    name        := NamePP ();
    main        := NEW (TextPP, found := TRUE);
    margin      := NEW (RealPP, name := "Margin", val := 2.0);
    leftalign   := NEW (BooleanPP, name := "LeftAlign");
    centeralign := NEW (BooleanPP, name := "Center");
    rightalign  := NEW (BooleanPP, name := "RightAlign");
    enum := NEW (EnumPP).init (
              KP3 {leftalign, centeralign, rightalign}, 1);
    from         := NEW (TextPP, name := "From");
    res : FVText;
  BEGIN
    ParseProps (cl, list, state, PP4 {name, main, margin, from},
                main := main, enums := EP1 {enum});
    IF main.val # NIL THEN      (* skip *)
    ELSIF from.val # NIL THEN
      main.val := TextFromFile (from.val, cl)
    ELSE
      RAISE Error ("Main property is missing")
    END;
    res := cl.fv.realize ("Text", name.val.name);
    EVAL res.init (main.val, bgFg := state.shadow,
                   fnt := state.labelFont,
                   halign := FLOAT (enum.chosen) * 0.5,
                   hmargin := Pts.ToMM (margin.val));
    AddNameProp (cl, res, name, state);
    RETURN res
  END pText;


(* ========================== Text editors ========================= *)

PROCEDURE pTextArea (cl: ParseClosure; VAR list: List.T; state: State): VBT.T
  RAISES {Error} =
  VAR
    name                 := NamePP ();
    value                := NEW (TextPP, name := "Value", val := "");
    readOnly             := NEW (BooleanPP, name := "ReadOnly");
    expandOnDemand       := NEW (BooleanPP, name := "ExpandOnDemand");
    clip                 := NEW (BooleanPP, name := "Clip");
    turnMargin           := NEW (RealPP, name := "TurnMargin", val := 2.0);
    from                 := NEW (TextPP, name := "From");
    port          : Port;
    res: FVTextArea;
  BEGIN
    ParseProps (cl, list, state, PP4 {name, value, from, turnMargin},
                KP2 {readOnly, expandOnDemand});
    AssertEmpty (list);
    res := cl.fv.realize ("TextArea", name.val.name);
    TYPECASE res.port OF
    | NULL => port := NEW (Port); res.port := port
    | Port (p) => port := p
    ELSE
      RAISE Error ("TextArea has a port that is not a subtype of Port")
    END;
    EVAL
      port.init (
        singleLine := FALSE, font := state.font, colorScheme := state.shadow,
        expandOnDemand := expandOnDemand.val, wrap := NOT clip.val,
        turnMargin := turnMargin.val, readOnly := readOnly.val);
    IF value.found OR from.val = NIL THEN
      TextPort.SetText (port, value.val)
    ELSE
      TextPort.SetText (port, TextFromFile (from.val, cl))
    END;
    EVAL res.init (TextEditVBT.Kind.MultiLine);
    VBT.SetCursor (res, Cursor.TextPointer);
    AddNameProp (cl, res, name, state);
    RETURN res
  END pTextArea;

PROCEDURE pTextEdit (cl: ParseClosure; VAR list: List.T; state: State): VBT.T
  RAISES {Error} =
  VAR
    name        := NamePP ();
    value       := NEW (TextPP, name := "Value", val := "");
    readOnly    := NEW (BooleanPP, name := "ReadOnly");
    clip        := NEW (BooleanPP, name := "Clip");
    turnMargin  := NEW (RealPP, name := "TurnMargin", val := 2.0);
    from        := NEW (TextPP, name := "From");
  VAR
    port: Port;
    res : FVTextEdit;
  BEGIN
    ParseProps (cl, list, state, PP4 {name, value, from, turnMargin},
                KP2 {readOnly, clip});
    AssertEmpty (list);
    res := cl.fv.realize ("TextEdit", name.val.name);
    TYPECASE res.port OF
    | NULL => port := NEW (Port); res.port := port
    | Port (p) => port := p
    ELSE
      RAISE Error ("TextEdit has a port that is not a subtype of Port")
    END;
    EVAL port.init (singleLine := FALSE, font := state.font,
                    colorScheme := state.shadow, readOnly := readOnly.val,
                    wrap := NOT clip.val, turnMargin := turnMargin.val);
    IF value.found OR from.val = NIL THEN
      TextPort.SetText (port, value.val)
    ELSE
      TextPort.SetText (port, TextFromFile (from.val, cl))
    END;
    EVAL res.init (TextEditVBT.Kind.Scrollable, shadow := state.shadow);
    VBT.SetCursor (res, Cursor.TextPointer);
    AddNameProp (cl, res, name, state);
    RETURN res
  END pTextEdit;

PROCEDURE pTypescript (cl: ParseClosure; VAR list: List.T; state: State):
  VBT.T RAISES {Error} =
  VAR
    name        := NamePP ();
    readOnly    := NEW (BooleanPP, name := "ReadOnly");
    clip        := NEW (BooleanPP, name := "Clip");
    turnMargin  := NEW (RealPP, name := "TurnMargin", val := 2.0);
  VAR
    port: TypescriptPort;
    res : FVTypescript;
  BEGIN
    ParseProps (cl, list, state, PP2 {name, turnMargin}, KP2 {readOnly, clip});
    AssertEmpty (list);
    res := cl.fv.realize ("Typescript", name.val.name);
    TYPECASE res.port OF
    | NULL => port := NEW (TypescriptPort); res.port := port
    | TypescriptPort (p) => port := p
    ELSE
      RAISE Error (
              "Typescript has a port that is not a subtype of TypescriptPort")
    END;
    EVAL port.init (font := state.font, colorScheme := state.shadow,
                    readOnly := readOnly.val, wrap := NOT clip.val,
                    turnMargin := turnMargin.val);
    EVAL res.init (shadow := state.shadow);
    VBT.SetCursor (res, Cursor.TextPointer);
    AddNameProp (cl, res, name, state);
    RETURN res
  END pTypescript;

PROCEDURE TextFromFile (filename: TEXT; cl: ParseClosure): TEXT
  RAISES {Error} =
  BEGIN
    TRY
      RETURN Rsrc.Get (filename, cl.fv.path);
    EXCEPT
    | Rd.Failure (ref) => RAISE Error (RdUtils.FailureText (ref))
    | Rd.EndOfFile => RAISE Error ("End of file")
    | Thread.Alerted => RAISE Error ("interrupted (Thread.Alerted)")
    | Rsrc.NotFound => RAISE Error ("No such resource: " & filename)
    END
  END TextFromFile;

PROCEDURE NewShadowStyle (default := Shadow.Style.Flat): EnumPP =
  VAR
    flat     := NEW (BooleanPP, name := "Flat");
    raised   := NEW (BooleanPP, name := "Raised");
    lowered  := NEW (BooleanPP, name := "Lowered");
    ridged   := NEW (BooleanPP, name := "Ridged");
    chiseled := NEW (BooleanPP, name := "Chiseled");
  BEGIN
    RETURN NEW (EnumPP).init (
             KP5 {flat, raised, lowered, ridged, chiseled}, ORD (default))
  END NewShadowStyle;

PROCEDURE pTypeIn (    cl   : ParseClosure;
                   VAR list : List.T;
                       state: State         ): VBT.T
  RAISES {Error} =
  VAR
    name       := NamePP ();
    value      := NEW (TextPP, name := "Value", val := "");
    readOnly   := NEW (BooleanPP, name := "ReadOnly");
    scrollable := NEW (BooleanPP, name := "Scrollable");
    clip       := NEW (BooleanPP, name := "Clip");
    turnMargin := NEW (RealPP, name := "TurnMargin", val := 0.0);
    kind       := TextEditVBT.Kind.SingleLine;
  VAR
    port: Port;
    res : FVTypeIn;
  BEGIN
    ParseProps (cl, list, state, PP2 {name, value},
                KP3 {readOnly, scrollable, clip});
    AssertEmpty (list);
    res := cl.fv.realize ("TypeIn", name.val.name);
    TYPECASE res.port OF
    | NULL => port := NEW (Port); res.port := port
    | Port (p) => port := p
    ELSE
      RAISE
        Error ("TypeIn has a port that is not a subtype of Port")
    END;
    EVAL
      port.init (
        singleLine := NOT scrollable.val, font := state.font,
        colorScheme := state.shadow, wrap := NOT clip.val,
        turnMargin := turnMargin.val, readOnly := readOnly.val);
    IF scrollable.val THEN
      kind := TextEditVBT.Kind.Scrollable
    END;
    TextPort.SetText (port, value.val);
    EVAL res.init (kind, shadow := state.shadow);
    VBT.SetCursor (res, Cursor.TextPointer);
    AddNameProp (cl, res, name, state);
    RETURN res
  END pTypeIn;

(* ======================== ZSplits & ZChildren ===================== *)

PROCEDURE pZSplit (cl: ParseClosure; VAR list: List.T; state: State): VBT.T
  RAISES {Error} =
  VAR name := NamePP ();
  VAR res: FVZSplit;
  BEGIN
    ParseProps (cl, list, state, PP1 {name});
    res := cl.fv.realize ("ZSplit", name.val.name);
    EVAL res.init ();
    state.zsplit := res;
    AddChildren (cl, res, list, state);
    AddNameProp (cl, res, name, state);
    RETURN res
  END pZSplit;

PROCEDURE pZBackground (cl: ParseClosure; VAR list: List.T; state: State):
  VBT.T RAISES {Error} =
  VAR
    name                        := NamePP ();
    res : FVZBackground;
    ch  : VBT.T;
  BEGIN
    IF state.zsplit = NIL THEN
      RAISE Error ("ZBackground must be inside a ZSplit.")
    END;
    ParseProps (cl, list, state, PP1 {name});
    ch := OneChild (cl, list, state);
    res := cl.fv.realize ("ZBackground", name.val.name);
    EVAL res.init (ch);
    AddNameProp (cl, res, name, state);
    RETURN res
  END pZBackground;

PROCEDURE pZChassis (cl: ParseClosure; VAR list: List.T; state: State): VBT.T
  RAISES {Error} =
  VAR
    name    := NamePP ();
    open    := NEW (BooleanPP, name := "Open");
    noClose := NEW (BooleanPP, name := "NoClose");
    title   := NEW (VBTPP, name := "Title");
    at      := NEW (AtSpecPP, name := "At");
  VAR
    res           : FVZChassis;
    titleChild, ch: VBT.T;
  BEGIN
    IF state.zsplit = NIL THEN
      RAISE Error ("ZChassis must be inside a ZSplit.")
    END;
    at.val := NIL;
    ParseProps (cl, list, state, PP3 {name, title, at}, KP2 {open, noClose});
    IF title.val = NIL THEN
      titleChild := TextVBT.New ("<Untitled>", fnt := state.labelFont,
                                 bgFg := state.shadow)
    ELSE
      titleChild := OneChild (cl, title.val, state)
    END;
    res := cl.fv.realize ("ZChassis", name.val.name);
    state.zchild := res;
    ch := OneChild (cl, list, state);
    EVAL res.init (
           ch, titleChild, state.shadow, NOT noClose.val, open.val, at.val);
    AddNameProp (cl, res, name, state);
    RETURN res
  END pZChassis;

PROCEDURE pZChild (cl: ParseClosure; VAR list: List.T; state: State): VBT.T
  RAISES {Error} =
  VAR
    name                   := NamePP ();
    open                   := NEW (BooleanPP, name := "Open");
    at                     := NEW (AtSpecPP, name := "At");
    res : FVZChild;
    ch  : VBT.T;
  BEGIN
    IF state.zsplit = NIL THEN
      RAISE Error ("ZChild must be inside a ZSplit.")
    END;
    at.val := NIL;
    ParseProps (cl, list, state, PP2 {name, at}, KP1 {open});
    res := cl.fv.realize ("ZChild", name.val.name);
    state.zchild := res;
    ch := OneChild (cl, list, state);
    EVAL res.init (ch, open.val, at.val);
    AddNameProp (cl, res, name, state);
    RETURN res
  END pZChild;

PROCEDURE pZGrow (cl: ParseClosure; VAR list: List.T; state: State): VBT.T
  RAISES {Error} =
  VAR
    name                  := NamePP ();
    res : FVZGrow;
    ch  : VBT.T;
  BEGIN
    ParseProps (cl, list, state, PP1 {name});
    ch := OneChild (cl, list, state);
    res := cl.fv.realize ("ZGrow", name.val.name);
    EVAL res.init (NEW (ShadowedFeedbackVBT.T).init (ch, state.shadow));
    AddNameProp (cl, res, name, state);
    RETURN res
  END pZGrow;

PROCEDURE pZMove (cl: ParseClosure; VAR list: List.T; state: State): VBT.T
  RAISES {Error} =
  VAR
    name                  := NamePP ();
    res : FVZMove;
    ch  : VBT.T;
  BEGIN
    ParseProps (cl, list, state, PP1 {name});
    ch := OneChild (cl, list, state);
    res := cl.fv.realize ("ZMove", name.val.name);
    EVAL res.init (NEW (ShadowedFeedbackVBT.T).init (ch, state.shadow));
    AddNameProp (cl, res, name, state);
    RETURN res
  END pZMove;


(* ====================================================================== *)
(* Parsing routines for inherited properties ("states") *)
(* ====================================================================== *)

PROCEDURE pMacro (list: List.T; VAR state: State) RAISES {Error} =
  (* (Macro name [BOA] (formals) bq-expr) *)
  BEGIN
    WITH m    = Macro.Parse (list),
         pair = List.AssocQ (state.macros, list.first) DO
      IF pair # NIL THEN
        pair.tail.first := m
      ELSE
        List.Push (state.macros, List.List2 (list.first, m))
      END
    END
  END pMacro;

(* Follow the guidelines in Kobara's book on Motif.  Nice background colors
   have RGB components that are each between 155 and 175 on a scale of 0-255.
   If the color is in that range, then the LightShadow should be computed "by
   multiplying the background color R, G, and B numbers each by 1.50".  Well,
   that arithmetic isn't quite right; 175 * 1.50 > 255.  So we just scale
   linearly so that 175 comes out at 0.95 ("not quite hitting white").
   Likewise, the DarkShadow should be computed by multiplying the BgColor
   values by 0.5.  The values in an RGB.T will be "gamma-corrected"
   by Trestle, so we use "true RGB" values here. *)
   
CONST
  rgb155      = 155.0 / 255.0; 
  rgb175      = 175.0 / 255.0;
  scaleLight  = 0.95 / rgb175;
  scaleDark   = 0.5;
    
PROCEDURE pBgColor (list: List.T; VAR state: State) RAISES {Error} =
  VAR nice := TRUE;
  BEGIN
    WITH r = ColorRGB (list, PaintOp.BW.UseBg) DO
      state.bgRGB := r.rgb;
      state.bgOp := r.op;
      FOR i := 0 TO 2 DO
        nice := nice AND rgb155 <= r.rgb [i] AND r.rgb [i] <= rgb175
      END;
      IF nice THEN
        FOR i := 0 TO 2 DO
          state.lightRGB [i] := r.rgb [i] * scaleLight;
          state.darkRGB [i]  := r.rgb [i] * scaleDark
        END;
        state.lightOp :=
          PaintOpCache.FromRGB (
            state.lightRGB, PaintOp.Mode.Accurate, -1.0, PaintOp.BW.UseFg);
        state.darkOp :=
          PaintOpCache.FromRGB (
            state.darkRGB, PaintOp.Mode.Accurate, -1.0, PaintOp.BW.UseFg)
      END
    END;
    state.shadow := Shadow.New (state.shadowSz, state.bgOp, state.fgOp,
                                state.lightOp, state.darkOp)
  END pBgColor;

PROCEDURE pColor (list: List.T; VAR state: State) RAISES {Error} =
  BEGIN
    WITH r = ColorRGB (list) DO state.fgRGB := r.rgb; state.fgOp := r.op END;
    state.shadow := Shadow.New (state.shadowSz, state.bgOp, state.fgOp,
                                state.lightOp, state.darkOp)
  END pColor;

PROCEDURE pLightShadow (list: List.T; VAR state: State) RAISES {Error} =
  BEGIN
    WITH r = ColorRGB (list) DO
      state.lightRGB := r.rgb;
      state.lightOp := r.op
    END;
    state.shadow := Shadow.New (state.shadowSz, state.bgOp, state.fgOp,
                                state.lightOp, state.darkOp)
  END pLightShadow;

PROCEDURE pDarkShadow (list: List.T; VAR state: State) RAISES {Error} =
  BEGIN
    WITH r = ColorRGB (list) DO
      state.darkRGB := r.rgb;
      state.darkOp := r.op
    END;
    state.shadow := Shadow.New (state.shadowSz, state.bgOp, state.fgOp,
                                state.lightOp, state.darkOp)
  END pDarkShadow;

EXCEPTION BadColorSpec;         (* internal *)

TYPE RgbOp = RECORD rgb: RGB.T; op: PaintOp.T END;

VAR
  qRGB := SxSymbol.FromName ("RGB");
  qHSV := SxSymbol.FromName ("HSV");
  qCIE := SxSymbol.FromName ("CIE");
  
PROCEDURE ColorRGB (list: List.T; bw := PaintOp.BW.UseFg): RgbOp
  RAISES {Error} =
  VAR
    original        := list;
    res     : RgbOp;
    rep := qRGB;
  BEGIN
    TRY
      IF list = NIL THEN RAISE BadColorSpec END;
      TYPECASE list.first OF
      | NULL => RAISE BadColorSpec
      | TEXT (t) =>
          IF list.tail # NIL THEN RAISE BadColorSpec END;
          res.rgb := ColorName.ToRGB (t)
      | REFANY =>
          IF List.Length (list) = 4 THEN
            TYPECASE List.Pop (list) OF
            | NULL => RAISE BadColorSpec
            | SxSymbol.T (s) =>
              IF s = qRGB OR s = qHSV OR s = qCIE THEN rep := s
              ELSE RAISE BadColorSpec
              END
            ELSE RAISE BadColorSpec
            END
          END;              
          IF List.Length (list) # 3 THEN RAISE BadColorSpec END;
          FOR i := 0 TO 2 DO
            TYPECASE List.Pop (list) OF
            | NULL => RAISE BadColorSpec
            | REF INTEGER (ri) =>
                IF ri^ = 0 THEN
                  res.rgb [i] := 0.0
                ELSIF ri^ = 1 THEN
                  res.rgb [i] := 1.0
                ELSE
                  RAISE BadColorSpec
                END
            | REF REAL (rr) => res.rgb [i] := rr^
            ELSE
              RAISE BadColorSpec
            END
          END
      END;
      IF rep = qHSV THEN
        res.rgb := HSV.RGBFromHSV (res.rgb)
      ELSIF rep = qCIE THEN
        res.rgb := CIE.RGBFromXYZ (res.rgb)
      END;
      res.op :=
        PaintOpCache.FromRGB (res.rgb, PaintOp.Mode.Accurate, -1.0, bw)
    EXCEPT
    | BadColorSpec => Gripe ("Illegal color-spec: ", original)
    | ColorName.NotFound => Gripe ("No such color: ", original)
    END;
    RETURN res
  END ColorRGB;

PROCEDURE pFont (list: List.T; VAR state: State) RAISES {Error} =
  BEGIN
    IF List.Length (list) = 1 AND ISTYPE (list.first, TEXT) THEN
      state.fontName := OneText (list)
    ELSE
      state.fontMetrics :=
        ParseFont (list, state.fontMetrics, DefaultFontMetrics);
      state.fontName := MetricsToName (state.fontMetrics)
    END;
    state.font := FVRuntime.FindFont (state.fontName)
  END pFont;

PROCEDURE pLabelFont (list: List.T; VAR state: State) RAISES {Error} =
  BEGIN
    IF List.Length (list) = 1 AND ISTYPE (list.first, TEXT) THEN
      state.labelFontName := OneText (list)
    ELSE
      state.labelFontMetrics :=
        ParseFont (list, state.labelFontMetrics, DefaultLabelFontMetrics);
      state.labelFontName := MetricsToName (state.labelFontMetrics)
    END;
    state.labelFont := FindFont (state.labelFontName)
  END pLabelFont;

PROCEDURE MetricsToName (metrics: List.T): TEXT =
  VAR
    wr           := TextWr.New ();
    pair: List.T;
  <* FATAL Wr.Failure, Thread.Alerted *>
  BEGIN
    FOR i := 0 TO LAST (MetricsProcs) DO
      Wr.PutChar (wr, '-');
      pair := List.Assoc (metrics, MetricsProcs [i].name);
      IF pair = NIL THEN
        Wr.PutChar (wr, '*')
      ELSE
        Wr.PutText (wr, List.Second (pair))
      END
    END;
    RETURN TextWr.ToText (wr)
  END MetricsToName;
    
PROCEDURE ParseFont (alist, metrics, default: List.T): List.T RAISES {Error} =
  VAR n: INTEGER;
  PROCEDURE gripe (x: REFANY) RAISES {Error} =
    BEGIN
      Gripe ("Bad font-spec: ", x)
    END gripe;
  BEGIN
    WHILE alist # NIL DO
      TYPECASE List.Pop (alist) OF
      | NULL => gripe (NIL)
      | SxSymbol.T (sym) =>
          IF Text.Equal (sym.name, "Reset") THEN
            metrics := List.Append (default, metrics)
          ELSE
            gripe (sym)
          END
      | List.T (pair) =>
          TYPECASE pair.first OF
          | NULL => gripe (pair)
          | SxSymbol.T (sym) =>
              IF MetricsNameTable.in (sym.name, n) THEN
                MetricsProcs [n].proc (sym.name, pair.tail, metrics)
              ELSE
                gripe (pair)
              END
          | REFANY => gripe (pair)
          END
      | REFANY (r) => gripe (r)
      END
    END;
    RETURN metrics
  END ParseFont;

PROCEDURE mText (name: TEXT; arglist: List.T; VAR metrics: List.T)
  RAISES {Error} =
  BEGIN
    List.Push (metrics, List.List2 (name, OneText (arglist)))
  END mText;

PROCEDURE mCardinal (name: TEXT; arglist: List.T; VAR metrics: List.T)
  RAISES {Error} =
  BEGIN
    IF List.Length (arglist) = 1 THEN (* gripe *)
      TYPECASE arglist.first OF
      | NULL =>                 (* gripe *)
      | TEXT (t) =>
          IF Text.Equal (t, "*") THEN
            List.Push (metrics, List.List2 (name, t));
            RETURN
          END
      | REF INTEGER (ri) =>
          IF ri^ >= 0 THEN
            List.Push (metrics, List.List2 (name, Fmt.Int (ri^)));
            RETURN
          END
      ELSE                      (* gripe *)
      END
    END;
    Gripe ("Bad font-spec: ", arglist)
  END mCardinal;

PROCEDURE pShadowSize (list: List.T; VAR state: State) RAISES {Error} =
  BEGIN
    state.shadowSz := OneReal (list);
    state.shadow := Shadow.New (state.shadowSz, state.bgOp, state.fgOp,
                                state.lightOp, state.darkOp)
  END pShadowSize;


(* ====================================================================== *)
(* Parsing routines for local properties *)
(* ====================================================================== *)

TYPE
  PP = OBJECT                   (* Property pair *)
         name  := "Main";
         found := FALSE
       METHODS
         set (form: List.T) RAISES {Error}
       END;
  KP0 = ARRAY [0 .. -1] OF BooleanPP;
  KP1 = ARRAY [0 .. 0] OF BooleanPP;
  KP2 = ARRAY [0 .. 1] OF BooleanPP;
  KP3 = ARRAY [0 .. 2] OF BooleanPP;
  KP4 = ARRAY [0 .. 3] OF BooleanPP;
  KP5 = ARRAY [0 .. 4] OF BooleanPP;
  KP6 = ARRAY [0 .. 5] OF BooleanPP;

  PP0 = ARRAY [0 .. -1] OF PP;
  PP1 = ARRAY [0 .. 0] OF PP;
  PP2 = ARRAY [0 .. 1] OF PP;
  PP3 = ARRAY [0 .. 2] OF PP;
  PP4 = ARRAY [0 .. 3] OF PP;
  PP5 = ARRAY [0 .. 4] OF PP;
  PP6 = ARRAY [0 .. 5] OF PP;

  EP0 = ARRAY [0 .. -1] OF EnumPP;
  EP1 = ARRAY [0 .. 0] OF EnumPP;
  EP2 = ARRAY [0 .. 1] OF EnumPP;

PROCEDURE ParseProps (         cl   : ParseClosure;
                      VAR      list : List.T;
                      VAR      state: State;
                      READONLY props: ARRAY OF PP        := PP0 {};
                      READONLY keys : ARRAY OF BooleanPP := KP0 {};
                               main : PP                 := NIL;
                      READONLY enums: ARRAY OF EnumPP    := EP0 {}  )
  RAISES {Error} =
  (* This is where we parse the properties in a component-list.  We keep
     scanning items until we reach something that isn't a known property.  The
     component-parser that called us is responsible for parsing all the
     remaining items on the list. *)
  VAR copy := list;
  BEGIN
    WHILE list # NIL DO
      copy := list;
      list := ParseProp (cl, list, state, props, keys, main, enums);
      IF list = copy THEN EXIT END
    END;
    IF main = NIL THEN          (* skip *)
    ELSIF list # NIL THEN
      main.set (list);
      list := NIL
    ELSIF NOT main.found THEN
      RAISE Error ("Missing Main property")
    END;
    (* Make sure they picked one in each enumeration. *)
    FOR i := FIRST (enums) TO LAST (enums) DO
      IF enums [i].chosen # -1 THEN (* skip *)
      ELSIF NOT enums [i].choices [enums [i].default].found THEN
        enums [i].choices [enums [i].default].val := TRUE;
        enums [i].chosen := enums [i].default
      ELSE
        Gripe ("Default marked #False, but no alternative was selected: ",
               enums [i].choices [enums [i].default].name)
      END
    END
  END ParseProps;

PROCEDURE ParseProp (         cl   : ParseClosure;
                     VAR      list : List.T;
                     VAR      state: State;
                     READONLY props: ARRAY OF PP;
                     READONLY keys : ARRAY OF BooleanPP;
                              main : PP;
                     READONLY enums: ARRAY OF EnumPP     ): List.T
  RAISES {Error} =
  VAR sProc: StateProc;
  BEGIN
    TYPECASE list.first OF
    | NULL =>
    | SxSymbol.T (sym) =>       (* Is it a "keyword", like MenuStyle? *)
        FOR i := FIRST (keys) TO LAST (keys) DO
          IF Text.Equal (sym.name, keys [i].name) THEN
            keys [i].val := TRUE;
            keys [i].found := TRUE;
            RETURN list.tail
          END
        END;
        (* It might be an enumeration keyword. *)
        FOR i := FIRST (enums) TO LAST (enums) DO
          FOR j := FIRST (enums [i].choices^) TO LAST (enums [i].choices^) DO
            IF Text.Equal (sym.name, enums [i].choices [j].name) THEN
              IF enums [i].chosen # -1 THEN
                Gripe ("Contradictory choices: ",
                       enums [i].choices [j].name & " "
                         & enums [i].choices [enums [i].chosen].name)
              ELSE
                enums [i].choices [j].val := TRUE;
                enums [i].choices [j].found := TRUE;
                enums [i].chosen := j;
                RETURN list.tail
              END
            END
          END
        END
      (* If it's not a keyword, it might be a symbol-component (Bar or
         Fill) *)
    | List.T (form) =>
        TYPECASE List.Pop (form) OF
        | NULL =>
        | SxSymbol.T (sym) =>
            (* Is it specific to this component?  E.g., (Height ...) *)
            FOR i := FIRST (props) TO LAST (props) DO
              IF Text.Equal (sym.name, props [i].name) THEN
                props [i].set (form); (* parse and set *)
                props [i].found := TRUE;
                RETURN list.tail
              END
            END;
            (* Is it a state like (BgColor ...)? *)
            sProc := FindStateProc (sym.name);
            IF sProc # NIL THEN sProc (form, state); RETURN list.tail END;
            (* Is it a macro?  Expand and re-test. *)
            WITH m = MacroFunction (sym, state) DO
              IF m # NIL THEN RETURN List.New (m.apply (form), list.tail) END
            END;
            (* Is it a Boolean for this component?  E.g., (Flex #True ...) *)
            FOR i := FIRST (keys) TO LAST (keys) DO
              IF Text.Equal (sym.name, keys [i].name) THEN
                keys [i].val := OneBoolean (form);
                keys [i].found := TRUE;
                RETURN list.tail
              END
            END;
            (* Is it an enumeration keyword? *)
            FOR i := FIRST (enums) TO LAST (enums) DO
              FOR j := FIRST (enums [i].choices^)
                  TO LAST (enums [i].choices^) DO
                IF Text.Equal (sym.name, enums [i].choices [j].name) THEN
                  IF OneBoolean (form) THEN
                    IF enums [i].chosen # -1 THEN
                      Gripe ("Contradictory choices: ",
                             enums [i].choices [j].name & " "
                               & enums [i].choices [enums [i].chosen].name)
                    ELSE
                      enums [i].choices [j].val := TRUE;
                      enums [i].choices [j].found := TRUE;
                      enums [i].chosen := j;
                      RETURN list.tail
                    END
                  ELSIF enums [i].chosen = j THEN
                    enums [i].choices [j].val := FALSE;
                    enums [i].choices [j].found := FALSE;
                    enums [i].chosen := -1;
                    RETURN list.tail
                  ELSE
                    RETURN list.tail
                  END
                END
              END
            END;
            (* Is it (Main ...)? *)
            IF main # NIL AND Text.Equal (sym.name, "Main") THEN
              main.set (form);
              main.found := TRUE;
              RETURN list.tail
            END;
            (* Is it Insert? *)
            IF Text.Equal (sym.name, "Insert") THEN
              RETURN List.AppendD (
                       InsertFile (OneText (form), cl.fv.path), list.tail)
            END
          (* It must a component like (HBox ...). *)
        ELSE
        END
    ELSE
    END;
    RETURN list
  END ParseProp;


TYPE
  AtSpecPP = PP OBJECT val: List.T := NIL OVERRIDES set := SetAtSpecPP END;
  BooleanPP = PP OBJECT val := FALSE OVERRIDES set := SetBooleanPP END;
  CardinalPP =
    PP OBJECT val: CARDINAL := 0 OVERRIDES set := SetCardinalPP END;
  CardinalListPP =
    PP OBJECT val: List.T := NIL OVERRIDES set := SetCardinalListPP END;
  EnumPP =
    PP OBJECT
      choices: REF ARRAY OF BooleanPP;
      chosen : [-1 .. LAST (CARDINAL)]  := -1;
      default: CARDINAL                 := 0
    METHODS
      init (READONLY a: ARRAY OF BooleanPP; default: CARDINAL): 
      EnumPP := InitEnumPP
    END;
  IntegerPP = PP OBJECT val := 0 OVERRIDES set := SetIntegerPP END;
  RealPP = PP OBJECT val := 0.0 OVERRIDES set := SetRealPP END;
  SizeRangePP =
    PP OBJECT val := FlexShape.Default OVERRIDES set := SetSizeRangePP END;
  SymbolPP =
    PP OBJECT val: SxSymbol.T := NIL OVERRIDES set := SetSymbolPP END;
  TextPP = PP OBJECT val: TEXT := NIL OVERRIDES set := SetTextPP END;
  TextListPP =
    PP OBJECT val: List.T := NIL OVERRIDES set := SetTextListPP END;
  VBTPP = PP OBJECT
            val  : List.T        := NIL;
          OVERRIDES
            set := SetVBTPP
          END;

PROCEDURE InitEnumPP (         pp     : EnumPP;
                      READONLY a      : ARRAY OF BooleanPP;
                               default: CARDINAL            ): EnumPP =
  BEGIN
    pp.choices := NEW (REF ARRAY OF BooleanPP, NUMBER (a));
    pp.choices^ := a;
    pp.default := default;
    RETURN pp
  END InitEnumPP;

(*
PROCEDURE SetEnumerationPP (pp: EnumerationPP; form: List.T)
  RAISES {Error} =
  BEGIN
    WITH choice = OneSymbol (form).name DO
      FOR i := FIRST (pp.ref^) TO LAST (pp.ref^) DO
        IF Text.Equal (choice, pp.ref [i]) THEN pp.val := i; RETURN END
      END;
      RAISE Error (Fmt.F ("Illegal value for %s: %s", pp.name, choice))
    END
  END SetEnumerationPP;
*)

PROCEDURE SetSymbolPP (pp: SymbolPP; form: List.T) RAISES {Error} =
  BEGIN
    pp.val := OneSymbol (form)
  END SetSymbolPP;

PROCEDURE SetBooleanPP (pp: BooleanPP; form: List.T) RAISES {Error} =
  BEGIN
    pp.val := OneBoolean (form)
  END SetBooleanPP;

PROCEDURE SetIntegerPP (pp: IntegerPP; form: List.T) RAISES {Error} =
  BEGIN
    pp.val := OneInteger (form)
  END SetIntegerPP;

PROCEDURE SetRealPP (pp: RealPP; form: List.T) RAISES {Error} =
  BEGIN
    pp.val := OneReal (form)
  END SetRealPP;

PROCEDURE SetCardinalPP (pp: CardinalPP; form: List.T) RAISES {Error} =
  BEGIN
    pp.val := OneCardinal (form)
  END SetCardinalPP;

PROCEDURE SetCardinalListPP (pp: CardinalListPP; form: List.T)
  RAISES {Error} =
  PROCEDURE cardinalp (ref: REFANY): BOOLEAN =
    BEGIN
      TYPECASE ref OF
      | NULL => RETURN FALSE
      | REF INTEGER (ri) => RETURN ri^ >= 0
      ELSE
        RETURN FALSE
      END
    END cardinalp;
  BEGIN
    pp.val := ListOfType (form, cardinalp, "cardinals ")
  END SetCardinalListPP;

PROCEDURE SetTextListPP (pp: TextListPP; form: List.T) RAISES {Error} =
  PROCEDURE textp (ref: REFANY): BOOLEAN =
    BEGIN
      RETURN ISTYPE (ref, TEXT)
    END textp;
  BEGIN
    pp.val := ListOfType (form, textp, "texts ")
  END SetTextListPP;

PROCEDURE ListOfType (form: List.T;
                      p   : (PROCEDURE (ref: REFANY): BOOLEAN);
                      name: TEXT                                ): List.T
  RAISES {Error} =
  PROCEDURE every (l: List.T): BOOLEAN =
    BEGIN
      WHILE l # NIL DO IF NOT p (List.Pop (l)) THEN RETURN FALSE END END;
      RETURN TRUE
    END every;
  BEGIN
    (** Allow form to be (1 2 3 ...) or ((1 2 3 ...)),
        since =(1 2 3) is read as (Value (1 2 3)), which is
        the same as (Value 1 2 3). *)
    IF every (form) THEN RETURN form END;
    TYPECASE form.first OF
    | List.T (l) => IF form.tail = NIL AND every (l) THEN RETURN l END
    ELSE
    END;
    Gripe ("Bad list of " & name, form)
  END ListOfType;

EXCEPTION BadAtSpec;

PROCEDURE SetAtSpecPP (pp: AtSpecPP; form: List.T) RAISES {Error} =
  (* The code in ZChildVBT.m3 actually parses the AtSpec into a structure.
     All we do here is to steal some of that code to check the syntax. *)
  VAR original := form;
  BEGIN
    TRY
      CASE List.Length (form) OF
      | 2 => GetAtSpecCoord (form, 2)
      | 3 => GetAtSpecCoord (form, 2); GetAtSpecHotSpot (form)
      | 4 => GetAtSpecCoord (form, 4)
      ELSE
        RAISE BadAtSpec
      END
    EXCEPT
    | BadAtSpec => Gripe ("Bad 'At' spec: ", original)
    END;
    pp.val := original
  END SetAtSpecPP;

PROCEDURE GetAtSpecCoord (VAR list: List.T; n: CARDINAL)
  RAISES {BadAtSpec} =
  BEGIN
    FOR i := 1 TO n DO
      TYPECASE List.Pop (list) OF
      | NULL => RAISE BadAtSpec
      | REF INTEGER, REF REAL =>
      ELSE
        RAISE BadAtSpec
      END
    END
  END GetAtSpecCoord;

PROCEDURE GetAtSpecHotSpot (list: List.T) RAISES {BadAtSpec} =
  CONST HotSpots = ARRAY OF TEXT {"NW", "NE", "SW", "SE"};
  BEGIN
    TYPECASE list.first OF
    | NULL =>
    | SxSymbol.T (sym) =>
        FOR i := 0 TO 3 DO
          IF Text.Equal (sym.name, HotSpots [i]) THEN RETURN END
        END
    ELSE
    END;
    RAISE BadAtSpec
  END GetAtSpecHotSpot;

PROCEDURE SetSizeRangePP (pp: SizeRangePP; form: List.T) RAISES {Error} =
  BEGIN
    pp.val := SizeRange (form)
  END SetSizeRangePP;

EXCEPTION BadSize;

PROCEDURE SizeRange (VAR list: List.T): FlexShape.SizeRange
  RAISES {Error} =
  VAR
    size     := FlexShape.Default;
    original := list;
  BEGIN
    TRY
      IF list = NIL THEN RAISE BadSize END;
      GetNatural (list, size);
      IF List.Length (list) = 4 THEN GetStretchOrShrink (list, size); END;
      IF List.Length (list) = 2 THEN GetStretchOrShrink (list, size); END;
      IF List.Length (list) # 0 THEN RAISE BadSize END;
      RETURN size;
    EXCEPT
    | BadSize => Gripe ("Illegal size", original)
    END;
  END SizeRange;

PROCEDURE GetNatural (VAR list: List.T; VAR size: FlexShape.SizeRange)
  RAISES {BadSize} =
  BEGIN
    TYPECASE list.first OF
    | NULL => RAISE BadSize
    | REF REAL, REF INTEGER => size.natural := GetNum (list);
    ELSE
      (* no leading number *)
    END;
  END GetNatural;

PROCEDURE GetStretchOrShrink (VAR list: List.T;
                              VAR size: FlexShape.SizeRange)
  RAISES {BadSize} =
  BEGIN
    TYPECASE List.Pop (list) OF
    | NULL => RAISE BadSize
    | SxSymbol.T (sym) =>
        IF Text.Equal (sym.name, "+") THEN
          size.stretch := GetNum (list, TRUE)
        ELSIF Text.Equal (sym.name, "-") THEN
          size.shrink := GetNum (list)
        ELSE
          RAISE BadSize
        END
    ELSE
      RAISE BadSize
    END
  END GetStretchOrShrink;

CONST
  InfinityNames = ARRAY OF
                    TEXT {"Inf", "inf", "INF", "Infinity", "infinity",
                          "INFINITY"};

PROCEDURE GetNum (VAR list: List.T; infOK: BOOLEAN := FALSE): REAL
  RAISES {BadSize} =
  BEGIN
    TYPECASE List.Pop (list) OF
    | NULL =>
    | REF REAL (rr) => RETURN rr^
    | REF INTEGER (ri) => RETURN FLOAT (ri^)
    | SxSymbol.T (sym) =>
        IF NOT infOK THEN RAISE BadSize END;
        FOR i := FIRST (InfinityNames) TO LAST (InfinityNames) DO
          IF Text.Equal (sym.name, InfinityNames [i]) THEN
            RETURN FlexShape.Infinity
          END
        END
    ELSE
    END;
    RAISE BadSize
  END GetNum;

PROCEDURE SetVBTPP (pp: VBTPP; form: List.T)
  RAISES {Error} =
  BEGIN
    pp.val := form
  END SetVBTPP;

(*
  VAR
    state := pp.state;
    name  := NamePP ();
  BEGIN
    ParseProps (form, state, PP1 {name});
    pp.val := OneChild (pp.cl, form, state);
    AddNameProp (pp.cl, pp.val, name, state)
  END SetVBTPP;
*)

PROCEDURE OneChild (cl: ParseClosure; list: List.T; state: State): VBT.T
  RAISES {Error} =
  BEGIN
    IF list = NIL THEN
      Gripe ("A compoonent is required here", "")
    ELSIF list.tail # NIL THEN
      Gripe (
        Fmt.F (
          "A single component is required here: %s",
          Sx.ToText (list, syntax := FVSyntax, elision := Sx.Elision {3, 4})))
    ELSE
      RETURN Item (cl, List.Pop (list), state)
    END
  END OneChild;

PROCEDURE SetTextPP (pp: TextPP; form: List.T) RAISES {Error} =
  BEGIN
    pp.val := OneText (form)
  END SetTextPP;

PROCEDURE AddChildren (cl   : ParseClosure;
                       v    : Multi.T;
                       list : List.T;
                       state: State         ) RAISES {Error} =
  BEGIN
    WHILE list # NIL DO
      TYPECASE List.Pop (list) OF
      | NULL => Gripe ("NIL is an illegal form")
      | List.T (a) =>
          TYPECASE a.first OF
          | NULL => Gripe ("(NIL ...) is an illegal form")
          | SxSymbol.T (sym) =>
              IF Text.Equal (sym.name, "Insert") THEN
                list := List.Append (
                          InsertFile (OneText (a.tail), cl.fv.path), list)
              ELSE
                Multi.AddChild (v, Item (cl, a, state))
              END
          ELSE
            Multi.AddChild (v, Item (cl, a, state))
          END
      | REFANY (ra) => Multi.AddChild (v, Item (cl, ra, state))
      END
    END
  END AddChildren;

PROCEDURE OneText (list: List.T): TEXT RAISES {Error} =
  BEGIN
    IF list # NIL THEN
      TYPECASE list.first OF
      | NULL =>
          (* Technically, this is illegal, but the FormsVBT prettyprinter
             in Ivy converts "" to (), and there's still some of that code
             around. *)
          IF list.tail = NIL THEN RETURN "" END
      | TEXT (t) => IF list.tail = NIL THEN RETURN t END
      ELSE
      END
    END;
    Gripe ("Bad text-form: ", list)
  END OneText;

PROCEDURE OneCardinal (list: List.T): CARDINAL RAISES {Error} =
  BEGIN
    IF list # NIL THEN
      TYPECASE list.first OF
      | NULL =>
      | REF INTEGER (ri) =>
          IF ri^ >= 0 AND list.tail = NIL THEN RETURN ri^ END
      ELSE
      END
    END;
    Gripe ("Expected a cardinal integer: ", list)
  END OneCardinal;

PROCEDURE OneInteger (list: List.T): INTEGER RAISES {Error} =
  BEGIN
    IF list # NIL THEN
      TYPECASE list.first OF
      | NULL =>
      | REF INTEGER (ri) => IF list.tail = NIL THEN RETURN ri^ END
      ELSE
      END
    END;
    Gripe ("Expected an integer: ", list)
  END OneInteger;

PROCEDURE OneReal (list: List.T): REAL RAISES {Error} =
  BEGIN
    IF list # NIL THEN
      TYPECASE list.first OF
      | NULL =>
      | REF INTEGER (ri) => IF list.tail = NIL THEN RETURN FLOAT (ri^) END
      | REF REAL (rr) => IF list.tail = NIL THEN RETURN rr^ END
      ELSE
      END
    END;
    Gripe ("Expected a real number: ", list)
  END OneReal;

PROCEDURE OneBoolean (form: List.T): BOOLEAN RAISES {Error} =
  BEGIN
    IF form # NIL AND form.tail = NIL THEN
      TYPECASE form.first OF
      | NULL =>
      | REF BOOLEAN (rb) => RETURN rb^ ELSE END
    END;
    Gripe ("Not a BOOLEAN: ", form)
  END OneBoolean;

PROCEDURE OneSymbol (form: List.T): SxSymbol.T RAISES {Error} =
  BEGIN
    IF form # NIL AND form.tail = NIL THEN
      TYPECASE form.first OF
      | NULL =>
      | SxSymbol.T (sym) => RETURN sym
      ELSE
      END
    END;
    Gripe ("Not a symbol: ", form)
  END OneSymbol;

PROCEDURE AssertEmpty (list: List.T) RAISES {Error} =
  BEGIN
    IF list # NIL THEN Gripe ("Extra junk in form: ", list) END
  END AssertEmpty;

(* ====================== Runtime Utilities ========================= *)

PROCEDURE AddNameProp (         cl   : ParseClosure;
                                v    : VBT.T;
                                pp   : SymbolPP;
                       READONLY state: State         ) RAISES {Error} =
  VAR stateRef: REF State;
  BEGIN
    IF Named (pp) THEN
      FVRuntime.SetVBT (cl.fv, pp.val.name, v);
      stateRef := NEW (REF State);
      stateRef^ := state;
      VBT.PutProp (v, stateRef);
    END
  END AddNameProp;

PROCEDURE AddForProp (cl: ParseClosure; v: VBT.T; pp: SymbolPP)
  RAISES {Error} =
  BEGIN
    IF pp.val = NIL THEN RAISE Error ("A name is required here.") END;
    cl.fixupList := NEW (FixupLink, targetName := pp.val.name,
                         sourceVBT := v, next := cl.fixupList)
  END AddForProp;


(* ========================== Table Lookup =========================== *)

PROCEDURE FindComponentProc (name: TEXT): ComponentProc =
  VAR n: INTEGER;
  BEGIN
    IF ComponentNameTable.in (name, n) THEN
      RETURN ComponentProcs [n]
    ELSE
      RETURN NIL
    END
  END FindComponentProc;

PROCEDURE FindRealizeProc (name: TEXT): RealizeProc RAISES {Error} =
  VAR n: INTEGER;
  BEGIN
    IF ComponentNameTable.in (name, n) THEN
      RETURN RealizeProcs [n]
    ELSE
      Gripe ("Unknown component: ", name)
    END
  END FindRealizeProc;

PROCEDURE FindStateProc (name: TEXT): StateProc =
  VAR n: INTEGER;
  BEGIN
    IF StateNameTable.in (name, n) THEN
      RETURN StateProcs [n]
    ELSE
      RETURN NIL
    END
  END FindStateProc;

CONST
  StateNames = ARRAY OF
                 TEXT {"BgColor", "Color", "DarkShadow", "Font",
                       "LabelFont", "LightShadow", "Macro", "ShadowSize"};

CONST
  StateProcs = ARRAY [0 .. LAST (StateNames)] OF
                 StateProc {pBgColor, pColor, pDarkShadow, pFont,
                            pLabelFont, pLightShadow, pMacro, pShadowSize};

(* NOTE: FVTypes contains type declarations corresponding to each
   component. When a new component is added, be sure to add an entry to
    Also, if the VBT class for a component changes (unlikely, but
   possible), be sure to modify the component's entry in FVTypes
   appropriately. *)

CONST
  ComponentNames = ARRAY OF TEXT { "Bar", "Boolean", "Border",
                     "Browser", "Button", "Chisel", "Choice",
                     "CloseButton", "DirMenu", "FileBrowser",
                     "Fill", "Filter", "Frame", "Generic",
                     "Glue", "Guard", "HBox", "HPackSplit",
                     "HTile", "Helper", "LinkButton", "MButton",
                     "Menu", "MenuBar", "MultiBrowser",
                     "Numeric", "PageButton", "Pixmap",
                     "PopButton", "PopMButton", "Radio", "Ridge",
                     "Rim", "Scale", "Scroller", "Shape",
                     "Source", "TSplit", "Target", "Text",
                     "TextArea", "TextEdit", "Texture",
                     "TrillButton", "TypeIn", "Typescript",
                     "VBox", "VPackSplit", "VTile", "Viewport",
                     "ZBackground", "ZChassis", "ZChild",
                     "ZGrow", "ZMove", "ZSplit"};

CONST
  ComponentProcs = ARRAY [0 ..  LAST (ComponentNames)] OF
                     ComponentProc { pBar, pBoolean, pBorder,
                     pBrowser, pButton, pChisel, pChoice, pCloseButton,
                     pDirMenu, pFileBrowser, pFill, pFilter,
                     pFrame, pGeneric, pGlue, pGuard, pHBox,
                     pHPackSplit, pHTile, pHelper, pLinkButton,
                     pMButton, pMenu, pMenuBar, pMultiBrowser,
                     pNumeric, pPageButton, pPixmap, pPopButton,
                     pPopMButton, pRadio, pRidge, pRim, pScale,
                     pScroller, pShape, pSource, pTSplit,
                     pTarget, pText, pTextArea, pTextEdit,
                     pTexture, pTrillButton, pTypeIn,
                     pTypescript, pVBox, pVPackSplit, pVTile,
                     pViewport, pZBackground, pZChassis, pZChild,
                     pZGrow, pZMove, pZSplit};

CONST
  RealizeProcs = ARRAY [0 ..  LAST (ComponentNames)] OF
                   RealizeProc { rBar, rBoolean, rBorder,
                   rBrowser, rButton, rChisel, rChoice, rCloseButton,
                   rDirMenu, rFileBrowser, rFill, rFilter,
                   rFrame, rGeneric, rGlue, rGuard, rHBox,
                   rHPackSplit, rHTile, rHelper, rLinkButton,
                   rMButton, rMenu, rMenuBar, rMultiBrowser,
                   rNumeric, rPageButton, rPixmap, rPopButton,
                   rPopMButton, rRadio, rRidge, rRim, rScale,
                   rScroller, rShape, rSource, rTSplit, rTarget,
                   rText, rTextArea, rTextEdit, rTexture,
                   rTrillButton, rTypeIn, rTypescript, rVBox,
                   rVPackSplit, rVTile, rViewport, rZBackground,
                   rZChassis, rZChild, rZGrow, rZMove, rZSplit};

TYPE
  mp = RECORD
         name                         : TEXT;
         proc                         : MetricsProc;
         fontDefault, labelFontDefault: TEXT
       END;

(* In the following table, we use "impossible" names to prevent the client
   from specifying AdStyle and PixelSize, so these will always be "*" in
   the font name. *)
CONST
  MetricsProcs = ARRAY [0 .. 13] OF
                   mp {mp {"Foundry", mText, "*", "*"},
                       mp {"Family", mText, "fixed", "helvetica"},
                       mp {"WeightName", mText, "medium", "bold"},
                       mp {"Slant", mText, "r", "r"},
                       mp {"Width", mText, "semicondensed", "*"},
                       mp {" -AdStyle- ", mText, "*", "*"},
                       mp {" -PixelSize- ", mCardinal, "*", "*"},
                       mp {"PointSize", mCardinal, "100", "100"},
                       mp {"HRes", mCardinal, "*", "*"},
                       mp {"VRes", mCardinal, "*", "*"},
                       mp {"Spacing", mText, "*", "*"},
                       mp {"AvgWidth", mCardinal, "*", "*"},
                       mp {"Registry", mText, "iso8859", "iso8859"},
                       mp {"Encoding", mText, "1", "1"}};
(* The 14 metrics-components must be in this order, so that we can generate
   the strings easily.  I have no idea what "AdStyle" is. *)
   
VAR StateNameTable, ComponentNameTable, MetricsNameTable: TxtIntTbl.T;

PROCEDURE InitParser () =
  BEGIN
    StateNameTable := TxtIntTbl.New (NUMBER (StateNames));
    ComponentNameTable := TxtIntTbl.New (NUMBER (ComponentNames));
    MetricsNameTable := TxtIntTbl.New (NUMBER (MetricsProcs));
    FOR i := FIRST (StateNames) TO LAST (StateNames) DO
      EVAL StateNameTable.put (StateNames [i], i)
    END;
    FOR i := FIRST (ComponentNames) TO LAST (ComponentNames) DO
      EVAL ComponentNameTable.put (ComponentNames [i], i)
    END;
    FOR i := FIRST (MetricsProcs) TO LAST (MetricsProcs) DO
      EVAL MetricsNameTable.put (MetricsProcs [i].name, i)
    END;
    DefaultFontMetrics := NIL;
    DefaultLabelFontMetrics := NIL;
    FOR i := 0 TO 13 DO
      WITH mp = MetricsProcs [i] DO
        List.Push (DefaultFontMetrics, List.List2 (mp.name, mp.fontDefault));
        List.Push (
          DefaultLabelFontMetrics, List.List2 (mp.name, mp.labelFontDefault))
      END
    END
  END InitParser;

BEGIN 
END FormsVBT.
