(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)
(*                                                                           *)
(* Last modified on Sat Oct  3 12:07:50 PDT 1992 by meehan                       *)

MODULE Macro;

(* In this module, we handle the reading and printing of S-expressions, as
   well as the implementation of macros. *)

IMPORT Char, Fmt, FormsVBT, FWr, IntRefTbl,List, Rd, SxSymbol, SxSyntax, Sx,
       Text, Thread, Wr;

REVEAL
  T = Public BRANDED OBJECT
        name    : SxSymbol.T;
        formals : List.T       := NIL; (* of symbols *)
        expander: Op;           (* compiled object *)
        boa: BOOLEAN            (* actuals are not named *)
      OVERRIDES
        apply := Apply
      END;

 
VAR                             (* CONST *)
  (* From the FormsVBT language itself: *)
  qBOA   := SxSymbol.FromName ("BOA");
  qName  := SxSymbol.FromName ("Name");
  qValue := SxSymbol.FromName ("Value");
  (* "Internal" symbols for macros: *)
  qBackquote   := SxSymbol.FromName (" backquote ");
  qComma       := SxSymbol.FromName (" comma ");
  qCommaAtsign := SxSymbol.FromName (" comma-atsign ");
  qQuote       := SxSymbol.FromName (" quote ");
  (* From the List interface: *)
  ListM     := SxSymbol.FromName ("List");
  qAppend   := SxSymbol.FromName ("Append", ListM);
  qCons     := SxSymbol.FromName ("New", ListM);
  qFirst    := SxSymbol.FromName ("First", ListM);
  qLength   := SxSymbol.FromName ("Length", ListM);
  qList     := SxSymbol.FromName ("List", ListM);
  qListStar := SxSymbol.FromName ("List*", ListM);
  qNth      := SxSymbol.FromName ("Nth", ListM);
  qNthTail  := SxSymbol.FromName ("NthTail", ListM);
  qSecond   := SxSymbol.FromName ("Second", ListM);
  qTail     := SxSymbol.FromName ("Tail", ListM);
  qThird    := SxSymbol.FromName ("Third", ListM);
  (* OK, now we're asking for REAL trouble. *)
  (* Extensions to the "base" language for macros (a.k.a.  Lisp): *)
  qAnd    := SxSymbol.FromName ("AND");
  qEquals := SxSymbol.FromName ("=");
  qGE     := SxSymbol.FromName (">=");
  qGT     := SxSymbol.FromName (">");
  qIf     := SxSymbol.FromName ("IF");
  qLE     := SxSymbol.FromName ("<=");
  qLT     := SxSymbol.FromName ("<");
  qMinus  := SxSymbol.FromName ("-");
  qNIL    := SxSymbol.FromName ("NIL");
  qNot    := SxSymbol.FromName ("NOT");
  qOr     := SxSymbol.FromName ("OR");
  qPlus   := SxSymbol.FromName ("+");
  (* From the Text interface: *)
  TextM       := SxSymbol.FromName ("Text");
  qCat        := SxSymbol.FromName ("Cat", TextM);
  qTextEmpty  := SxSymbol.FromName ("Empty", TextM);
  qTextEqual  := SxSymbol.FromName ("Equal", TextM);
  qTextLength := SxSymbol.FromName ("Length", TextM);
  qTextSub    := SxSymbol.FromName ("Sub", TextM);
  (* From the SxSymbol interface: *)
  SxSymbolM := SxSymbol.FromName ("SxSymbol");
  qFromName := SxSymbol.FromName ("FromName", SxSymbolM);

PROCEDURE Parse (list: List.T): T RAISES {FormsVBT.Error} =
  (* list = (name [BOA] formals bqexp). *)
  VAR
    formals: List.T;
    res             := NEW (T);
    n               := List.Length (list);
  PROCEDURE err (msg: TEXT; x: REFANY := "") RAISES {FormsVBT.Error} =
    BEGIN
      RAISE
        FormsVBT.Error (Fmt.F ("Illegal Macro form: %s %s", msg, ToText (x)))
    END err;
  BEGIN
    res.boa := n = 4 AND list.tail.first = qBOA;
    IF NOT res.boa AND NOT n = 3 THEN err ("Syntax error") END;
    TYPECASE List.Pop (list) OF
    | NULL => err ("Macro name is NIL")
    | SxSymbol.T (s) => res.name := s
    | REFANY (r) => err ("Macro name isn't a symbol: ", r)
    END;
    IF res.boa THEN list := list.tail END;
    TYPECASE List.Pop (list) OF
    | List.T (x) => formals := x
    | REFANY (x) => err ("Bad list of formals: ", x)
    END;
    WHILE formals # NIL DO
      TYPECASE List.Pop (formals) OF
      | NULL => err ("Null formal")
      | SxSymbol.T (s) =>
          IF List.AssocQ (res.formals, s) # NIL THEN
            err ("Duplicate formal: ", s.name)
          ELSE
            List.Push (res.formals, List.List2 (s, NoDefault))
          END
      | List.T (pair) =>
          IF List.Length (pair) # 2 THEN
            err ("Bad formal", pair)
          ELSE
            TYPECASE pair.first OF
            | SxSymbol.T (s) =>
                IF List.AssocQ (res.formals, s) # NIL THEN
                  err ("Duplicate formal: ", s.name)
                ELSE
                  List.Push (res.formals, List.List2 (s, pair.tail.first))
                END
            ELSE
              err ("Bad formal", pair)
            END
          END
      | REFANY (r) => err ("Formals must be symbols: ", r)
      END
    END;
    res.formals := List.ReverseD (res.formals);
    res.expander := Compile (list.first, res.formals, RefanyTC);
    RETURN res
  END Parse;

CONST RefanyTC = -1;
VAR
  TextTC    := TYPECODE (TEXT);
  ListTC    := TYPECODE (List.T);
  IntegerTC := TYPECODE (REF INTEGER);
  RealTC    := TYPECODE (REF REAL);
  NullTC    := TYPECODE (NULL);
  BooleanTC := TYPECODE (REF BOOLEAN);
  SymbolTC  := TYPECODE (SxSymbol.T);
  NullOp    := NEW (QuoteOp, arg := NIL, tc := NullTC);

CONST LastTypeIndex = 7;

TYPE TypeIndex = [0 .. LastTypeIndex];

VAR TypeCodes: ARRAY TypeIndex OF INTEGER;

PROCEDURE TypeCodeIndex (tc: INTEGER): TypeIndex =
  BEGIN
    FOR i := FIRST (TypeIndex) TO LAST (TypeIndex) DO
      IF tc = TypeCodes [i] THEN RETURN i END
    END;
    <* ASSERT FALSE *>
    END TypeCodeIndex;

PROCEDURE InitTypeCodes () =
  PROCEDURE OK (a, b: TypeIndex) =
    BEGIN
      ComparableTypes [a, b] := TRUE;
      ComparableTypes [b, a] := TRUE
    END OK;
  BEGIN
    TypeCodes := ARRAY TypeIndex OF
                   INTEGER {RefanyTC, TextTC, ListTC, IntegerTC, RealTC,
                            NullTC, BooleanTC, SymbolTC};
    FOR i := FIRST (TypeIndex) TO LAST (TypeIndex) DO
      FOR j := FIRST (TypeIndex) TO LAST (TypeIndex) DO
        ComparableTypes [i, j] := i = j
      END
    END;
    WITH ref     = TypeCodeIndex (RefanyTC),
         text    = TypeCodeIndex (TextTC),
         list    = TypeCodeIndex (ListTC),
         integer = TypeCodeIndex (IntegerTC),
         real    = TypeCodeIndex (RealTC),
         null    = TypeCodeIndex (NullTC),
         boolean = TypeCodeIndex (BooleanTC),
         symbol  = TypeCodeIndex (SymbolTC)  DO
      OK (ref, text);
      OK (ref, list);
      OK (ref, null);
      OK (ref, symbol);
      OK (text, null);
      OK (list, null);
    END;
  END InitTypeCodes;

VAR ComparableTypes: ARRAY TypeIndex, TypeIndex OF BOOLEAN;

VAR VarOps := ARRAY [0 .. 5] OF VarOp {NIL, ..};

PROCEDURE Comparable (a, b: INTEGER): BOOLEAN =
  BEGIN
    RETURN ComparableTypes [TypeCodeIndex (a), TypeCodeIndex (b)]
  END Comparable;

PROCEDURE Compile (exp: REFANY; formals: List.T; tc := RefanyTC): Op
  RAISES {FormsVBT.Error} =
  VAR
    value: REFANY;
    c    : Compiler;
  BEGIN
    TYPECASE exp OF
    | NULL =>
        Check (tc, NullTC);
        RETURN NullOp
    | SxSymbol.T (s) =>
        WITH p = Position (formals, s) DO
          IF p = -1 THEN
            RAISE FormsVBT.Error ("Unbound variable: " & s.name)
          ELSIF p < NUMBER (VarOps) THEN
            RETURN VarOps [p]
          ELSE
            RETURN NEW (VarOp, index := p)
          END
        END
    | TEXT =>
        Check (tc, TextTC);
        RETURN NEW (QuoteOp, arg := exp, tc := TextTC)
    | REF INTEGER =>
        Check (tc, IntegerTC);
        RETURN NEW (QuoteOp, arg := exp, tc := IntegerTC)
    | REF REAL =>
        Check (tc, RealTC);
        RETURN NEW (QuoteOp, arg := exp, tc := RealTC)
    | REF BOOLEAN =>
        Check (tc, BooleanTC);
        RETURN NEW (QuoteOp, arg := exp, tc := BooleanTC)
    | List.T (x) =>
        WITH f    = x.first,
             args = x.tail,
             n    = List.Length (args) DO
          TYPECASE f OF
          | SxSymbol.T (s) =>
              IF Ctable.in (s.number, value) THEN
                c := value;
                Check (tc, c.tc, c.n, n);
                RETURN c.compile (args, formals, tc)
              END
          ELSE
          END
        END
    ELSE
    END;
    RAISE FormsVBT.Error (
            "Illegal expression in macro definition:" & ToText (exp))
  END Compile;

TYPE Display = REF ARRAY OF REFANY;

PROCEDURE ToText (x: REFANY): TEXT =
  BEGIN
    TYPECASE x OF
    | NULL => RETURN "NIL"
    | TEXT (t) => RETURN t
    ELSE
      TRY
        RETURN Sx.ToText (x, syntax := FormsVBT.FVSyntax)
      EXCEPT
      | Thread.Alerted, Sx.PrintError => RETURN "<Unprintable expression>"
      END
    END
  END ToText;

PROCEDURE Fault (typeName: TEXT; arg: REFANY): REFANY RAISES {FormsVBT.Error} =
  BEGIN
    RAISE FormsVBT.Error (
            Fmt.F ("A %s was required here: %s", typeName, ToText (arg)))
  END Fault;
  
PROCEDURE Apply (m: T; actuals: List.T): REFANY RAISES {FormsVBT.Error} =
  PROCEDURE err (msg: TEXT; actuals: REFANY := "") RAISES {FormsVBT.Error} =
    BEGIN
      RAISE FormsVBT.Error (Fmt.F ("Error in call to macro %s: %s %s",
                                   m.name.name, msg, ToText (actuals)))
    END err;
  VAR
    ac           := List.Length (actuals);
    fc           := List.Length (m.formals);
    d            := NEW (Display, fc);
    vars: List.T := NIL;
    pair: List.T;
  BEGIN
    IF ac > fc THEN err ("Too many arguments: ", Sx.NewInteger (ac)) END;
    IF m.boa THEN
      FOR i := 0 TO ac - 1 DO d [i] := List.Pop (actuals) END;
      FOR i := ac TO fc - 1 DO
        pair := List.Nth (m.formals, i);
        IF pair.tail.first = NoDefault THEN
          err ("Argument has no default: ", pair.first)
        ELSE
          d [i] := pair.tail.first
        END
      END
    ELSE
      IF ac # fc THEN
        FOR i := 0 TO fc - 1 DO
          pair := List.Nth (m.formals, i);
          d [i] := pair.tail.first
        END
      END;
      WHILE actuals # NIL DO
        TYPECASE List.Pop (actuals) OF
        | NULL => err ("NIL argument")
        | List.T (y) =>
            IF List.Length (y) # 2 THEN
              err ("Illegal argument: ", y)
            ELSE
              WITH p = Position (m.formals, y.first) DO
                IF p = -1 THEN
                  err ("Unknown variable: ", y.first)
                ELSIF List.MemberQ (vars, y.first) THEN
                  err ("Argument passed twice: ", y.first)
                ELSE
                  d [p] := y.tail.first;
                  List.Push (vars, y.first)
                END
              END
            END
        | REFANY (r) => err ("Illegal argument: ", r)
        END
      END;
      IF fc # ac THEN
        FOR i := 0 TO fc - 1 DO
          IF d [i] = NoDefault THEN
            pair := List.Nth (m.formals, i);
            err ("No value was supplied for ", pair.first)
          END
        END
      END
    END;
    RETURN m.expander.eval (d)
  END Apply;

PROCEDURE Eval (op: Op; d: Display): REFANY RAISES {FormsVBT.Error} =
  BEGIN
    RETURN op.eval (d)
  END Eval;

TYPE
  Compiler = OBJECT
               tc: INTEGER;     (* the typecode of the result *)
               n : CARDINAL;    (* the number of parameters *)
             METHODS
               compile (args: List.T; formals: List.T; tc: INTEGER): Op
                        RAISES {FormsVBT.Error}
             END;
  Op = OBJECT
         tc           := RefanyTC;
         args: List.T
       METHODS
         eval (d: Display): REFANY RAISES {FormsVBT.Error}
       END;
  AppendOp = Op OBJECT OVERRIDES eval := EvalAppend END;
  CatOp = Op OBJECT OVERRIDES eval := EvalCat END;
  ConsOp = Op OBJECT OVERRIDES eval := EvalCons END;
  EmptyOp = Op OBJECT OVERRIDES eval := EvalEmpty END;
  EqualsOp = Op OBJECT OVERRIDES eval := EvalEquals END;
  IfOp = Op OBJECT OVERRIDES eval := EvalIf END;
  ListOp = Op OBJECT OVERRIDES eval := EvalList END;
  ListStarOp = Op OBJECT OVERRIDES eval := EvalListStar END;
  QuoteOp = Op OBJECT arg: REFANY OVERRIDES eval := EvalQuote END;
  SubOp = Op OBJECT OVERRIDES eval := EvalSub END;
  TextEqualOp = Op OBJECT OVERRIDES eval := EvalTextEqual END;
  TextLengthOp = Op OBJECT OVERRIDES eval := EvalTextLength END;
  VarOp = Op OBJECT index: CARDINAL OVERRIDES eval := EvalVar END;
  ArithOp = Op OBJECT type: INTEGER END;
  PlusOp = ArithOp OBJECT OVERRIDES eval := EvalPlus END;
  MinusOp = ArithOp OBJECT OVERRIDES eval := EvalMinus END;
  FromNameOp = Op OBJECT OVERRIDES eval := EvalFromName END;
  
VAR Ctable := IntRefTbl.New (20); (* Maps symbol.number -> compiler *)

PROCEDURE InitCompilers () =
  PROCEDURE f (s: SxSymbol.T; c: Compiler) =
    BEGIN
      EVAL Ctable.put (s.number, c)
    END f;
  BEGIN
    f (qQuote, NEW (Compiler, tc := RefanyTC, n := 1, compile := CompileQuote));
    f (qCons, NEW (Compiler, tc := ListTC, n := 2, compile :=
       CompileCons));
    f (qList, NEW (Compiler, tc := ListTC, n := LAST (CARDINAL),
                   compile := CompileList));
    f (qListStar, NEW (Compiler, tc := ListTC, n := LAST (CARDINAL),
                       compile := CompileListStar));
    f (qAppend, NEW (Compiler, tc := ListTC, n := 2, compile := CompileAppend));
    f (qBackquote,
       NEW (Compiler, tc := RefanyTC, n := 1, compile := CompileBackquote));
    f (qIf, NEW (Compiler, tc := RefanyTC, n := 3, compile := CompileIf));
    f (qEquals,
       NEW (Compiler, tc := BooleanTC, n := 2, compile := CompileEquals));
    f (qCat, NEW (Compiler, tc := TextTC, n := LAST (CARDINAL),
                  compile := CompileCat));
    f (qTextEmpty,
       NEW (Compiler, tc := BooleanTC, n := 1, compile := CompileEmpty));
    f (qTextSub, NEW (Compiler, tc := TextTC, n := 3, compile := CompileSub));
    f (qTextLength,
       NEW (Compiler, tc := IntegerTC, n := 1, compile := CompileTextLength));
    f (qTextEqual,
       NEW (Compiler, tc := BooleanTC, n := 2, compile := CompileTextEqual));
    f (qPlus, NEW (Compiler, tc := RefanyTC, n := LAST (CARDINAL),
                   compile := CompilePlus));
    f (qMinus, NEW (Compiler, tc := RefanyTC, n := LAST (CARDINAL),
                    compile := CompileMinus));
    f (qFromName,
       NEW (Compiler, tc := SymbolTC, n := 1, compile := CompileFromName));
    (* Many more to come ... *)
  END InitCompilers;

PROCEDURE Check (TCwanted, TCgonnaGet       : INTEGER;
                 argCountWanted, argCountGot: CARDINAL  := 0)
  RAISES {FormsVBT.Error} =
  BEGIN
    IF argCountWanted # argCountGot AND argCountWanted # LAST (CARDINAL) THEN
      RAISE FormsVBT.Error (
              Fmt.F ("Wrong number of args: %s instead of %s",
                     Fmt.Int (argCountGot), Fmt.Int (argCountWanted)))
    ELSIF TCwanted # RefanyTC AND TCgonnaGet # NullTC
            AND TCgonnaGet # TCwanted
            AND TCgonnaGet # RefanyTC (* NARROW at runtime *)
      THEN
      RAISE FormsVBT.Error ("Invalid type")
    END
  END Check;
  
PROCEDURE CompileQuote (<* UNUSED *> self   : Compiler;
                                     args   : List.T;
                        <* UNUSED *> formals: List.T;
                                     tc     : INTEGER   ): Op
  RAISES {FormsVBT.Error} =
  BEGIN
    WITH arg      = args.first,
         actualTC = TYPECODE (arg) DO
      Check (tc, actualTC);
      RETURN NEW (QuoteOp, arg := arg, tc := actualTC)
    END
  END CompileQuote;
    
PROCEDURE EvalQuote (x: QuoteOp; <* UNUSED *> d: Display): REFANY =
  BEGIN
    RETURN x.arg
  END EvalQuote;

PROCEDURE CompileCons (<* UNUSED *> self   : Compiler;
                                    args   : List.T;
                                    formals: List.T;
                                    tc     : INTEGER   ): Op
  RAISES {FormsVBT.Error} =
  BEGIN
    args.first := Compile (args.first, formals, RefanyTC);
    args.tail.first := Compile (args.tail.first, formals, ListTC);
    RETURN NEW (ConsOp, args := args, tc := tc)
  END CompileCons;

PROCEDURE EvalCons (x: ConsOp; d: Display): REFANY RAISES {FormsVBT.Error} =
  BEGIN
    RETURN
      List.New (Eval (x.args.first, d), GetList (Eval (x.args.tail.first, d)))
  END EvalCons;

PROCEDURE CompileList (<* UNUSED *> self   : Compiler;
                                    args   : List.T;
                                    formals: List.T;
                                    tc     : INTEGER   ): Op
  RAISES {FormsVBT.Error} =
  VAR res := NEW (ListOp, args := args, tc := tc);
  BEGIN
    WHILE args # NIL DO
      args.first := Compile (args.first, formals);
      args := args.tail
    END;
    RETURN res
  END CompileList;

PROCEDURE EvalList (x: ListOp; d: Display): REFANY RAISES {FormsVBT.Error} =
  VAR
    res: List.T := NIL;
    ops         := x.args;
  BEGIN
    WHILE ops # NIL DO List.Push (res, Eval (List.Pop (ops), d)) END;
    RETURN List.ReverseD (res)
  END EvalList;

PROCEDURE CompileListStar (<* UNUSED *> self   : Compiler;
                                        args   : List.T;
                                        formals: List.T;
                                        tc     : INTEGER   ): Op
  RAISES {FormsVBT.Error} =
  VAR res := NEW (ListStarOp, args := args, tc := tc);
  BEGIN
    WHILE args # NIL DO
      args.first := Compile (args.first, formals);
      args := args.tail
    END;
    RETURN res
  END CompileListStar;

PROCEDURE EvalListStar (x: ListStarOp; d: Display): REFANY
  RAISES {FormsVBT.Error} =
  VAR
    ops       := x.args;
    op   : Op := List.Pop (ops);
    first     := List.List1 (op.eval (d));
    last      := first;
  BEGIN
    WHILE ops.tail # NIL DO
      op := List.Pop (ops);
      List.Push (last.tail, op.eval (d));
      last := last.tail
    END;
    op := ops.first;
    last.tail := GetList (op.eval (d));
    RETURN first
  END EvalListStar;

PROCEDURE CompileAppend (<* UNUSED *> self   : Compiler;
                                      args   : List.T;
                                      formals: List.T;
                                      tc     : INTEGER   ): Op
  RAISES {FormsVBT.Error} =
  VAR res := NEW (AppendOp, args := args, tc := tc);
  BEGIN
    WHILE args # NIL DO
      args.first := Compile (args.first, formals, ListTC);
      args := args.tail
    END;
    RETURN res
  END CompileAppend;

PROCEDURE EvalAppend (x: AppendOp; d: Display): REFANY
  RAISES {FormsVBT.Error} =
  VAR
    res : List.T := NIL;
    args         := List.Reverse (x.args);
  BEGIN
    WHILE args # NIL DO
      res := List.Append (GetList (Eval (args.first, d)), res);
      args := args.tail
    END;
    RETURN res
  END EvalAppend;

PROCEDURE CompileIf (<* UNUSED *> self   : Compiler;
                                  args   : List.T;
                                  formals: List.T;
                                  tc     : INTEGER   ): Op
  RAISES {FormsVBT.Error} =
  BEGIN
    List.SetNth (args, 0, Compile (List.Nth (args, 0), formals, BooleanTC));
    List.SetNth (args, 1, Compile (List.Nth (args, 1), formals, RefanyTC));
    List.SetNth (args, 2, Compile (List.Nth (args, 2), formals, RefanyTC));
    RETURN NEW (IfOp, args := args, tc := tc)
  END CompileIf;

PROCEDURE EvalIf (x: IfOp; d: Display): REFANY RAISES {FormsVBT.Error} =
  BEGIN
    IF GetRefBoolean (Eval (List.Nth (x.args, 0), d))^ THEN
      RETURN Eval (List.Nth (x.args, 1), d)
    ELSE
      RETURN Eval (List.Nth (x.args, 2), d)
    END
  END EvalIf;

PROCEDURE CompileBackquote (<* UNUSED *> self   : Compiler;
                                         args   : List.T;
                                         formals: List.T;
                            <* UNUSED *> tc     : INTEGER   ): Op
  RAISES {FormsVBT.Error} =
  BEGIN
    (* There is no EvalBackquote.  Backquoted S-expressions simply expand into
       other S-expressions, which are in turn compiled. *)
    RETURN Compile (Backquote (List.First (args)), formals)
  END CompileBackquote;

PROCEDURE Backquote (exp: REFANY): REFANY RAISES {FormsVBT.Error} =
  (* This returns a Lisp-like S-expression that can be passed to Eval to
     produce a new FormsVBT expression.  The only operators are QUOTE,
     LIST, LIST*, and APPEND. *)
  BEGIN
    TYPECASE exp OF
    | NULL => RETURN NIL
    | List.T (list) =>
        IF list.first = qComma THEN
          RETURN list.tail.first
        ELSIF list.first = qBackquote THEN
          RETURN Backquote (Backquote (list.tail.first))
        ELSE
          TYPECASE list.first OF
          | NULL =>
          | List.T (sublist) =>
              IF sublist.first = qCommaAtsign THEN
                RETURN List.List3 (qAppend, sublist.tail.first,
                                   Backquote (list.tail))
              END
          ELSE
          END;
          RETURN Combine (Backquote (list.first), Backquote (list.tail))
        END
    ELSE
    END;
    RETURN List.List2 (qQuote, exp)
  END Backquote;

PROCEDURE Combine (car, cdr: REFANY): REFANY =
  BEGIN
    (* This implementation attempts to recycle cons-cells wherever possible. *)
    TYPECASE car OF
    | NULL =>
        TYPECASE cdr OF
        | NULL =>
            (* (cons NIL NIL) -> (QUOTE (NIL)) *)
            RETURN List.List2 (qQuote, List.List1 (NIL))
        | List.T (cdr) =>
            IF cdr.first = qQuote THEN
              (* (cons NIL (QUOTE x)) -> (QUOTE (NIL .  x)) *)
              cdr.tail.first := List.New (NIL, cdr.tail.first);
              RETURN cdr
            END
        ELSE
        END
    | List.T (car) =>
        IF car.first = qQuote THEN
          TYPECASE cdr OF
          | NULL =>
              (* (cons (QUOTE x) NIL) -> (QUOTE (x)) *)
              car.tail := List.List1 (car.tail);
              RETURN car
          | List.T (cdr) =>
              IF cdr.first = qQuote THEN
                (* (cons (QUOTE x) (QUOTE y)) -> (QUOTE (x .  y)) *)
                car.tail.tail := cdr.tail.first;
                cdr.first := car.tail;
                cdr.tail := NIL;
                car.tail := cdr;
                RETURN car
                (* RETURN List.List2 ( qQuote, List.New (car.tail.first,
                   cdr.tail.first)) *)
              ELSIF cdr.first = qList OR cdr.first = qListStar THEN
                List.Push (cdr.tail, car);
                RETURN cdr
              END
          ELSE
          END
        ELSE
          TYPECASE cdr OF
          | NULL =>
              (* (cons x NIL) -> (LIST x) *)
              RETURN List.List2 (qList, car)
          | List.T (cdr) =>
              IF cdr.first = qList OR cdr.first = qListStar THEN
                (* (cons x (LIST .  y)) -> (LIST x .  y) *)
                List.Push (cdr.tail, car);
                RETURN cdr
              END
          END
        END
    ELSE
    END;
    RETURN List.List3 (qListStar, car, cdr)
  END Combine;

PROCEDURE CompileEquals (<* UNUSED *> self   : Compiler;
                                      args   : List.T;
                                      formals: List.T;
                                      tc     : INTEGER   ): Op
  RAISES {FormsVBT.Error} =
  VAR
    res     := NEW (EqualsOp, args := args, tc := tc);
    op : Op;
  BEGIN
    args.first := Compile (args.first, formals);
    op := args.first;
    args.tail.first := Compile (args.tail.first, formals, op.tc);
    RETURN res
  END CompileEquals;

PROCEDURE EvalEquals (x: EqualsOp; d: Display): REFANY
  RAISES {FormsVBT.Error} =
  VAR
    op1: Op := x.args.first;
    op2: Op := x.args.tail.first;
    a       := op1.eval (d);
    b       := op2.eval (d);
  BEGIN
    IF a = b THEN
      RETURN Sx.True
    ELSIF NOT Comparable (op1.tc, op2.tc) THEN
      RAISE FormsVBT.Error ("Invalid comparison")
    ELSIF x.tc = IntegerTC THEN
      RETURN BooleanRefs [GetRefInteger (a)^ = GetRefInteger (b)^]
    ELSIF x.tc = RealTC THEN
      RETURN BooleanRefs [GetRefReal (a)^ = GetRefReal (b)^]
    ELSE
      (* If a and b are non-numeric refs, and we got here, then a # b. *)
      RETURN Sx.False
    END
  END EvalEquals;

PROCEDURE CompilePlus (<* UNUSED *> self   : Compiler;
                                    args   : List.T;
                                    formals: List.T;
                       <* UNUSED *> tc     : INTEGER   ): Op
  RAISES {FormsVBT.Error} =
  VAR
    foundType     := FALSE;
    type          := RefanyTC;
    res           := NEW (PlusOp, args := args);
    op       : Op;
  BEGIN
    IF args = NIL THEN RAISE FormsVBT.Error ("(+) isn't defined.") END;
    REPEAT
      op := Compile (args.first, formals);
      args.first := op;
      args := args.tail;
      IF foundType THEN
        IF (op.tc = IntegerTC OR op.tc = RealTC) AND op.tc # type THEN
          RAISE FormsVBT.Error ("Invalid argument to +")
        END
      ELSIF op.tc = IntegerTC OR op.tc = RealTC THEN
        foundType := TRUE;
        type := op.tc
      ELSIF op.tc # RefanyTC THEN
        RAISE FormsVBT.Error ("Invalid argument to +")
      END
    UNTIL args = NIL;
    res.type := type;
    RETURN res
  END CompilePlus;

PROCEDURE EvalPlus (x: PlusOp; d: Display): REFANY RAISES {FormsVBT.Error} =
  VAR
    ops     := x.args;
    op : Op;
  PROCEDURE AddIntegers (isum: INTEGER): REFANY RAISES {FormsVBT.Error} =
    BEGIN
      WHILE ops # NIL DO
        op := List.Pop (ops);
        isum := isum + GetRefInteger (op.eval (d))^
      END;
      RETURN Sx.NewInteger (isum)
    END AddIntegers;
  PROCEDURE AddReals (rsum: REAL): REFANY RAISES {FormsVBT.Error} =
    BEGIN
      WHILE ops # NIL DO
        op := List.Pop (ops);
        rsum := rsum + GetRefReal (op.eval (d))^
      END;
      RETURN Sx.NewReal (rsum)
    END AddReals;
  BEGIN
    IF x.type = IntegerTC THEN
      RETURN AddIntegers (0)
    ELSIF x.type = RealTC THEN
      RETURN AddReals (0.0)
    ELSE
      op := List.Pop (ops);
      TYPECASE op.eval (d) OF
      | NULL => RETURN Fault ("number", NIL)
      | REF INTEGER (ri) => RETURN AddIntegers (ri^)
      | REF REAL (rr) => RETURN AddReals (rr^)
      | REFANY (ref) => RETURN Fault ("number", ref)
      END
    END
  END EvalPlus;

PROCEDURE CompileMinus (<* UNUSED *> self   : Compiler;
                                     args   : List.T;
                                     formals: List.T;
                        <* UNUSED *> tc     : INTEGER   ): Op
  RAISES {FormsVBT.Error} =
  VAR
    foundType         := FALSE;
    type              := RefanyTC;
    res               := NEW (MinusOp, args := args);
    op       : Op;
  BEGIN
    IF args = NIL THEN RAISE FormsVBT.Error ("(-) isn't defined.") END;
    REPEAT
      op := Compile (args.first, formals);
      args.first := op;
      args := args.tail;
      IF foundType THEN
        IF (op.tc = IntegerTC OR op.tc = RealTC) AND op.tc # type THEN
          RAISE FormsVBT.Error ("Invalid argument to -")
        END
      ELSIF op.tc = IntegerTC OR op.tc = RealTC THEN
        foundType := TRUE;
        type := op.tc
      ELSIF op.tc # RefanyTC THEN
        RAISE FormsVBT.Error ("Invalid argument to -")
      END
    UNTIL args = NIL;
    res.type := type;
    RETURN res
  END CompileMinus;

PROCEDURE EvalMinus (x: MinusOp; d: Display): REFANY RAISES {FormsVBT.Error} =
  VAR
    ops     := x.args;
    op : Op;
  PROCEDURE SubIntegers (isum: INTEGER): REFANY RAISES {FormsVBT.Error} =
    BEGIN
      WHILE ops # NIL DO
        op := List.Pop (ops);
        isum := isum - GetRefInteger (op.eval (d))^
      END;
      RETURN Sx.NewInteger (isum)
    END SubIntegers;
  PROCEDURE SubReals (rsum: REAL): REFANY RAISES {FormsVBT.Error} =
    BEGIN
      WHILE ops # NIL DO
        op := List.Pop (ops);
        rsum := rsum - GetRefReal (op.eval (d))^
      END;
      RETURN Sx.NewReal (rsum)
    END SubReals;
  BEGIN
    IF x.type = IntegerTC THEN
      RETURN SubIntegers (0)
    ELSIF x.type = RealTC THEN
      RETURN SubReals (0.0)
    ELSE
      op := List.Pop (ops);
      TYPECASE op.eval (d) OF
      | NULL => RETURN Fault ("number", NIL)
      | REF INTEGER (ri) => RETURN SubIntegers (ri^)
      | REF REAL (rr) => RETURN SubReals (rr^)
      | REFANY (ref) => RETURN Fault ("number", ref)
      END
    END
  END EvalMinus;

PROCEDURE CompileCat (<* UNUSED *> self   : Compiler;
                                   args   : List.T;
                                   formals: List.T;
                                   tc     : INTEGER   ): Op
  RAISES {FormsVBT.Error} =
  VAR res := NEW (CatOp, args := args, tc := tc);
  BEGIN
    WHILE args # NIL DO
      args.first := Compile (args.first, formals, tc);
      args := args.tail
    END;
    RETURN res
  END CompileCat;

PROCEDURE EvalCat (x: CatOp; d: Display): REFANY RAISES {FormsVBT.Error} =
  VAR
    res     := "";
    ops     := x.args;
  BEGIN
    WHILE ops # NIL DO res := res & GetText (Eval (List.Pop (ops), d)) END;
    RETURN res
  END EvalCat;

PROCEDURE CompileFromName (<* UNUSED *> self   : Compiler;
                                        args   : List.T;
                                        formals: List.T;
                                        tc     : INTEGER   ): Op
  RAISES {FormsVBT.Error} =
  BEGIN
    args.first := Compile (args.first, formals, TextTC);
    RETURN NEW (FromNameOp, args := args, tc := tc)
  END CompileFromName;

PROCEDURE EvalFromName (x: FromNameOp; d: Display): REFANY
  RAISES {FormsVBT.Error} =
  BEGIN
    RETURN SxSymbol.FromName (GetText (Eval (x.args.first, d)))
  END EvalFromName;

PROCEDURE CompileEmpty (<* UNUSED *> self   : Compiler;
                                     args   : List.T;
                                     formals: List.T;
                                     tc     : INTEGER   ): Op
  RAISES {FormsVBT.Error} =
  BEGIN
    args.first := Compile (args.first, formals,TextTC);
    RETURN NEW (EmptyOp, args := args, tc := tc)
  END CompileEmpty;

VAR BooleanRefs := ARRAY BOOLEAN OF REF BOOLEAN {Sx.False, Sx.True};

PROCEDURE EvalEmpty (x: EmptyOp; d: Display): REFANY RAISES {FormsVBT.Error} =
  BEGIN
    RETURN BooleanRefs [Text.Empty (GetText (Eval (x.args.first, d)))]
  END EvalEmpty;

PROCEDURE CompileSub (<* UNUSED *> self   : Compiler;
                                   args   : List.T;
                                   formals: List.T;
                                   tc     : INTEGER   ): Op
  RAISES {FormsVBT.Error} =
  BEGIN
    List.SetNth (args, 0, Compile (List.Nth (args, 0), formals, tc));
    List.SetNth (args, 1, Compile (List.Nth (args, 1), formals, IntegerTC));
    List.SetNth (args, 2, Compile (List.Nth (args, 2), formals, IntegerTC));
    RETURN NEW (SubOp, args := args, tc := tc)
  END CompileSub;

PROCEDURE EvalSub (x: SubOp; d: Display): REFANY RAISES {FormsVBT.Error} =
  BEGIN
    RETURN Text.Sub (GetText (Eval (List.Nth (x.args, 0), d)),
                     GetRefCardinal (Eval (List.Nth (x.args, 1), d))^,
                     GetRefCardinal (Eval (List.Nth (x.args, 2), d))^)
  END EvalSub;

PROCEDURE CompileTextLength (<* UNUSED *> self   : Compiler;
                                          args   : List.T;
                                          formals: List.T;
                                          tc     : INTEGER   ): Op
  RAISES {FormsVBT.Error} =
  BEGIN
    args.first := Compile (args.first, formals, TextTC);
    RETURN NEW (TextLengthOp, args := args, tc := tc)
  END CompileTextLength;

PROCEDURE EvalTextLength (x: TextLengthOp; d: Display): REFANY
  RAISES {FormsVBT.Error} =
  BEGIN
    RETURN Sx.NewInteger (Text.Length (GetText (Eval (x.args.first, d))))
  END EvalTextLength;

PROCEDURE CompileTextEqual (<* UNUSED *> self   : Compiler;
                                         args   : List.T;
                                         formals: List.T;
                                         tc     : INTEGER   ): Op
  RAISES {FormsVBT.Error} =
  BEGIN
    List.SetNth (args, 0, Compile (List.Nth (args, 0), formals, TextTC));
    List.SetNth (args, 1, Compile (List.Nth (args, 1), formals, TextTC));
    RETURN NEW (TextEqualOp, args := args, tc := tc)
  END CompileTextEqual;

PROCEDURE EvalTextEqual (x: TextEqualOp; d: Display): REFANY
  RAISES {FormsVBT.Error} =
  BEGIN
    RETURN BooleanRefs [Text.Equal (GetText (Eval (List.Nth (x.args, 0), d)),
                                    GetText (Eval (List.Nth (x.args, 1), d)))]
  END EvalTextEqual;


PROCEDURE EvalVar (x: VarOp; d: Display): REFANY =
  BEGIN
    RETURN d [x.index]
  END EvalVar;

(* ******** Safe retrieval functions ******* *)

PROCEDURE GetText (ref: REFANY): TEXT RAISES {FormsVBT.Error} =
  BEGIN
    TYPECASE ref OF
    | NULL => RETURN Fault ("text", NIL)
    | TEXT (t) => RETURN t
    ELSE
      RETURN Fault ("text", ref)
    END
  END GetText;

PROCEDURE GetRefBoolean (ref: REFANY): REF BOOLEAN RAISES {FormsVBT.Error} =
  BEGIN
    TYPECASE ref OF
    | NULL => RETURN Fault ("boolean", NIL)
    | REF BOOLEAN (t) => RETURN t
    ELSE
      RETURN Fault ("boolean", ref)
    END
  END GetRefBoolean;
  
PROCEDURE GetRefInteger (ref: REFANY): REF INTEGER RAISES {FormsVBT.Error} =
  BEGIN
    TYPECASE ref OF
    | NULL => RETURN Fault ("integer", NIL)
    | REF INTEGER (t) => RETURN t
    ELSE
      RETURN Fault ("integer", ref)
    END
  END GetRefInteger;
  
PROCEDURE GetRefCardinal (ref: REFANY): REF INTEGER RAISES {FormsVBT.Error} =
  BEGIN
    TYPECASE ref OF
    | NULL => RETURN Fault ("integer", NIL)
    | REF INTEGER (t) => (* All Sx-integers are REF INTEGER *)
        IF t^ >= 0 THEN RETURN t ELSE RETURN Fault ("cardinal", t) END
    ELSE
      RETURN Fault ("integer", ref)
    END
  END GetRefCardinal;
  
PROCEDURE GetRefReal (ref: REFANY): REF REAL RAISES {FormsVBT.Error} =
  BEGIN
    TYPECASE ref OF
    | NULL => RETURN Fault ("real", NIL)
    | REF REAL (t) => RETURN t
    ELSE
      RETURN Fault ("real", ref)
    END
  END GetRefReal;
  
PROCEDURE GetList (ref: REFANY): List.T RAISES {FormsVBT.Error} =
  BEGIN
    TYPECASE ref OF
    | List.T (t) => RETURN t    (* NIL is OK here *)
    ELSE
      RETURN Fault ("list", ref)
    END
  END GetList;

PROCEDURE Position (list: List.T; item: REFANY): [-1 .. LAST (CARDINAL)] =
  VAR i: CARDINAL := 0;
  BEGIN
    LOOP
      IF list = NIL THEN RETURN -1
      ELSIF List.First (List.Pop (list)) = item THEN RETURN i
      ELSE INC (i)
      END
    END
  END Position;

(****************** Syntax for reading/writing %foo, =baz ******************)

TYPE
  Parser = SxSyntax.Parser OBJECT
             bqLevel: CARDINAL := 0
           OVERRIDES
             apply := FVSymbolParser
           END;
   
PROCEDURE FVSymbolParser (p     : Parser;
                          rd    : Rd.T;
                          c     : CHAR;
                          root  : SxSymbol.T;
                          syntax: SxSyntax.T  ): REFANY
  RAISES {Sx.ReadError, Rd.Failure, Thread.Alerted} =
  BEGIN
    TRY
      CASE c OF
      | '=' =>
          IF Rd.GetChar (rd) IN Char.Spaces THEN
            RETURN qEquals
          ELSE
            Rd.UnGetChar (rd);
            RETURN List.List2 (qValue, Sx.Read (rd, root, syntax))
          END
      | '%' => RETURN List.List2 (qName, Sx.Read (rd, root, syntax))
      | '\'' => RETURN List.List2 (qQuote, Sx.Read (rd, root, syntax))
      | '`' =>
          INC (p.bqLevel);
          TRY
            RETURN List.List2 (qBackquote, Sx.Read (rd, root, syntax))
          FINALLY
            DEC (p.bqLevel)
          END
      | ',' =>
          IF p.bqLevel = 0 THEN
            RAISE Sx.ReadError ("comma not inside backquote")
          ELSE
            DEC (p.bqLevel);
            TRY
              IF Rd.GetChar (rd) = '@' THEN
                RETURN List.List2 (qCommaAtsign, Sx.Read (rd, root, syntax))
              ELSE
                Rd.UnGetChar (rd);
                RETURN List.List2 (qComma, Sx.Read (rd, root, syntax))
              END
            FINALLY
              INC (p.bqLevel)
            END
          END
      ELSE
      END
    EXCEPT
    | Rd.EndOfFile => RAISE Sx.ReadError ("Premature EOF")
    END
  END FVSymbolParser;

PROCEDURE FVListPrinter (<* UNUSED *> p      : SxSyntax.Printer;
                                      fwr    : FWr.T;
                                      value  : REFANY;
                                      elision: Sx.Elision;
                                      root   : SxSymbol.T;
                                      syntax : SxSyntax.T;       )
  RAISES {Sx.PrintError, Wr.Failure, Thread.Alerted} =
  VAR x: List.T := value;
  PROCEDURE Default () RAISES {Sx.PrintError, Wr.Failure, Thread.Alerted} =
    BEGIN
      FWr.Begin (fwr, 2);
      Wr.PutChar (fwr, '(');
      LOOP
        syntax.Print (fwr, List.Pop (x), elision, root);
        IF x = NIL THEN EXIT END;
        Wr.PutChar (fwr, ' ');
        FWr.UnitedBreak (fwr, 0)
      END;
      Wr.PutChar (fwr, ')');
      FWr.End (fwr)
    END Default;
  BEGIN
    IF List.Length (x) # 2 THEN
      Default ()
    ELSIF x.first = qName THEN
      Wr.PutChar (fwr, '%');
      syntax.Print (fwr, x.tail.first, elision, root)
    ELSIF x.first = qValue THEN
      Wr.PutChar (fwr, '=');
      syntax.Print (fwr, x.tail.first, elision, root)
    ELSIF x.first = qQuote THEN
      Wr.PutChar (fwr, '\'');
      syntax.Print (fwr, x.tail.first, elision, root)
    ELSIF x.first = qBackquote THEN
      Wr.PutChar (fwr, '`');
      syntax.Print (fwr, x.tail.first, elision, root)
    ELSIF x.first = qComma THEN
      Wr.PutChar (fwr, ',');
      syntax.Print (fwr, x.tail.first, elision, root)
    ELSIF x.first = qCommaAtsign THEN
      Wr.PutText (fwr, ",@");
      syntax.Print (fwr, x.tail.first, elision, root)
    ELSE
      Default ()
    END
  END FVListPrinter;

PROCEDURE Initialize () =
  BEGIN
    (* Use a special syntax table to handle %name and =value *)
    FormsVBT.FVSyntax := SxSyntax.Standard ();
    FormsVBT.FVParser := NEW (Parser);

    SxSyntax.SetCharParser (FormsVBT.FVSyntax, '=', FormsVBT.FVParser);
    SxSyntax.SetCharParser (FormsVBT.FVSyntax, '%', FormsVBT.FVParser);
    SxSyntax.SetCharParser (FormsVBT.FVSyntax, '\'', FormsVBT.FVParser);
    SxSyntax.SetCharParser (FormsVBT.FVSyntax, '`', FormsVBT.FVParser);
    SxSyntax.SetCharParser (FormsVBT.FVSyntax, ',', FormsVBT.FVParser);

    FormsVBT.FVPrinter := NEW (SxSyntax.Printer, apply := FVListPrinter);
    SxSyntax.SetRefPrinter (
      FormsVBT.FVSyntax, TYPECODE (List.T), FormsVBT.FVPrinter);
  END Initialize;

VAR NoDefault: REFANY;

BEGIN
  Initialize ();
  NoDefault := NEW (REF CARDINAL); (* Any unique ref will do. *)
  InitCompilers ();
  InitTypeCodes ();
  FOR i := FIRST (VarOps) TO LAST (VarOps) DO
    VarOps [i] := NEW (VarOp, index := i)
  END                   
END Macro.
