(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)

(* Last modified on Tue Feb 11 16:18:58 PST 1992 by muller                   *)

INTERFACE File;

IMPORT Text;

(* The methods in this interface raise an exception if the requested
   operation cannot be performed.  In general, the exception Error is raised
   and is an indication that an error was reported by the operating system. In
   some cases, a more specific exception can be raised, to give a better
   idea of the problem; however, an implementation can chose raise Error
   rather than the more specific one, and clients of this interface must be
   prepared for that. *) 
  
EXCEPTION
  Error;
  CannotAccess;
  EOF;
  NoSpace;
  NotSeekable;


TYPE 
  F <: OBJECT METHODS
         close () RAISES {Error};
           (* close the file 'self'. *)

         seek (pos: CARDINAL) RAISES {Error};
           (* Change the current position for 'self' to be 'pos'. *)

         isaTTY (): BOOLEAN RAISES {Error};
           (* determine is 'self' is a terminal device.  This routine can
              actually answer FALSE even if the file is a terminal device; it
              is only intended as hint. *)

         isSeekable (): BOOLEAN RAISES {Error};
           (* determine if the file 'self' is seekable. *) END;

  R <: F OBJECT METHODS
         new (name: Text.T): R RAISES {CannotAccess, Error};
           (* open the file named 'name' for reading, set the current position
              at the beginning of it and associate self with it. If self is
              NIL, allocate a NEW R. 
              If the file cannot be accessed for reading, RAISE CannotAccess. *)

         read (VAR a: ARRAY OF CHAR; VAR n: CARDINAL) RAISES {EOF, Error};
           (* Read characters from the current position in 'self'.  If there
              are no characters before the end of file, raise the exception EOF.
              Otherwise, read as many as possible to fill 'a'; the number of
              characters actually read is returned in 'n' and the current
              position is advanced correspondingly . *)

         eof (): BOOLEAN RAISES ANY;
           (* return TRUE if the end of the file has been reached *) END;

  W <: F OBJECT METHODS       
         new (name: Text.T): W RAISES {CannotAccess, Error};
           (* open the file named 'name' for writing, set the current position
              at the beginning of it and associate self with it.  If self is 
              NIL, allocate a NEW W. The previous contents of the file is lost.
              If the file cannot be accessed for writing, raise CannotAccess.  *)

         write (READONLY a: ARRAY OF CHAR) RAISES {NoSpace, Error};
           (* Write the characters of 'a' at the current position in 'self'.  If
              there is no space on the physical device to store those
              characters, raise the exception NoSpace.  Otherwise, store those
              characters and advance the current position correspondingly. *) END;

END File.

