INTERFACE FileOp;

(***************************************************************************)
(*                      Copyright (C) Olivetti 1989                        *)
(*                          All Rights reserved                            *)
(*                                                                         *)
(* Use and copy of this software and preparation of derivative works based *)
(* upon this software are permitted to any person, provided this same      *)
(* copyright notice and the following Olivetti warranty disclaimer are     *) 
(* included in any copy of the software or any modification thereof or     *)
(* derivative work therefrom made by any person.                           *)
(*                                                                         *)
(* This software is made available AS IS and Olivetti disclaims all        *)
(* warranties with respect to this software, whether expressed or implied  *)
(* under any law, including all implied warranties of merchantibility and  *)
(* fitness for any purpose. In no event shall Olivetti be liable for any   *)
(* damages whatsoever resulting from loss of use, data or profits or       *)
(* otherwise arising out of or in connection with the use or performance   *)
(* of this software.                                                       *)
(***************************************************************************)

IMPORT Text;
IMPORT TimeDate, OSError;


(* Provides a portable interface to common file system operations. As this is
intended to be a portable interface the operations it provides will typically
be a subset of the operations available on most systems; hopefully it is a
useful subset.
  A system specific interface (called FileOp_<system> where <system> is the
system name) should be provided for more system specific file operations. *)


(* First enquiry functions. This interface provides a way to obtain a handle on
a file or directory. The handle can then be queried to find out about the file
or directory it represents. *)

TYPE
  AccessBit = {Readable, Writeable, Executable, Traversable};
  Access = SET OF AccessBit;

(* a simplification of the access control available on most systems. Only the
access of the current program is considered e.g. 'Readable' indicates only that
the current program can read the given file/directory - no information is given
about anyone else's access.
  The first three access bits mainly apply to files and are fairly obvious.
'Traversable' only applies to directories - 'Traversable' access to a directory
means that the program has access to look something up in that directory *)

  Info_public = OBJECT
  METHODS
    (* enquiry methods *)
    name(): Text.T RAISES {};             (* name *)
    isFile(): BOOLEAN RAISES {};          (* file or directory? *)
    access(): Access RAISES {};           (* access *)
    (* the following are not always meaningful for directories *)
    lastModified(): TimeDate.Stamp RAISES {}; (* last modified stamp *)
    length(): CARDINAL RAISES {};         (* length *)
  END;

  Info <: Info_public;


PROCEDURE GetInfo(name: Text.T; mustExist := TRUE): Info RAISES {OSError.E};
(* 'GetInfo' trys to returns an information handle for the named file. If the
operation succeeds the handle is returned and its methods can then be used to
find out more about the named file.
  If the operation fails 'GetInfo' raises 'OSError.E'. An exception to this
rule occurs if 'mustExist' is FALSE and the operation fails because no object
called 'name' exists. In this special case no exception is raised and NIL is
returned.
  The 'name' argument can be the null text in which case information about the
current directory is read. If the 'name' argument is NIL a checked runtime
error will occur. *)

PROCEDURE Same(info1, info2: Info): BOOLEAN RAISES {};
(* returns TRUE if 't1' and 't2' are handles on the same file system object. *)


PROCEDURE Accessible(name: Text.T): BOOLEAN RAISES {};
(* Checks if an object called 'name' exists in the file systems is accessible
in some way. Makes no promises as to how accessible 'name' is!. 'name' may be
null, indicating the current directory, but may not be NIL. If it is a checked
runtime error will occur *)

PROCEDURE Remove(name: Text.T; mustExist := TRUE) RAISES {OSError.E};
(* Remove the named file. Raises 'OSError.E' if the operation fails. If
'mustExist' is FALSE and the remove fails because 'name' does not exist no
exception is raised.
  If 'name' is a directory name the effect of 'Remove' is undefined. The
'DirOp' interface provides a directory 'Remove' operation. *)

PROCEDURE Rename(from, to: Text.T) RAISES {OSError.E};
(* Rename 'from' as 'to'. Raises 'OSError.E' if the operation fails. If 'from'
and 'to' are directory names the effect of 'Rename' is undefined. The
'DirOp' interface provides a directory 'Rename' operation. *)

END FileOp.
