(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)
(*                                                                           *)
(* Last modified on Tue Jun 30 23:16:58 PDT 1992 by meehan                   *)
<* PRAGMA LL *>

MODULE ClipboardVBT;

IMPORT Filter, VBT;

TYPE
  Public = Filter.T OBJECT
       METHODS
         <* LL.sup <= VBT.mu *>
         get (time: VBT.TimeStamp): TEXT;
         put (val: TEXT; time: VBT.TimeStamp)
       END;
       
REVEAL
  T = Public BRANDED OBJECT
        mu: MUTEX;
        <* LL = self.mu *>
        text := ""
      OVERRIDES
        init  := Init;
        read  := Read;
        write := Write;
        get := Get;
        put := Put
      END;

PROCEDURE Init (c: T; ch: VBT.T): Filter.T =
  <* LL.sup <= VBT.mu *>
  BEGIN
    c.mu := NEW (MUTEX);
    RETURN Filter.T.init (c, ch)
  END Init;

PROCEDURE Get (c: T; time: VBT.TimeStamp): TEXT =
  BEGIN
    LOCK c.mu DO
      TRY VBT.Acquire (c, VBT.Source, time) EXCEPT VBT.Error => END;
      RETURN c.text
    END
  END Get;

PROCEDURE Put (c: T; val: TEXT; time: VBT.TimeStamp) = 
  BEGIN
    LOCK c.mu DO
      TRY VBT.Acquire (c, VBT.Source, time) EXCEPT VBT.Error => END;
      c.text := val
    END
  END Put; 

PROCEDURE Read (c: T; s: VBT.Selection; typecode: CARDINAL): VBT.Value
  RAISES {VBT.Error} =
  BEGIN
    IF typecode # TYPECODE (TEXT) THEN
      RAISE VBT.Error (VBT.ErrorCode.WrongType)
    ELSIF s # VBT.Source THEN
      RAISE VBT.Error (VBT.ErrorCode.Unreadable)
    ELSE
      LOCK c.mu DO RETURN VBT.FromRef (c.text) END
    END
  END Read;

PROCEDURE Write (c: T; s: VBT.Selection; value: VBT.Value; typecode: CARDINAL)
  RAISES {VBT.Error} =
  BEGIN
    IF typecode # TYPECODE (TEXT) THEN
      RAISE VBT.Error (VBT.ErrorCode.WrongType)
    ELSIF s # VBT.Source THEN
      RAISE VBT.Error (VBT.ErrorCode.Unwritable)
    ELSE
      TYPECASE value.toRef () OF
      | TEXT (text) => LOCK c.mu DO c.text := text END
      ELSE
        RAISE VBT.Error (VBT.ErrorCode.WrongType)
      END
    END
  END Write;

PROCEDURE Find (v: VBT.T): T =
  BEGIN
    LOOP
      TYPECASE v OF
      | NULL => RETURN NIL
      | T (c) => RETURN c
      ELSE
        v := VBT.Parent (v)
      END
    END
  END Find;

BEGIN END ClipboardVBT.
