(* Copyright (C) 1991-1992, Digital Equipment Corporation                    *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)
(*                                                                           *)
(* Last modified on Fri Jul 31 17:51:01 PDT 1992 by meehan *)
<* PRAGMA LL *>

MODULE EditCmd EXPORTS EditCmd, TextPort, TextPortPrivate;

IMPORT Char, ClipboardVBT, MText, MTextRd, MTextUnit, Range, Rd, Text, Thread,
       VBT, VTDef, VText;

<* FATAL VTDef.Error *>(* Raised when vtext = NIL or closed *)

PROCEDURE ToNextWord (v: T; time: VBT.TimeStamp) =
  VAR
    right := LocateNextWordBoundary (v);
    left  := MTextUnit.StartOfRun (v.vtext.mtext, right);
  BEGIN
    IF left >= 0 THEN Select (v, time, left, right) END
  END ToNextWord;

PROCEDURE ToPrevWord (v: T; time: VBT.TimeStamp) =
  VAR
    left  := LocateNextWordBoundary (v, reverse := TRUE);
    right := MTextUnit.EndOfRun (v.vtext.mtext, left);
  BEGIN
    IF right >= 0 THEN
      Select (v, time, left, right, caretEnd := WhichEnd.Left)
    END
  END ToPrevWord;

PROCEDURE DeleteToEndOfWord (v: T; time: VBT.TimeStamp) =
  VAR
    start := VText.CaretIndex (v.vtext);
    end   := LocateNextWordBoundary (v);
  BEGIN
    Replace (v, start, end, "");
    Select (v, time, start, start)
  END DeleteToEndOfWord;

PROCEDURE DeleteToStartOfWord (v: T; time: VBT.TimeStamp) =
  VAR
    end   := VText.CaretIndex (v.vtext);
    start := LocateNextWordBoundary (v, reverse := TRUE);
  BEGIN
    Replace (v, start, end, "");
    Select (v, time, start, start)
  END DeleteToStartOfWord;

PROCEDURE LocateNextWordBoundary (v: T; reverse := FALSE): CARDINAL =
  <* FATAL Range.Error *> (* can't happen *)
  VAR
    index       := VText.CaretIndex (v.vtext);
    rd          := MTextRd.New (v.vtext.mtext, index, reverse := reverse);
    c    : CHAR;
    count       := 0;
  BEGIN
    TRY
      TRY
        REPEAT
          c := Rd.GetChar (rd);
          INC (count);
        UNTIL c IN Char.AlphaNumerics;
        REPEAT
          c := Rd.GetChar (rd);
          INC (count);
        UNTIL NOT c IN Char.AlphaNumerics;
        DEC (count)
      FINALLY
        Rd.Close (rd)
      END
    EXCEPT
      Rd.EndOfFile, Rd.Failure, Thread.Alerted =>
    END;
    IF reverse THEN RETURN index - count ELSE RETURN index + count END
  END LocateNextWordBoundary;

PROCEDURE Clear (v: T; time: VBT.TimeStamp) =
  VAR left, right: CARDINAL;
  BEGIN
    Selection (v, left, right);
    IF v.readOnly OR left < v.typeinStart THEN RETURN END;
    Replace (v, left, right, "");
    Select (v, time, left, left)
  END Clear;

PROCEDURE Copy (v: T; time: VBT.TimeStamp) =
  VAR c: ClipboardVBT.T;
  BEGIN
    LOCK v.mu DO
      c := v.clipboard;
      IF c = NIL THEN
        c := ClipboardVBT.Find (v);
        IF c = NIL THEN RETURN ELSE v.clipboard := c END
      END
    END;
    TRY
      VBT.Acquire (c, VBT.Source, time);
      VBT.Write (v, VBT.Source, time, VBT.Read (v, VBT.Target, time))
    EXCEPT
    | VBT.Error =>
    END
  END Copy;
        
PROCEDURE Cut (v: T; time: VBT.TimeStamp) =
  BEGIN
    Copy (v, time);
    Clear (v, time)
  END Cut;

PROCEDURE Paste (v: T; time: VBT.TimeStamp) =
  BEGIN
    TRY
      VBT.Write (v, VBT.Target, time, VBT.Read (v, VBT.Source, time))
    EXCEPT
    | VBT.Error => RETURN
    END
  END Paste;

PROCEDURE Move (v: T; time: VBT.TimeStamp) =
  BEGIN
    TRY
      VBT.Write (v, VBT.Target, time, VBT.Read (v, VBT.Source, time));
      VBT.Write (v, VBT.Source, time, VBT.FromRef (""))
    EXCEPT
    | VBT.Error =>
    END
  END Move;

PROCEDURE Swap (v: T; time: VBT.TimeStamp) =
  VAR primaryValue, secondaryValue: REFANY;
  BEGIN
    TRY
      primaryValue := VBT.Read (v, VBT.Target, time).toRef ();
      secondaryValue := VBT.Read (v, VBT.Source, time).toRef ();
      VBT.Write (v, VBT.Target, time, VBT.FromRef (secondaryValue));
      VBT.Write (v, VBT.Source, time, VBT.FromRef (primaryValue))
    EXCEPT
    | VBT.Error => RETURN
    END;
    VBT.Mark (v)
  END Swap;

PROCEDURE Undo (v: T) =
  <* LL < v.mu *>
  BEGIN
    LOCK v.mu DO
      IF v.cur.prev # NIL THEN v.cur := v.cur.prev; Exchange (v) END
    END
  END Undo;

PROCEDURE UndoUndo (v: T) =
  <* LL < v.mu *>
  BEGIN
    LOCK v.mu DO
      IF v.cur.next # NIL THEN Exchange (v); v.cur := v.cur.next END
    END
  END UndoUndo;

PROCEDURE UndoCount (v: T): CARDINAL =
  <* LL < v.mu *>
  VAR
    n: CARDINAL := 0;
    r: UndoRec;
  BEGIN
    LOCK v.mu DO
      r := v.cur;
      WHILE r.prev # NIL DO INC (n); r := r.prev END;
      RETURN n
    END
  END UndoCount;
  
PROCEDURE UndoUndoCount (v: T): CARDINAL =
  <* LL < v.mu *>
  VAR
    n: CARDINAL := 0;
    r: UndoRec;
  BEGIN
    LOCK v.mu DO
      r := v.cur;
      WHILE r.next # NIL DO INC (n); r := r.next END;
      RETURN n
    END
  END UndoUndoCount;

PROCEDURE ResetUndo (v: T) =
  <* LL < v.mu *>
  BEGIN
    LOCK v.mu DO v.cur := NEW (UndoRec) END
  END ResetUndo;

PROCEDURE Exchange (v: T) =
  <* LL = v.mu *>
  VAR
    prev := "";
    r    := v.cur;
  BEGIN
    IF r.begin < r.end AND r.begin < MText.Length (v.vtext.mtext) THEN
      prev := MText.GetText (v.vtext.mtext, r.begin, r.end)
    END;
    Normalize (v, r.begin);
    TRY
      VText.Replace (v.vtext, r.begin, r.end, r.text)
    EXCEPT
    | Thread.Alerted, Rd.Failure, Rd.EndOfFile => RETURN
    END;
    r.end := r.begin + Text.Length (r.text);
    r.text := prev;
    MarkAndUpdate (v);
    TraceUndo (v)
  END Exchange;
  
BEGIN END EditCmd.
