(* Copyright (C) 1991-1992, Digital Equipment Corporation                    *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)
(*                                                                           *)
(* Last modified on Wed Aug 12 22:59:04 PDT 1992 by meehan                   *)
(*      modified on Tue May 19 16:21:47 1992 by mhb                          *)
(*      modified on Thu May 14 01:45:33 1992 by steveg                       *)
(*      modified on Thu Mar 14 12:16:11 PST 1991 by brooks                   *)
(*      modified on Thu Feb  7 14:03:27 PST 1991 by chan                     *)
<* PRAGMA LL *>

MODULE TextPort EXPORTS TextPort, TextPortPrivate;

IMPORT Axis, Char, EditCmd, Font, KeyboardKey, KeyTrans, MText, MTextRd,
       MTextUnit, PaintOp, Palette, Pts, Range, Rd, RdUtils, Rect,
       Region, ScrollerVBT, Text, Thread, VBT, VBTRep, VTDef, VText;

IMPORT Fmt, SmallIO;

CONST
  Space     = Char.SP;
  Backspace = Char.BS;
  Tab       = Char.HT;
  Return    = Char.NL;
  Del       = Char.DEL;

TYPE
  SelectionRecord = RECORD
                      owned         : BOOLEAN;
                      interval      : VText.Interval;
                      mode          : VText.SelectionMode;
                      lastRightClick: VText.WhichEnd;
                    END;
REVEAL
  T = Private BRANDED OBJECT
        <* LL = v.mu *>
        (* short-term data for dragging *)
        dragging      : BOOLEAN;
        dragButton    : VBT.Button;
        dragSel       : SelectionType;
        pivotL, pivotR: CARDINAL;       (* bounds of the original interval *)
        fixed         : CARDINAL;       (* fixed pivot *)
        (* for ^O and ^P *)
        lastCmdKind: CommandKind;
        wishCol    : CARDINAL;
        (* durable user interface data *)
        selection: ARRAY SelectionType OF SelectionRecord;
        modifiedP: BOOLEAN;
        hasFocus : BOOLEAN;
        swapping : BOOLEAN;
        location: RECORD
                    margin: ARRAY Axis.T OF REAL;
                    align : ARRAY Axis.T OF REAL;
                  END;
        focusTime: VBT.TimeStamp;  (* event time when we last took KBFocus *)
        pattern  : TEXT;           (* last pattern for "find" commands *)
        <* LL.sup = VBT.mu.self *>
        lastNonEmptyWidth := 0;
      OVERRIDES
        (* VBT.T overrides *)
        repaint   := Repaint;
        reshape   := Reshape;
        rescreen  := Rescreen;
        redisplay := Redisplay;
        mouse     := Mouse;
        position  := Position;
        key       := Key;
        misc      := Misc;
        read      := Read;
        write     := Write;
        shape     := Shape;
        (* These are TextPort.Public overrides *)
        returnAction   := ReturnAction;
        tabAction      := Insert4spaces;
        defaultAction  := IgnoreKey;
        scrollUpdate   := UpdateScrollbar;
        init           := Init;
        getFont        := GetFont;
        setFont        := SetFont;
        getColorScheme := GetColorScheme;
        setColorScheme := SetColorScheme;
        focus          := IgnoreFocus;
        modified       := IgnoreModification;
        filter         := NoOpFilter
      END;

TYPE
  CommandKind = {VertCommand, OtherCommand};

VAR
  selectionModes: ARRAY [VBT.Modifier.MouseL .. VBT.Modifier.MouseM],
                    [0 .. 4] OF
                    VText.SelectionMode;
  standardStyle, specialStyle: ARRAY SelectionType OF VText.IntervalStyle;
  SwapHighlights := VBT.GetMiscCodeType ("SwapHighlights");

PROCEDURE Init (v             : T;
                singleLine                          := FALSE;
                hMargin                             := 1.5;
                vMargin                             := 1.5;
                font                                := Font.BuiltIn;
                colorScheme   : PaintOp.ColorScheme := NIL;
                expandOnDemand                      := FALSE;
                wrap                                := TRUE;
                readOnly                            := FALSE;
                turnMargin                          := 2.0           ): T =
  BEGIN
    TRY
      IF colorScheme = NIL THEN colorScheme := PaintOp.bgFg END;
      VAR
        vFont := VText.MakeVFont (
                   font := font, printable := (Char.All - Char.Controls)
                                                + SET OF CHAR {'\t'},
                   whiteTabs := TRUE);
        vOptions := VText.MakeVOptions (
                      vFont := vFont, leftMargin := hMargin,
                      rightMargin := hMargin, turnMargin := turnMargin,
                      topMargin := vMargin, leading := 0.0,
                      whiteBlack := colorScheme, whiteStroke := colorScheme,
                      leftOffset := 0.0, wrap := wrap AND NOT singleLine,
                      eob := FALSE, intervalStylePrecedence := NIL);
        iOptions := VText.MakeIntervalOptions (
                      style := standardStyle [SelectionType.Primary],
                      whiteBlack := colorScheme, whiteStroke := colorScheme,
                      leading := colorScheme.bg);
      BEGIN
        v.font := font;
        v.colors := colorScheme;
        v.mu := NEW (MUTEX);
        v.vtext :=
          VText.New (MText.New ("", 256), v, VBT.Domain (v), vOptions);
        FOR sel := FIRST (SelectionType) TO LAST (SelectionType) DO
          v.selection [sel].owned := FALSE;
          v.selection [sel].interval :=
            VText.CreateInterval (
              vtext := v.vtext, indexL := 0, indexR := 0, options := iOptions)
        END
      END;

      v.singleLine := singleLine;
      v.readOnly := readOnly;
      v.modifiedP := FALSE;
      v.hasFocus := FALSE;
      v.swapping := FALSE;
      v.visible := TRUE;
      v.location.margin [Axis.T.Hor] := hMargin;
      v.location.margin [Axis.T.Ver] := vMargin;
      v.location.align [Axis.T.Hor] := 0.0;
      v.location.align [Axis.T.Ver] := 0.0;
      v.typeinStart := 0;
      v.pattern := "";
      v.expandOnDemand := expandOnDemand AND NOT singleLine;
      v.cur := NEW (UndoRec);
      RETURN v
    EXCEPT
    | VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted => RETURN NIL
    END
  END Init;

PROCEDURE SetScrollBar (v: T; scrollBar: Scroller)  =
  BEGIN
    v.scrollBar := scrollBar;
    scrollBar.textport := v
  END SetScrollBar;


(***************************  Client Interface  ***************************)

PROCEDURE SetReadOnly (v: T; flag: BOOLEAN) =
  BEGIN
    v.readOnly := flag
  END SetReadOnly;

PROCEDURE SetWrap (v: T; wrap: BOOLEAN) =
  BEGIN
    LOCK v.mu DO
      IF v.vtext.vOptions.wrap # wrap THEN
        v.vtext.vOptions.wrap := wrap;
        TRY
          VText.ChangeVOptions (v.vtext, v.vtext.vOptions)
        EXCEPT
        | VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted =>
        END;
        VBT.Mark (v)
      END
    END
  END SetWrap;

PROCEDURE Length (v: T): CARDINAL =
  BEGIN
    RETURN MText.Length (v.vtext.mtext)
  END Length;
  
PROCEDURE GetText (v    : T;
                   begin: CARDINAL := 0;
                   end  : CARDINAL := LAST (CARDINAL)): TEXT =
  <* LL= VBT.mu *>
  BEGIN
    RETURN MText.GetText (v.vtext.mtext, begin, end)
  END GetText;

PROCEDURE SetText (v: T; t: TEXT) =
  <* LL <= VBT.mu *>
  BEGIN
    LOCK v.mu DO
      ReplaceInVText (v, 0, LAST (CARDINAL), t);
      TRY
        VText.SetStart (v.vtext, 0, 0)
      EXCEPT
      | VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted =>
      END;
      (* VBT.NewShape (v); *)
      MarkAndUpdate (v)
    END
  END SetText;

PROCEDURE PutText (v: T; t: TEXT) =
  <* LL <= VBT.mu *>
  BEGIN
    LOCK v.mu DO
      ReplaceInVText (v, LAST (CARDINAL), LAST (CARDINAL), t);
      (* VBT.NewShape (v); *)
      MarkAndUpdate (v)
    END
  END PutText;

PROCEDURE GetFont (v: T): Font.T  =
  BEGIN
    LOCK v.mu DO
      RETURN v.font
    END
  END GetFont; 

PROCEDURE SetFont (v: T; font: Font.T) =
  (* By the book, we should call ExplodeVText, ExplodeVOptions, ExplodeVFont,
     MakeVFont, and MakeVOptions before calling ChangeVOptions, but we cheat
     by looking at the implementation and consing only a new VFont. *)
  VAR
    vtext   : VText.T;
    vOptions: VText.VOptions;
    vFont   : VText.VFont;
  BEGIN
    LOCK v.mu DO
      IF font = v.font THEN RETURN END;
      vtext := v.vtext;
      vOptions := vtext.vOptions;
      vFont := vOptions.vFontxxx;
      TRY
        vOptions.vFontxxx :=
          VText.MakeVFont (font := font, printable := vFont.vFont.printable,
                           whiteTabs := vFont.vFont.whiteTabs);
        v.font := font;         (* For convenience only *)
        VText.ChangeVOptions (vtext, vOptions)
      EXCEPT
      | VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted =>
      END;
      SetFontDimensions (v);
      VBT.NewShape (v);
      MarkAndUpdate (v)
    END
  END SetFont;
      
PROCEDURE GetColorScheme (v: T): PaintOp.ColorScheme  =
  BEGIN
    LOCK v.mu DO
      RETURN v.vtext.vOptions.whiteBlack (* one of several choices *)
    END
  END GetColorScheme;

PROCEDURE SetColorScheme (v: T; colorScheme: PaintOp.ColorScheme) =
  VAR vOptions: VText.VOptions;
  BEGIN
    LOCK v.mu DO
      vOptions := v.vtext.vOptions;
      IF v.colors = colorScheme THEN RETURN END;
      vOptions.whiteBlack := colorScheme;
      vOptions.whiteStroke := colorScheme;
      TRY
        VText.ChangeVOptions (v.vtext, vOptions)
      EXCEPT
      |  VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted =>
      END;
      FOR sel := FIRST (SelectionType) TO LAST (SelectionType) DO
        WITH options = v.selection [sel].interval.options DO
          options.whiteBlack := colorScheme;
          options.whiteStroke := colorScheme;
          options.leading := colorScheme.bg
        END
      END;
      v.colors := colorScheme;
      VBT.Mark (v)
    END
  END SetColorScheme;

PROCEDURE SetFontDimensions (v: T) =
  <* LL = v.mu *>
  BEGIN
    (* metrics := FontClass.FontMetrics(vbt, v.font); *)
    WITH st = VBT.ScreenTypeOf (v) DO
      IF st # NIL THEN
        WITH bounds = Palette.ResolveFont (
                        st, v.font).metrics.maxBounds,
             box = bounds.boundingBox DO
          v.fontHeight := Rect.VerSize (box);
          v.charWidth := bounds.printWidth
          (* not "Rect.HorSize (box)", alas *)
        END
      END
    END
  END SetFontDimensions;

PROCEDURE Width (v: T): CARDINAL =
  BEGIN
    WITH n = v.shape (Axis.T.Hor, 0).pref DO
      LOCK v.mu DO
        IF v.charWidth = 0 THEN
          RETURN 0
        ELSE
          RETURN n DIV v.charWidth
        END
      END
    END
  END Width;
      
PROCEDURE VertSize (v: T): CARDINAL =
  BEGIN
    RETURN v.shape (Axis.T.Ver, 0).pref
  END VertSize;

PROCEDURE TryFocus (v: T; t: VBT.TimeStamp): BOOLEAN =
  BEGIN
    (* Force all pending redisplays: *)
    VBTRep.Redisplay ();
    IF NOT Rect.IsEmpty (VBT.Domain (v)) AND GetKFocus (v, t)
         AND TakeSelection (v, t, SelectionType.Primary) THEN
      VBT.Mark (v);
      RETURN TRUE
    ELSE
      RETURN FALSE
    END
  END TryFocus;

PROCEDURE GetKFocus (v: T; t: VBT.TimeStamp): BOOLEAN =
  BEGIN
    IF NOT v.hasFocus THEN
      v.focus (TRUE, t);
      LOCK v.mu DO
        TRY
          VBT.Acquire (v, VBT.KBFocus, t);
          VText.SwitchCaret (v.vtext, VText.OnOffState.On);
          v.hasFocus := TRUE;
          v.focusTime := t
        EXCEPT
        | VBT.Error, VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted =>
        END
      END
    END;
    RETURN v.hasFocus
  END GetKFocus;

PROCEDURE TakeSelection (v: T; t: VBT.TimeStamp; sel: SelectionType):
  BOOLEAN =
  VAR s: VBT.Selection;
  BEGIN
    LOCK v.mu DO
      WITH r = v.selection [sel] DO (* writable designator *)
        IF NOT r.owned THEN
          IF sel = SelectionType.Primary THEN
            s := VBT.Target
          ELSE
            s := VBT.Source
          END;
          TRY VBT.Acquire (v, s, t) EXCEPT VBT.Error => RETURN FALSE END;
          r.owned := TRUE;
          TRY
            VText.SwitchInterval (r.interval, VText.OnOffState.On)
          EXCEPT
          | VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted =>
          END
        END
      END
    END;
    RETURN TRUE
  END TakeSelection;

PROCEDURE HasFocus (v: T): BOOLEAN  =
  BEGIN
    RETURN v.hasFocus
  END HasFocus;

PROCEDURE Select (v          : T;
                  time       : VBT.TimeStamp;
                  begin, end : CARDINAL;
                  sel        : SelectionType   := SelectionType.Primary;
                  replaceMode: BOOLEAN         := FALSE;
                  caretEnd   : WhichEnd        := WhichEnd.Right         ) =
  BEGIN
    TRY
      IF NOT TakeSelection (v, time, sel)
           OR sel = SelectionType.Primary AND NOT v.readOnly
                AND NOT GetKFocus (v, time) THEN
        RETURN
      END;
      WITH length   = MText.Length (v.vtext.mtext),
           end      = MIN (end, length),
           begin    = MIN (begin, end),
           z        = v.selection [sel],
           interval = z.interval,
           options  = interval.options              DO
        z.mode := VText.SelectionMode.CharSelection;
        IF sel = SelectionType.Primary AND replaceMode AND NOT v.readOnly THEN
          options.style := specialStyle [sel]
        ELSE
          options.style := standardStyle [sel]
        END;
        VText.MoveInterval (interval, begin, end);
        IF sel = SelectionType.Primary THEN
          IF caretEnd = WhichEnd.Right THEN
            VText.MoveCaret (v.vtext, end)
          ELSE
            VText.MoveCaret (v.vtext, begin)
          END
        END
      END;
    EXCEPT
    | VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted =>
    END;
    VBT.Mark (v)
  END Select;

PROCEDURE Unselect (v: T; <* UNUSED *> t: VBT.TimeStamp) =
  BEGIN
    TRY
      VText.SwitchCaret (v.vtext, VText.OnOffState.Off);
      VText.SwitchInterval (
        v.selection [SelectionType.Primary].interval, VText.OnOffState.Off);
      VBT.Mark (v)
    EXCEPT
    | VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted =>
    END
  END Unselect;

PROCEDURE HasSelection (v: T; sel := SelectionType.Primary):
  BOOLEAN = <* LL.sup = VBT.mu *>
  BEGIN
    LOCK v.mu DO RETURN v.selection [sel].owned END
  END HasSelection;

PROCEDURE Selection (              v         : T;
                     VAR (* out *) begin, end: VText.Index;
                     sel := SelectionType.Primary) =
  BEGIN
    LOCK v.mu DO
      WITH z = v.selection [sel].interval DO begin := z.l; end := z.r END
    END
  END Selection;
  
PROCEDURE CaretPosition (v: T): CARDINAL =
  BEGIN
    TRY RETURN VText.CaretIndex (v.vtext) EXCEPT | VTDef.Error => RETURN 0 END
  END CaretPosition;

PROCEDURE StandardInsert (v: T; text: TEXT) =
  BEGIN
    Insert (v, text)
  END StandardInsert;

PROCEDURE IsVisible (v: T; pos: CARDINAL): BOOLEAN =
  BEGIN
    TRY
      RETURN VText.InRegion (v.vtext, 0, pos)
    EXCEPT
    | VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted => RETURN FALSE
    END
  END IsVisible;

PROCEDURE IsModified (v: T): BOOLEAN =
  BEGIN
    RETURN v.modifiedP
  END IsModified;

PROCEDURE SetModified (v: T; modified: BOOLEAN) =
  BEGIN
    v.modifiedP := modified
  END SetModified;

PROCEDURE GetVText (v: T): VText.T =
  BEGIN
    RETURN v.vtext
  END GetVText;

(*********************  Events and User Interface  **********************)

PROCEDURE Misc (v: T; READONLY cd: VBT.MiscRec) =
  VAR
    sel           : SelectionType;
    indexL, indexR: VText.Index;
    type                          := cd.type;
    s                             := cd.selection;
  BEGIN
    TRY
      IF type = SwapHighlights THEN
        v.swapping := TRUE
      ELSIF type = VBT.Lost THEN
        (* We try for a model in which KBFocus and Target (Primary) always go
           together, but we have to keep track of them separately because
           other programs may not adhere to that model. *)
         (* Actually, we don't do that any more, so that we can use
            "mouse accelerators", where you select text in window A and
            then middle-click on the Find button, for example, in window B,
            which starts a search for the selected text. If we re-acquire
            VBT.Target, then we'll lose the text we want to search for. *)
        IF s = VBT.KBFocus THEN
          v.hasFocus := FALSE;
          VText.SwitchCaret (v.vtext, VText.OnOffState.Off);
          v.focus (FALSE, cd.time)
        ELSIF s = VBT.Target THEN
          sel := SelectionType.Primary
        ELSIF s = VBT.Source THEN
          sel := SelectionType.Secondary;
          IF v.swapping THEN Selection (v, indexL, indexR, sel) END
        ELSE
          RETURN
        END;
        IF s # VBT.KBFocus THEN
          WITH z = v.selection [sel] DO
            VText.SwitchInterval (z.interval, VText.OnOffState.Off);
            z.owned := FALSE
          END
        END;
        IF v.swapping THEN
          (* After losing the secondary, grab primary! *)
          IF TryFocus (v, cd.time) THEN
            VText.MoveInterval (
              v.selection [SelectionType.Primary].interval, indexL, indexR);
            VText.MoveCaret (v.vtext, indexR)
          END;
          v.swapping := FALSE
        END;
        VBT.Mark (v)
      ELSIF type = VBT.TakeSelection AND s = VBT.KBFocus THEN
        (* EVAL TryFocus (v, cd.time) *)
        EVAL GetKFocus (v, cd.time)
      ELSE                      (* ignore other codes *)
      END
    EXCEPT
    | VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted =>
    END
  END Misc;


PROCEDURE Mouse (v: T; READONLY cd: VBT.MouseRec) =
  <* LL.sup = VBT.mu *>
  (* React to a mouse-click.  Essentially, the click is located (using
     VText.Pounce) and the selected piece of text either becomes the new
     selection or is added to the existing one. *)
  CONST
    Sel = ARRAY BOOLEAN OF
            SelectionType {SelectionType.Primary, SelectionType.Secondary};
  VAR previouslyOwned: BOOLEAN;
  BEGIN
    WITH sel      = Sel [VBT.Modifier.Control IN cd.modifiers],
         rec      = v.selection [sel],
         interval = rec.interval,
         options  = interval.options                            DO

      (* Filter out gone or up transitions *)
      IF cd.clickType = VBT.ClickType.LastUp THEN
        v.dragging := FALSE;
        RETURN
      END;
      IF cd.clickType # VBT.ClickType.FirstDown THEN RETURN END;

      (* Do we won the selection? *)
      previouslyOwned := rec.owned;
      (* Get the selection (and focus if primary-selecting) *)
      IF NOT TakeSelection (v, cd.time, sel)
           OR sel = SelectionType.Primary AND NOT GetKFocus (v, cd.time) THEN
        RETURN
      END;

      (* If we right-click on a window that did NOT own the primary before the
         call to TakeSelection, then restore its primary-selection highlight,
         and return. *)
      (* [This code used to test whether we had the keyboard focus, but now we
         always get the keyboard focus when the mouse moves back in the
         window.  See the reference to VBT.TakeSelection in Misc.] *)
      TRY
        IF sel = SelectionType.Primary AND NOT previouslyOwned
             AND cd.whatChanged = VBT.Modifier.MouseR THEN
          VText.SwitchInterval (interval, VText.OnOffState.On);
          VBT.Mark (v);
          RETURN
        END;

        (* Translate the click to a selection mode and highlight style. *)
        IF cd.whatChanged = VBT.Modifier.MouseL
             OR cd.whatChanged = VBT.Modifier.MouseM THEN
          rec.mode := selectionModes [cd.whatChanged, MIN (cd.clickCount, 4)];
          options.style := standardStyle [sel];
          VText.ChangeIntervalOptions (interval, options)
        ELSIF cd.clickCount < 2 THEN (* a single-click *)
          (* MouseR: use pending-delete style if possible *)
          (* TYPESCRIPT SPECIAL TEST *)
          IF interval.l >= v.typeinStart AND NOT v.readOnly THEN
            options.style := specialStyle [sel];
            VText.ChangeIntervalOptions (interval, options)
          END;
          (* The current interval-bounds become the pivots. *)
          v.pivotL := interval.l;
          v.pivotR := interval.r
        ELSIF rec.mode > FIRST (VText.SelectionMode) THEN
          (* multi-clicking: make the selection-mode smaller. *)
          DEC (rec.mode)
        END
      EXCEPT
      | VTDef.Error =>
      END;

      (* Do the selection; side-effect is to set up mouse cage to track
         drags *)
      v.dragButton := cd.whatChanged;
      v.dragSel := sel;
      MakeSelection (v, cd.cp);
      v.dragging := TRUE
    END
  END Mouse;

TYPE IRange = RECORD left, middle, right: CARDINAL END;

PROCEDURE MakeSelection (v: T; cp: VBT.CursorPosition) =
  VAR
    sel  := v.dragSel;
    mode := v.selection [sel].mode;
    r    := GetRange (v.vtext, cp, mode);
  PROCEDURE h (left, middle, right: CARDINAL) =
    BEGIN
      TRY
        IF sel = SelectionType.Primary THEN
          VText.MoveCaret (v.vtext, middle)
        END;
        VText.MoveInterval (v.selection [sel].interval, left, right)
      EXCEPT
      | VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted =>
      END;
      VBT.Mark (v)
    END h;
  BEGIN
    IF mode = VText.SelectionMode.CharSelection
         AND (v.dragButton # VBT.Modifier.MouseR OR v.dragging) THEN
      (* Note that dragging is FALSE for initial right-down.  This makes
         CharSelection extension work better, and same as Ivy *)
      r.left := r.middle;
      r.right := r.middle
    END;
    IF v.dragButton # VBT.Modifier.MouseR THEN
      h (r.left, r.middle, r.right)
    ELSIF NOT v.dragging THEN
      (* extend: move one end, keeping the other fixed *)
      IF r.left < v.pivotL THEN
        v.fixed := MAX (r.right, v.pivotR);
        h (r.left, r.left, v.fixed)
      ELSE
        v.fixed := MIN (r.left, v.pivotL);
        h (v.fixed, r.right, r.right)
      END
    ELSE
      IF r.left < v.fixed THEN
        h (r.left, r.left, v.fixed)
      ELSE
        h (v.fixed, r.right, r.right)
      END
    END
  END MakeSelection;

PROCEDURE Position (v: T; READONLY cd: VBT.PositionRec) =
  BEGIN
    IF NOT v.dragging THEN (* skip *)
    ELSIF cd.cp.gone THEN
      VBT.SetCage (v, VBT.GoneCage)
    ELSE
      MakeSelection (v, cd.cp)
    END
  END Position;

PROCEDURE GetRange (vt  : VText.T;
                    cp  : VBT.CursorPosition;
                    mode: VText.SelectionMode ): IRange =
  VAR
    whichEnd  : VText.WhichEnd;
    rect      : Rect.T;
    lineNum   : CARDINAL;
    ch        : CHAR;
    atEnd     : BOOLEAN;
    lt, md, rt: CARDINAL;
    e         : MTextUnit.Extent;
  BEGIN
    TRY
      VText.PounceLocate (vt, 0, cp.pt, lt, rt, lineNum, ch);
      atEnd := lt = rt;
      IF atEnd AND lt > 0 THEN DEC (lt) END;
      CASE mode OF
      | VText.SelectionMode.ParagraphSelection =>
          (* paragraph strategy differs from VText's strategy *)
          e := MTextUnit.ParagraphExtent (vt.mtext, lt);
          lt := e.left;
          rt := e.right
      | VText.SelectionMode.LineSelection =>
          e := MTextUnit.LineExtent (vt.mtext, lt);
          lt := e.left;
          rt := e.right
      ELSE
        VText.PounceExtend (vt, 0, lt, rt, lineNum, ch, mode)
      END;
      whichEnd := VText.PounceEncage (vt, 0, cp.pt, lt, md, rt, rect);
      VBT.SetCage (vt.vbt, VBT.CageFromRect (rect, cp));
      IF (mode = VText.SelectionMode.CharSelection
            OR mode = VText.SelectionMode.WordSelection) AND ch # Char.NL
           AND (whichEnd = WhichEnd.Right OR atEnd) THEN
        md := rt
      ELSE
        md := lt
      END
    EXCEPT
    | VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted =>
    END;
    RETURN IRange {lt, md, rt}
  END GetRange;

(***************************  Replace  ****************************)

PROCEDURE IsReplaceMode (v: T): BOOLEAN =
  BEGIN
    RETURN v.selection [SelectionType.Primary].interval.options.style
             = specialStyle [SelectionType.Primary]
  END IsReplaceMode;

PROCEDURE Replace (v: T; begin, end: CARDINAL; newText: TEXT)  =
  BEGIN
    ReplaceInVText(v, begin, end, newText);
    (* VBT.NewShape (v); *)
    MarkAndUpdate (v)
  END Replace;

PROCEDURE Replace1 (v: T; t: TEXT; sel: SelectionType) =
  <* LL.sup <= VBT.mu *>
  VAR begin, end: CARDINAL;
  BEGIN
    LOCK v.mu DO
      WITH intv = v.selection [sel].interval DO
        begin := intv.l;
        ReplaceInVText (v, intv.l, intv.r, t);
        intv.options.style := standardStyle [sel];
        end := begin + Text.Length (t);
        TRY
          VText.MoveInterval (intv, begin, end)
        EXCEPT
        | VTDef.Error, Rd.EndOfFile, Rd.Failure,
              Thread.Alerted =>
        END;
        (* VBT.NewShape (v); *)
        MarkAndUpdate (v)
      END
    END
  END Replace1;

PROCEDURE ReplaceInVText (v         : T;
                          begin, end: CARDINAL;
                          newText   : TEXT      ) =
  <* LL = v.mu *>
  BEGIN
    end := MIN (end, MText.Length (v.vtext.mtext));
    begin := MIN (begin, end);
    AddToUndo (v, begin, end, newText);
    TRY
      VText.Replace (v.vtext, begin, end, newText)
    EXCEPT
    | VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted =>
    END;
    IF NOT v.modifiedP THEN
      v.modifiedP := TRUE;
      v.modified ()
    END
  END ReplaceInVText;

PROCEDURE AddToUndo (v: T; begin, end: CARDINAL; newText: TEXT) =
  <* LL = v.mu *>
  VAR
    n := Text.Length (newText);
    r := v.cur;
  BEGIN
    IF v.readOnly OR begin = end AND n = 0 THEN RETURN END;
    IF r.prev # NIL AND begin = end AND n = 1 AND r.prev.end = begin
         AND Text.GetChar (newText, 0) IN Char.Graphics THEN
      (* It's straight typing.  Extend the previous record. *)
      INC (r.prev.end)
    ELSE
      r.begin := begin;
      r.end := begin + n;
      r.text := MText.GetText (v.vtext.mtext, begin, end);
      IF r.next = NIL THEN r.next := NEW (UndoRec, prev := r) END;
      v.cur := r.next
    END;
    TraceUndo (v)
  END AddToUndo;

VAR tracingUndo := FALSE;

PROCEDURE TraceUndo (v: T) =
  <* LL = v.mu *>
  VAR
    r       := v.cur;
    t: TEXT;
  BEGIN
    IF NOT tracingUndo THEN RETURN END;
    WHILE r.prev # NIL DO r := r.prev END;
    REPEAT
      t := r.text;
      IF Text.Length (t) > 20 THEN t := Text.Sub (t, 0, 20) & "..." END;
      SmallIO.PutText (
        SmallIO.stderr,
        Fmt.F ("%s -> %s: \"%s\"\n", Fmt.Int (r.begin), Fmt.Int (r.end), t));
      r := r.next
    UNTIL r = NIL;
    SmallIO.PutText (SmallIO.stderr, "-------------------\n")
  END TraceUndo;
  
PROCEDURE Insert (v: T; t: TEXT; isBackspace: BOOLEAN := FALSE) =
  VAR begin, end: CARDINAL;
  BEGIN
    IF IsReplaceMode (v) THEN
      Replace1 (v, t, SelectionType.Primary)
    ELSE
      TRY
        LOCK v.mu DO
          begin := VText.CaretIndex (v.vtext);
          (* TYPESCRIPT SPECIAL *)
          IF begin < v.typeinStart THEN
            begin := MText.Length (v.vtext.mtext);
            VText.MoveCaret (v.vtext, begin)
          END;
          end := begin;
          IF isBackspace THEN
            IF begin <= v.typeinStart THEN RETURN END;
            DEC (begin)
          END;
          ReplaceInVText (v, begin, end, t);
          begin := VText.CaretIndex (v.vtext);
          VText.MoveInterval (
            v.selection [SelectionType.Primary].interval,
            begin, begin);
          MarkAndUpdate (v)
        END
      EXCEPT
      | VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted =>
      END
    END
  END Insert;

PROCEDURE InsertLocked (v: T; t: TEXT) =
  (* InsertLocked is a simplified form of Insert, to be called
     only with v.mu already held. *)
  VAR begin, end: CARDINAL;
  BEGIN
    TRY
      begin := VText.CaretIndex (v.vtext);
      (* TYPESCRIPT SPECIAL *)
      IF begin < v.typeinStart THEN
        begin := MText.Length (v.vtext.mtext)
      END;
      end := begin;
      ReplaceInVText (v, begin, end, t);
      begin := VText.CaretIndex (v.vtext);
      VText.MoveInterval (
        v.selection [SelectionType.Primary].interval, begin,
        begin);
    EXCEPT
    | VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted =>
    END;
    MarkAndUpdate (v)
  END InsertLocked;

(************************  Shape of current text  *************************)

PROCEDURE Shape (v: T; ax: Axis.T; n: CARDINAL):
  VBT.SizeRange =
  <* LL = VBT.mu.v *>
  VAR
    extraHeight, lines: CARDINAL;
    pref              : INTEGER;
  BEGIN
    IF VBT.ScreenTypeOf (v) = NIL THEN
      RETURN VBT.DefaultShape
    END;
    IF v.fontHeight = 0 THEN    (* ScreenType just became
                                   non-NIL. *)
      LOCK v.mu DO
        SetFontDimensions (v)   (* Sets v.fontHeight and
                                   v.charWidth *)
      END
    END;
    IF ax = Axis.T.Hor THEN
      pref := v.lastNonEmptyWidth;
      IF pref = 0 THEN
        LOCK v.mu DO pref := 30 * v.charWidth END
      END;
      RETURN VBT.SizeRange {0, pref, 99999}
    END;
    (* ax = Axis.T.Ver *)
    IF n = 0 THEN n := v.shape (Axis.T.Hor, 0).pref END;
    LOCK v.mu DO
      extraHeight :=
        2 * Pts.ToScreenPixels (
          v, v.location.margin [Axis.T.Ver], Axis.T.Ver);
      IF v.singleLine THEN
        pref := v.fontHeight + extraHeight
      ELSE
        (* How many lines would it take to display the whole
           vtext?  Make sure there is room for at least one
           line (vertically) or vtext gets very confused.
           width (res.pref) = 0 => vtext has not been reshaped
           yet *)
        IF n = 0 THEN
          lines := 1
        ELSE
          TRY
            lines := 1 + VText.LinesBetween (
                           v.vtext, 0, LAST (CARDINAL),
                           LAST (CARDINAL),
                           (* fudge n appropriately *)
                           n - (v.vtext.leftMargin
                                  + v.vtext.rightMargin
                                  + 2 * v.vtext.turnMargin))
          EXCEPT
          | VTDef.Error, Rd.EndOfFile, Rd.Failure,
                Thread.Alerted =>
              lines := 1
          END
        END;
        (* How many pixels is that? *)
        pref := lines * v.vtext.lineSpacing + v.vtext.topMargin
                  + extraHeight
      END
    END;
    RETURN VBT.SizeRange {pref, pref, pref + 1}
  END Shape;

PROCEDURE Reshape (v: T; READONLY cd: VBT.ReshapeRec) =
  VAR
    newRect                                 := cd.new;
    dividers: ARRAY [0 .. 0] OF VText.Coord;
  BEGIN
    IF newRect = cd.prev AND NOT cd.marked
         OR Rect.IsEmpty (newRect) THEN
      RETURN
    END;
    IF Rect.HorSize (newRect) # v.lastNonEmptyWidth THEN
      v.lastNonEmptyWidth := Rect.HorSize (newRect);
      VBT.NewShape (v)
    END;
    LOCK v.mu DO
      IF NOT v.vtext.vOptions.wrap THEN
        newRect.east := LAST (INTEGER) DIV 2
      END;
      TRY
        VText.Move (v.vtext, newRect, cd.saved, dividers);
        VText.Update (v.vtext);
        v.linesShown :=
          1 + VText.WhichLine (v.vtext, 0, cd.new.south);
        (* if it will all fit, normalize to fit *)
        IF Rect.IsEmpty (cd.prev) AND NOT Rect.IsEmpty (cd.new)
             AND VText.LinesBetween (
                   v.vtext, 0, MText.Length (v.vtext.mtext),
                   v.linesShown) < v.linesShown THEN
          Normalize (v, 0)      (* Normalize calls
                                   MarkAndUpdate *)
        ELSE
          MarkAndUpdate (v)
        END
      EXCEPT
      | VTDef.Error, Rd.EndOfFile, Rd.Failure,
            Thread.Alerted =>
      END
    END
  END Reshape;

PROCEDURE ShapeInfo (v: T; VAR lineCount, lineLength: INTEGER) =
  VAR
    e      := MTextUnit.Extent {0, 0, TRUE};
    length := MText.Length (v.vtext.mtext);
  BEGIN
    lineCount := 0;
    lineLength := 0;
    IF length = 0 THEN RETURN END;
    WHILE e.right < length DO
      e := MTextUnit.LineExtent (v.vtext.mtext, e.right);
      INC (lineCount);
      lineLength := MAX (lineLength, e.right - e.left - 1)
    END;
    (* adjust for last line: if ends with \n, increment
       lineCount; otherwise, len of last line is right-left, not
       right-left-1. *)
    IF MText.GetChar (v.vtext.mtext, length - 1) = '\n' THEN
      INC (lineCount);
      lineLength := MAX (lineLength, e.right - e.left - 1)
    ELSE
      lineLength := MAX (lineLength, e.right - e.left)
    END
  END ShapeInfo;

PROCEDURE Key (v: T; READONLY cd: VBT.KeyRec) =
  BEGIN
    (* "cd" must be a VALUE parameter in KeyCode1 so that a filter proc can
       change it. *)
    IF NOT v.expandOnDemand THEN
      KeyCode1 (v, cd)
    ELSE
      WITH oldVsizeRange = Shape (v, Axis.T.Ver, 0) DO
        KeyCode1 (v, cd);
        IF Shape (v, Axis.T.Ver, 0) # oldVsizeRange THEN
          (* Scroll back to the top, so we can see the whole text. *)
          Normalize (v, 0);
          VBT.NewShape (v)
        END                     (* IF *)
      END                       (* WITH *)
    END                         (* IF *)
  END Key;
  
PROCEDURE KeyCode1 (v: T; VALUE cd: VBT.KeyRec) =
  VAR ch: CHAR;
  BEGIN
    IF NOT cd.wentDown OR NOT v.hasFocus OR Rect.IsEmpty (VBT.Domain (v)) THEN
      RETURN
    END;
    v.filter (cd);
    IF cd.whatChanged = VBT.NoKey THEN RETURN END;
    ch := KeyTrans.Latin1 (cd.whatChanged);
    IF VBT.Modifier.Control IN cd.modifiers THEN
      ControlChord (v, ch, cd)
    ELSIF VBT.Modifier.Option IN cd.modifiers THEN
      OptionChord (v, ch, cd)
    ELSIF KeyboardKey.Left <= cd.whatChanged
            AND cd.whatChanged <= KeyboardKey.Down THEN
      (* Map cursor keys (Left, Up, Right, Down) into control keys. *)
      ControlChord (v, ARRAY OF
                         CHAR {'j', 'o', 'k', 'p'} [
                         cd.whatChanged - KeyboardKey.Left], cd)
    ELSIF v.readOnly THEN
      RETURN
    ELSIF ch = Backspace OR ch = Del THEN
      Insert (v, "", TRUE);
      Normalize (v);
      (* If the primary selection was highlit, it is now empty! *)
      v.selection [SelectionType.Primary].interval.options.style :=
        standardStyle [SelectionType.Primary];
      v.lastCmdKind := CommandKind.OtherCommand
    ELSIF ch = Tab THEN
      v.tabAction (cd);
      v.lastCmdKind := CommandKind.OtherCommand
    ELSIF ch = Return THEN
      IF VBT.Modifier.Shift IN cd.modifiers THEN
        Newline (v, cd)
      ELSE
        v.returnAction (cd)
      END;
      v.lastCmdKind := CommandKind.OtherCommand;
    ELSIF ch IN Char.Graphics OR ch = Space THEN (* real typing *)
      Insert (v, Text.FromChar (ch));
      Normalize (v);
      v.lastCmdKind := CommandKind.OtherCommand
    ELSE
      (* including NullKey, for untranslatable keys *)
      v.defaultAction (cd);
      v.lastCmdKind := CommandKind.OtherCommand
    END
  END KeyCode1;

PROCEDURE ControlChord (v: T; ch: CHAR; READONLY cd: VBT.KeyRec) =
  VAR
    index, length : CARDINAL;
    indexL, indexR: VText.Index;
    left, right   : INTEGER;
    e             : MTextUnit.Extent;
  VAR
    time  := cd.time;
    vtext := v.vtext;
  BEGIN
    TRY
      v.lastCmdKind := CommandKind.OtherCommand;
      CASE ch OF
      | 'i' => EditCmd.ToNextWord (v, time)
      | 'u' => EditCmd.ToPrevWord (v, time)
      | 'f' => EditCmd.DeleteToEndOfWord (v, time)
      | 'd' => EditCmd.DeleteToStartOfWord (v, time)
      | 'q' => EditCmd.Clear (v, time)
      | 'w' => EditCmd.Paste (v, time)
      | 'e' => EditCmd.Move (v, time)
      | 'r' => EditCmd.Swap (v, time)
      | 'z' => EditCmd.Undo (v) (* control-Z *)
      | 'Z' => EditCmd.UndoUndo (v) (* control-shift-Z *)
      | 'h' =>                  (* Exchange the primary and
                                   secondary. *)
          WITH primary = v.selection [SelectionType.Primary],
               intvl_1 = primary.interval,
               secondary = v.selection [
                             SelectionType.Secondary],
               intvl_2 = secondary.interval DO
            index := VText.CaretIndex (vtext);
            (* This VBT owns the primary.  Does it own both? *)
            IF secondary.owned THEN
              IF index = intvl_1.l THEN
                index := intvl_2.l
              ELSE
                index := intvl_2.r
              END;
              VText.MoveInterval (
                intvl_1, intvl_2.l, intvl_2.r);
              VText.MoveCaret (vtext, index)
            ELSE
              (* This is more complex.  Must tell the owner of
                 the secondary that this is going to be a swap,
                 and then grab the secondary. *)
              TRY
                VBT.Put (v, VBT.Source, time, SwapHighlights);
                VText.SwitchInterval (
                  intvl_1, VText.OnOffState.Off);
                IF NOT TakeSelection (
                         v, time, SelectionType.Secondary) THEN
                  RETURN
                END
              EXCEPT            (* fail due to no secondary
                                   selection *)
              | VBT.Error => RETURN
              END
            END;
            (* Now, move the secondary selection to what used
               to be the primary interval. *)
            VText.MoveInterval (intvl_2, intvl_1.l, intvl_1.r)
          END;
          v.lastCmdKind := CommandKind.OtherCommand

      | 'j', 'k', 'l', ';', 'y' =>
          (* Caret- and interval-twiddling *)
          length := MText.Length (vtext.mtext);
          index := VText.CaretIndex (vtext);
          Selection (v, indexL, indexR, SelectionType.Primary);
          CASE ch OF
          | 'j', 'k' =>
              IF ch = 'j' THEN
                IF index > 0 THEN DEC (index) END
              ELSE
                INC (index)
              END;
              indexL := index;
              indexR := index
          | 'l' =>
              VAR
                line := MTextUnit.LineInfo (vtext.mtext, index);
              BEGIN
                IF indexL = line.left AND indexR = line.right
                     AND index = line.left AND line.left > 0 THEN
                  line := MTextUnit.LineInfo (
                            vtext.mtext, line.left - 1)
                END;
                index := line.left;
                indexL := line.left;
                indexR := line.right
              END
          | ';' =>
              VAR
                line := MTextUnit.LineInfo (vtext.mtext, index);
              BEGIN
                IF indexL = line.left
                     AND indexR = line.rightEnd
                     AND index = line.rightEnd
                     AND line.rightEnd < length THEN
                  line := MTextUnit.LineInfo (
                            vtext.mtext, line.right)
                END;
                index := line.rightEnd;
                indexL := line.left;
                indexR := line.rightEnd
              END
          | 'y' =>
              IF index = indexL THEN
                index := indexR
              ELSE
                index := indexL
              END
          ELSE
          END;                  (* inner CASE *)
          WITH intv = v.selection [
                        SelectionType.Primary].interval DO
            intv.options.style :=
              standardStyle [SelectionType.Primary];
            VText.ChangeIntervalOptions (intv, intv.options);
            VText.MoveInterval (intv, indexL, indexR)
          END;
          VText.MoveCaret (vtext, index);
          v.lastCmdKind := CommandKind.OtherCommand

      | 'o', 'p' =>
          (* Vertical movement commands *)
          index := VText.CaretIndex (vtext);
          e := MTextUnit.LineExtent (vtext.mtext, index);
          left := e.left;
          right := e.right;
          IF v.lastCmdKind # CommandKind.VertCommand THEN
            v.wishCol := ColumnOf (vtext.mtext, left, index)
          END;
          v.lastCmdKind := CommandKind.VertCommand;
          IF ch = 'o' THEN
            IF left = 0 THEN RETURN END;
            e := MTextUnit.LineExtent (vtext.mtext, left - 1);
            left := e.left;
            right := e.right
          ELSE
            left := right
          END;
          index := ToColumn (vtext.mtext, left, v.wishCol);
          VText.MoveInterval (
            v.selection [SelectionType.Primary].interval,
            index, index);
          VText.MoveCaret (vtext, index)

      | 'a', 's', 'g', 'c', 'v', 'b' =>
          (* Deletion commands *)
          IF v.readOnly THEN
            RETURN
          END;                  (* no edits permitted *)
          length := MText.Length (vtext.mtext);
          index := VText.CaretIndex (vtext);
          CASE ch OF
          | 'a' =>              (* delete previous char *)
              IF index <= v.typeinStart THEN RETURN END;
              ReplaceInVText (v, index - 1, index, "");
              DEC (index)
          | 's' =>              (* delete next char *)
              IF index < v.typeinStart OR index >= length THEN
                RETURN
              END;
              ReplaceInVText (v, index, index + 1, "")
          | 'g' =>              (* delete current word *)
              IF index >= length OR index < v.typeinStart THEN
                RETURN
              END;
              WordAt (vtext.mtext, index, left, right);
              left := MAX (left, v.typeinStart);
              ReplaceInVText (v, left, right, "");
              index := left
          | 'c' =>              (* delete to bol, or current
                                   line if we're at bol *)
              left := MAX (
                        MTextUnit.StartOfLine (
                          vtext.mtext, index), v.typeinStart);
              IF index < left THEN (* Caret is in the read-only
                                      part of the line *)
                RETURN
              ELSIF index > left THEN
                ReplaceInVText (v, left, index, "");
                index := left
              ELSIF left > v.typeinStart THEN (* index = left =
                                                 bol *)
                ReplaceInVText (v, left - 1, left, "");
                DEC (index)
              ELSE
                RETURN
              END
          | 'v' =>
              (* delete to eol, including newline if we're at
                 eol *)
              VAR
                info := MTextUnit.LineInfo (vtext.mtext, index);
                end := info.rightEnd;
              BEGIN
                IF index = info.rightEnd THEN
                  end := info.right
                END;
                ReplaceInVText (
                  v, MAX (index, v.typeinStart), end, "")
              END
          | 'b' =>              (* delete current line *)
              WITH e = MTextUnit.LineExtent (
                         vtext.mtext, index) DO
                left := e.left;
                right := e.right;
                index := MAX (left, v.typeinStart);
                ReplaceInVText (v, index, right, "")
              END
          ELSE                  <* ASSERT FALSE *>
          END;                  (* inner CASE *)
          WITH interval = v.selection [
                            SelectionType.Primary].interval DO
            interval.options.style :=
              standardStyle [SelectionType.Primary];
            VText.ChangeIntervalOptions (
              interval, interval.options);
            VText.MoveInterval (interval, index, index)
          END;
          VText.MoveCaret (vtext, index);
          v.lastCmdKind := CommandKind.OtherCommand
      | 'm' =>
          Find (v, time, FindWhere.FindPrev, selected := FALSE)
      | 'n' =>
          Find (v, time, FindWhere.FindNext, selected := TRUE)
      | ',' =>
          Find (v, time, FindWhere.FindNext, selected := FALSE)
      | ' ' =>
        (* normalize *)
      ELSE
        v.defaultAction (cd);
        v.lastCmdKind := CommandKind.OtherCommand;
        (* don't normalize if unknown ctrl chord, including
           just ctrl itself: *)
        RETURN
      END;                      (* outer CASE *)
    EXCEPT
    | VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted =>
    END;
    Normalize (v)
  END ControlChord;

PROCEDURE OptionChord (v: T; ch: CHAR; READONLY cd: VBT.KeyRec) =
  BEGIN
    TRY
      v.lastCmdKind := CommandKind.OtherCommand;
      CASE ch OF
      | 'm' => Find (v, cd.time, FindWhere.FindPrev, TRUE)
      | 'n' => Find (v, cd.time, FindWhere.FindFromTop, TRUE)
      | 'c' => EditCmd.Copy (v, cd.time)
      | 'v' => EditCmd.Paste (v, cd.time)
      | 'x' => EditCmd.Cut (v, cd.time)
      | Backspace, Del =>
          (* Option-BS: transpose the two characters behind the
             caret *)
          IF v.readOnly THEN RETURN END;
          VAR
            two: ARRAY [0 .. 1] OF CHAR;
            indexR := VText.CaretIndex (v.vtext);
            indexL := indexR - 2;
            intvl := v.selection [
                       SelectionType.Primary].interval;
          BEGIN
            IF indexL >= v.typeinStart THEN
              two [1] := MText.GetChar (v.vtext.mtext, indexL);
              two [0] :=
                MText.GetChar (v.vtext.mtext, indexL + 1);
              VText.MoveInterval (intvl, indexL, indexR);
              intvl.options.style :=
                specialStyle [SelectionType.Primary];
              Insert (v, Text.FromChars (two))
            END
          END
      | Return =>
          IF v.readOnly THEN RETURN END;
          Insert (v, "\n");
          VAR
            intvl := v.selection [
                       SelectionType.Primary].interval;
            l := intvl.l - 1;
          BEGIN
            VText.MoveInterval (intvl, l, l);
            VText.MoveCaret (v.vtext, l)
          END
      ELSE
        v.defaultAction (cd);
        (* don't normalize if unknown chord, including just
           option itself: *)
        RETURN
      END;                      (* outer CASE *)
    EXCEPT
    | VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted =>
    END;
    Normalize (v)
  END OptionChord;

PROCEDURE Newline (v: T; <* UNUSED *> READONLY event: VBT.KeyRec)  =
  BEGIN
    Insert (v, "\n");
    Normalize (v)
  END Newline;

PROCEDURE NewlineAndIndent (v: T; READONLY event: VBT.KeyRec) =
  VAR
    index, left, leftMargin, rightMargin, rightEnd, right: INTEGER;
  BEGIN
    TRY
      index := VText.CaretIndex (v.vtext);
      MTextUnit.LineFacts (
        v.vtext.mtext, index, left, leftMargin, rightMargin,
        rightEnd, right);
      IF leftMargin = rightEnd AND index = rightEnd
           AND NOT IsReplaceMode (v) THEN
        (* We're at the end of an all-blank line. *)
        ReplaceInVText (v, left, left, "\n");
        Select (v, event.time, index + 1, index + 1)
      ELSIF leftMargin = rightMargin THEN (* line is all
                                             blanks *)
        Insert (v, "\n")
      ELSE                      (* Copy all the leading blanks
                                   onto the new line. *)
        Insert (v, "\n" & MText.GetText (
                            v.vtext.mtext, left, leftMargin))
      END;
    EXCEPT
    | VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted => RETURN
    END;
    Normalize (v)
  END NewlineAndIndent;
   
PROCEDURE WordAt (             mtext      : MText.T;
                               index      : INTEGER;
                  VAR (* out*) left, right: INTEGER  ) =
  (** A word is
      - a run of alphanumerics
      - a run of blanks
      - any other single character
     We find a word such that left <= index < right.
  **)
  VAR e: MTextUnit.Extent;
  BEGIN
    e := MTextUnit.RunExtent (mtext, index, Char.AlphaNumerics);
    IF e.inside THEN
      left := e.left;
      right := e.right;
      RETURN
    END;
    e := MTextUnit.RunExtent (mtext, index, Char.Spaces);
    IF e.inside THEN
      left := e.left;
      right := e.right;
      RETURN
    END;
    left := index;
    right := index + 1
  END WordAt;

PROCEDURE ColumnOf ( <* UNUSED *>t: MText.T; base, index: CARDINAL):
  CARDINAL =
  BEGIN
    RETURN index - base
  END ColumnOf;


PROCEDURE ToColumn (t: MText.T; base, column: CARDINAL):
  CARDINAL =
  BEGIN
    RETURN
      MIN (base + column, MTextUnit.LineInfo (t, base).rightEnd)
  END ToColumn;
  
TYPE
  FindWhere = {FindNext, FindPrev, FindFromTop};

PROCEDURE Find (v       : T;
                time    : VBT.TimeStamp;
                where   : FindWhere;
                selected: BOOLEAN        ) =
  VAR indexL, indexR: VText.Index;
  <* FATAL Range.Error *>
  BEGIN
    Selection (v, indexL, indexR, SelectionType.Primary);
    TRY
      IF selected THEN
        WITH newPattern = MText.GetText (
                            v.vtext.mtext, indexL, indexR) DO
          IF NOT Text.Empty (newPattern) THEN
            v.pattern := newPattern
          END
        END
      END;
      WITH len = Text.Length (v.pattern) DO
        IF len = 0 THEN RETURN END;
        CASE where OF
        | FindWhere.FindNext =>
            WITH rd = MTextRd.New (
                        v.vtext.mtext, start := indexR,
                        rangeStart := 0,
                        rangeEnd := LAST (CARDINAL)),
                 found = RdUtils.Find (rd, v.pattern) DO
              IF found >= 0 THEN
                Select (
                  v, time, found, found + len,
                  SelectionType.Primary, replaceMode := TRUE);
                v.lastCmdKind := CommandKind.OtherCommand
              END;
              Rd.Close (rd)
            END
        | FindWhere.FindFromTop =>
            WITH rd = MTextRd.New (v.vtext.mtext, start := 0,
                                   rangeStart := 0,
                                   rangeEnd := LAST (CARDINAL)),
                 found = RdUtils.Find (rd, v.pattern) DO
              IF found >= 0 THEN
                Select (
                  v, time, found, found + len,
                  SelectionType.Primary, replaceMode := TRUE);
                v.lastCmdKind := CommandKind.OtherCommand
              END;
              Rd.Close (rd)
            END
        | FindWhere.FindPrev =>
            (* Use a reverse-reader, and look for
               reverse(pattern).  Previous implementation
               started searching from the beginning and kept
               matching until we reached the current
               position. *)
            WITH rd = MTextRd.New (
                        v.vtext.mtext, start := indexL,
                        rangeStart := 0, rangeEnd := indexL,
                        reverse := TRUE),
                 found = RdUtils.Find (
                           rd, TextReverse (v.pattern)) DO
              IF found >= 0 THEN
                Select (v, time, indexL - found - len,
                        indexL - found, SelectionType.Primary,
                        replaceMode := TRUE);
                v.lastCmdKind := CommandKind.OtherCommand
              END;
              Rd.Close (rd)
            END
        END
      END
    EXCEPT
    | VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted =>
    END
  END Find;

PROCEDURE TextReverse (t: TEXT): TEXT =
  VAR
    len : CARDINAL          := Text.Length (t);
    buf : REF ARRAY OF CHAR;
    i, j: CARDINAL;
    c   : CHAR;
  BEGIN
    buf := NEW (REF ARRAY OF CHAR, len);
    Text.SetChars (buf^, t);
    i := 0;
    j := len - 1;
    WHILE i < j DO
      c := buf [i];
      buf [i] := buf [j];
      buf [j] := c;
      INC (i);
      DEC (j)
    END;
    RETURN Text.FromChars (buf^)
  END TextReverse;

PROCEDURE Repaint (v: T; READONLY rgn: Region.T) =
  BEGIN
    TRY
      LOCK v.mu DO
        VText.Bad (v.vtext, Region.BoundingBox (rgn));
        VText.Update (v.vtext)
      END
    EXCEPT
    | VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted =>
    END
  END Repaint;

PROCEDURE Rescreen (v: T; READONLY cd: VBT.RescreenRec) =
  BEGIN
    LOCK v.mu DO
      VText.Rescreen (v.vtext, cd);
      SetFontDimensions (v);
      VBT.NewShape (v)
    END
  END Rescreen;

PROCEDURE Redisplay (v: T) =
  BEGIN
    TRY
      LOCK v.mu DO VText.Update (v.vtext) END
    EXCEPT
    | VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted =>
    END
  END Redisplay;

PROCEDURE MarkAndUpdate (v: T)  =
  BEGIN
    VBT.Mark (v);
    ScrollUpdate (v)
  END MarkAndUpdate;

(***************************  Global Text  ***************************)

PROCEDURE Read (v: T; s: VBT.Selection; typecode: CARDINAL): VBT.Value
  RAISES {VBT.Error} =
  VAR sel: SelectionType;
  BEGIN
    IF v = NIL THEN
      RAISE VBT.Error (VBT.ErrorCode.Unreadable)
    ELSIF typecode # TYPECODE (TEXT) THEN
      RAISE VBT.Error (VBT.ErrorCode.WrongType)
    ELSE
      IF s = VBT.Target THEN
        sel := SelectionType.Primary
      ELSIF s = VBT.Source THEN
        sel := SelectionType.Secondary
      ELSE
        RAISE VBT.Error (VBT.ErrorCode.Unreadable)
      END;
      LOCK v.mu DO
        WITH intv = v.selection [sel].interval DO
          RETURN VBT.FromRef (MText.GetText (v.vtext.mtext, intv.l, intv.r))
        END
      END
    END
  END Read;

PROCEDURE Write (             v       : T;
                              s       : VBT.Selection;
                              value   : VBT.Value;
                 <* UNUSED *> typecode: CARDINAL       ) RAISES {VBT.Error} =
  <* LL.sup <= VBT.mu *>
  VAR sel: SelectionType;
  BEGIN
    IF v = NIL THEN RAISE VBT.Error (VBT.ErrorCode.Unreadable) END;
    IF s = VBT.Target THEN
      sel := SelectionType.Primary
    ELSIF s = VBT.Source THEN
      sel := SelectionType.Secondary
    ELSE
      RAISE VBT.Error (VBT.ErrorCode.Unwritable)
    END;
    TYPECASE value.toRef () OF
    | TEXT (text) =>
        LOCK v.mu DO
          IF v.readOnly OR v.selection [sel].interval.l < v.typeinStart THEN
            RAISE VBT.Error (VBT.ErrorCode.Unwritable)
          END
        END;
        IF sel = SelectionType.Primary THEN
          Insert (v, text)
        ELSE
          Replace1 (v, text, sel)
        END
    ELSE
      RAISE VBT.Error (VBT.ErrorCode.WrongType)
    END
  END Write;

(******************************  Scrolling  ******************************)

REVEAL
  Scroller = ScrollerVBT.T BRANDED OBJECT
               textport: T
             OVERRIDES
               scroll     := Scroll;
               autoScroll := AutoScroll;
               thumb      := Thumb
             END;

PROCEDURE ScrollUpdate (v: T) =
  BEGIN
    TRY
      IF v.scrollBar = NIL THEN RETURN END;
      WITH start = VText.StartIndex (v.vtext, 0) DO
        v.scrollUpdate (
          start, start + VText.CharsInRegion (v.vtext, 0),
          MText.Length (v.vtext.mtext))
      END
    EXCEPT
    | VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted =>
    END
  END ScrollUpdate;

CONST
  NearEdge = 13;
    (* Thumbing closer than this to top/bottom of scroll bar is treated as
       being exactly at the top/bottom. *)

PROCEDURE Scroll (                      s     : Scroller;
                  <* UNUSED *> READONLY cd    : VBT.MouseRec;
                                        part  : INTEGER;
                  <* UNUSED *>          height: INTEGER;
                  towardsEOF: BOOLEAN) =
  VAR distance: INTEGER;
  BEGIN
    TRY
      WITH v     = s.textport,
           vtext = v.vtext     DO
        distance := MAX (1, VText.WhichLine (vtext, 0, part));
        IF NOT towardsEOF THEN distance := -distance END;
        VText.Scroll (vtext, 0, distance);
        VText.Update (vtext);
        ScrollUpdate (v)
      END
    EXCEPT
    | VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted =>
    END
  END Scroll;

PROCEDURE AutoScroll (                      s : Scroller;
                      <* UNUSED *> READONLY cd: VBT.MouseRec;
                      linesToScroll: CARDINAL;
                      towardsEOF   : BOOLEAN   ) =
  VAR distance: INTEGER := linesToScroll;
  BEGIN
    TRY
      IF NOT towardsEOF THEN distance := -distance END;
      WITH v     = s.textport,
           vtext = v.vtext     DO
        VText.Scroll (vtext, 0, distance);
        VText.Update (vtext);
        ScrollUpdate (v)
      END
    EXCEPT
    | VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted =>
    END
  END AutoScroll;

PROCEDURE Thumb (                      s     : Scroller;
                 <* UNUSED *> READONLY cd    : VBT.MouseRec;
                                       part  : INTEGER;
                                       height: INTEGER       ) =
  VAR position: INTEGER;
  BEGIN
    TRY
      WITH v      = s.textport,
           vtext  = v.vtext,
           length = MText.Length (vtext.mtext) DO
        IF length = 0 OR part < NearEdge THEN
          position := 0
        ELSIF part + NearEdge > height THEN
          position := length - 1
        ELSE
          position := (part * length) DIV height
        END;
        VText.SetStart (vtext, 0, position);
        VText.Update (vtext);
        ScrollUpdate (v)
      END
    EXCEPT
    | VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted =>
    END
  END Thumb;

(****************************   methods  ****************************)

PROCEDURE ReturnAction (v: T; READONLY event: VBT.KeyRec) =
  BEGIN
    IF v.singleLine THEN
      v.defaultAction (event)
    ELSE
      NewlineAndIndent (v, event)
    END
  END ReturnAction;

PROCEDURE Insert4spaces (v: T; <* UNUSED *> READONLY event: VBT.KeyRec) =
  BEGIN
    Insert (v, "    ");
  END Insert4spaces;
  
PROCEDURE IgnoreKey (<* UNUSED *> v: T;
                     <* UNUSED *> READONLY event: VBT.KeyRec) =
  BEGIN
  END IgnoreKey;

(*************************  Miscellany  ************************)

PROCEDURE UpdateScrollbar (v: T; viewStart, viewEnd, length: INTEGER) =
  BEGIN
    IF v.scrollBar = NIL THEN RETURN END;
    ScrollerVBT.Update (v.scrollBar, viewStart, viewEnd, length)
  END UpdateScrollbar;

PROCEDURE Normalize (v: T; to: INTEGER := -1) =
  VAR point: CARDINAL;
  BEGIN
    TRY
      IF to < 0 THEN
        point := VText.CaretIndex (v.vtext)
      ELSE
        point := MIN (to, MText.Length (v.vtext.mtext))
      END;
      IF NOT VText.InRegion (v.vtext, 0, point) THEN
        VText.SetStart (v.vtext, 0, point, v.linesShown DIV 2)
      END;
      MarkAndUpdate (v)
    EXCEPT
    | VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted =>
    END
  END Normalize;

PROCEDURE IgnoreFocus (<* UNUSED *> v:     T;
                       <* UNUSED *> gaining: BOOLEAN;
                       <* UNUSED *> time   : VBT.TimeStamp) =
  BEGIN
  END IgnoreFocus;

PROCEDURE IgnoreModification (<* UNUSED *> v: T)  =
  BEGIN
  END IgnoreModification;

PROCEDURE NoOpFilter (<* UNUSED *> v: T;
                      <* UNUSED *> VAR (* inOut*) event: VBT.KeyRec) =
  BEGIN
  END NoOpFilter;

(*************************  Module Initialization  ************************)

BEGIN
  selectionModes [VBT.Modifier.MouseL, 0] :=
    VText.SelectionMode.CharSelection;
  selectionModes [VBT.Modifier.MouseL, 1] :=
    VText.SelectionMode.CharSelection;
  selectionModes [VBT.Modifier.MouseL, 2] :=
    VText.SelectionMode.LineSelection;
  selectionModes [VBT.Modifier.MouseL, 3] :=
    VText.SelectionMode.LineSelection;
  selectionModes [VBT.Modifier.MouseL, 4] :=
    VText.SelectionMode.AllSelection;
  selectionModes [VBT.Modifier.MouseM, 0] :=
    VText.SelectionMode.WordSelection;
  selectionModes [VBT.Modifier.MouseM, 1] :=
    VText.SelectionMode.WordSelection;
  selectionModes [VBT.Modifier.MouseM, 2] :=
    VText.SelectionMode.ParagraphSelection;
  selectionModes [VBT.Modifier.MouseM, 3] :=
    VText.SelectionMode.ParagraphSelection;
  selectionModes [VBT.Modifier.MouseM, 4] :=
    VText.SelectionMode.AllSelection;
    
  standardStyle [SelectionType.Primary] :=
    VText.IntervalStyle.UnderlineStyle;
  standardStyle [SelectionType.Secondary] :=
    VText.IntervalStyle.GrayUnderlineStyle;
  specialStyle [SelectionType.Primary] :=
    VText.IntervalStyle.InverseStyle;
  specialStyle [SelectionType.Secondary] :=
    VText.IntervalStyle.GrayUnderlineStyle;
END TextPort.



