(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)
(*                                                                           *)
(* Last modified on Mon Aug 10  0:30:52 PDT 1992 by meehan     *)
(*      modified on Tue Jun 16 13:08:06 PDT 1992 by muller     *)
(*      modified on Mon Jun 15 22:39:17 1992 by mhb        *)
(*      modified on Fri Mar 20 22:37:08 1992 by steveg         *)
(*      modified on Fri Feb 8 15:32:36 PST 1991 by brooks      *)
<* PRAGMA LL *>

(* A "TypescriptVBT" is a subtype of "TextEditVBT", with
   additional features to make it serve as a ``glass teletype''
   with a memory.  To the client, it provides a reader to get
   input from the user, and a writer to print output.  The reader
   is intermittent and unseekable; the writer is buffered and
   unseekable.  The writer is paired with the reader, such that
   the writer is flushed whenever a seek blocks on the reader.
   The writer is not flushed at every newline.

   All input to the typescript, once it has been read, and all
   output, become part of the ``history'' of the typescript,
   and is not modifiable; it remains until the user deletes it
   by calling "ClearHistory".  Selections that lie fully or
   partially within the history region are never ``replace-mode''
   selections (see Section~\ref{ReplaceMode},
   page~\pageref{ReplaceMode}).  Any attempt to type or insert
   text in the history region becomes an insertion at the end
   of the typescript instead.

   A typescript's textport, "v.port", must be of type
   "TypescriptVBT.Port" (which is a subtype of the type "TextPort.T").
   The type "TypescriptVBT.Port" overrides "TextPort.T.returnAction":
   it makes the text of the current type-in region available to the
   reader and no longer editable.

   The client may, in general, use all the "TextEditVBT"
   procedures on a "TypescriptVBT", but should not call
   "TextPort.SetReadOnly(v.port, FALSE)".

*)

INTERFACE TypescriptVBT;

IMPORT Font, PaintOp, Rd, Shadow, TextEditVBT, TextPort,
       Thread, VBT, Wr;

TYPE
  T <: Public;
  Public = TextEditVBT.T OBJECT
           METHODS
             <* LL.sup <= VBT.mu or 0 if v is disconnected *>
             init (shadow: Shadow.T := NIL;
                   style            := Shadow.Style.Flat): T;
             interrupt       (time: VBT.TimeStamp);
             handleInterrupt (time: VBT.TimeStamp);
             setThread       (thread: Thread.T := NIL);
             terminate       ();
           END;

(* The call "v.init(...)" initializes "v" as an empty typescript.
   The parameter "shadow" defaults to "Shadow.None".

   If "v.port" is "NIL" when "v.init(...)" is called, then a default 
   "TextPort" will be created and assigned to "v.port".  It is a 
   checked runtime error if "v.port" is not of type "TypescriptVBT.Port", 
   which is a subtype of "TextPort.T". 

   The call "v.interrupt(time)" simulates an interrupt by
   flushing pending type-in, writing the characters "^C", and then
   calling "v.handleInterrupt(time)".

   The call "v.handleInterrupt(time)" alerts the thread that is
   currently waiting on input from "v"'s reader, or if none, the
   thread that most recently waited.  If no thread has ever read
   from "v"'s reader, this call has no effect.

   The call "v.setThread(thread)" sets the thread that is to be
   alerted on subsequent calls to "v.interrupt(time)".  (So does
   reading.) "thread" defaults to "Thread.Self()".

   The call "v.terminate()" causes "v" to report end of file on
   its input stream.  This is appropriate when "v" is being
   thrown away.  The call causes "v" to become unresponsive to
   further user input, but it continues to display output written
   to its output stream. *)

TYPE
  Points = REAL;
  Port <: PublicPort;
  PublicPort =
    TextPort.T OBJECT
    METHODS
      init (hMargin, vMargin: Points := 1.5;
            font                     := Font.BuiltIn;
            colorScheme: PaintOp.ColorScheme := NIL;
            wrap                             := TRUE;
            readOnly                         := FALSE;
            turnMargin : Points              := 2.0    ): Port;
               END;

  Reader <: PublicReader;
  PublicReader = Rd.T OBJECT METHODS typescript (): T END;

  Writer <: PublicWriter;
  PublicWriter = Wr.T OBJECT METHODS typescript (): T END;

PROCEDURE GetRd (v: T): Reader;
(* Get the input stream for "v".  Always returns the same reader
   for a given "v". *)

PROCEDURE GetWr (v: T): Writer;
(* Get the output stream for "v".  Always returns the same writer
   for a given "v". *)

PROCEDURE GetHistory (v: T): TEXT; <* LL <= VBT.mu *>
(* Return the ``history'' text of "v". *)

PROCEDURE ClearHistory (v: T);  <*  LL <= VBT.mu *>
(* Clear the ``history'' text of "v". *)

END TypescriptVBT.


